﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

//#include <nw/g3d.h>
//#include <nw/ut.h>
//#include <nw/dev.h>

#include <nn/g3d.h>
#include <nn/vfx/vfx_Heap.h>

namespace nw      {
namespace eftdemo {


class SimpleShader;

//------------------------------------------------------------------------------
// ※ グラフィックスに関する処理は、g3dのデモを参考にしてください。
//------------------------------------------------------------------------------
enum MemoryPoolType
{
    MemoryPoolType_CpuInvisible,
    MemoryPoolType_CpuCached,
    MemoryPoolType_CpuUncached
};

//------------------------------------------------------------------------------
//     シェーダを取得します。
//------------------------------------------------------------------------------
nn::gfx::Shader* getSimpleShader() NN_NOEXCEPT;

//------------------------------------------------------------------------------
//     シェーダを取得します。
//------------------------------------------------------------------------------
nn::gfx::Shader* getSkinningShader() NN_NOEXCEPT;

//------------------------------------------------------------------------------
//　　　シェーダリソースを初期化処理を行います。
//------------------------------------------------------------------------------
void InitializeG3dShaderResource( nn::gfx::Device* pDevice, nn::vfx::Heap* pHeap ) NN_NOEXCEPT;

//------------------------------------------------------------------------------
//　　　シェーダリソースの終了処理を行います。
//------------------------------------------------------------------------------
//void FinalizeG3dShaderResource( nw::ut::IAllocator* allocator );
void FinalizeG3dShaderResource( nn::gfx::Device* pDevice, nn::vfx::Heap* pHeap ) NN_NOEXCEPT;


#if 0
//------------------------------------------------------------------------------
//　　　シェーダリソースを取得します。
//------------------------------------------------------------------------------
void GetG3dShaderResource( nw::g3d::GfxShader::SetupArg* setupArg ) NN_NOEXCEPT;



//------------------------------------------------------------------------------
//　　　モデルの生成を行います。
//------------------------------------------------------------------------------
//nn::g3d::ModelObj* BuildModel( nw::ut::IAllocator* allocator, nw::g3d::res::ResModel* rMdl );
nn::g3d::ModelObj* BuildModel( nn::vfx::Heap* pHeap, nn::g3d::ResModel* rMdl ) NN_NOEXCEPT;


//------------------------------------------------------------------------------
//　　　モデルの破棄を行います。
//------------------------------------------------------------------------------
//void DestroyModel( nw::ut::IAllocator* allocator, nw::g3d::ModelObj* mdl );
void DestroyModel( nn::vfx::Heap* pHeap, nn::g3d::ModelObj* mdl ) NN_NOEXCEPT;


//------------------------------------------------------------------------------
//      スケルタルアニメの生成を行います。
//------------------------------------------------------------------------------
//nn::g3d::SkeletalAnimObj* BuildSklAnim( nw::ut::IAllocator* allocator, nw::g3d::res::ResSkeletalAnim* rSklAnim, int maxSklCount );
nn::g3d::SkeletalAnimObj* BuildSklAnim( nn::vfx::Heap* pHeap, nn::g3d::ResSkeletalAnim* rSklAnim, int maxSklCount ) NN_NOEXCEPT;


//------------------------------------------------------------------------------
//　    パラメータアニメの生成を行います。
//------------------------------------------------------------------------------
nn::g3d::ShaderParamAnimObj* BuildParamAnim( nw::ut::IAllocator* allocator, nw::g3d::res::ResShaderParamAnim* rParamAnim, int maxMatCount ) NN_NOEXCEPT;


//------------------------------------------------------------------------------
//　    テクスチャアニメの生成を行います。
//------------------------------------------------------------------------------
nn::g3d::TexPatternAnimObj* BuildTexAnim( nw::ut::IAllocator* allocator, nw::g3d::res::ResTexPatternAnim* rTexAnim, int maxMatCount ) NN_NOEXCEPT;


//------------------------------------------------------------------------------
//　    ビジビリティアニメの生成を行います。
//------------------------------------------------------------------------------
nn::g3d::VisibilityAnimObj* BuildVisAnim( nw::ut::IAllocator* allocator, nn::g3d::res::ResVisibilityAnim* rVisAnim, int maxCount ) NN_NOEXCEPT;


//------------------------------------------------------------------------------
//　    シェイプアニメの生成を行います。
//------------------------------------------------------------------------------
nn::g3d::ShapeAnimObj* BuildShapeAnim( nw::ut::IAllocator* allocator, nw::g3d::res::ResShapeAnim* rShapeAnim, int maxShapeCount ) NN_NOEXCEPT;


//------------------------------------------------------------------------------
//　    アニメの破棄を行います。
//------------------------------------------------------------------------------
void DestroyModelAnim( nw::ut::IAllocator* allocator, nn::g3d::ModelAnimObj* anim ) NN_NOEXCEPT;


//------------------------------------------------------------------------------
//　    フェッチシェーダの生成を行います。
//------------------------------------------------------------------------------
int BuildFetchShader( nw::ut::IAllocator* allocator, const nw::g3d::ModelObj* mdl, nw::g3d::GfxFetchShader** pFetchShaderArray ) NN_NOEXCEPT;


//------------------------------------------------------------------------------
//　    フェッチシェーダの破棄を行います。
//------------------------------------------------------------------------------
void DestroyFetchShader( nw::ut::IAllocator* allocator, nw::g3d::GfxFetchShader** pFetchShaderArray, int numShader ) NN_NOEXCEPT;


//------------------------------------------------------------------------------
//　    モデルにシェーダを紐づけします。
//------------------------------------------------------------------------------
int BindShader( nw::ut::IAllocator* allocator, const SimpleShader& shader, nw::g3d::ModelObj* model, const char** blockId ) NN_NOEXCEPT;


//------------------------------------------------------------------------------
//　    シェーダの破棄を行います。
//------------------------------------------------------------------------------
void DestroyShader( nw::ut::IAllocator* allocator, SimpleShader& shader ) NN_NOEXCEPT;


//------------------------------------------------------------------------------
//　　　シンプルシェーダクラス
//------------------------------------------------------------------------------
class SimpleShader : public nw::g3d::GfxShader
{
public:
    enum UniformBlockType
    {
        MATERIAL_BLOCK,
        SHAPE_BLOCK,
        SKELETON_BLOCK,
        VIEW_BLOCK,
        USER_SHAPE_BLOCK
    };

    struct ViewBlock
    {
//        nw::g3d::Mtx44 projMtx;
//        nw::g3d::Mtx34 cameraMtx;
//        nw::g3d::Vec4  lightPos;
        nn::util::Matrix4x4f projMtx;
        nn::util::Matrix4x4f cameraMtx;
        nn::util::Vector4f   lightPos;
    };

    enum
    {
        // シェーダで使用する各項目の最大数を設定します。
        // 必要数を事前に決定しておくことでロケーションを記録するメモリを節約することも可能です。
        NUM_SHADER_ATTRIB   = 8,
        NUM_SHADER_SAMPLER  = 8,
        NUM_SHADER_BLOCK    = 5,

        // モデルに使用するテクスチャのアライメントの最大値を設定します。
        // リソースファイルごとのアライメントを事前に取得しておくことで、
        // 必要なメモリのアライメントを最小限に抑えてメモリを節約することも可能です。
        TEXTURE_ALIGNMENT   = 8192
    };


    enum ViewId
    {
        VIEW_ID_LIGHT      = 0,
        VIEW_ID_CAMERA     = 1,
        VIEW_ID_CUBEMAP_PX = 2,
        VIEW_ID_CUBEMAP_NX = 3,
        VIEW_ID_CUBEMAP_PY = 4,
        VIEW_ID_CUBEMAP_NY = 5,
        VIEW_ID_CUBEMAP_PZ = 6,
        VIEW_ID_CUBEMAP_NZ = 7,
        VIEW_ID_MAX        = 8,
    };

    struct Path
    {
        const char* gsh;
        const char* vertexShader;
        const char* geometryShader;
        const char* fragmentShader;
    };

    struct Location
    {
    //    s8 vertex;
    //    s8 geometry;
    //    s8 fragment;
    //    u8 reserved;
        int8_t vertex;
        int8_t geometry;
        int8_t fragment;
        uint8_t reserved;
    };


    //---------------------------------------------------------------------------
    //! @brief      サンプラ―ロケーションの初期化を行います。
    //---------------------------------------------------------------------------
    void InitSamplerLocations(int numSampler, const char* name[]) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief      ブロックロケーションの初期化を行います。
    //---------------------------------------------------------------------------
    void InitBlockLocations(int numBlock, const char* name[]) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief      ビューブロックの初期化を行います。
    //---------------------------------------------------------------------------
    void InitViewBlock( nw::ut::IAllocator* allocator ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief      ビューブロックの終了処理を行います。
    //---------------------------------------------------------------------------
    void FinalizeViewBlock( nw::ut::IAllocator* allocator ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief
    //---------------------------------------------------------------------------
    int GetSamplerCount() const NN_NOEXCEPT { return mNumSampler; }

    //---------------------------------------------------------------------------
    //! @brief
    //---------------------------------------------------------------------------
    int GetBlockCount() const NN_NOEXCEPT { return mNumBlock; }

    //---------------------------------------------------------------------------
    //! @brief
    //---------------------------------------------------------------------------
    nw::g3d::GfxBuffer& GetViewBlock( u32 viewId ) NN_NOEXCEPT { return mViewBlock[viewId]; };

    //---------------------------------------------------------------------------
    //! @brief
    //---------------------------------------------------------------------------
    const Location& GetSamplerLocation(int index) const NN_NOEXCEPT
    {
        //NW_G3D_ASSERT_INDEX_BOUNDS(index, mNumSampler);
        return mSamplerLocation[index];
    }

    //---------------------------------------------------------------------------
    //! @brief
    //---------------------------------------------------------------------------
    const Location& GetBlockLocation(int index) const NN_NOEXCEPT
    {
        //NW_G3D_ASSERT_INDEX_BOUNDS(index, mNumBlock);
        return mBlockLocation[index];
    }

private:
//    u8                  mNumSampler;
//    u8                  mNumBlock;
//    u16                 reserved;
    uint8_t             mNumSampler;
    uint8_t             mNumBlock;
    uint16_t            reserved;

    Location            mSamplerLocation[NUM_SHADER_SAMPLER];
    Location            mBlockLocation[NUM_SHADER_BLOCK];
    nw::g3d::GfxBuffer  mViewBlock[VIEW_ID_MAX];
};
#endif

// シェーダで使用する各項目の最大数を設定します。
// 必要数を事前に決定しておくことでロケーションを記録するメモリを節約することも可能です。
enum
{
    ShaderSamplerCount = 8,
};

//------------------------------------------------------------------------------
//　　　サンプラー対応情報
//------------------------------------------------------------------------------
struct SamplerTable
{
    int index[ShaderSamplerCount];
};

//------------------------------------------------------------------------------
//　　　ビューの Uniform Block
//------------------------------------------------------------------------------
struct ViewBlock
{
    nn::util::FloatColumnMajor4x4 projMtx;
    nn::util::FloatColumnMajor4x3 cameraMtx;
};

//------------------------------------------------------------------------------
//　　　シンプルモデルクラス
//------------------------------------------------------------------------------
class SimpleModel
{
public:
    enum
    {
        MODEL_ANIM_MAX = 4
    };

    //---------------------------------------------------------------------------
    //! @brief     コンストラクタ
    //---------------------------------------------------------------------------
    SimpleModel() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief     デストラクタ
    //---------------------------------------------------------------------------
    ~SimpleModel() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief     初期化処理を行います。
    //---------------------------------------------------------------------------
    bool Initialize( void* modelResource, size_t binarySize, nn::vfx::Heap* pHeap, nn::gfx::Device* pDevice) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief     終了処理を行います。
    //---------------------------------------------------------------------------
    void Finalize( nn::gfx::Device* pDevice, nn::vfx::Heap* pHeap ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief     計算処理を行います。
    //---------------------------------------------------------------------------
    void Calc() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief     ブロックの更新処理を行います。
    //---------------------------------------------------------------------------
    void CalcBlock( const nn::util::Matrix4x4fType* projMtx, const nn::util::Matrix4x3fType* viewMtx, const nn::util::Vector3fType* lightPos, const float animaSpeed = 1.0f ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief     描画処理を行います。
    //---------------------------------------------------------------------------
    void Draw( nn::gfx::CommandBuffer* pCommandBuffer, nn::vfx::Heap* pHeap ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief     描画マトリクスを設定します。
    //---------------------------------------------------------------------------
    void SetDrawMatrix( const nn::util::Matrix4x3fType& drawMatrix ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief     アニメーションフレームを設定します。
    //---------------------------------------------------------------------------
    void SetAnimationFrame( float frame ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief     アニメーションインデックスを設定します。
    //---------------------------------------------------------------------------
    void SetAnimationIndex( uint32_t index ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        モデル名を取得します。
    //---------------------------------------------------------------------------
    const char* GetModelName() const NN_NOEXCEPT
    {
        return mModelObj->GetResource()->GetName();
    }

    //---------------------------------------------------------------------------
    //! @brief        ボーン数を取得します。
    //---------------------------------------------------------------------------
    int32_t GetBoneNum() const NN_NOEXCEPT
    {
        nn::g3d::SkeletonObj* skeleton = mModelObj->GetSkeleton();
        if ( !skeleton ) return 0;
        return skeleton->GetBoneCount();
    }

    //---------------------------------------------------------------------------
    //! @brief        ボーン名を取得します。
    //---------------------------------------------------------------------------
    const char* GetBoneName( int index ) const NN_NOEXCEPT
    {
        nn::g3d::SkeletonObj* skeleton = mModelObj->GetSkeleton();
        return skeleton->GetBoneName( index );
    }

    //---------------------------------------------------------------------------
    //! @brief        ボーンのワールドマトリックスを取得します。
    //---------------------------------------------------------------------------
    const nn::util::Matrix4x3fType* GetBoneWorldMatrix(int index) NN_NOEXCEPT
    {
        nn::g3d::SkeletonObj* skeleton = mModelObj->GetSkeleton();
        int boneCount = skeleton->GetBoneCount();

        if (boneCount <= index)
            return NULL;

        return &skeleton->GetWorldMtxArray()[index];
    }
    //---------------------------------------------------------------------------
    //! @brief        ディスクリプタスロットを更新します。
    //---------------------------------------------------------------------------
    void UpdateDescriptorSlot() NN_NOEXCEPT;

private:
    //---------------------------------------------------------------------------
    //! @brief        メモリプールから領域を確保します
    //---------------------------------------------------------------------------
    ptrdiff_t AllocateFromMemoryPool(size_t size, size_t alignment) NN_NOEXCEPT;
private:
    nn::g3d::ResFile*           mResFile;
    nn::g3d::ModelObj*          mModelObj;
    nn::g3d::ModelAnimObj*      mModelAnimObjArray[MODEL_ANIM_MAX];
    nn::gfx::BlendState         mBlendState;
    nn::gfx::DepthStencilState  mDepthStencilState;
    nn::gfx::VertexState*       mpVertexStateArray;
    SamplerTable*               mpSamplerTableArray;
    nn::g3d::ResSkeletalAnim*   mpResSkeletalAnim;
    nn::g3d::SkeletalAnimObj    mSkeletalAnimObj;
    nn::g3d::ResBoneVisibilityAnim*   mpResBoneVisibilityAnim;
    nn::g3d::BoneVisibilityAnimObj    mBoneVisibilityAnimObj;

    nn::gfx::Buffer             m_ViewBlock[2];

    void*                       m_pVertexStateBuffer;

    void*   m_pMemoryPoolPtr;
    nn::gfx::MemoryPool m_MemoryPool;
    ptrdiff_t   m_MemoryPoolOffset;

    nn::util::Matrix4x3fType    mDrawMatrix;

    float                       mAnimFrame;
    uint32_t                    mAnimIndex;
    int                         m_CurrentFace;
};


} // namespace eftdemo
} // namespace nw
