﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Runtime.InteropServices;
using System.Text;
using EffectMaker.BusinessLogic.IO;

namespace EffectMaker.UIControls.EffectBrowser.Utilities
{
    using System;
    using System.Collections.Concurrent;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;

    using EffectMaker.UIControls.EffectBrowser.Data;

    using FileInfo = System.IO.FileInfo;

    /// <summary>
    /// The path utility.
    /// </summary>
    public static class PathUtility
    {
        /// <summary>
        /// アクセス不可パス
        /// </summary>
        private static readonly ConcurrentDictionary<string, bool> InaccessiblePaths =
            new ConcurrentDictionary<string, bool>();

        /// <summary>
        /// 参照されるアセットのディレクトリパスを作る
        /// </summary>
        /// <param name="baseDirPath">
        /// The base dir path.
        /// </param>
        /// <param name="isBaseSearchPathFiltering">
        /// The is base search path filtering.
        /// </param>
        /// <param name="kind">
        /// The kind.
        /// </param>
        /// <returns>
        /// The <see cref="IEnumerable"/>.
        /// </returns>
        public static IEnumerable<string> MakeChildFileDirPaths(
            string baseDirPath,
            bool isBaseSearchPathFiltering,
            FileKindType kind)
        {
            System.Diagnostics.Debug.Assert(
                (kind == FileKindType.TextureFile) ||
                (kind == FileKindType.PrimitiveFile) ||
                (kind == FileKindType.CombinerFile),
                "アセットファイルタイプではない");

            var folder = (kind == FileKindType.TextureFile) ? IOConstants.TextureFolderName :
                (kind == FileKindType.PrimitiveFile) ? IOConstants.PrimitiveFolderName :
                IOConstants.CombinerShaderFolderName;

            yield return baseDirPath;
            yield return Path.Combine(baseDirPath, folder);

            if (isBaseSearchPathFiltering == false)
            {
                var path = baseDirPath;

                while (IsRootPath(path) == false)
                {
                    path = NormalizePathString(Path.Combine(path, ".."));

                    if (IsRootPath(path) && path.Length == 2)
                    {
                        path += @"\";
                    }

                    var childFolder = Path.Combine(path, folder);

                    if (DirectoryExists(childFolder))
                    {
                        yield return childFolder;
                    }
                }
            }
        }

        /// <summary>
        /// 参照されるアセットの検索対象パスを作る
        /// </summary>
        /// <param name="baseDirPath">
        /// The base dir path.
        /// </param>
        /// <param name="isBaseSearchPathFiltering">
        /// The is base search path filtering.
        /// </param>
        /// <param name="kind">
        /// The kind.
        /// </param>
        /// <returns>
        /// The <see cref="IEnumerable"/>.
        /// </returns>
        public static IEnumerable<string> MakeReferenceSearchDirPaths(
            string baseDirPath,
            bool isBaseSearchPathFiltering,
            FileKindType kind)
        {
            // 基点パス以下の全ディレクトリを列挙します。
            var downPaths = PathUtility.EnumerateDirectories(
                baseDirPath,
                "*",
                SearchOption.AllDirectories);

            // 基点パスとそれより上階層のアセットディレクトリを列挙します。
            var upperPaths = PathUtility.MakeChildFileDirPaths(
                baseDirPath,
                isBaseSearchPathFiltering,
                kind);

            // 2つを結合し、重複を省いてサーチ対象とします。
            return downPaths.Concat(upperPaths).Distinct();
        }

#if false

    // 存在する参照ファイルパスを作る
        public static string MakeExistsReferenceFilePath(string filename, string basePath, bool isBaseSearchPathFiltering, FileKindType kind)
        {
            return
                MakeChildFileDirPaths(basePath, isBaseSearchPathFiltering, kind).
                Select(dirPath => Path.Combine(dirPath, filename)).
                FirstOrDefault(FileExists);
        }
#else

        /// <summary>
        /// 存在する参照ファイルパスを作る
        /// </summary>
        /// <param name="filename">
        /// The filename.
        /// </param>
        /// <param name="childFileDirPaths">
        /// The child file dir paths.
        /// </param>
        /// <returns>
        /// The <see cref="string"/>.
        /// </returns>
        public static string MakeExistsReferenceFilePath(string filename, IEnumerable<string> childFileDirPaths)
        {
            return childFileDirPaths.Select(dirPath => Path.Combine(dirPath, filename)).FirstOrDefault(FileExists);
        }

#endif

        /// <summary>
        /// The is root path.
        /// </summary>
        /// <param name="path">
        /// The path.
        /// </param>
        /// <returns>
        /// The <see cref="bool"/>.
        /// </returns>
        public static bool IsRootPath(string path)
        {
            return path == Path.GetPathRoot(path);
        }

        /// <summary>
        /// The make file mask pattern.
        /// </summary>
        /// <param name="kind">
        /// The kind.
        /// </param>
        /// <returns>
        /// The <see cref="string"/>.
        /// </returns>
        public static string MakeFileMaskPattern(FileKindType kind)
        {
            return "*" + Constants.FileKindInfos[kind].ExtName;
        }

        /// <summary>
        /// The normalize path string.
        /// </summary>
        /// <param name="path">
        /// The path.
        /// </param>
        /// <returns>
        /// The <see cref="string"/>.
        /// </returns>
        public static string NormalizePathString(string path)
        {
            var normalizedPath = Path.GetFullPath(path);

            if (Directory.Exists(normalizedPath))
            {
                if (normalizedPath.Last() == '\\')
                {
                    normalizedPath = normalizedPath.Substring(0, normalizedPath.Length - 1);
                }
            }

            return normalizedPath;
        }

        /// <summary>
        /// The is valid filepath string.
        /// </summary>
        /// <param name="src">
        /// The src.
        /// </param>
        /// <returns>
        /// The <see cref="bool"/>.
        /// </returns>
        public static bool IsValidFilepathString(string src)
        {
            if (src == string.Empty)
            {
                return false;
            }

            return src.IndexOfAny(Path.GetInvalidPathChars())==-1;
        }

        /// <summary>
        /// The setup unused directory.
        /// </summary>
        public static void SetupUnusedDirectory()
        {
            if (DirectoryExists(Constants.TexturesUnusedDirectory) == false)
            {
                Directory.CreateDirectory(Constants.TexturesUnusedDirectory);
            }

            if (DirectoryExists(Constants.PrimitivesUnusedDirectory) == false)
            {
                Directory.CreateDirectory(Constants.PrimitivesUnusedDirectory);
            }

            if (BusinessLogic.Options.OptionStore.ProjectConfig.IsEftCombinerEditorEnabled)
            {
                if (DirectoryExists(Constants.CombinersUnusedDirectory) == false)
                {
                    Directory.CreateDirectory(Constants.CombinersUnusedDirectory);
                }
            }
        }

        /// <summary>
        /// The directory exists.
        /// </summary>
        /// <param name="src">
        /// The src.
        /// </param>
        /// <returns>
        /// The <see cref="bool"/>.
        /// </returns>
        public static bool DirectoryExists(string src)
        {
            if (IsMatchToNetworkDrive(src))
            {
                return EnumerateDirectories(src).Any();
            }
            else
            {
                return IsValidFilepathString(src) && Directory.Exists(src);
            }
        }

        /// <summary>
        /// The file exists.
        /// </summary>
        /// <param name="src">
        /// The src.
        /// </param>
        /// <returns>
        /// The <see cref="bool"/>.
        /// </returns>
        public static bool FileExists(string src)
        {
            return IsValidFilepathString(src) && File.Exists(src);
        }

        /// <summary>
        /// The is local drive path.
        /// </summary>
        /// <param name="src">
        /// The src.
        /// </param>
        /// <returns>
        /// The <see cref="bool"/>.
        /// </returns>
        public static bool IsLocalDrivePath(string src)
        {
            if (src.Length < 2)
            {
                return false;
            }

            return src[1] == ':';
        }

        /// <summary>
        /// The is networkpath.
        /// </summary>
        /// <param name="src">
        /// The src.
        /// </param>
        /// <returns>
        /// The <see cref="bool"/>.
        /// </returns>
        public static bool IsNetworkPath(string src)
        {
            if (src.Length < 3)
            {
                return false;
            }

            return src[0] == '\\' && src[1] == '\\';
        }

        /// <summary>
        /// ネットワークドライブをさすパスかチェックする
        /// </summary>
        /// <param name="src">ファイルパス</param>
        /// <returns>ネットワークドライブならtrue, そうでないならfalse</returns>
        public static bool IsMatchToNetworkDrive(string src)
        {
            return System.Text.RegularExpressions.Regex.IsMatch(src, @"^\\\\[^\\]+\\?$");
        }

        /// <summary>
        /// ネットワーク上のActive Directoryをさすパスかチェックする
        /// </summary>
        /// <param name="src">ファイルパス</param>
        /// <returns>Active Directoryならtrue, そうでないならfalse</returns>
        public static bool IsMatchToNetworkActiveDirectory(string src)
        {
            return System.Text.RegularExpressions.Regex.IsMatch(src, @"^\\\\[^\\]+\\[^\\]+\\?$");
        }

        /// <summary>
        /// The enumerate directories.
        /// </summary>
        /// <param name="dirPath">
        /// The dir path.
        /// </param>
        /// <returns>
        /// The <see cref="IEnumerable"/>.
        /// </returns>
        public static IEnumerable<string> EnumerateDirectories(string dirPath)
        {
            if (IsAccessibleDirectory(dirPath) == false)
            {
                return new string[0];
            }

            // ネットワークドライブ直下の場合は、Win32APIを用いてActiveDirectory一覧を取得する。
            if (IsMatchToNetworkDrive(dirPath))
            {
                return new string[0];

                // ActiveDirectoryを調べるコード

                //int totalentries;
                //int resume_handle = 0;
                //int entriesread;
                //IntPtr bufPtr = IntPtr.Zero;

                //// ActiveDirectory一覧を取得する.
                //int result = NetShareEnum(
                //     new StringBuilder(dirPath), DataInformationLevel.Level1, ref bufPtr
                //     , MAX_PREFERRED_LENGTH, out entriesread, out totalentries, ref resume_handle
                //   );

                //// bufPtrにあるActiveDirectoryの配列を、shareInfoリストに移し替える。
                //IntPtr currentPtr = bufPtr; // ActiveDirectoryの配列をさすポインタ
                //List<SHARE_INFO_1> shareInfos = new List<SHARE_INFO_1>(); // ActiveDirectoryの情報を格納するリスト
                //long nStructSize = Marshal.SizeOf(typeof(SHARE_INFO_1)); // ActiveDirectoryの情報を格納する構造体SHARE_INFO_1のサイズ

                //// butPtrの内容を、shareInfosリストに移し替えていく
                //for (int i = 0; i < entriesread; i++)
                //{
                //    SHARE_INFO_1 shi1 = (SHARE_INFO_1)Marshal.PtrToStructure(currentPtr, typeof(SHARE_INFO_1));
                //    shareInfos.Add(shi1);

                //    if (IntPtr.Size == 4)
                //    {
                //        // 32bit OSの場合
                //        currentPtr = new IntPtr(currentPtr.ToInt32() + (int)nStructSize);
                //    }
                //    else
                //    {
                //        // 64bit OSの場合は、ポインタも64bitで扱う必要がある。
                //        currentPtr = new IntPtr(currentPtr.ToInt64() + nStructSize);
                //    }
                //}

                //NetApiBufferFree(bufPtr);

                //string tmpDirPath = dirPath;

                //// 末尾が@'\'じゃなかったら、追加しておく
                //if (dirPath.EndsWith("\\") == false)
                //{
                //    tmpDirPath += "\\";
                //}

                //// disk deriveであり、アクセスが許可されているパスだけ集める
                //var dirs = shareInfos.Where(x => x.shi1_type == 0 && IsAccessibleDirectory(tmpDirPath + x.ToString()))
                //                     .Select(x => tmpDirPath + x.ToString());
                //return dirs;
            }

            try
            {
                return IsValidFilepathString(dirPath)
                           ? Directory.EnumerateDirectories(dirPath).Where(IsAvailability)
                           : new string[0];
            }
            catch (UnauthorizedAccessException)
            {
                return new string[0];
            }
            catch (IOException)
            {
                return new string[0];
            }
        }

        /// <summary>
        /// The enumerate directories.
        /// </summary>
        /// <param name="dirPath">
        /// The dir path.
        /// </param>
        /// <param name="searchPattern">
        /// The search pattern.
        /// </param>
        /// <param name="searchOption">
        /// The search option.
        /// </param>
        /// <returns>
        /// The <see cref="IEnumerable"/>.
        /// </returns>
        public static IEnumerable<string> EnumerateDirectories(
            string dirPath,
            string searchPattern,
            SearchOption searchOption)
        {
            if (IsAccessibleDirectory(dirPath) == false)
            {
                return new string[0];
            }

            try
            {
                return IsValidFilepathString(dirPath)
                           ? Directory.EnumerateDirectories(dirPath, searchPattern, searchOption).Where(IsAvailability)
                           : new string[0];
            }
            catch (UnauthorizedAccessException)
            {
                return new string[0];
            }
            catch (IOException)
            {
                return new string[0];
            }
        }

        /// <summary>
        /// The enumerate files.
        /// </summary>
        /// <param name="dirPath">
        /// The dir path.
        /// </param>
        /// <returns>
        /// The <see cref="IEnumerable"/>.
        /// </returns>
        public static IEnumerable<string> EnumerateFiles(string dirPath)
        {
            if (IsMatchToNetworkDrive(dirPath) ||
                IsAccessibleDirectory(dirPath) == false)
            {
                return new string[0];
            }

            try
            {
                return IsValidFilepathString(dirPath)
                           ? Directory.EnumerateFiles(dirPath).Where(IsAvailability)
                           : new string[0];
            }
            catch (UnauthorizedAccessException)
            {
                return new string[0];
            }
            catch (IOException)
            {
                return new string[0];
            }
        }

        /// <summary>
        /// The enumerate files.
        /// </summary>
        /// <param name="dirPath">
        /// The dir path.
        /// </param>
        /// <param name="searchPattern">
        /// The search pattern.
        /// </param>
        /// <returns>
        /// The <see cref="IEnumerable"/>.
        /// </returns>
        public static IEnumerable<string> EnumerateFiles(string dirPath, string searchPattern)
        {
            if (IsAccessibleDirectory(dirPath) == false)
            {
                return new string[0];
            }

            try
            {
                return IsValidFilepathString(dirPath)
                           ? Directory.EnumerateFiles(dirPath, searchPattern).Where(IsAvailability)
                           : new string[0];
            }
            catch (UnauthorizedAccessException)
            {
                return new string[0];
            }
            catch (IOException)
            {
                return new string[0];
            }
        }

        /// <summary>
        /// The enumerate files.
        /// </summary>
        /// <param name="dirPath">
        /// The dir path.
        /// </param>
        /// <param name="searchPattern">
        /// The search pattern.
        /// </param>
        /// <param name="searchOption">
        /// The search option.
        /// </param>
        /// <returns>
        /// The <see cref="IEnumerable"/>.
        /// </returns>
        public static IEnumerable<string> EnumerateFiles(
            string dirPath,
            string searchPattern,
            SearchOption searchOption)
        {
            if (IsAccessibleDirectory(dirPath) == false)
            {
                return new string[0];
            }

            try
            {
                if (IsValidFilepathString(dirPath))
                {
                    var result = new List<string>();
                    result.AddRange(Directory.EnumerateFiles(dirPath, searchPattern));
                    try
                    {
                        foreach (var dir in Directory.EnumerateDirectories(dirPath, "*", searchOption))
                        {
                            try
                            {
                                if (IsAccessibleDirectory(dir))
                                {
                                    result.AddRange(Directory.EnumerateFiles(dir, searchPattern));
                                }

                            }
                            catch
                            {
                                // アクセスできなかったパスはスキップ
                            }
                        }
                    }
                    catch
                    {
                        // アクセスできなかったパスはスキップ
                    }

                    return result;
                }
                else
                {
                    return new string[0];
                }
            }
            catch (UnauthorizedAccessException)
            {
                return new string[0];
            }
            catch (IOException)
            {
                return new string[0];
            }
        }

        /// <summary>
        /// サブディレクトリを持っているか
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>サブディレクトリがあるならtrue, 無いならfalse</returns>
        public static bool HasSubDirectory(string filePath)
        {

            if (!IsMatchToNetworkDrive(filePath) && !System.IO.Directory.Exists(filePath))
            {
                return false;
            }

            return EnumerateDirectories(filePath).Any();
        }

        /// <summary>
        /// The is accessible directory.
        /// </summary>
        /// <param name="filepath">
        /// The filepath.
        /// </param>
        /// <returns>
        /// The <see cref="bool"/>.
        /// </returns>
        public static bool IsAccessibleDirectory(string filepath)
        {
            if (string.IsNullOrEmpty(filepath) == true)
            {
                return false;
            }

            if (InaccessiblePaths.ContainsKey(filepath))
            {
                return false;
            }

            if (IsMatchToNetworkDrive(filepath))
            {
                return false;

                // ActiveDirectoryを調べるコード

                //int totalentries;
                //int resume_handle = 0;
                //int entriesread;
                //IntPtr bufPtr = IntPtr.Zero;

                //int result = NetShareEnum(
                //     new StringBuilder(filepath), DataInformationLevel.Level1, ref bufPtr
                //     , MAX_PREFERRED_LENGTH, out entriesread, out totalentries, ref resume_handle
                //   );

                //return result == NERR_Success;
            }

            // 「ドライブにディスクがありません。ディスクをドライブXに挿入してください。」
            // の警告メッセージ回避
            if (!System.IO.Directory.Exists(filepath))
            {
                return false;
            }

            try
            {
                Directory.GetAccessControl(filepath);

                return true;
            }
            catch (Exception)
            {
                InaccessiblePaths[filepath] = true;

                return false;
            }
        }

        /// <summary>
        /// The is availability.
        /// </summary>
        /// <param name="filepath">
        /// The filepath.
        /// </param>
        /// <returns>
        /// The <see cref="bool"/>.
        /// </returns>
        private static bool IsAvailability(string filepath)
        {
            return ((new FileInfo(filepath)).Attributes & (FileAttributes.Hidden | FileAttributes.System)) == 0;
        }

        enum DataInformationLevel
        {
            /// <summary>
            /// 共有の名前を取得します。関数から制御が返ると、bufptr パラメータが指すバッファに、複数の 構造体からなる 1 つの配列が格納されます。
            /// </summary>
            Level0 = 0,
            /// <summary>
            /// リソースの名前、タイプ、リソースに関連付けられているコメントなど、共有リソースに関する情報を取得します。
            /// 関数から制御が返ると、bufptr パラメータが指すバッファに、複数の 構造体からなる 1 つの配列が格納されます。
            /// </summary>
            Level1 = 1,
            /// <summary>
            /// リソースの名前、タイプ、アクセス許可、パスワード、接続の数など、共有リソースに関する情報を取得します。関数から制御が返ると、bufptr パラメータが指すバッファに、複数の 構造体からなる 1 つの配列が格納されます。
            /// </summary>
            Level2 = 2,
            /// <summary>
            /// リソースの名前、タイプ、アクセス許可、接続の数、他の固有情報など、共有リソースに関する情報を取得します。関数から制御が返ると、bufptr パラメータが指すバッファに、複数の 構造体からなる 1 つの配列が格納されます。
            /// </summary>
            Level502 = 502,
        }

        /// <summary>
        /// The SHARE_INFO_1 structure contains information about the shared resource, including the name and type of the resource, and a comment associated with the resource.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Unicode)]
        public struct SHARE_INFO_1
        {
            /// <summary>
            /// Pointer to a Unicode string specifying the share name of a resource. Calls to the NetShareSetInfo function ignore this member.
            /// </summary>
            public string shi1_netname;
            /// <summary>
            /// A bitmask of flags that specify the type of the shared resource. Calls to the NetShareSetInfo function ignore this member.
            /// </summary>
            public uint shi1_type;
            /// <summary>
            /// Pointer to a Unicode string specifying an optional comment about the shared resource.
            /// </summary>
            public string shi1_remark;

            public SHARE_INFO_1(string sharename, uint sharetype, string remark)
            {
                shi1_netname = sharename;
                shi1_type = sharetype;
                shi1_remark = remark;
            }

            public override string ToString()
            {
                return shi1_netname;
            }
        }
        const int NERR_Success = 0;
        const uint MAX_PREFERRED_LENGTH = 0xFFFFFFFF;

        [DllImport("Netapi32.dll", CharSet = CharSet.Unicode)]
        static extern int NetShareEnum(
          StringBuilder serverName,
          DataInformationLevel level,
          ref IntPtr bufPtr,
          uint prefmaxlen,
          out int entriesread,
          out int totalentries,
          ref int resume_handle
        );

        /// <summary>
        /// NetApiBufferAllocate が割り当てたメモリを解放します。Windows NT/2000 上で他のネットワーク管理関数が返したメモリを解放するには、この関数を使ってください。
        /// </summary>
        /// <param name="Buffer">既に他のネットワーク管理関数が返したバッファへのポインタを指定します。</param>
        /// <returns>関数が成功すると、NERR_Success が返ります。</returns>
        [DllImport("Netapi32.dll", SetLastError = true)]
        static extern int NetApiBufferFree(IntPtr Buffer);
    }
}
