﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace EffectMaker.UIControls.EffectBrowser.Utilities
{
    using System;
    using System.Drawing;
    using System.Globalization;
    using System.Runtime.InteropServices;

    using Microsoft.Win32;

    /// <summary>
    /// The image utility.
    /// </summary>
    public static class ImageUtility
    {
        #region Public Methods and Operators

        /// <summary>
        /// The get directory icon.
        /// </summary>
        /// <param name="path">
        /// The path.
        /// </param>
        /// <returns>
        /// The <see cref="Bitmap"/>.
        /// </returns>
        public static Bitmap GetDirectoryIcon(string path)
        {
            var handle = GetDirectoryIconHandle(path);

            return handle == IntPtr.Zero ? SystemIcons.WinLogo.ToBitmap() : Icon.FromHandle(handle).ToBitmap();
        }

        /// <summary>
        /// The get shell icon.
        /// </summary>
        /// <param name="shellIconIndex">
        /// The shell icon index.
        /// </param>
        /// <returns>
        /// The <see cref="Bitmap"/>.
        /// </returns>
        public static Bitmap GetShellIcon(int shellIconIndex)
        {
            var registryKeys = new[] { "SOFTWARE", "Microsoft", "Windows", "CurrentVersion", "Explorer", "Shell Icons" };

            try
            {
                RegistryKey key = null;
                {
                    foreach (var keyName in registryKeys)
                    {
                        key = Registry.LocalMachine.OpenSubKey(keyName, false);
                        if (key == null)
                        {
                            return GetDefaultShellIcon(shellIconIndex);
                        }
                    }
                }

                System.Diagnostics.Debug.Assert(key != null, "key != null");

                var iconFileDesc = key.GetValue(shellIconIndex.ToString(CultureInfo.InvariantCulture)) as string;
                {
                    if ((iconFileDesc == null) || (iconFileDesc.Length <= 0))
                    {
                        return GetDefaultShellIcon(shellIconIndex);
                    }
                }

                key.Close();

                var tokens = iconFileDesc.Split(',');
                {
                    if (tokens.Length != 2)
                    {
                        return GetDefaultShellIcon(shellIconIndex);
                    }
                }

                int iconIndex;
                {
                    if (int.TryParse(tokens[1], out iconIndex) == false)
                    {
                        return GetDefaultShellIcon(shellIconIndex);
                    }
                }

                var iconS = IntPtr.Zero;
                {
                    var iconL = IntPtr.Zero;
                    ExtractIconEx(tokens[0], iconIndex, ref iconL, ref iconS, 1);
                }

                return Icon.FromHandle(iconS).ToBitmap();
            }
            catch
            {
                return GetDefaultShellIcon(shellIconIndex);
            }
        }

        #endregion

        #region Methods

        /// <summary>
        /// The extract icon ex.
        /// </summary>
        /// <param name="pszPath">
        /// The psz path.
        /// </param>
        /// <param name="iconIndex">
        /// The i icon index.
        /// </param>
        /// <param name="largeIcon">
        /// The h large icon.
        /// </param>
        /// <param name="smallIcon">
        /// The h small icon.
        /// </param>
        /// <param name="icons">
        /// The i icons.
        /// </param>
        /// <returns>
        /// The <see cref="uint"/>.
        /// </returns>
        [DllImport("shell32.dll", CharSet = CharSet.Auto, SetLastError = true)]
        private static extern uint ExtractIconEx(
            string pszPath,
            int iconIndex,
            ref IntPtr largeIcon,
            ref IntPtr smallIcon,
            int icons);

        /// <summary>
        /// The get default shell icon.
        /// </summary>
        /// <param name="shellIconIndex">
        /// The shell icon index.
        /// </param>
        /// <returns>
        /// The <see cref="Bitmap"/>.
        /// </returns>
        private static Bitmap GetDefaultShellIcon(int shellIconIndex)
        {
            try
            {
                var iconS = IntPtr.Zero;
                {
                    var iconL = IntPtr.Zero;

                    ExtractIconEx("shell32.dll", shellIconIndex, ref iconL, ref iconS, 1);
                }

                var icon = Icon.FromHandle(iconS);

                var newIcon = new Icon(icon, icon.Width, icon.Height);
                icon.Dispose();
                icon = newIcon;

                return icon.ToBitmap();
            }
            catch
            {
                return SystemIcons.WinLogo.ToBitmap();
            }
        }

        /// <summary>
        /// The get directory icon handle.
        /// </summary>
        /// <param name="path">
        /// The path.
        /// </param>
        /// <returns>
        /// The <see cref="IntPtr"/>.
        /// </returns>
        private static IntPtr GetDirectoryIconHandle(string path)
        {
            var shfi = new SHFILEINFO();
            {
                const uint Flags = SHGFI.ICON | SHGFI.USEFILEATTRIBUTES | SHGFI.SMALLICON;

                // Get file info, 0x10 = FILE_ATTRIBUTE_NORMAL
                SHGetFileInfo(path, 0x10, ref shfi, (uint)Marshal.SizeOf(shfi), Flags);
            }

            return shfi.Hicon;
        }

        /// <summary>
        /// The sh get file info.
        /// </summary>
        /// <param name="pszPath">
        /// The psz path.
        /// </param>
        /// <param name="fileAttributes">
        /// The dw file Attributes.
        /// </param>
        /// <param name="psfi">
        /// The psfi.
        /// </param>
        /// <param name="sizeFileInfo">
        /// The cb size file info.
        /// </param>
        /// <param name="flags">
        /// The u flags.
        /// </param>
        /// <returns>
        /// The <see cref="IntPtr"/>.
        /// </returns>
        [DllImport("shell32.dll")]
        private static extern IntPtr SHGetFileInfo(
            string pszPath,
            uint fileAttributes,
            ref SHFILEINFO psfi,
            uint sizeFileInfo,
            uint flags);

        #endregion

        // ReSharper disable InconsistentNaming
        // ReSharper disable UnusedMember.Local
        // ReSharper disable FieldCanBeMadeReadOnly.Local
        // ReSharper disable MemberCanBePrivate.Local

        /// <summary>
        /// The shfileinfo.
        /// </summary>
        [StructLayout(LayoutKind.Sequential)]
        private struct SHFILEINFO
        {
            /// <summary>
            /// The h icon.
            /// </summary>
            public IntPtr Hicon;

            /// <summary>
            /// The i icon.
            /// </summary>
            public int Iicon;

            /// <summary>
            /// The Attributes.
            /// </summary>
            public uint Attributes;

            /// <summary>
            /// The sz display name.
            /// </summary>
            [MarshalAs(UnmanagedType.ByValTStr, SizeConst = MaxPath)]
            public string DisplayName;

            /// <summary>
            /// The sz type name.
            /// </summary>
            [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 80)]
            public string TypeName;

            /// <summary>
            /// The ma x_ path.
            /// </summary>
            private const int MaxPath = 260;
        }

        /// <summary>
        /// The shell icon indices.
        /// </summary>
        public static class ShellIconIndices
        {
            #region Constants

            /// <summary>
            /// The s i_ audi o_ cd.
            /// </summary>
            public const int AUDIOCD = 40;

            /// <summary>
            /// The s i_ cdrom.
            /// </summary>
            public const int CDROM = 11;

            /// <summary>
            /// The s i_ controlpanel.
            /// </summary>
            public const int CONTROLPANEL = 35;

            /// <summary>
            /// The s i_ de f_ application.
            /// </summary>
            public const int DEFAPPLICATION = 2;

            /// <summary>
            /// The s i_ de f_ document.
            /// </summary>
            public const int DEFDOCUMENT = 1;

            /// <summary>
            /// The s i_ desktop.
            /// </summary>
            public const int DESKTOP = 34;

            /// <summary>
            /// The s i_ dun.
            /// </summary>
            public const int DUN = 33;

            /// <summary>
            /// The s i_ favorites.
            /// </summary>
            public const int FAVORITES = 43;

            /// <summary>
            /// The s i_ flopp y_35.
            /// </summary>
            public const int FLOPPY35 = 6;

            /// <summary>
            /// The s i_ flopp y_514.
            /// </summary>
            public const int FLOPPY514 = 5;

            /// <summary>
            /// The s i_ folde r_ closed.
            /// </summary>
            public const int FOLDERCLOSED = 3;

            /// <summary>
            /// The s i_ folde r_ open.
            /// </summary>
            public const int FOLDEROPEN = 4;

            /// <summary>
            /// The s i_ font.
            /// </summary>
            public const int FONT = 38;

            /// <summary>
            /// The s i_ hdd.
            /// </summary>
            public const int HDD = 8;

            /// <summary>
            /// The s i_ hibernate.
            /// </summary>
            public const int HIBERNATE = 48;

            /// <summary>
            /// The s i_ lock.
            /// </summary>
            public const int LOCK = 47;

            /// <summary>
            /// The s i_ logoff.
            /// </summary>
            public const int LOGOFF = 44;

            /// <summary>
            /// The s i_ mycomputer.
            /// </summary>
            public const int MYCOMPUTER = 15;

            /// <summary>
            /// The s i_ network.
            /// </summary>
            public const int NETWORK = 13;

            /// <summary>
            /// The s i_ networkdrive.
            /// </summary>
            public const int NETWORKDRIVE = 9;

            /// <summary>
            /// The s i_ networkdriv e_ disconnected.
            /// </summary>
            public const int NETWORKDRIVEDISCONNECTED = 10;

            /// <summary>
            /// The s i_ networ k_ neighborhood.
            /// </summary>
            public const int NETWORKNEIGHBORHOOD = 17;

            /// <summary>
            /// The s i_ networ k_ workgroup.
            /// </summary>
            public const int NETWORKWORKGROUP = 18;

            /// <summary>
            /// The s i_ printer.
            /// </summary>
            public const int PRINTER = 37;

            /// <summary>
            /// The s i_ printe r_ default.
            /// </summary>
            public const int PRINTERDEFAULT = 30;

            /// <summary>
            /// The s i_ printmanager.
            /// </summary>
            public const int PRINTMANAGER = 16;

            /// <summary>
            /// The s i_ programgroups.
            /// </summary>
            public const int PROGRAMGROUPS = 36;

            /// <summary>
            /// The s i_ ramdisk.
            /// </summary>
            public const int RAMDISK = 12;

            /// <summary>
            /// The s i_ recyclebi n_ empty.
            /// </summary>
            public const int RECYCLEBINEMPTY = 31;

            /// <summary>
            /// The s i_ recyclebi n_ full.
            /// </summary>
            public const int RECYCLEBINFULL = 32;

            /// <summary>
            /// The s i_ removable.
            /// </summary>
            public const int REMOVABLE = 7;

            /// <summary>
            /// The s i_ share.
            /// </summary>
            public const int SHARE = 28;

            /// <summary>
            /// The s i_ shortcut.
            /// </summary>
            public const int SHORTCUT = 29;

            /// <summary>
            /// The s i_ startmen u_ docking.
            /// </summary>
            public const int STARTMENUDOCKING = 26;

            /// <summary>
            /// The s i_ startmen u_ documents.
            /// </summary>
            public const int STARTMENUDOCUMENTS = 20;

            /// <summary>
            /// The s i_ startmen u_ find.
            /// </summary>
            public const int STARTMENUFIND = 22;

            /// <summary>
            /// The s i_ startmen u_ help.
            /// </summary>
            public const int STARTMENUHELP = 23;

            /// <summary>
            /// The s i_ startmen u_ programs.
            /// </summary>
            public const int STARTMENUPROGRAMS = 19;

            /// <summary>
            /// The s i_ startmen u_ run.
            /// </summary>
            public const int STARTMENURUN = 24;

            /// <summary>
            /// The s i_ startmen u_ settings.
            /// </summary>
            public const int STARTMENUSETTINGS = 21;

            /// <summary>
            /// The s i_ startmen u_ shutdown.
            /// </summary>
            public const int STARTMENUSHUTDOWN = 27;

            /// <summary>
            /// The s i_ startmen u_ suspend.
            /// </summary>
            public const int STARTMENUSUSPEND = 25;

            /// <summary>
            /// The s i_ taskbar.
            /// </summary>
            public const int TASKBAR = 39;

            /// <summary>
            /// The s i_ unknown.
            /// </summary>
            public const int UNKNOWN = 0;

            #endregion
        }

        /// <summary>
        /// The shgfi.
        /// </summary>
        private static class SHGFI
        {
            #region Constants

            /// <summary>
            /// The shgf i_ addoverlays.
            /// </summary>
            public const uint ADDOVERLAYS = 0x000000020; // apply the appropriate overlays

            /// <summary>
            /// The shgf i_ Attributes.
            /// </summary>
            public const uint ATTRIBUTES = 0x000000800; // get Attributes

            /// <summary>
            /// The shgf i_ att r_ specified.
            /// </summary>
            public const uint ATTRSPECIFIED = 0x000020000; // get only specified Attributes

            /// <summary>
            /// The shgf i_ displayname.
            /// </summary>
            public const uint DISPLAYNAME = 0x000000200; // get display name

            /// <summary>
            /// The shgf i_ exetype.
            /// </summary>
            public const uint EXETYPE = 0x000002000; // return exe type

            /// <summary>
            /// The shgf i_ icon.
            /// </summary>
            public const uint ICON = 0x000000100; // get icon

            /// <summary>
            /// The shgf i_ iconlocation.
            /// </summary>
            public const uint ICONLOCATION = 0x000001000; // get icon location

            /// <summary>
            /// The shgf i_ largeicon.
            /// </summary>
            public const uint LARGEICON = 0x000000000; // get large icon

            /// <summary>
            /// The shgf i_ linkoverlay.
            /// </summary>
            public const uint LINKOVERLAY = 0x000008000; // put a link overlay on icon

            /// <summary>
            /// The shgf i_ openicon.
            /// </summary>
            public const uint OPENICON = 0x000000002; // get open icon

            /// <summary>
            /// The shgf i_ overlayindex.
            /// </summary>
            public const uint OVERLAYINDEX = 0x000000040;

            // Get the index of the overlay in the upper 8 bits of the Iicon

            /// <summary>
            /// The shgf i_ pidl.
            /// </summary>
            public const uint PIDL = 0x000000008; // pszPath is a pidl

            /// <summary>
            /// The shgf i_ selected.
            /// </summary>
            public const uint SELECTED = 0x000010000; // show icon in selected state

            /// <summary>
            /// The shgf i_ shelliconsize.
            /// </summary>
            public const uint SHELLICONSIZE = 0x000000004; // get shell size icon

            /// <summary>
            /// The shgf i_ smallicon.
            /// </summary>
            public const uint SMALLICON = 0x000000001; // get small icon

            /// <summary>
            /// The shgf i_ sysiconindex.
            /// </summary>
            public const uint SYSICONINDEX = 0x000004000; // get system icon index

            /// <summary>
            /// The shgf i_ typename.
            /// </summary>
            public const uint TYPENAME = 0x000000400; // get type name

            /// <summary>
            /// The shgf i_ usefileattributes.
            /// </summary>
            public const uint USEFILEATTRIBUTES = 0x000000010; // use passed dwFileAttribute

            #endregion
        }

        // ReSharper restore MemberCanBePrivate.Local
        // ReSharper restore FieldCanBeMadeReadOnly.Local
        // ReSharper restore UnusedMember.Local
        // ReSharper restore InconsistentNaming
    }
}
