﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Linq;
using EffectMaker.UIControls.EffectBrowser.Utilities;
using EffectMaker.UIControls.EffectBrowser.Data.FileCache;

namespace EffectMaker.UIControls.EffectBrowser.Data
{
    /// <summary>
    /// The file info.
    /// </summary>
    public partial class FileInfo
    {
        #region Public Methods and Operators

        /// <summary>
        /// The load from file.
        /// </summary>
        /// <param name="kinds">
        /// 読み込み対象とするファイル種別の配列
        /// </param>
        public void LoadFromFile(FileKindType[] kinds)
        {
            if (this.IsInternalLoaded)
            {
                return;
            }

            if (kinds.Contains(this.FileKind))
            {
                this.LoadFromFileInternal();
            }

            if (this.HasChildren)
            {
                foreach (var child in this.Children.Where(child => kinds.Contains(child.FileKind)))
                {
                    child.LoadFromFileInternal();
                }
            }
        }

        #endregion

        #region Methods

        /// <summary>
        /// The load from file internal.
        /// </summary>
        private void LoadFromFileInternal()
        {
            switch (this.FileKind)
            {
                case FileKindType.EsetFile:
                    this.LoadFromFileInternalEsetFile();
                    break;

                case FileKindType.TextureFile:
                case FileKindType.PrimitiveFile:
                case FileKindType.CombinerFile:
                case FileKindType.PreviewFile:
                case FileKindType.WorkspaceFile:
                    this.LoadFromFileInternalOtherFile();
                    break;
            }
        }

        /// <summary>
        /// The load from file internal eset file.
        /// </summary>
        private void LoadFromFileInternalEsetFile()
        {
            var cache = FileCacheManager.Instance.Get(this.FileFullPath, true);
            if (cache.IsValid)
            {
                Func<string, FileKindType, FileInfo> createFileInfo = (path, kind) =>
                    {
                        // esetから参照するアセットを探す時は基点以下縛りをしてはいけない
                        var childFileDirPaths = PathUtility.MakeChildFileDirPaths(this.DirName, false, kind);

                        var fullPath = PathUtility.MakeExistsReferenceFilePath(path, childFileDirPaths);

                        if (string.IsNullOrEmpty(fullPath))
                        {
                            return null;
                        }

                        var newCache = FileCacheManager.Instance.Get(fullPath, true);

                        var fileInfo = new FileInfo
                                           {
                                               DirName = Path.GetDirectoryName(fullPath),
                                               FileName = Path.GetFileName(fullPath),
                                               UpdateTimestamp = newCache.UpdateTimestamp,
                                               CreateTimestamp = newCache.CreateTimestamp,
                                               FileKind = kind,
                                               ByteSize = newCache.ByteSize,
                                               Depth = this.Depth + 1,
                                               Parent = this,
                                               Icon = newCache.Icon,
                                               LargeIcon = newCache.LargeIcon,
                                               IsOldData = newCache.IsOldData,
                                               ThumnailFilePath = newCache.ThumnailFilePath,
                                           };

                        return fileInfo;
                    };

                var textures =
                    cache.Textures.Select(x => createFileInfo(x, FileKindType.TextureFile)).Where(x => x != null);
                var primitives =
                    cache.Primitives.Select(x => createFileInfo(x, FileKindType.PrimitiveFile)).Where(x => x != null);
                var combiners =
                    cache.Combiners.Select(x => createFileInfo(x, FileKindType.CombinerFile)).Where(x => x != null);

                this.Icon = cache.IsValid ? cache.Icon : null;
                this.LargeIcon = cache.IsValid ? cache.LargeIcon : null;
                this.Children = textures.Concat(primitives).Concat(combiners).ToArray();
                this.Comment = cache.Comment;
                this.EmitterNames = cache.EmitterNames;
                this.LabelColor = cache.LabelColor;
                this.IsOldData = cache.IsOldData;  // 旧形式データであることのマーキング
                this.ThumnailFilePath = cache.ThumnailFilePath;
            }
            else
            {
                this.Icon = cache.IsValid ? cache.Icon : null;
                this.LargeIcon = cache.IsValid ? cache.LargeIcon : null;
                this.Children = null;
                this.Comment = cache.Comment;
                this.EmitterNames = cache.EmitterNames;
                this.LabelColor = LabelColorType.Color0;
                this.IsOldData = false;
                this.ThumnailFilePath = cache.ThumnailFilePath;
            }
        }

        /// <summary>
        /// エミッタセット以外の(ぶら下がりのない)ファイルの読み込み処理です。
        /// 基本的にキャッシュからのデータをスルーするだけです。
        /// </summary>
        private void LoadFromFileInternalOtherFile()
        {
            var cache = FileCacheManager.Instance.Get(this.FileFullPath, true);

            this.Icon = cache.IsValid ? cache.Icon : null;
            this.LargeIcon = cache.IsValid ? cache.LargeIcon : null;
            this.Comment = cache.Comment;
            this.EmitterNames = cache.EmitterNames;
            this.LabelColor = cache.LabelColor;
            this.IsOldData = cache.IsOldData;
            this.ThumnailFilePath = cache.ThumnailFilePath;
        }

        #endregion
    }
}
