﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Collections.Generic;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Xml;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Texture;

namespace EffectMaker.UIControls.EffectBrowser.Data.FileCache
{
    /// <summary>
    /// The file cache value.
    /// </summary>
    public partial class FileCacheValue
    {
        /// <summary>
        /// The load eset.
        /// </summary>
        /// <returns>
        /// The <see cref="bool"/>.
        /// </returns>
        private bool LoadEset()
        {
            this.Textures = new string[0];
            this.Primitives = new string[0];
            this.Combiners = new string[0];
            this.EmitterNames = new string[0];
            this.LabelColor = LabelColorType.Color0;
            this.Comment = string.Empty;

            return this.FastLoadEset();
        }

#if false

    // エフェクトメーカ本体で実装されている方法でEsetファイルを読む
        private bool SafeLoadEset()
        {
            // デシリアライズを実行
            DataModelBase data;
            var result         = SerializerFactory.Deserialize(FileFullPath, out data);
            var emitterSetData = data as EmitterSetData;

            // エラー処理
            if (result != DataModelSerializationResults.Success || emitterSetData == null)
            {
                IsValid = false;
                return false;
            }

            Textures   = emitterSetData.TexturePaths.ToArray();
            Primitives = emitterSetData.PrimitivePaths.ToArray();
            LabelColor = (LabelColorType)emitterSetData.EmitterSetBasicData.LabelColor;
            Comment    = emitterSetData.EmitterSetBasicData.Comment;

            return true;
        }
#endif

        /// <summary>
        /// エフェクトブラウザ独自実装した方法でEsetファイルを読む
        /// </summary>
        /// <returns>
        /// The <see cref="bool"/>.
        /// </returns>
        private bool FastLoadEset()
        {
            if (Utilities.PathUtility.FileExists(this.FileFullPath) == false)
            {
                this.IsValid = false;
                return false;
            }

            var emitterNames = new List<string>();
            var textures = new List<string>();
            var primitives = new List<string>();
            var combiners = new List<string>();
            try
            {
                using (var fs = new FileStream(this.FileFullPath, FileMode.Open, FileAccess.Read))
                {
                    var reader = XmlReader.Create(
                        fs,
                        new XmlReaderSettings {IgnoreComments = true, IgnoreWhitespace = true});

                    var inEmitterData = false;
                    var inName = false;
                    var inEmitterSetBasicData = false;
                    var inComment = false;
                    var inLabelColor = false;
                    var inEmitterTextureFileData = false;
                    var inPrimitiveFilePath = false;
                    var inFilePath = false;
                    var inCombinerEditorProjectPath = false;

                    while (reader.Read())
                    {
                        if (reader.IsEmptyElement)
                        {
                            continue;
                        }

                        var name = reader.Name;

                        switch (name)
                        {
                            case "EmitterSet":
                                // 旧形式であることのマーキング
                                this.IsOldData = true;
                                return true;

                            case "EmitterData":
                                switch (reader.NodeType)
                                {
                                    case XmlNodeType.Element:
                                        inEmitterData = true;
                                        break;

                                    case XmlNodeType.EndElement:
                                        inEmitterData = false;
                                        break;
                                }

                                break;

                            case "EmitterSetBasicData":
                                switch (reader.NodeType)
                                {
                                    case XmlNodeType.Element:
                                        inEmitterSetBasicData = true;
                                        break;

                                    case XmlNodeType.EndElement:
                                        inEmitterSetBasicData = false;
                                        break;
                                }

                                break;

                            case "EmitterTextureFileData":
                                switch (reader.NodeType)
                                {
                                    case XmlNodeType.Element:
                                        inEmitterTextureFileData = true;
                                        break;

                                    case XmlNodeType.EndElement:
                                        inEmitterTextureFileData = false;
                                        break;
                                }

                                break;

                            case "PrimitiveFilePath":
                                switch (reader.NodeType)
                                {
                                    case XmlNodeType.Element:
                                        inPrimitiveFilePath = true;
                                        break;

                                    case XmlNodeType.EndElement:
                                        inPrimitiveFilePath = false;
                                        break;
                                }

                                break;

                            case "CombinerEditorProjectPath":
                                switch (reader.NodeType)
                                {
                                    case XmlNodeType.Element:
                                        inCombinerEditorProjectPath = true;
                                        break;

                                    case XmlNodeType.EndElement:
                                        inCombinerEditorProjectPath = false;
                                        break;
                                }

                                break;
                        }

                        if (inEmitterData)
                        {
                            switch (name)
                            {
                                case "Name":
                                    switch (reader.NodeType)
                                    {
                                        case XmlNodeType.Element:
                                            inName = true;
                                            break;

                                        case XmlNodeType.EndElement:
                                            inName = false;
                                            break;
                                    }

                                    break;
                            }

                            if (inName)
                            {
                                if (reader.NodeType == XmlNodeType.Text)
                                {
                                    emitterNames.Add(reader.Value);
                                }
                            }
                        }

                        if (inEmitterSetBasicData)
                        {
                            switch (name)
                            {
                                case "Comment":
                                    switch (reader.NodeType)
                                    {
                                        case XmlNodeType.Element:
                                            inComment = true;
                                            break;

                                        case XmlNodeType.EndElement:
                                            inComment = false;
                                            break;
                                    }

                                    break;

                                case "LabelColor":
                                    switch (reader.NodeType)
                                    {
                                        case XmlNodeType.Element:
                                            inLabelColor = true;
                                            break;

                                        case XmlNodeType.EndElement:
                                            inLabelColor = false;
                                            break;
                                    }

                                    break;
                            }

                            if (inComment)
                            {
                                if (reader.NodeType == XmlNodeType.Text)
                                {
                                    this.Comment = reader.Value;
                                }
                            }
                            else if (inLabelColor)
                            {
                                if (reader.NodeType == XmlNodeType.Text)
                                {
                                    this.LabelColor = (LabelColorType)int.Parse(reader.Value);
                                }
                            }
                        }

                        if (inEmitterTextureFileData)
                        {
                            if (name == "FilePath")
                            {
                                switch (reader.NodeType)
                                {
                                    case XmlNodeType.Element:
                                        inFilePath = true;
                                        break;

                                    case XmlNodeType.EndElement:
                                        inFilePath = false;
                                        break;
                                }
                            }

                            if (inFilePath)
                            {
                                textures.Add(reader.Value);
                            }
                        }
                        else if (inPrimitiveFilePath)
                        {
                            if (reader.NodeType == XmlNodeType.Text)
                            {
                                primitives.Add(reader.Value);
                            }
                        }
                        else if (inCombinerEditorProjectPath)
                        {
                            if (reader.NodeType == XmlNodeType.Text)
                            {
                                combiners.Add(reader.Value);
                            }
                        }
                    }
                }
            }
            catch
            {
                // 読み込み失敗時は失敗として続行
                Logger.Log("LogView", LogLevels.Error, Properties.Resources.ErrorEsetLoadFailed, this.FileFullPath);
                this.IsValid = false;
                return false;
            }

            this.EmitterNames = emitterNames.ToArray();

            this.Textures =
                textures.Where(
                    x => (string.IsNullOrEmpty(x) == false) && Path.GetExtension(x) == Constants.ExtTextureFile)
                    .Select(Path.GetFileName)
                    .Distinct()
                    .ToArray();
            this.Primitives =
                primitives.Where(
                    x => (string.IsNullOrEmpty(x) == false) && Path.GetExtension(x) == Constants.ExtPrimitiveFile)
                    .Select(Path.GetFileName)
                    .Distinct()
                    .ToArray();

            if (OptionStore.ProjectConfig.IsEftCombinerEditorEnabled)
            {
                this.Combiners =
                    combiners.Where(
                        x => (string.IsNullOrEmpty(x) == false) && Path.GetExtension(x) == Constants.ExtCombinerFile)
                        .Select(Path.GetFileName)
                        .Distinct()
                        .ToArray();
            }

            // サムネイル画像を読み込む
            {
                string dirPath = Path.GetDirectoryName(this.FileFullPath);
                string fileName = Path.GetFileName(this.FileFullPath);

                string jpgPath = Path.Combine(dirPath, ".NwThumbs", fileName + ".jpg");
                string pngPath = Path.Combine(dirPath, ".NwThumbs", fileName + ".png");

                string targetPath = string.Empty;

                if (File.Exists(jpgPath))
                {
                    targetPath = jpgPath;
                }
                else if (File.Exists(pngPath))
                {
                    targetPath = pngPath;
                }

                if (string.IsNullOrEmpty(targetPath) == false)
                {
                    using (FileStream fs = new FileStream(targetPath, FileMode.Open, FileAccess.Read))
                    {
                        Bitmap tmpBitmap = new Bitmap(fs);
                        Size iconSize = new Size(Constants.IconSize, Constants.IconSize);
                        Bitmap resizeBmp = TextureData.ResizeBitmap(tmpBitmap, iconSize);
                        this.Icon = tmpBitmap == resizeBmp ? (Bitmap)resizeBmp.Clone() : resizeBmp;
                        this.LargeIcon = tmpBitmap;
                        fs.Close();
                    }

                    this.ThumnailFilePath = targetPath;
                }
            }

            return true;
        }
    }
}
