﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace EffectMaker.UIControls.EffectBrowser.Data.FileCache
{
    using System;
    using System.IO;

    /// <summary>
    /// The file cache manager.
    /// </summary>
    public class FileCacheManager
    {
        #region Static Fields

        /// <summary>
        /// The singleton instance.
        /// </summary>
        private static readonly FileCacheManager SingletonInstance = new FileCacheManager();

        #endregion

        #region Fields

        /// <summary>
        /// The file cache.
        /// </summary>
        private readonly FileCache fileCache = new FileCache();

        #endregion

        #region Public Properties

        /// <summary>
        /// Gets the instance.
        /// </summary>
        public static FileCacheManager Instance
        {
            get
            {
                return SingletonInstance;
            }
        }

        #endregion

        #region Public Methods and Operators

        /// <summary>
        /// The contains deep loaded.
        /// </summary>
        /// <param name="filePath">
        /// The file path.
        /// </param>
        /// <returns>
        /// The <see cref="bool"/>.
        /// </returns>
        public bool ContainsDeepLoaded(string filePath)
        {
            var value = this.fileCache.Get(filePath);
            if (value == null)
            {
                return false;
            }

            return value.IsDeepLoaded;
        }

        /// <summary>
        /// The get.
        /// </summary>
        /// <param name="filePath">
        /// The file path.
        /// </param>
        /// <param name="isDeepLoad">
        /// The is deep load.
        /// </param>
        /// <returns>
        /// The <see cref="FileCacheValue"/>.
        /// </returns>
        public FileCacheValue Get(string filePath, bool isDeepLoad)
        {
            var isDirectory = Utilities.PathUtility.DirectoryExists(filePath);

            if ((isDirectory == false) && (Utilities.PathUtility.FileExists(filePath) == false))
            {
                return null;
            }

            var value = this.fileCache.Get(filePath);
            if (value != null)
            {
                // 大文字小文字の変更がある可能性があるので更新する
                value.FileFullPath = filePath;

                var updateTimestamp = File.GetLastWriteTime(filePath);

                // キャッシュ済みだけど日付が違うので読みなおしてそれを返す
                if (value.UpdateTimestamp != updateTimestamp)
                {
                    value.ReadDeepLoaded();
                    value.Load(isDirectory, updateTimestamp, isDeepLoad);
                }

                if (isDeepLoad && (value.IsDeepLoaded == false))
                {
                    value.Load(isDirectory, updateTimestamp, true);
                }
            }
            else
            {
                // キャッシュに無いので新規で作ってそれを返す
                value = new FileCacheValue(filePath, isDeepLoad);

                this.fileCache.Add(filePath, value);
            }

            return value;
        }

        /// <summary>
        /// キャッシュされている状態か？
        /// キャッシュに含まれていても日付が対応していなければ false
        /// </summary>
        /// <param name="filePath">
        /// The file path.
        /// </param>
        /// <param name="updatTimestamp">
        /// The updat timestamp.
        /// </param>
        /// <returns>
        /// The <see cref="bool"/>.
        /// </returns>
        public bool IsCached(string filePath, DateTime updatTimestamp)
        {
            var value = this.fileCache.Get(filePath);
            if (value == null)
            {
                return false;
            }

            return value.UpdateTimestamp == updatTimestamp;
        }

        /// <summary>
        /// 指定したファイルパスのキャッシュを破棄します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        public void Discard(string filePath)
        {
            this.fileCache.Remove(filePath);
        }

        /// <summary>
        /// 全てのキャッシュを破棄します。
        /// </summary>
        public void DiscardAll()
        {
            this.fileCache.Clear();
        }

        #endregion
    }
}
