﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using EffectMaker.BusinessLogic.IO;

namespace EffectMaker.UIControls.EffectBrowser.Data
{
    using System.Collections.Generic;
    using System.Drawing;
    using System.Drawing.Imaging;
    using System.IO;
    using System.Reflection;

    using EffectMaker.UIControls.EffectBrowser.Utilities;

    /// <summary>
    /// The constants.
    /// </summary>
    public static class Constants
    {
        #region Constants

        /// <summary>
        /// The expander margin.
        /// </summary>
        public const int ExpanderMargin = 6;

        /// <summary>
        /// The icon margin.
        /// </summary>
        public const int IconMargin = 1;

        /// <summary>
        /// アイコンの幅、高さ
        /// </summary>
        public const int IconSize = 16;

        /// <summary>
        /// 大きなアイコンの幅、高さ
        /// </summary>
        public const int LargeIconSize = 96;

        #endregion

        #region Static Fields

        /// <summary>
        /// The breadcrumb ctrl item img.
        /// </summary>
        public static readonly Bitmap BreadcrumbCtrlItemImg = Properties.Resources.BreadcrumbCtrlItemImg;

        /// <summary>
        /// The child expander.
        /// </summary>
        public static readonly Dictionary<bool, Bitmap> ChildExpander =
            new Dictionary<bool, Bitmap>
            {
                { false, Properties.Resources.Icon_Collapsed },
                { true, Properties.Resources.Icon_Expanded }
            };

        /// <summary>
        /// The ext eset file.
        /// </summary>
        public static readonly string ExtEsetFile = ".eset";

        /// <summary>
        /// The ext preview file.
        /// </summary>
        public static readonly string ExtPreviewFile = ".prev";

        /// <summary>
        /// The ext primitive file.
        /// </summary>
        public static readonly string ExtPrimitiveFile = ".fmdb";

        /// <summary>
        /// The ext texture file.
        /// </summary>
        public static readonly string ExtTextureFile = ".ftxb";

        /// <summary>
        /// The ext combiner file.
        /// </summary>
        public static readonly string ExtCombinerFile = ".ecmb";

        /// <summary>
        /// The ext workspace file.
        /// </summary>
        public static readonly string ExtWorkspaceFile = ".work";

        /// <summary>
        /// The file kind infos.
        /// </summary>
        public static readonly Dictionary<FileKindType, FileKindInfo> FileKindInfos =
            new Dictionary<FileKindType, FileKindInfo>
            {
                {
                    FileKindType.EsetFile,
                    new FileKindInfo
                    {
                        Icon =
                            Properties.Resources.Icon_EmitterSet,
                        LargeIcon =
                            Properties.Resources.Icon_EmitterSet_LSize,
                        Name =
                            Properties.Resources.FileKind_EmitterSetFile,
                        ShortName =
                            Properties.Resources.FileKind_EmitterSet,
                        ExtName = ExtEsetFile
                    }
                },
                {
                    FileKindType.TextureFile,
                    new FileKindInfo
                    {
                        Icon =
                            Properties.Resources.Icon_Textures,
                        LargeIcon =
                            Properties.Resources.Icon_Textures_LSize,
                        Name =
                            Properties.Resources.FileKind_TextureFile,
                        ShortName =
                            Properties.Resources.FileKind_Texture,
                        ExtName = ExtTextureFile
                    }
                },
                {
                    FileKindType.PreviewFile,
                    new FileKindInfo
                    {
                        Icon =
                            Properties.Resources.Icon_PreviewSetting,
                        LargeIcon =
                            Properties.Resources.Icon_Preview_LSize,
                        Name =
                            Properties.Resources.FileKind_PreviewFile,
                        ShortName =
                            Properties.Resources.FileKind_Preview,
                        ExtName = ExtPreviewFile
                    }
                },
                {
                    FileKindType.PrimitiveFile,
                    new FileKindInfo
                    {
                        Icon =
                            Properties.Resources.Icon_Primitives,
                        LargeIcon =
                            Properties.Resources.Icon_Primitive_LSize,
                        Name =
                            Properties.Resources.FileKind_PrimitiveFile,
                        ShortName =
                            Properties.Resources.FileKind_Primitive,
                        ExtName = ExtPrimitiveFile
                    }
                },
                {
                    FileKindType.CombinerFile,
                    new FileKindInfo
                    {
                        Icon =
                            Properties.Resources.Icon_ECMB,
                        LargeIcon =
                            Properties.Resources.Icon_ECMB_LSize,
                        Name =
                            Properties.Resources.FileKind_CombinerFile,
                        ShortName =
                            Properties.Resources.FileKind_Combiner,
                        ExtName = ExtCombinerFile
                    }
                },
                {
                    FileKindType.WorkspaceFile,
                    new FileKindInfo
                    {
                        Icon =
                            Properties.Resources.Icon_WorkFile,
                        LargeIcon =
                            Properties.Resources.Icon_WorkFile_LSize,
                        Name =
                            Properties.Resources.FileKind_WorkspaceFile,
                        ShortName =
                            Properties.Resources.FileKind_Workspace,
                        ExtName = ExtWorkspaceFile
                    }
                },
                {
                    FileKindType.Folder,
                    new FileKindInfo
                    {
                        Icon =
                            ImageUtility.GetShellIcon(ImageUtility.ShellIconIndices.FOLDEROPEN),
                        LargeIcon =
                            Properties.Resources.Icon_Folder_LSize,
                        Name =
                            Properties.Resources.FileKind_Folder,
                        ShortName =
                            Properties.Resources.FileKind_Folder,
                        ExtName = null
                    }
                }
            };

        /// <summary>
        /// The label color colors.
        /// </summary>
        public static readonly Dictionary<LabelColorType, Color> LabelColorColors =
            new Dictionary<LabelColorType, Color>
            {
                { LabelColorType.Color0, UIConstants.LabelColorColor0 },
                { LabelColorType.Color1, UIConstants.LabelColorColor1 },
                { LabelColorType.Color2, UIConstants.LabelColorColor2 },
                { LabelColorType.Color3, UIConstants.LabelColorColor3 },
                { LabelColorType.Color4, UIConstants.LabelColorColor4 },
                { LabelColorType.Color5, UIConstants.LabelColorColor5 },
                { LabelColorType.Color6, UIConstants.LabelColorColor6 },
                { LabelColorType.Color7, UIConstants.LabelColorColor7 },
            };

        /// <summary>
        /// The label color icon 0.
        /// </summary>
        public static readonly Bitmap LabelColorIcon0 =
            new Bitmap(IconSize, IconSize, PixelFormat.Format32bppArgb);

        /// <summary>
        /// The label color icon 1.
        /// </summary>
        public static readonly Bitmap LabelColorIcon1 =
            new Bitmap(IconSize, IconSize, PixelFormat.Format32bppArgb);

        /// <summary>
        /// The label color icon 2.
        /// </summary>
        public static readonly Bitmap LabelColorIcon2 =
            new Bitmap(IconSize, IconSize, PixelFormat.Format32bppArgb);

        /// <summary>
        /// The label color icon 3.
        /// </summary>
        public static readonly Bitmap LabelColorIcon3 =
            new Bitmap(IconSize, IconSize, PixelFormat.Format32bppArgb);

        /// <summary>
        /// The label color icon 4.
        /// </summary>
        public static readonly Bitmap LabelColorIcon4 =
            new Bitmap(IconSize, IconSize, PixelFormat.Format32bppArgb);

        /// <summary>
        /// The label color icon 5.
        /// </summary>
        public static readonly Bitmap LabelColorIcon5 =
            new Bitmap(IconSize, IconSize, PixelFormat.Format32bppArgb);

        /// <summary>
        /// The label color icon 6.
        /// </summary>
        public static readonly Bitmap LabelColorIcon6 =
            new Bitmap(IconSize, IconSize, PixelFormat.Format32bppArgb);

        /// <summary>
        /// The label color icon 7.
        /// </summary>
        public static readonly Bitmap LabelColorIcon7 =
            new Bitmap(IconSize, IconSize, PixelFormat.Format32bppArgb);

        /// <summary>
        /// The label color icons.
        /// </summary>
        public static readonly Dictionary<LabelColorType, Bitmap> LabelColorIcons =
            new Dictionary<LabelColorType, Bitmap>
            {
                { LabelColorType.Color0, LabelColorIcon0 },
                { LabelColorType.Color1, LabelColorIcon1 },
                { LabelColorType.Color2, LabelColorIcon2 },
                { LabelColorType.Color3, LabelColorIcon3 },
                { LabelColorType.Color4, LabelColorIcon4 },
                { LabelColorType.Color5, LabelColorIcon5 },
                { LabelColorType.Color6, LabelColorIcon6 },
                { LabelColorType.Color7, LabelColorIcon7 },
            };

        /// <summary>
        /// The supported ext types.
        /// </summary>
        public static readonly Dictionary<string, FileKindType> SupportedExtTypes =
            new Dictionary<string, FileKindType>
            {
                { ExtEsetFile, FileKindType.EsetFile },
                { ExtTextureFile, FileKindType.TextureFile },
                { ExtPreviewFile, FileKindType.PreviewFile },
                { ExtPrimitiveFile, FileKindType.PrimitiveFile },
                { ExtCombinerFile, FileKindType.CombinerFile },
                { ExtWorkspaceFile, FileKindType.WorkspaceFile },
            };

        #endregion

        #region Constructors and Destructors

        /// <summary>
        /// Initializes static members of the <see cref="Constants"/> class.
        /// </summary>
        static Constants()
        {
            var rect = new Rectangle(0, 0, IconSize - 1, IconSize - 1);

            using (var g = Graphics.FromImage(LabelColorIcon0))
            {
                g.Clear(UIConstants.LabelColorColor0);
                g.DrawRectangle(Pens.LightGray, rect);
            }

            using (var g = Graphics.FromImage(LabelColorIcon1))
            {
                g.Clear(UIConstants.LabelColorColor1);
                g.DrawRectangle(Pens.LightGray, rect);
            }

            using (var g = Graphics.FromImage(LabelColorIcon2))
            {
                g.Clear(UIConstants.LabelColorColor2);
                g.DrawRectangle(Pens.LightGray, rect);
            }

            using (var g = Graphics.FromImage(LabelColorIcon3))
            {
                g.Clear(UIConstants.LabelColorColor3);
                g.DrawRectangle(Pens.LightGray, rect);
            }

            using (var g = Graphics.FromImage(LabelColorIcon4))
            {
                g.Clear(UIConstants.LabelColorColor4);
                g.DrawRectangle(Pens.LightGray, rect);
            }

            using (var g = Graphics.FromImage(LabelColorIcon5))
            {
                g.Clear(UIConstants.LabelColorColor5);
                g.DrawRectangle(Pens.LightGray, rect);
            }

            using (var g = Graphics.FromImage(LabelColorIcon6))
            {
                g.Clear(UIConstants.LabelColorColor6);
                g.DrawRectangle(Pens.LightGray, rect);
            }

            using (var g = Graphics.FromImage(LabelColorIcon7))
            {
                g.Clear(UIConstants.LabelColorColor7);
                g.DrawRectangle(Pens.LightGray, rect);
            }
        }

        #endregion

        #region Public Properties

        /// <summary>
        /// Gets the eset pattern.
        /// </summary>
        public static string EsetPattern
        {
            get
            {
                return PathUtility.MakeFileMaskPattern(FileKindType.EsetFile);
            }
        }

        /// <summary>
        /// Gets the favorites filepath.
        /// </summary>
        public static string FavoritesFilepath
        {
            get
            {
                var location = Path.GetDirectoryName(Assembly.GetEntryAssembly().Location);
                System.Diagnostics.Debug.Assert(location != null, "location != null");

                return Path.Combine(location, "EffectBrowserFavorites.xml");
            }
        }

        /// <summary>
        /// Gets the primitives unused directory.
        /// </summary>
        public static string PrimitivesUnusedDirectory
        {
            get
            {
                var location = Path.GetDirectoryName(Assembly.GetEntryAssembly().Location);
                System.Diagnostics.Debug.Assert(location != null, "location != null");
                return Path.Combine(location, "UnUsed", IOConstants.PrimitiveFolderName);
            }
        }

        /// <summary>
        /// Gets the combiners unused directory.
        /// </summary>
        public static string CombinersUnusedDirectory
        {
            get
            {
                var location = Path.GetDirectoryName(Assembly.GetEntryAssembly().Location);
                System.Diagnostics.Debug.Assert(location != null, "location != null");
                return Path.Combine(location, "UnUsed", IOConstants.CombinerShaderFolderName);
            }
        }

        /// <summary>
        /// Gets the setting filepath.
        /// </summary>
        public static string SettingFilepath
        {
            get
            {
                var location = Path.GetDirectoryName(Assembly.GetEntryAssembly().Location);
                System.Diagnostics.Debug.Assert(location != null, "location != null");

                return Path.Combine(location, "EffectBrowserSetting.xml");
            }
        }

        /// <summary>
        /// Gets the textures unused directory.
        /// </summary>
        public static string TexturesUnusedDirectory
        {
            get
            {
                var location = Path.GetDirectoryName(Assembly.GetEntryAssembly().Location);
                System.Diagnostics.Debug.Assert(location != null, "location != null");
                return Path.Combine(location, "UnUsed", IOConstants.TextureFolderName);
            }
        }

        #endregion

        /// <summary>
        /// The file kind info.
        /// </summary>
        public class FileKindInfo
        {
            #region Public Properties

            /// <summary>
            /// Gets or sets the ext name.
            /// </summary>
            public string ExtName { get; set; }

            /// <summary>
            /// Gets or sets the icon.
            /// </summary>
            public Bitmap Icon { get; set; }

            /// <summary>
            /// 大きなアイコンを取得または設定します。
            /// </summary>
            public Bitmap LargeIcon { get; set; }

            /// <summary>
            /// Gets or sets the name.
            /// </summary>
            public string Name { get; set; }

            /// <summary>
            /// Gets or sets the short name.
            /// </summary>
            public string ShortName { get; set; }

            #endregion
        }
    }
}
