﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

using SysDebug = System.Diagnostics.Debug;

namespace EffectMaker.UIControls.EffectBrowser.Data
{
    /// <summary>
    /// ファイルの裏読みを行うクラスです。
    /// </summary>
    public class BackgroundLoader
    {
        /// <summary>
        /// このローダーで対象とするファイル種別の配列です。
        /// </summary>
        private readonly FileKindType[] fileKinds;

        /// <summary>
        /// UIスレッドの同期コンテキストです。
        /// </summary>
        private readonly SynchronizationContext syncContext;

        /// <summary>
        /// タスク情報です。
        /// この参照値はUIスレッドからのみ変更を行います。
        /// 参照先のオブジェクトの値は非同期スレッドからも変更を行います。
        /// </summary>
        private TaskInfo taskInfo;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="fileKinds">対象とするファイル種別</param>
        /// <param name="syncContext">同期コンテキスト</param>
        public BackgroundLoader(FileKindType[] fileKinds, SynchronizationContext syncContext)
        {
            this.fileKinds = fileKinds;
            this.syncContext = syncContext;
        }

        /// <summary>
        /// 裏読み処理をキャンセルします。
        /// </summary>
        public void Cancel()
        {
            SysDebug.Assert(SynchronizationContext.Current == this.syncContext, "SynchronizationContext.Current == this.syncContext");

            if (this.taskInfo == null)
            {
                return;
            }

            // 裏読みタスクをキャンセルする
            // キャンセル待ちをするとデッドロックになるため自然に任せる
            this.taskInfo.LoadTaskCancel.Cancel();
            this.taskInfo.LoadTaskCancel.Dispose();

            // タスク情報を更新
            this.taskInfo = null;
        }

        /// <summary>
        /// 裏読み処理を開始します。
        /// </summary>
        /// <param name="targetFiles">読み込みを行うファイルのリスト</param>
        /// <param name="onLoaded">ファイル毎の読み込みが終わったときの処理</param>
        /// <param name="onFinished">読み込みが完了したときの処理</param>
        public void StartLoading(IEnumerable<FileInfo> targetFiles, Action<FileInfo> onLoaded, Action onFinished)
        {
            SysDebug.Assert(SynchronizationContext.Current == this.syncContext, "SynchronizationContext.Current == this.syncContext");

            // 作成済みのタスクをキャンセルする
            if (this.taskInfo != null)
            {
                this.Cancel();
            }

            // タスク情報を作成
            TaskInfo taskInfo = new TaskInfo()
            {
                LoadTaskCancel = new CancellationTokenSource(),
                LockTargetFileList = new object()
            };

            // ターゲットファイルのリストを作成
            {
                taskInfo.TargetFileList = new LinkedList<FileInfo>(targetFiles.Where(x => this.IsTargetKind(x.FileKind)));

                taskInfo.TargetFileDict = new Dictionary<string, LinkedListNode<FileInfo>>(taskInfo.TargetFileList.Count);

                LinkedListNode<FileInfo> node = taskInfo.TargetFileList.First;

                while (node != null)
                {
                    // 同じテクスチャを参照するesetを展開したときに重複するテクスチャパスが渡されることがあるので
                    // 重複をチェックしてから値を追加する
                    if (taskInfo.TargetFileDict.ContainsKey(node.Value.FileFullPath) == false)
                    {
                        taskInfo.TargetFileDict.Add(node.Value.FileFullPath, node);
                        node = node.Next;
                    }
                    else
                    {
                        // 重複する値をリストから削除する
                        LinkedListNode<FileInfo> nextNode = node.Next;
                        taskInfo.TargetFileList.Remove(node);
                        node = nextNode;
                    }
                }
            }

            // タスクで行う読み込み処理を設定
            Action loadAction = () =>
            {
#if false
                foreach (var fileInfo in clonedOpendAllFiles)
                {
                    fileInfo.LoadFromFile(this.fileKinds);
                    onLoaded();
                }

                onFinished();
#else

                ParallelOptions options = new ParallelOptions()
                {
                    CancellationToken = taskInfo.LoadTaskCancel.Token,
                    MaxDegreeOfParallelism = Math.Max(1, Math.Min(4, Environment.ProcessorCount - 1))  // EffectMakerの操作が重くなるのでスレッド数を制限
                };

                try
                {
                    ParallelLoopResult loopResult = Parallel.For(0, taskInfo.TargetFileDict.Count, options, (fileInfo, loopState) =>
                    {
                        LinkedListNode<FileInfo> targetNode;

                        // 処理するファイルを取得
                        lock (taskInfo.LockTargetFileList)
                        {
                            targetNode = taskInfo.TargetFileList.First;
                            taskInfo.TargetFileList.RemoveFirst();
                            taskInfo.TargetFileDict[targetNode.Value.FileFullPath] = null;
                        }

                        // 読み込み処理を実行
                        targetNode.Value.LoadFromFile(this.fileKinds);

                        if (loopState.ShouldExitCurrentIteration || loopState.IsExceptional)
                        {
                            loopState.Stop();
                            return;
                        }

                        // ファイル毎の読み込み終了処理を呼び出す
                        this.syncContext.Post(_ =>
                        {
                            if (taskInfo.LoadTaskCancel.IsCancellationRequested == false)
                            {
                                onLoaded(targetNode.Value);
                            }
                        }, null);
                    });

                    // 読み込み完了処理を呼び出す
                    if (loopResult.IsCompleted)
                    {
                        this.syncContext.Post(_ =>
                        {
                            if (taskInfo.LoadTaskCancel.IsCancellationRequested == false)
                            {
                                onFinished();
                            }
                        }, null);
                    }
                }
                catch (OperationCanceledException)
                {
                }
#endif
            };

            // 裏読みタスクを実行
            taskInfo.Task = Task.Factory.StartNew(loadAction, taskInfo.LoadTaskCancel.Token);

            // タスク情報を更新
            this.taskInfo = taskInfo;
        }

        /// <summary>
        /// 指定したファイルの読み込みを最優先に行うようにします。
        /// </summary>
        /// <param name="fileInfo">ファイル情報</param>
        public void MoveToTop(FileInfo fileInfo)
        {
            SysDebug.Assert(SynchronizationContext.Current == this.syncContext, "SynchronizationContext.Current == this.syncContext");

            if (this.taskInfo == null)
            {
                return;
            }

            lock (this.taskInfo.LockTargetFileList)
            {
                // ファイル情報に対応するノードを取得
                LinkedListNode<FileInfo> targetNode;
                this.taskInfo.TargetFileDict.TryGetValue(fileInfo.FileFullPath, out targetNode);

                // ノードをリストの先頭に移動
                if (targetNode != null)
                {
                    this.taskInfo.TargetFileList.Remove(targetNode);
                    this.taskInfo.TargetFileList.AddFirst(targetNode);
                }
            }
        }

        /// <summary>
        /// このローダーが対象とするファイル種別かどうかを返します。
        /// </summary>
        /// <param name="kind">ファイル種別</param>
        /// <returns>対象ならtrue,そうでなければfalse.</returns>
        public bool IsTargetKind(FileKindType kind)
        {
            return this.fileKinds.Contains(kind);
        }

        /// <summary>
        /// タスク情報です。
        /// </summary>
        private class TaskInfo
        {
            /// <summary>
            /// 裏読みを行うタスクを取得または設定します。
            /// </summary>
            public Task Task { get; set; }

            /// <summary>
            /// タスクのキャンセルイベントを取得または設定します。
            /// </summary>
            public CancellationTokenSource LoadTaskCancel { get; set; }

            /// <summary>
            /// ターゲットファイルのディクショナリを取得または設定します。
            /// </summary>
            public Dictionary<string, LinkedListNode<FileInfo>> TargetFileDict { get; set; }

            /// <summary>
            /// 読み込み優先順に並べ替えたターゲットファイルのリストを取得または設定します。
            /// </summary>
            public LinkedList<FileInfo> TargetFileList { get; set; }

            /// <summary>
            /// TargetFileListのロックオブジェクトを取得または設定します。
            /// </summary>
            public object LockTargetFileList { get; set; }
        }
    }
}
