﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using EffectMaker.BusinessLogic.Options;
using EffectMaker.UIControls.EffectBrowser.Controls.Basic;

namespace EffectMaker.UIControls.EffectBrowser.Controls.SearchTab
{
    using System;
    using System.ComponentModel;
    using System.Windows.Forms;

    using EffectMaker.UIControls.EffectBrowser.Controls.FileListView;
    using EffectMaker.UIControls.EffectBrowser.Data;
    using EffectMaker.UIControls.EffectBrowser.Utilities;

    /// <summary>
    /// The search panel.
    /// </summary>
    public partial class SearchPanel : UserControl
    {
        #region Fields

        /// <summary>
        /// The directory.
        /// </summary>
        private string directory = string.Empty;

        /// <summary>
        /// アセット種別ごとのファイル検索パネルを配列化します。
        /// </summary>
        private SearchFilePanel[] searchFilePanels = null;

        #endregion

        #region Constructors and Destructors

        /// <summary>
        /// Initializes a new instance of the <see cref="SearchPanel"/> class.
        /// </summary>
        public SearchPanel()
        {
            this.InitializeComponent();

            this.rbnTexture.Tag = FileKindType.TextureFile;
            this.rbnPrimitive.Tag = FileKindType.PrimitiveFile;
            this.rbnCombiner.Tag = FileKindType.CombinerFile;

            this.SelectedSearchModeChanged += (s, e) => this.UpdateHelpToolTip();
            this.SelectedSearchModeChanged += (s, e) => this.UpdateBaseFilterCheckBoxVisible();

            // オプションでコンバイナ連携がON/OFFされた時に追従する
            OptionStore.OptionChanged += (s, e) => this.UpdateRadioButtons();

            // 初回表示時のラジオボタン配置で余白が出てしまうのを回避する
            this.pnlFileKind.VisibleChanged += this.UpdateRadioButtonsOnce;
        }

        #endregion

        #region Public Events

        /// <summary>
        /// The directory changed.
        /// </summary>
        public event EventHandler DirectoryChanged;

        /// <summary>
        /// The kind changed.
        /// </summary>
        public event EventHandler KindChanged;

        /// <summary>
        /// The move all files button click.
        /// </summary>
        public event EventHandler MoveAllFilesButtonClick
        {
            add
            {
                foreach (var sfp in this.SearchFilePanels)
                {
                    sfp.MoveAllFilesButtonClick += value;
                }
            }

            remove
            {
                foreach (var sfp in this.SearchFilePanels)
                {
                    sfp.MoveAllFilesButtonClick -= value;
                }
            }
        }

        /// <summary>
        /// The move selected files button click.
        /// </summary>
        public event EventHandler MoveSelectedFilesButtonClick
        {
            add
            {
                foreach (var sfp in this.SearchFilePanels)
                {
                    sfp.MoveSelectedFilesButtonClick += value;
                }
            }

            remove
            {
                foreach (var sfp in this.SearchFilePanels)
                {
                    sfp.MoveSelectedFilesButtonClick -= value;
                }
            }
        }

        /// <summary>
        /// The reference file path changed.
        /// </summary>
        public event EventHandler ReferenceFilePathChanged
        {
            add
            {
                foreach (var sfp in this.SearchFilePanels)
                {
                    sfp.ReferenceFilePathChanged += value;
                }
            }

            remove
            {
                foreach (var sfp in this.SearchFilePanels)
                {
                    sfp.ReferenceFilePathChanged -= value;
                }
            }
        }

        /// <summary>
        /// The reset button click.
        /// </summary>
        public event EventHandler ResetButtonClick
        {
            add
            {
                this.btnReset.Click += value;
            }

            remove
            {
                this.btnReset.Click -= value;
            }
        }

        /// <summary>
        /// The search button click.
        /// </summary>
        public event EventHandler SearchButtonClick
        {
            add
            {
                this.btnSearch.Click += value;
            }

            remove
            {
                this.btnSearch.Click -= value;
            }
        }

        /// <summary>
        /// The selected search mode changed.
        /// </summary>
        public event EventHandler SelectedSearchModeChanged
        {
            add
            {
                foreach (var sfp in this.SearchFilePanels)
                {
                    sfp.SelectedSearchModeChanged += value;
                }
            }

            remove
            {
                foreach (var sfp in this.SearchFilePanels)
                {
                    sfp.SelectedSearchModeChanged -= value;
                }
            }
        }

        #endregion

        #region Public Properties

        /// <summary>
        /// Gets or sets the directory.
        /// </summary>
        [Browsable(false)]
        public string Directory
        {
            get
            {
                return this.directory;
            }

            set
            {
                if (this.directory == value)
                {
                    return;
                }

                this.directory = value;

                foreach (var sfp in this.SearchFilePanels)
                {
                    sfp.Directory = this.directory;
                }

                this.UpdateButtons();
                this.UpdateHelpToolTip();

                if (ControlEventSuppressBlock.Enabled == false)
                {
                    if (this.DirectoryChanged != null)
                    {
                        this.DirectoryChanged(this, EventArgs.Empty);
                    }
                }
            }
        }

        /// <summary>
        /// Sets the files count.
        /// </summary>
        public int FilesCount
        {
            set
            {
                foreach (var sfp in this.SearchFilePanels)
                {
                    sfp.MoveAllFilesButtonEnabled = value > 0;
                }
            }
        }

        /// <summary>
        /// Gets the reference file path.
        /// </summary>
        public string ReferenceFilePath
        {
            get
            {
                return this.SelectedSearchFilePanel.ReferenceFilePath;
            }
        }

        /// <summary>
        /// Gets or sets a value indicating whether reset button enabled.
        /// </summary>
        public bool ResetButtonEnabled
        {
            get
            {
                return this.btnReset.Enabled;
            }

            set
            {
                this.btnReset.Enabled = value;
            }
        }

        /// <summary>
        /// Gets or sets a value indicating whether search button enabled.
        /// </summary>
        public bool SearchButtonEnabled
        {
            get
            {
                return this.btnSearch.Enabled;
            }

            set
            {
                this.btnSearch.Enabled = value;
            }
        }

        /// <summary>
        /// Gets the search file info.
        /// </summary>
        public SearchFileInfo SearchFileInfo
        {
            get
            {
                return new SearchFileInfo
                {
                    FileKind = this.SelectedSearchFilePanel.FileKind,
                    SearchMode = this.SelectedSearchFilePanel.SearchMode,
                    ReferenceFilePath = this.SelectedSearchFilePanel.FilePath
                };
            }
        }

        /// <summary>
        /// Gets the search mode.
        /// </summary>
        public SearchFileInfo.SearchModeKind SearchMode
        {
            get
            {
                return this.SelectedSearchFilePanel.SearchMode;
            }
        }

        /// <summary>
        /// Gets the selected file kind.
        /// </summary>
        public FileKindType SelectedFileKind
        {
            get
            {
                if (this.rbnTexture.Checked)
                {
                    return FileKindType.TextureFile;
                }

                if (this.rbnPrimitive.Checked)
                {
                    return FileKindType.PrimitiveFile;
                }

                return FileKindType.CombinerFile;
            }
        }

        /// <summary>
        /// Sets the selected files count.
        /// </summary>
        public int SelectedFilesCount
        {
            set
            {
                foreach (var sfp in this.SearchFilePanels)
                {
                    sfp.MoveSelectedFilesButtonEnabled = value > 0;
                }
            }
        }

        /// <summary>
        /// 検索を基点以下に絞るチェックボックス
        /// </summary>
        public EBCheckBox CbxBaseSearchPathFiltering
        {
            get { return this.cbxBaseSearchPathFiltering; }
        }

        #endregion

        #region Properties

        /// <summary>
        /// Gets the selected search file panel.
        /// </summary>
        private SearchFilePanel SelectedSearchFilePanel
        {
            get
            {
                if (this.rbnTexture.Checked)
                {
                    return this.sfpTexture;
                }

                if (this.rbnPrimitive.Checked)
                {
                    return this.sfpPrimitive;
                }

                return this.sfpCombiner;
            }
        }

        /// <summary>
        /// アセット種別ごとのファイル検索パネル配列を取得します。
        /// </summary>
        private SearchFilePanel[] SearchFilePanels
        {
            get
            {
                return this.searchFilePanels ?? (this.searchFilePanels = new[]
                {
                    this.sfpTexture, this.sfpPrimitive, this.sfpCombiner,
                });
            }
        }

        #endregion

        #region Public Methods and Operators

        /// <summary>
        /// The refresh view.
        /// </summary>
        public void RefreshView()
        {
            this.UpdateView();

            foreach (var sfp in this.SearchFilePanels)
            {
                sfp.RefreshView();
            }
        }

        #endregion

        #region Methods

        /// <summary>
        /// The on load.
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);

            this.sfpTexture.Initialize(FileKindType.TextureFile);
            this.sfpPrimitive.Initialize(FileKindType.PrimitiveFile);
            this.sfpCombiner.Initialize(FileKindType.CombinerFile);

            this.UpdateView();
            this.UpdateButtons();
            this.UpdateHelpToolTip();
        }

        /// <summary>
        /// 初回表示時にラジオボタンの余白を調整するハンドラです。
        /// </summary>
        /// <param name="sender">使用しません</param>
        /// <param name="args">使用しません</param>
        private void UpdateRadioButtonsOnce(object sender, EventArgs args)
        {
            this.UpdateRadioButtons();
            this.pnlFileKind.VisibleChanged -= this.UpdateRadioButtonsOnce;
        }

        /// <summary>
        /// The kind_ checked changed.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// The e.
        /// </param>
        private void Kind_CheckedChanged(object sender, EventArgs e)
        {
            this.UpdateView();
            this.UpdateHelpToolTip();

            if (this.KindChanged != null)
            {
                this.KindChanged(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// The update buttons.
        /// </summary>
        private void UpdateButtons()
        {
            var isExistsDirectory = PathUtility.DirectoryExists(this.Directory);

            this.btnSearch.Enabled = isExistsDirectory;
            this.btnReset.Enabled = isExistsDirectory;

            foreach (var sfp in this.SearchFilePanels)
            {
                sfp.Enabled = isExistsDirectory;
            }

            this.rbnTexture.Enabled = isExistsDirectory;
            this.rbnPrimitive.Enabled = isExistsDirectory;
            this.rbnCombiner.Enabled = isExistsDirectory;
        }

        /// <summary>
        /// The update help tool tip.
        /// </summary>
        private void UpdateHelpToolTip()
        {
            if (PathUtility.DirectoryExists(this.Directory) == false)
            {
                // ttpHelp.SetToolTip(pnlHelp, Properties.Resources.SearchHelpToolTip_InputSearchBasePath);
                this.lblHelp.Text = Properties.Resources.SearchHelpToolTip_InputSearchBasePath;
            }
            else
            {
                var fileKind = Constants.FileKindInfos[this.SelectedSearchFilePanel.FileKind];

                switch (this.SearchMode)
                {
                    case SearchFileInfo.SearchModeKind.ReferenceFile:

                        // ttpHelp.SetToolTip(pnlHelp, string.Format(Properties.Resources.SearchHelpToolTip_SearchMode_ReferenceFile, fileName));
                        this.lblHelp.Text =
                            string.Format(Properties.Resources.SearchHelpToolTip_SearchMode_ReferenceFile, fileKind.ShortName);
                        break;

                    case SearchFileInfo.SearchModeKind.UnusedFile:

                        // ttpHelp.SetToolTip(pnlHelp, string.Format(Properties.Resources.SearchHelpToolTip_SearchMode_UnusedFile, fileName));
                        this.lblHelp.Text = string.Format(
                            Properties.Resources.SearchHelpToolTip_SearchMode_UnusedFile,
                            fileKind.ShortName);
                        break;

                    case SearchFileInfo.SearchModeKind.EmitterSetUsedFile:

                        // ttpHelp.SetToolTip(pnlHelp, string.Format(Properties.Resources.SearchHelpToolTip_SearchMode_EmitterSetUsedFile, fileName));
                        this.lblHelp.Text =
                            string.Format(
                                Properties.Resources.SearchHelpToolTip_SearchMode_EmitterSetUsedFile,
                                fileKind.ShortName);
                        break;
                }
            }
        }

        /// <summary>
        /// The update view.
        /// </summary>
        private void UpdateView()
        {
            this.UpdateRadioButtons();
            this.UpdateBaseFilterCheckBoxVisible();

            using (new LockWindowUpdateBlock(this))
            using (new LayoutSuspendBlock(this))
            using (new ControlEventSuppressBlock())
            {
                switch (this.SelectedFileKind)
                {
                    case FileKindType.TextureFile:
                        this.sfpTexture.Visible = true;
                        this.sfpPrimitive.Visible = false;
                        this.sfpCombiner.Visible = false;
                        break;

                    case FileKindType.PrimitiveFile:
                        this.sfpPrimitive.Visible = true;
                        this.sfpTexture.Visible = false;
                        this.sfpCombiner.Visible = false;
                        break;

                    case FileKindType.CombinerFile:
                        this.sfpCombiner.Visible = true;
                        this.sfpTexture.Visible = false;
                        this.sfpPrimitive.Visible = false;
                        break;
                }
            }
        }

        /// <summary>
        /// ラジオボタンの選択肢をコンバイナ連携の状況にあわせて更新します。
        /// </summary>
        private void UpdateRadioButtons()
        {
            this.pnlFileKind.Controls.Clear();
            this.pnlFileKind.Width = 0;

            if (OptionStore.ProjectConfig.IsEftCombinerEditorEnabled)
            {
                this.pnlFileKind.Controls.Add(this.rbnCombiner);
                this.pnlFileKind.Controls.Add(this.rbnPrimitive);
                this.pnlFileKind.Controls.Add(this.rbnTexture);
            }
            else
            {
                this.pnlFileKind.Controls.Add(this.rbnPrimitive);
                this.pnlFileKind.Controls.Add(this.rbnTexture);
            }
        }

        /// <summary>
        /// 検索モードに応じて基点パス以下絞り込みチェックボックスの表示状態を更新します。
        /// </summary>
        private void UpdateBaseFilterCheckBoxVisible()
        {
            switch (this.SearchMode)
            {
                case SearchFileInfo.SearchModeKind.ReferenceFile:
                case SearchFileInfo.SearchModeKind.UnusedFile:
                    this.cbxBaseSearchPathFiltering.Visible = true;
                    break;
                case SearchFileInfo.SearchModeKind.EmitterSetUsedFile:
                    this.cbxBaseSearchPathFiltering.Visible = false;
                    break;
            }
        }

        /// <summary>
        /// The btn reset_ click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// The e.
        /// </param>
        private void BtnReset_Click(object sender, EventArgs e)
        {
            switch (this.SelectedFileKind)
            {
                case FileKindType.TextureFile:
                    this.sfpTexture.Reset();
                    break;

                case FileKindType.PrimitiveFile:
                    this.sfpPrimitive.Reset();
                    break;

                case FileKindType.CombinerFile:
                    this.sfpCombiner.Reset();
                    break;
            }
        }

        #endregion
    }
}
