﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace EffectMaker.UIControls.EffectBrowser.Controls.FileTreeView
{
    using System.Collections.Generic;
    using System.Linq;
    using System.Windows.Forms;

    /// <summary>
    /// The eb file tree node.
    /// </summary>
    public class EBFileTreeNode : TreeNode
    {
        #region Constructors and Destructors

        /// <summary>
        /// Initializes a new instance of the <see cref="EBFileTreeNode"/> class.
        /// </summary>
        public EBFileTreeNode()
        {
            this.NodeKind = NodeKindType.Unknown;
        }

        /// <summary>
        /// Initializes a new instance of the <see cref="EBFileTreeNode"/> class.
        /// </summary>
        /// <param name="kind">
        /// The kind.
        /// </param>
        public EBFileTreeNode(NodeKindType kind)
        {
            this.NodeKind = kind;
        }

        #endregion

        #region Enums

        /// <summary>
        /// The node kind type.
        /// </summary>
        public enum NodeKindType
        {
            /// <summary>
            /// The directory.
            /// </summary>
            Directory,

            /// <summary>
            /// The bookmark.
            /// </summary>
            Bookmark,

            /// <summary>
            /// The fav folder.
            /// </summary>
            FavFolder,

            /// <summary>
            /// The expander dummy.
            /// </summary>
            ExpanderDummy, // [+]表示用

            /// <summary>
            /// The unknown.
            /// </summary>
            Unknown
        }

        #endregion

        #region Public Properties

        /// <summary>
        /// Gets a value indicating whether can rename.
        /// </summary>
        public bool CanRename
        {
            get
            {
                return (this.NodeKind == NodeKindType.FavFolder) || (this.NodeKind == NodeKindType.Bookmark);
            }
        }

        // 注意:
        // フィールド、プロパティを変更した時はCloneメソッドを変更する

        /// <summary>
        /// Gets the child nodes.
        /// </summary>
        public IEnumerable<EBFileTreeNode> ChildNodes
        {
            get
            {
                return this.Nodes.OfType<EBFileTreeNode>();
            }
        }

        /// <summary>
        /// Gets a value indicating whether disable nodes clear on expand.
        /// </summary>
        public bool DisableNodesClearOnExpand { get; private set; }

        /// <summary>
        /// Gets or sets the filepath.
        /// </summary>
        public string Filepath { get; set; }

        /// <summary>
        /// Gets a value indicating whether is config info.
        /// </summary>
        public bool IsConfigInfo
        {
            get
            {
                return (this.NodeKind == NodeKindType.FavFolder) || (this.NodeKind == NodeKindType.Bookmark);
            }
        }

        /// <summary>
        /// Gets or sets a value indicating whether is expanded.
        /// </summary>
        public new bool IsExpanded
        {
            get
            {
                return base.IsExpanded;
            }

            set
            {
                if (value)
                {
                    this.Expand();
                }
                else
                {
                    this.Collapse();
                }
            }
        }

        /// <summary>
        /// ツリー展開時に子ノードのアップデートは必要か
        /// </summary>
        public bool IsNeedNodesUpdateOnTreeExpand
        {
            get
            {
                return (this.NodeKind == NodeKindType.Directory) || (this.NodeKind == NodeKindType.Bookmark);
            }
        }

        /// <summary>
        /// Gets or sets a value indicating whether is root.
        /// </summary>
        public bool IsRoot { get; set; }

        /// <summary>
        /// Gets the node kind.
        /// </summary>
        public NodeKindType NodeKind { get; private set; }

        #endregion

        #region Public Methods and Operators

        /// <summary>
        /// The clone.
        /// </summary>
        /// <returns>
        /// The <see cref="object"/>.
        /// </returns>
        public override object Clone()
        {
            var dst = base.Clone() as EBFileTreeNode;

            System.Diagnostics.Debug.Assert(dst != null, "dst != null");

            dst.Filepath = this.Filepath;
            dst.IsRoot = this.IsRoot;
            dst.NodeKind = this.NodeKind;

            return dst;
        }

        /// <summary>
        /// The expand without node clear.
        /// </summary>
        public void ExpandWithoutNodeClear()
        {
            this.DisableNodesClearOnExpand = true;
            this.Expand();
            this.DisableNodesClearOnExpand = false;
        }

        #endregion
    }
}
