﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using System.Windows.Forms;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.FileMonitor;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Debugging.Profiling;
using EffectMaker.UIControls.EffectBrowser.Data.FileCache;
using EffectMaker.UIControls.EffectBrowser.Data;
using EffectMaker.UIControls.EffectBrowser.Utilities;

using SysDebug = System.Diagnostics.Debug;
using FileInfo = EffectMaker.UIControls.EffectBrowser.Data.FileInfo;

namespace EffectMaker.UIControls.EffectBrowser.Controls.FileListView.Base
{
    /// <summary>
    /// The eb file list view.
    /// </summary>
    public partial class EBFileListView
    {
        /// <summary>
        /// The update state names.
        /// </summary>
        private static readonly Dictionary<UpdateStateKind, string> UpdateStateNames =
            new Dictionary<UpdateStateKind, string>
            {
                { UpdateStateKind.Wait, string.Empty },
                // {UpdateStateKind.Wait, Properties.Resources.FileListView_UpdateState_Wait},
                { UpdateStateKind.FileEnum, Properties.Resources.FileListView_UpdateState_FileEnum },
                { UpdateStateKind.Search, Properties.Resources.FileListView_UpdateState_Search },
                { UpdateStateKind.EditDataLoad, Properties.Resources.FileListView_UpdateState_EditDataLoad },
                { UpdateStateKind.AssetsLoad, Properties.Resources.FileListView_UpdateState_AssetsLoad },
                { UpdateStateKind.CloseOpen, Properties.Resources.FileListView_UpdateState_CloseOpen },
                { UpdateStateKind.Resort, Properties.Resources.FileListView_UpdateState_Resort },
            };

        /// <summary>
        /// The monitor manager.
        /// </summary>
        private readonly FileMonitorManager monitorManager = new FileMonitorManager(Application.OpenForms["MainForm"]);

        /// <summary>
        /// 同期コンテキスト
        /// </summary>
        private SynchronizationContext syncContext;

        /// <summary>
        /// エフェクトメーカーで編集するファイルの裏読みローダーです。
        /// </summary>
        private BackgroundLoader editDataLoader;

        /// <summary>
        /// アセットファイルの裏読みローダーです。
        /// </summary>
        private BackgroundLoader assetDataLoader;

        /// <summary>
        /// 現在のフォルダにあるフィルタリング前のファイルリストです。
        /// </summary>
        private FileInfo[] unfilteredFiles = new FileInfo[0];

        /// <summary>
        /// 検索フィルタを通したファイルリストです。
        /// </summary>
        private FileInfo[] searchedFiles = new FileInfo[0];

        /// <summary>
        /// 検索フィルタを通して選択した子供を展開したファイルリストです。
        /// </summary>
        private FileInfo[] openedFiles = new FileInfo[0];

        /// <summary>
        /// 更新ステートです。
        /// </summary>
        private UpdateStateKind updateState = UpdateStateKind.Wait;

        /// <summary>
        /// 裏読みが完了したエディットデータの数です。
        /// </summary>
        private int loadedEditDataCount = 0;

        /// <summary>
        /// 裏読みが完了したアセットデータの数です。
        /// </summary>
        private int loadedAssetDataCount = 0;

        /// <summary>
        /// アップデートの時間を計測するタイマーです。
        /// </summary>
        private ProfileTimer updateTimer = null;

        /// <summary>
        /// The items count changed.
        /// </summary>
        public event EventHandler ItemsCountChanged;

        /// <summary>
        /// 更新ステートが変わったときに発生します。
        /// </summary>
        public event EventHandler UpdateStateChanged;

        /// <summary>
        /// 裏読みが完了したエディットデータの数が変わったときに発生します。
        /// </summary>
        public event EventHandler LoadedEditDataCountChanged;

        /// <summary>
        /// 裏読みが完了したアセットデータの数が変わったときに発生します。
        /// </summary>
        public event EventHandler LoadedAssetDataCountChanged;

        /// <summary>
        /// 更新ステートです。
        /// </summary>
        public enum UpdateStateKind
        {
            /// <summary>
            /// 待ち状態
            /// </summary>
            Wait,

            /// <summary>
            /// The file enum.
            /// </summary>
            FileEnum, // ファイル列挙

            /// <summary>
            /// The search.
            /// </summary>
            Search, // 再検索

            /// <summary>
            /// The resort.
            /// </summary>
            Resort, // 再ソート

            /// <summary>
            /// The edit data load.
            /// </summary>
            EditDataLoad, // 編集データ読み込み

            /// <summary>
            /// The assets load.
            /// </summary>
            AssetsLoad, // アセット読み込み

            /// <summary>
            /// The close open.
            /// </summary>
            CloseOpen, // 開閉
        }

        /// <summary>
        /// 裏読みが完了したエディットデータの数を取得します。
        /// </summary>
        public int LoadedEditDataCount
        {
            get { return this.loadedEditDataCount; }

            private set
            {
                if (value != this.loadedEditDataCount)
                {
                    this.loadedEditDataCount = value;

                    this.LoadedEditDataCountChanged?.Invoke(this, EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// 裏読みが完了したアセットデータの数を取得します。
        /// </summary>
        public int LoadedAssetDataCount
        {
            get { return this.loadedAssetDataCount; }

            private set
            {
                if (value != this.loadedAssetDataCount)
                {
                    this.loadedAssetDataCount = value;

                    this.LoadedAssetDataCountChanged?.Invoke(this, EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// リストに表示する項目の数を取得します。
        /// </summary>
        public int ItemsCount
        {
            get { return this.VirtualListSize; }

            private set
            {
                if (value != this.VirtualListSize)
                {
                    this.VirtualListSize = value;

                    this.UpdateBackgroundImage();

                    this.ItemsCountChanged?.Invoke(this, EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// 更新ステートに対応した文字列を取得します。
        /// </summary>
        public string UpdateStateName
        {
            get { return UpdateStateNames[this.UpdateState]; }
        }

        /// <summary>
        /// 更新ステートを取得します。
        /// </summary>
        public UpdateStateKind UpdateState
        {
            get { return this.updateState; }

            private set
            {
                if (this.updateState != value)
                {
                    this.updateState = value;

                    this.UpdateStateChanged?.Invoke(this, EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// 更新処理関係のデータを初期化します。
        /// </summary>
        private void Initialize_Update()
        {
            this.syncContext = SynchronizationContext.Current;

            {
                FileKindType[] fileKinds = new []
                {
                    FileKindType.EsetFile,
                    FileKindType.PreviewFile,
                    FileKindType.WorkspaceFile,
                };

                this.editDataLoader = new BackgroundLoader(fileKinds, this.syncContext);
            }

            {
                FileKindType[] fileKinds = new []
                {
                    FileKindType.TextureFile,
                    FileKindType.PrimitiveFile,
                    FileKindType.CombinerFile,
                };

                this.assetDataLoader = new BackgroundLoader(fileKinds, this.syncContext);
            }
        }

        /// <summary>
        /// 更新処理を行います。
        /// </summary>
        /// <param name="updateState"></param>
        private void RunUpdate(UpdateStateKind updateState)
        {
            SysDebug.Assert(SynchronizationContext.Current == this.syncContext, "SynchronizationContext.Current == this.syncContext");

            if (updateState == this.UpdateState)
            {
                return;
            }

            if (this.updateTimer == null && updateState != UpdateStateKind.Wait)
            {
                this.updateTimer = new ProfileTimer("EffectBrowser update.");
            }

            this.UpdateState = updateState;

            if (this.UpdateState == UpdateStateKind.FileEnum)
            {
                this.UpdateList_FileEnum();
                this.UpdateState = UpdateStateKind.Search;
            }

            if (this.UpdateState == UpdateStateKind.Search)
            {
                this.UpdateList_Search();
                this.UpdateState = UpdateStateKind.EditDataLoad;
            }

            if (this.UpdateState == UpdateStateKind.Resort)
            {
                this.UpdateList_Resort();
                this.UpdateState = UpdateStateKind.EditDataLoad;
            }

            /// エフェクトメーカーで編集するデータの情報を読み込む
            if (this.UpdateState == UpdateStateKind.EditDataLoad)
            {
                Action onFinished = () => this.RunUpdate(UpdateStateKind.AssetsLoad);
                this.UpdateList_EditDataLoad(onFinished);

                return;  // 別スレッドで処理するため一旦抜ける
            }

            /// アセットデータの情報を読み込む
            if (this.UpdateState == UpdateStateKind.AssetsLoad)
            {
                Action onFinished = () => this.RunUpdate(UpdateStateKind.Wait);
                this.UpdateList_AssetsLoad(onFinished);

                return;  // 別スレッドで処理するため一旦抜ける
            }

            if (this.UpdateState == UpdateStateKind.CloseOpen)
            {
                this.UpdateList_CloseOpen();
                this.UpdateState = UpdateStateKind.Wait;
            }

            if (this.updateTimer != null) {
                this.updateTimer.Dispose();
                this.updateTimer = null;
            }

            SysDebug.Assert(this.UpdateState == UpdateStateKind.Wait, "this.UpdateState == UpdateStateKind.Wait");
        }

        /// <summary>
        /// The on gamma correction changed.
        /// </summary>
        public void OnGammaCorrectionChanged()
        {
            this.RunUpdate(UpdateStateKind.EditDataLoad);
        }

        /// <summary>
        /// The clear items.
        /// </summary>
        public void ClearItems()
        {
            this.monitorManager.AllClearMonitor();

            this.unfilteredFiles = new FileInfo[0];
            this.searchedFiles = new FileInfo[0];
            this.openedFiles = new FileInfo[0];

            this.ItemsCount = 0;

            this.SelectedIndices.Clear();

            this.Invalidate();
        }

        /// <summary>
        /// The request update.
        /// </summary>
        protected void RequestUpdate()
        {
            this.CancelAsyncJob();

            this.RunUpdate(UpdateStateKind.FileEnum);
        }

        /// <summary>
        /// リストビューの描画でリストビューアイテムが必要になったときの処理を行います。
        /// </summary>
        /// <param name="e">イベント情報</param>
        /// <exception cref="NotSupportedException">NotSupportedException</exception>
        protected override void OnRetrieveVirtualItem(RetrieveVirtualItemEventArgs e)
        {
            base.OnRetrieveVirtualItem(e);

            string[] columns;

            {
                // このアサートはSIGLO-29713の修正で引っかからなくなっているはずです。
                SysDebug.Assert(this.openedFiles.Length > e.ItemIndex, "まだ開いていないファイルに対するイベントが発生しました。");

                var fileInfo = this.openedFiles[e.ItemIndex];

                // MTT!
                if (this.editDataLoader.IsTargetKind(fileInfo.FileKind))
                {
                    this.editDataLoader.MoveToTop(fileInfo);
                }
                else if (this.assetDataLoader.IsTargetKind(fileInfo.FileKind))
                {
                    this.assetDataLoader.MoveToTop(fileInfo);
                }

                //                           "0123456789ABCD"
                const string ForIconWidth1 = "     ";
                const string ForIconWidth2 = "         ";
                const string ForIconWidth3 = "              ";

                var depthWidth = new string(' ', fileInfo.Depth * 3);

                switch (this.ViewType)
                {
                    case ViewTypeKind.FileList:
                        columns = new []
                        {
                            fileInfo.DisplayFileName + (this.WithExpander ? ForIconWidth3 : ForIconWidth1)
                            + depthWidth,
                            fileInfo.DisplayUpdateTimestamp, fileInfo.DisplayCreateTimestamp,
                            fileInfo.DisplayFileKind, fileInfo.DisplayByteSize,
                            fileInfo.DisplayLabelColor + ForIconWidth2, fileInfo.DisplayComment
                        };
                        break;

                    case ViewTypeKind.ChildFileList:
                        columns = new []
                        {
                            fileInfo.DisplayFileName + (this.WithExpander ? ForIconWidth3 : ForIconWidth1)
                            + depthWidth,
                            fileInfo.DisplayPath, fileInfo.DisplayByteSize,
                            fileInfo.DisplayUpdateTimestamp
                        };
                        break;

                    default:
                        throw new NotSupportedException();
                }
            }

            // リストビューアイテムを作成
            e.Item = new ListViewItem(columns)
            {
                Tag = this.openedFiles[e.ItemIndex]
            };

            for (var i = 0; i != this.Columns.Count; ++i)
            {
                e.Item.SubItems[i].Tag = this.Columns[i].Tag;
            }
        }

        /// <summary>
        /// ファイル列挙
        /// </summary>
        private void UpdateList_FileEnum()
        {
            // 待機カーソルを表示してファイルを列挙する
            using (new WaitCursorBlock(this))
            {
                // 展開中のファイルを復元用に取得
                var opendParentFiles = new HashSet<string>(this.unfilteredFiles.Where(x => x.IsOpend).Select(x => x.FileFullPath));

                // ソートしたファイルリストを取得
                this.unfilteredFiles = this.SortByColumnState(this.Files).ToArray();

                // 展開中のファイルを復元
                foreach (var file in this.unfilteredFiles.Where(file => opendParentFiles.Contains(file.FileFullPath)))
                {
                    file.IsOpend = true;
                }
            }
        }

        /// <summary>
        /// 検索
        /// </summary>
        private void UpdateList_Search()
        {
            // 選択中のファイルのパスを復元用に取得
            var selectedFilePaths = new HashSet<string>(this.SelectedFiles.Select(x => x.FileFullPath));

            // ファイルモニターの登録を全て解除
            this.monitorManager.AllClearMonitor();

            // 検索ワードでフィルタしたファイルリストを更新
            this.searchedFiles = this.SearchByCurrentWord(this.unfilteredFiles).ToArray();

            // ファイルリストをファイルモニターに登録
            foreach (FileInfo info in this.searchedFiles)
            {
                this.RegisterFileMonitor(info);
            }

            this.UpdateListAllFiles(false);

            // 選択を復元する
            // 仕組み上、復元するのは親の選択状態に限る
            using (new LockWindowUpdateBlock(this))
            using (new ControlEventSuppressBlock())
            {
                for (int i = 0; i < this.searchedFiles.Length; ++i)
                {
                    this.searchedFiles[i].Index = i;
                }

                this.SelectedIndices.Clear();

                var selectedFiles = this.searchedFiles.Where(x => selectedFilePaths.Contains(x.FileFullPath));
                selectedFiles.ForEach(f => this.SelectedIndices.Add(f.Index));
            }
        }

        /// <summary>
        /// 列のソート方法が変わったときや列を表示しなくなったときのソート処理を行います。
        /// </summary>
        private void UpdateList_Resort()
        {
            // 選択中のファイルのパスを復元用に取得
            var selectedFiles = this.SelectedFiles.ToArray();

            // 再ソートを実行
            this.unfilteredFiles = this.SortByColumnState(this.unfilteredFiles).ToArray();

            // 検索ワードでフィルタしたファイルのリストを更新
            // 順番が変わるだけで数は変わらないはずなのでファイルモニタは更新しない
            this.searchedFiles = this.SearchByCurrentWord(this.unfilteredFiles).ToArray();

            // 選択を復元する
            // 仕組み上、復元するのは親の選択状態に限る
            using (new LockWindowUpdateBlock(this))
            using (new ControlEventSuppressBlock())
            {
                for (int i = 0; i < this.searchedFiles.Length; ++i)
                {
                    this.searchedFiles[i].Index = i;
                }

                this.SelectedIndices.Clear();

                foreach (FileInfo f in selectedFiles)
                {
                    this.SelectedIndices.Add(f.Index);
                }
            }

            this.UpdateListAllFiles(false);
        }

        /// <summary>
        /// エフェクトメーカーで編集するデータの情報を読み込みます。
        /// </summary>
        /// <param name="onFinished">読み込みが完了したときの処理</param>
        private void UpdateList_EditDataLoad(Action onFinished)
        {
            var sw = new Stopwatch();
            sw.Start();

#if DEBUG
            int fileCount = 0;
            Logger.Log(LogLevels.Debug, "Beginning EBFileListView.UpdateList_EditDataLoad...");
            var profileTimer = new ProfileTimer("EBFileListView.UpdateList_EditDataLoad");
#endif

            var targetFiles = this.openedFiles.Where(x => x.IsInternalLoaded == false);

            Action<FileInfo> onLoadedAction = (fileInfo) =>
            {
                ++this.LoadedEditDataCount;

                // ファイルロードの進捗を表示します。
#if DEBUG
                FileInfo[] targetFileArray = targetFiles.ToArray();
                fileCount++;
                StringBuilder message = new StringBuilder();
                int progress = 0;
                int interval = Math.Max(1, targetFileArray.Length / 10);
                if (targetFileArray.Length != 0 && (fileCount % interval == 0 || fileCount == targetFileArray.Length))
                {
                    progress = (int)Math.Round(100.0 * fileCount / targetFileArray.Length);

                    message.Append("EBFileListView.UpdateList_Search ");
                    message.Append(progress.ToString() + "%");
                    message.Append(" ");
                    message.Append(this.LoadedEditDataCount.ToString());
                    message.Append("/");
                    message.Append(targetFileArray.Length.ToString());

                    Logger.Log(message.ToString(), LogLevels.Profile);
                }

                if (fileCount == targetFileArray.Length)
                {
                    profileTimer.Stop();
                }
#endif

                if (sw.ElapsedMilliseconds > 500)
                {
                    this.Invalidate();
                    sw.Restart();
                }

                this.RegisterThumbnailMoniter(fileInfo);
            };

            Action onFinishedAction = () =>
            {
                this.Invalidate();
                onFinished();
            };

            this.LoadedEditDataCount = 0;

　          this.editDataLoader.StartLoading(targetFiles, onLoadedAction, onFinishedAction);
        }

        /// <summary>
        /// アセットデータの情報を読み込みます。
        /// </summary>
        /// <param name="onFinished">読み込みが完了したときの処理</param>
        private void UpdateList_AssetsLoad(Action onFinished)
        {
            Stopwatch sw = new Stopwatch();
            sw.Start();

            var targetFiles = this.openedFiles.Where(x => x.IsInternalLoaded == false);

            Action<FileInfo> onLoadedAction = (fileInfo) =>
            {
                ++this.LoadedAssetDataCount;

                if (sw.ElapsedMilliseconds > 500)
                {
                    this.Invalidate();
                    sw.Restart();
                }

                this.RegisterThumbnailMoniter(fileInfo);
            };

            Action onFinishedAction = () =>
            {
                this.Invalidate();
                onFinished();
            };

            this.LoadedAssetDataCount = 0;

            this.assetDataLoader.StartLoading(targetFiles, onLoadedAction, onFinishedAction);
        }

        /// <summary>
        /// 開閉
        /// </summary>
        private void UpdateList_CloseOpen()
        {
            this.UpdateListAllFiles(false);
        }

        /// <summary>
        /// allFiles を元に更新する
        /// </summary>
        /// <param name="isFirstItemSelect">
        /// The is first item select.
        /// </param>
        private void UpdateListAllFiles(bool isFirstItemSelect)
        {
            this.openedFiles = this.searchedFiles.SelectMany(x => x.IsOpend ? x.AllItems : new[] { x }).ToArray();
            this.ItemsCount = this.openedFiles.Count();

            if (isFirstItemSelect && this.openedFiles.Any())
            {
                this.SelectedIndices.Clear();
                this.SelectedIndices.Add(0);
                this.Items[0].Focused = true;
            }

            this.Invalidate();
        }

        /// <summary>
        /// The cancel async job.
        /// </summary>
        private void CancelAsyncJob()
        {
            this.editDataLoader.Cancel();
            this.assetDataLoader.Cancel();

            this.RunUpdate(UpdateStateKind.Wait);
        }

        /// <summary>
        /// 監視中のファイルに変更があったときの処理を行います。
        /// </summary>
        /// <param name="path">変更されたパス</param>
        /// <param name="userData">ユーザーデータ</param>
        /// <param name="changeType">変更の種類</param>
        private void OnFileChanged(string path, object userData, WatcherChangeTypes changeType)
        {
            // ファイルを再列挙する
            this.RunUpdate(UpdateStateKind.FileEnum);
        }

        /// <summary>
        /// 監視中のサムネイルファイルに変更があったときの処理を行います。
        /// </summary>
        /// <param name="path">変更されたパス</param>
        /// <param name="userData">ユーザーデータ</param>
        /// <param name="changeType">変更の種類</param>
        private void OnThumbnailFileChanged(string path, object userData, WatcherChangeTypes changeType)
        {
            // サムネイル画像のキャッシュを破棄
            FileInfo fileInfo = (FileInfo)userData;
            FileCacheManager.Instance.Discard(fileInfo.FileFullPath);

            // ファイルを再列挙する
            this.RunUpdate(UpdateStateKind.FileEnum);
        }

        /// <summary>
        /// ファイルの監視登録を行います。
        /// </summary>
        /// <param name="info">ファイル情報</param>
        private void RegisterFileMonitor(FileInfo info)
        {
            this.monitorManager.RegisterMonitor(info.FileFullPath, null, this.OnFileChanged);

            if (info.IsInternalLoaded)
            {
                this.RegisterThumbnailMoniter(info);
            }
        }

        /// <summary>
        /// サムネイルファイルの監視登録を行います。
        /// </summary>
        /// <param name="info">ファイル情報</param>
        private void RegisterThumbnailMoniter(FileInfo info)
        {
            var cache = FileCacheManager.Instance.Get(info.FileFullPath, true);

            if (string.IsNullOrEmpty(cache.ThumnailFilePath) == false)
            {
                info.ThumnailFilePath = cache.ThumnailFilePath;
            }

            if (string.IsNullOrEmpty(info.ThumnailFilePath) == false)
            {
                this.monitorManager.RegisterMonitorPreCopyPath(info.ThumnailFilePath, info, this.OnThumbnailFileChanged);
            }
        }

        /// <summary>
        /// ファイル監視を解除します。
        /// </summary>
        /// <param name="info">ファイル情報</param>
        private void UnregisterMonitor(FileInfo info)
        {
            // ファイルの監視を解除
            this.monitorManager.UnregisterMonitor(info.FileFullPath);

            // プリミティブ用のサムネイルファイルの監視を解除
            if (!string.IsNullOrEmpty(info.ThumnailFilePath))
            {
                this.monitorManager.UnregisterMonitor(info.ThumnailFilePath);
            }
        }
    }
}
