﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace EffectMaker.UIControls.EffectBrowser.Controls.Basic
{
    using System;
    using System.Drawing;
    using System.Drawing.Drawing2D;
    using System.Drawing.Imaging;
    using System.Windows.Forms;

    using EffectMaker.UIControls.EffectBrowser.Utilities;

    /// <summary>
    /// The eb text box.
    /// </summary>
    public class EBTextBox : TextBox
    {
        #region Fields

        /// <summary>
        /// The back image.
        /// </summary>
        private Bitmap backImage;

        /// <summary>
        /// The background icon.
        /// </summary>
        private Image backgroundIcon;

        /// <summary>
        /// The background icon layout.
        /// </summary>
        private BackgroundIconLayoutKind backgroundIconLayout;

        /// <summary>
        /// The is draw background.
        /// </summary>
        private bool isDrawBackground = true;

        /// <summary>
        /// The prompt text.
        /// </summary>
        private string promptText = string.Empty;

        #endregion

        #region Constructors and Destructors

        /// <summary>
        /// Initializes a new instance of the <see cref="EBTextBox"/> class.
        /// </summary>
        public EBTextBox()
        {
            this.DoubleBuffered = true;
        }

        #endregion

        #region Enums

        /// <summary>
        /// The background icon layout kind.
        /// </summary>
        public enum BackgroundIconLayoutKind
        {
            /// <summary>
            /// The left.
            /// </summary>
            Left,

            /// <summary>
            /// The right.
            /// </summary>
            Right
        }

        #endregion

        #region Public Properties

        /// <summary>
        /// Gets or sets the background icon.
        /// </summary>
        public Image BackgroundIcon
        {
            get
            {
                return this.backgroundIcon;
            }

            set
            {
                this.backgroundIcon = value;
                this.Invalidate();
            }
        }

        /// <summary>
        /// Gets or sets the background icon layout.
        /// </summary>
        public BackgroundIconLayoutKind BackgroundIconLayout
        {
            get
            {
                return this.backgroundIconLayout;
            }

            set
            {
                this.backgroundIconLayout = value;
                this.Invalidate();
            }
        }

        /// <summary>
        /// Gets or sets the prompt text.
        /// </summary>
        public string PromptText
        {
            get
            {
                return this.promptText;
            }

            set
            {
                this.promptText = value;
                this.backImage = null;
                this.Invalidate();
            }
        }

        #endregion

        #region Properties

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                return !ControlEventSuppressBlock.Enabled && base.CanRaiseEvents;
            }
        }

        /// <summary>
        /// Gets or sets a value indicating whether double buffered.
        /// </summary>
        protected override sealed bool DoubleBuffered
        {
            get
            {
                return base.DoubleBuffered;
            }

            set
            {
                base.DoubleBuffered = value;
            }
        }

        #endregion

        #region Methods

        /// <summary>
        /// The on font changed.
        /// </summary>
        /// <param name="e">
        /// イベント引数
        /// </param>
        protected override void OnFontChanged(EventArgs e)
        {
            base.OnFontChanged(e);

            this.Invalidate();
        }

        /// <summary>
        /// The wnd proc.
        /// </summary>
        /// <param name="m">
        /// The m.
        /// </param>
        protected override void WndProc(ref Message m)
        {
            const int WmPaint = 15;
            const int WmSetfocus = 7;
            const int WmKillfocus = 8;

            switch (m.Msg)
            {
                case WmSetfocus:
                    this.isDrawBackground = false;
                    break;

                case WmKillfocus:
                    this.isDrawBackground = true;
                    break;
            }

            base.WndProc(ref m);

            if (m.Msg == WmPaint)
            {
                if (this.isDrawBackground && string.IsNullOrEmpty(this.Text))
                {
                    using (var g = this.CreateGraphics())
                    {
                        this.DrawBack(g);
                    }
                }
            }
        }

        /// <summary>
        /// The draw back.
        /// </summary>
        /// <param name="g">
        /// The g.
        /// </param>
        private void DrawBack(Graphics g)
        {
            if (this.backImage == null)
            {
                this.backImage = new Bitmap(this.Width, this.Height, PixelFormat.Format24bppRgb);

                using (var bg = Graphics.FromImage(this.backImage))
                {
                    bg.InterpolationMode = InterpolationMode.HighQualityBicubic;
                    bg.SmoothingMode = SmoothingMode.HighQuality;

                    bg.Clear(Color.White);
                    this.DrawBackgroundIcon(bg);
                    this.DrawPrompt(bg);
                }
            }

            g.DrawImage(this.backImage, 0, 0);
        }

        /// <summary>
        /// The draw background icon.
        /// </summary>
        /// <param name="g">
        /// The g.
        /// </param>
        private void DrawBackgroundIcon(Graphics g)
        {
            if (this.BackgroundIcon == null)
            {
                return;
            }

            var x = (this.BackgroundIconLayout == BackgroundIconLayoutKind.Left)
                        ? 0
                        : this.ClientRectangle.Width - this.BackgroundIcon.Width;
            var y = (this.ClientRectangle.Height - this.BackgroundIcon.Height) / 2;

            g.DrawImage(this.BackgroundIcon, new Point(x, y));
        }

        /// <summary>
        /// The draw prompt.
        /// </summary>
        /// <param name="g">
        /// The g.
        /// </param>
        private void DrawPrompt(Graphics g)
        {
            if (string.IsNullOrEmpty(this.PromptText))
            {
                return;
            }

            if (string.IsNullOrEmpty(this.Text) == false)
            {
                return;
            }

            using (var promptFont = new Font(this.Font, FontStyle.Italic))
            {
                const TextFormatFlags LeftFlag = TextFormatFlags.Left | TextFormatFlags.VerticalCenter;

                TextRenderer.DrawText(
                    g,
                    this.PromptText,
                    promptFont,
                    this.ClientRectangle,
                    Color.Gray,
                    Color.White,
                    LeftFlag);
            }
        }

        #endregion
    }
}
