﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using EffectMaker.UIDialogs.GeneralControls;

namespace EffectMaker.UIDialogs.MessageDialogs
{
    /// <summary>
    /// Enum for extended message box buttons.
    /// </summary>
    public enum MessageBoxButtonsEx
    {
        /// <summary>OK button.</summary>
        OK = 1,

        /// <summary>OK and Detail buttons.</summary>
        OKDetail,

        /// <summary>OK and Cancel buttons.</summary>
        OKCancel,

        /// <summary>OK, Cancel and Detail buttons.</summary>
        OKCancelDetail,

        /// <summary>OK, Retry and Ignore buttons.</summary>
        AbortRetryIgnore,

        /// <summary>Yes, No and Cancel buttons.</summary>
        YesNoCancel,

        /// <summary>Yes and No buttons.</summary>
        YesNo,

        /// <summary>Retry and Cancel buttons.</summary>
        RetryCancel,
    }

    /// <summary>
    /// メッセージボックスダイアログクラス。
    /// </summary>
    public partial class MessageBox : ModalDialog
    {
        /// <summary>Flag indicating whether the message box is an optional message box.</summary>
        private bool isOptionalMessageBox = false;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="message">The message to show on the message box.</param>
        /// <param name="buttons">The buttons to show on the message box.</param>
        /// <param name="icon">The icon for the message box.</param>
        /// <param name="defaultButton">The default button of the message box.</param>
        private MessageBox(
            string message,
            MessageBoxButtonsEx buttons,
            MessageBoxIcon icon,
            MessageBoxDefaultButton defaultButton)
        {
            this.InitializeComponent();

            Debug.Assert(message != null, "The message must be not null.");

            // タイトル
            this.Text = Application.ProductName;

            // アイコン＆サウンド効果
            switch (icon)
            {
                //// 情報
                case MessageBoxIcon.Information:
                    this.pctIcon.Image = SystemIcons.Information.ToBitmap();
                    this.SoundEffect = DialogSoundEffect.Information;
                    break;
                //// 警告
                case MessageBoxIcon.Warning:
                    this.pctIcon.Image = SystemIcons.Warning.ToBitmap();
                    this.SoundEffect = DialogSoundEffect.Warning;
                    break;
                //// エラー
                case MessageBoxIcon.Error:
                    this.pctIcon.Image = SystemIcons.Error.ToBitmap();
                    this.SoundEffect = DialogSoundEffect.Error;
                    break;
                //// 問い合わせ
                case MessageBoxIcon.Question:
                    this.pctIcon.Image = SystemIcons.Question.ToBitmap();
                    this.SoundEffect = DialogSoundEffect.Question;
                    break;
                default:
                    break;
            }

            // メッセージ
            // ２行以下の場合は上部にマージンを設定する
            this.lblMessage.Text = message;
            {
                string[] split = message.Split('\n');
                if (split.Length <= 2)
                {
                    Padding margin = this.lblMessage.Margin;
                    margin.Top = split.Length == 1 ? 8 : 4;
                    this.lblMessage.Margin = margin;
                }
            }

            // ボタン
            switch (buttons)
            {
                case MessageBoxButtonsEx.OK:
                    this.btnResult1.Text = Properties.Resources.ControlsDialogResultOk;
                    this.btnResult1.DialogResult = DialogResult.OK;
                    this.btnResult2.Visible = false;
                    this.btnResult2.Enabled = false;
                    this.btnResult3.Visible = false;
                    this.btnResult3.Enabled = false;
                    this.CancelButton = this.btnResult1;
                    break;
                case MessageBoxButtonsEx.OKDetail:
                    this.btnResult1.Text = Properties.Resources.ControlsDialogResultOk;
                    this.btnResult1.DialogResult = DialogResult.OK;
                    this.btnResult2.Text = Properties.Resources.ControlsDialogResultDetail;
                    this.btnResult2.DialogResult = DialogResult.None;
                    this.btnResult3.Visible = false;
                    this.btnResult3.Enabled = false;
                    this.CancelButton = this.btnResult1;

                    this.btnResult2.Click += (s, e) =>
                    {
                        if (DetailButtonClicked != null)
                        {
                            DetailButtonClicked(this, e);
                        }
                    };

                    break;
                case MessageBoxButtonsEx.OKCancel:
                    this.btnResult1.Text = Properties.Resources.ControlsDialogResultIgnore;
                    this.btnResult1.DialogResult = DialogResult.OK;
                    this.btnResult2.Text = Properties.Resources.ControlsDialogResultCancel;
                    this.btnResult2.DialogResult = DialogResult.Cancel;
                    this.btnResult3.Visible = false;
                    this.btnResult3.Enabled = false;
                    this.CancelButton = this.btnResult2;
                    break;
                case MessageBoxButtonsEx.YesNo:
                    this.btnResult1.Text = Properties.Resources.ControlsDialogResultYes;
                    this.btnResult1.DialogResult = DialogResult.Yes;
                    this.btnResult2.Text = Properties.Resources.ControlsDialogResultNo;
                    this.btnResult2.DialogResult = DialogResult.No;
                    this.btnResult3.Visible = false;
                    this.btnResult3.Enabled = false;
                    this.CancelButton = this.btnResult2;
                    break;
                case MessageBoxButtonsEx.YesNoCancel:
                    this.btnResult1.Text = Properties.Resources.ControlsDialogResultYes;
                    this.btnResult1.DialogResult = DialogResult.Yes;
                    this.btnResult2.Text = Properties.Resources.ControlsDialogResultNo;
                    this.btnResult2.DialogResult = DialogResult.No;
                    this.btnResult3.Text = Properties.Resources.ControlsDialogResultCancel;
                    this.btnResult3.DialogResult = DialogResult.Cancel;
                    this.CancelButton = this.btnResult3;
                    break;
                case MessageBoxButtonsEx.RetryCancel:
                    this.btnResult1.Text = Properties.Resources.ControlsDialogResultRetry;
                    this.btnResult1.DialogResult = DialogResult.Retry;
                    this.btnResult2.Text = Properties.Resources.ControlsDialogResultCancel;
                    this.btnResult2.DialogResult = DialogResult.Cancel;
                    this.btnResult3.Visible = false;
                    this.btnResult3.Enabled = false;
                    this.CancelButton = this.btnResult2;
                    break;
                case MessageBoxButtonsEx.AbortRetryIgnore:
                    this.btnResult1.Text = Properties.Resources.ControlsDialogResultAbort;
                    this.btnResult1.DialogResult = DialogResult.Abort;
                    this.btnResult2.Text = Properties.Resources.ControlsDialogResultRetry;
                    this.btnResult2.DialogResult = DialogResult.Retry;
                    this.btnResult3.Text = Properties.Resources.ControlsDialogResultIgnore;
                    this.btnResult3.DialogResult = DialogResult.Ignore;
                    this.CancelButton = this.btnResult3;
                    break;
                default: break;
            }

            // デフォルトボタン
            switch (defaultButton)
            {
                case MessageBoxDefaultButton.Button1:
                    this.AcceptButton = this.btnResult1;
                    break;
                case MessageBoxDefaultButton.Button2:
                    this.AcceptButton = this.btnResult2.Visible ? this.btnResult2 : this.btnResult1;
                    break;
                case MessageBoxDefaultButton.Button3:
                    this.AcceptButton = this.btnResult3.Visible ? this.btnResult3 : this.btnResult1;
                    break;
                default:
                    break;
            }

            // MessageBoxは常に最前面に表示する.
            this.TopMost = true;
        }

        /// <summary>
        /// Triggered when the message box is showing.
        /// </summary>
        public static event EventHandler<MessageBoxShowingEventArgs> MessageBoxShowing = null;

        /// <summary>
        /// Triggered when the message box is closing.
        /// </summary>
        public static event CancelEventHandler MessageBoxClosing = null;

        /// <summary>
        /// Triggered when the detail button is clicked.
        /// </summary>
        public static event EventHandler DetailButtonClicked = null;

        /// <summary>
        /// Get or set the flag indicating whether this message box is an optional message box.
        /// </summary>
        public bool IsOptionalMessageBox
        {
            get
            {
                return this.isOptionalMessageBox;
            }

            set
            {
                this.isOptionalMessageBox = value;
                this.chkEnableMsgBox.Visible = value;
            }
        }

        /// <summary>
        /// Get the flag indicating whether the message box should be displayed next time.
        /// </summary>
        public bool ShouldShowMessageBoxNextTime
        {
            get
            {
                if (this.IsOptionalMessageBox == false)
                {
                    return true;
                }
                else
                {
                    return !this.chkEnableMsgBox.Checked;
                }
            }
        }

        /// <summary>
        /// Assign Click event handler for the detail button.
        /// </summary>
        /// <param name="handler">The event handler.</param>
        public static void AssignDetailButtonClickHandler(EventHandler handler)
        {
            DetailButtonClicked = handler;
        }

        /// <summary>
        /// Show the message box.
        /// </summary>
        /// <param name="message">The message to show on the message box.</param>
        /// <param name="buttons">The buttons to show on the message box.</param>
        /// <param name="icon">The icon for the message box.</param>
        /// <param name="defaultButton">The default button of the message box.</param>
        /// <returns>The return value of the message box.</returns>
        public static DialogResult Show(
            string message,
            MessageBoxButtons buttons,
            MessageBoxIcon icon,
            MessageBoxDefaultButton defaultButton)
        {
            MessageBoxButtonsEx msgButtons = MessageBoxButtonsEx.OK;

            switch (buttons)
            {
                case MessageBoxButtons.OK:
                    msgButtons = MessageBoxButtonsEx.OK;
                    break;
                case MessageBoxButtons.OKCancel:
                    msgButtons = MessageBoxButtonsEx.OKCancel;
                    break;
                case MessageBoxButtons.AbortRetryIgnore:
                    msgButtons = MessageBoxButtonsEx.AbortRetryIgnore;
                    break;
                case MessageBoxButtons.YesNo:
                    msgButtons = MessageBoxButtonsEx.YesNo;
                    break;
                case MessageBoxButtons.YesNoCancel:
                    msgButtons = MessageBoxButtonsEx.YesNoCancel;
                    break;
                case MessageBoxButtons.RetryCancel:
                    msgButtons = MessageBoxButtonsEx.RetryCancel;
                    break;
            }

            return Show(message, msgButtons, icon, defaultButton);
        }

        /// <summary>
        /// Show the message box.
        /// </summary>
        /// <param name="message">The message to show on the message box.</param>
        /// <param name="buttons">The buttons to show on the message box.</param>
        /// <param name="icon">The icon for the message box.</param>
        /// <param name="defaultButton">The default button of the message box.</param>
        /// <returns>The return value of the message box.</returns>
        public static DialogResult Show(
            string message,
            MessageBoxButtonsEx buttons,
            MessageBoxIcon icon,
            MessageBoxDefaultButton defaultButton)
        {
            // Create the message box.
            MessageBox dialog = new MessageBox(
                message,
                buttons,
                icon,
                defaultButton);

////            // Show the message box at the center of the screen.
////            Form activeForm = Form.ActiveForm;
////            if (activeForm == null)
////            {
////                dialog.StartPosition = FormStartPosition.CenterScreen;
////            }

            // Show the message box at the center of the screen.
            dialog.StartPosition = FormStartPosition.CenterScreen;

            // Trigger message box showing event.
            if (MessageBoxShowing != null)
            {
                var args = new MessageBoxShowingEventArgs(false);
                MessageBoxShowing(dialog, args);
                if (args.Cancel == true)
                {
                    return args.DialogResultOnCanceling;
                }
            }

            // Show the message box and return the result.
            return dialog.ShowDialog();
        }

        /// <summary>
        /// Handle Shown event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnShown(EventArgs e)
        {
            // ボタンの横方向中央配置
            // 全てのレイアウト調整が完了した後で行う
            if (this.pnlClient.Width > this.pnlButtons.Width)
            {
                Padding margin = this.pnlButtons.Margin;
                margin.Left = (this.pnlClient.Width - this.pnlButtons.Width) / 2;
                this.pnlButtons.Margin = margin;
            }

            base.OnShown(e);
        }

        /// <summary>
        /// Handle FormClosing event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnFormClosing(FormClosingEventArgs e)
        {
            if (e.Cancel == false && MessageBoxClosing != null)
            {
                var args = new CancelEventArgs(false);
                MessageBoxClosing(this, args);
                if (args.Cancel == true)
                {
                    e.Cancel = true;
                }
            }

            base.OnFormClosing(e);
        }

        /// <summary>
        /// Event argument class for MessageBoxShowing event.
        /// </summary>
        public class MessageBoxShowingEventArgs : CancelEventArgs
        {
            /// <summary>
            /// Constructor.
            /// </summary>
            public MessageBoxShowingEventArgs() :
                base()
            {
                this.DialogResultOnCanceling = DialogResult.None;
            }

            /// <summary>
            /// Constructor.
            /// </summary>
            /// <param name="cancel">Flag indicating if should cancel.</param>
            public MessageBoxShowingEventArgs(bool cancel) :
                base(cancel)
            {
                this.DialogResultOnCanceling = DialogResult.None;
            }

            /// <summary>
            /// Get the message box result to return when canceling showing the message box.
            /// </summary>
            public DialogResult DialogResultOnCanceling { get; set; }
        }
    }
}
