﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Diagnostics.CodeAnalysis;
using System.Drawing;
using System.Drawing.Imaging;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using System.Windows.Forms.VisualStyles;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.Foundation.Collections;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Texture;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.EffectBrowser.Data;
using EffectMaker.UIControls.Layout;
using EffectMaker.UIDialogs.MessageDialogs;
using EffectMaker.UIDialogs.Properties;
using PathUtility = EffectMaker.UIControls.EffectBrowser.Utilities.PathUtility;

namespace EffectMaker.UIDialogs.ExportDialog
{
    /// <summary>
    /// アセットファイルのコピー先を選択するダイアログを表示します
    /// </summary>
    public partial class ExportDialog : Form
    {
        /// <summary>
        /// エミッタセットタイプ
        /// </summary>
        private const string EsetStr = "eset";

        /// <summary>
        /// プレビュータイプ
        /// </summary>
        private const string PrevStr = "prev";

        /// <summary>
        /// テクスチャタイプ
        /// </summary>
        private const string FtxbStr = "ftxb";

        /// <summary>
        /// プリミティブタイプ
        /// </summary>
        private const string FmdbStr = "fmdb";

        /// <summary>
        /// コンバイナシェーダタイプ
        /// </summary>
        private const string EcmbStr = "ecmb";

        /// <summary>
        /// オリジナルのパスを保持する辞書
        /// </summary>
        private readonly Dictionary<ListViewItem, string> originalPaths = new Dictionary<ListViewItem, string>();

        /// <summary>
        /// テクスチャアイコン辞書
        /// </summary>
        private readonly Dictionary<ListViewItem, Image> texImages = new Dictionary<ListViewItem, Image>();

        /// <summary>
        /// アイテムソーター
        /// </summary>
        private ListViewItemComparer itemSorter = null;

        /// <summary>
        /// コンバイナエディタが有効か否か
        /// </summary>
        private bool enableCombiner = false;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="targetFiles">
        /// コピー元のファイルリスト
        /// </param>
        public ExportDialog(IEnumerable<string> targetFiles)
        {
            this.InitializeComponent();

            this.enableCombiner = OptionStore.ProjectConfig.IsEftCombinerEditorEnabled;

            this.EsetSaveList = new List<FileCopyInfo>();
            this.PrevSaveList = new List<FileCopyInfo>();
            this.FileCopyList = new List<FileCopyInfo>();

            // ローカライズされたテキストをセット
            this.Text = Resources.ExportDialogCaption;
            this.btnOK.Text = Resources.ControlsDialogResultOk;
            this.btnCancel.Text = Resources.ControlsDialogResultCancel;
            this.destinationLabel.Text = Resources.ExportDialogDestinationFolder;
            this.typeCheckButton.Text = Resources.ExportDialogCheck;
            this.fileListLabel.Text = Resources.ExportDialogFileListView;
            this.typeClearButton.Text = Resources.ExportDialogClear;
            this.fileCountLabel.Text = Resources.ExportDialogFileCount;
            this.optionGroup.Text = Resources.ExportDialogSaveOption;
            this.textureCheckBox.Text = Resources.ExportDialogTextures;
            this.primitiveCheckBox.Text = Resources.ExportDialogPrimitives;

            this.combinerCheckBox.Text = Resources.ExportDialogCombiners;
            if (!this.enableCombiner)
            {
                int moveH = this.combinerCheckBox.Height + this.combinerCheckBox.Margin.Top;

                this.optionGroup.Controls.Remove(this.combinerCheckBox);
                this.optionGroup.Height -= moveH;

                var loc = this.optionGroup.Location;
                loc.Y += moveH;
                this.optionGroup.Location = loc;

                var size = this.fileListView.Size;
                size.Height += moveH;
                this.fileListView.Size = size;

                var sizeC = this.fileCountView.Size;
                sizeC.Height += moveH;
                this.fileCountView.Size = sizeC;
            }

            // ドロップダウンボタンにアイテムを追加
            this.AddDropDown(this.typeCheckButton, Resources.ExportDialogAllFiles, null, 0, true);
            this.AddDropDown(this.typeCheckButton, EsetStr, Resources.Icon_EmitterSet, 1, true);
            this.AddDropDown(this.typeCheckButton, PrevStr, Resources.Icon_PreviewSetting, 2, true);
            this.AddDropDown(this.typeCheckButton, FtxbStr, Resources.Icon_Textures, 3, true);
            this.AddDropDown(this.typeCheckButton, FmdbStr, Resources.Icon_Primitives, 4, true);
            if (this.enableCombiner)
            {
                this.AddDropDown(this.typeCheckButton, EcmbStr, Resources.Icon_Primitives, 5, true);
            }

            this.AddDropDown(this.typeClearButton, Resources.ExportDialogAllFiles, null, 0, false);
            this.AddDropDown(this.typeClearButton, EsetStr, Resources.Icon_EmitterSet, 1, false);
            this.AddDropDown(this.typeClearButton, PrevStr, Resources.Icon_PreviewSetting, 2, false);
            this.AddDropDown(this.typeClearButton, FtxbStr, Resources.Icon_Textures, 3, false);
            this.AddDropDown(this.typeClearButton, FmdbStr, Resources.Icon_Primitives, 4, false);
            if (this.enableCombiner)
            {
                this.AddDropDown(this.typeClearButton, EcmbStr, Resources.Icon_Primitives, 5, true);
            }

            this.fileCountView.View = View.Details;
            this.fileCountView.HeaderStyle = ColumnHeaderStyle.None;
            this.fileCountView.Columns.Add(string.Empty);
            this.fileCountView.Columns.Add(string.Empty);
            this.fileCountView.Columns.Add(string.Empty);
            this.fileCountView.Columns.Add(string.Empty);
            this.fileCountView.Columns[0].TextAlign = HorizontalAlignment.Left;
            this.fileCountView.Columns[1].TextAlign = HorizontalAlignment.Right;
            this.fileCountView.Columns[2].TextAlign = HorizontalAlignment.Center;
            this.fileCountView.Columns[3].TextAlign = HorizontalAlignment.Right;
            this.fileCountView.Columns[0].Width = 40;
            this.fileCountView.Columns[1].Width = 30;
            this.fileCountView.Columns[2].Width = 20;
            this.fileCountView.Columns[3].Width = 30;
            string[] row = { EsetStr, "0", "/", "0" };
            this.fileCountView.Items.Add(new ListViewItem(row));
            this.fileCountView.Items.Add(new ListViewItem(row));
            this.fileCountView.Items.Add(new ListViewItem(row));
            this.fileCountView.Items.Add(new ListViewItem(row));
            this.fileCountView.Items.Add(new ListViewItem(row));
            this.fileCountView.Items[0].SubItems[0].Text = EsetStr + ":";
            this.fileCountView.Items[1].SubItems[0].Text = PrevStr + ":";
            this.fileCountView.Items[2].SubItems[0].Text = FtxbStr + ":";
            this.fileCountView.Items[3].SubItems[0].Text = FmdbStr + ":";
            if (this.enableCombiner)
            {
                this.fileCountView.Items.Add(new ListViewItem(row));
                this.fileCountView.Items[4].SubItems[0].Text = EcmbStr + ":";
                this.fileCountView.Items[5].SubItems[0].Text = Resources.ExportDialogTotal;
            }
            else
            {
                this.fileCountView.Items[4].SubItems[0].Text = Resources.ExportDialogTotal;
            }

            // ファイルリストビューのカラム設定
            this.fileListView.Columns.Add(Resources.ExportDialogFileName);
            this.fileListView.Columns.Add(Resources.ExportDialogFileType);
            this.fileListView.Columns.Add(Resources.ExportDialogTimestamp);
            this.fileListView.Columns.Add(Resources.ExportDialogFilePath);
            this.fileListView.Columns.Add(string.Empty);
            this.fileListView.Columns[0].Width = 180;
            this.fileListView.Columns[1].Width = 70;
            this.fileListView.Columns[2].Width = 200;
            this.fileListView.Columns[3].Width = 250;

            // 初期状態記憶用非表示列
            this.fileListView.Columns[4].Width = 0;
            this.fileListView.ColumnWidthChanging += (s, e) =>
            {
                // 幅0からリサイズ禁止
                if (e.ColumnIndex == 4)
                {
                    e.NewWidth = 0;
                    e.Cancel = true;
                }
            };

            // ソートの仕込み
            this.itemSorter = new ListViewItemComparer
            {
                ColumnModes = new[]
                {
                    ListViewItemComparer.ComparerMode.String,
                    ListViewItemComparer.ComparerMode.String,
                    ListViewItemComparer.ComparerMode.String,
                    ListViewItemComparer.ComparerMode.String,
                    ListViewItemComparer.ComparerMode.Integer,
                }
            };
            this.fileListView.ListViewItemSorter = this.itemSorter;
            this.fileListView.ColumnClick += (s, e) =>
            {
                this.itemSorter.Column = e.Column;
                this.fileListView.Sort();
                ColumnUtil.UpdateSortIcon(this.fileListView, e.Column, this.itemSorter.Order);
            };

            this.fileListView.CheckBoxes = true;

            // OSに任せず自力で描画
            this.fileListView.OwnerDraw = true;

            // と言いつつカラムヘッダとアイテム全体の描画はデフォルト
            this.fileListView.DrawColumnHeader += (s, e) => e.DrawDefault = true;
            this.fileListView.DrawItem += (s, e) => e.DrawDefault = true;

            // サブアイテムの描画だけオーバーライド
            this.fileListView.DrawSubItem += this.OnDrawSubItem;

            // ファイルリストビューへのアイテム追加
            ListViewItem currentEsetItem = null;

            // esetアイテムのTagにはアセットアイテムのリストを保持する。
            // prev,ftxb,fmdbアイテムのTagには所属するesetアイテムを保持する。
            // 大元のソースのパスはDictionaryに保持する。
            int count = 0;
            foreach (var targetFile in targetFiles)
            {
                var fileName = Path.GetFileName(targetFile);
                var extName = Path.GetExtension(targetFile).Remove(0, 1).ToLower();

                string[] row1 =
                {
                    fileName,
                    extName,
                    string.Empty,
                    string.Empty,
                    count.ToString("D"),
                };
                var item = new ListViewItem(row1)
                {
                    UseItemStyleForSubItems = false
                };

                if (extName != EsetStr)
                {
                    for (int i = 0; i < 4; ++i)
                    {
                        item.SubItems[i].ForeColor = Color.Gray;
                    }
                }

                if (extName == EsetStr)
                {
                    item.Checked = true;
                    item.Tag = new List<ListViewItem>();
                    currentEsetItem = item;
                }
                else if (extName == PrevStr)
                {
                    item.Tag = currentEsetItem;
                }
                else if (extName == FtxbStr || extName == FmdbStr || extName == EcmbStr)
                {
                    // ファイルが存在しなかったらスキップ
                    if (!File.Exists(targetFile))
                    {
                        continue;
                    }

                    // 同じエミッタセット内に同一アセットがある場合はスキップ
                    bool sameAsset = this.fileListView.Items.OfType<ListViewItem>().Any(
                        regItem => regItem.Tag == currentEsetItem && this.originalPaths[regItem] == targetFile);

                    if (sameAsset)
                    {
                        continue;
                    }

                    if (extName == FtxbStr)
                    {
                        this.texImages.Add(item, this.LoadTexture(targetFile));
                    }

                    item.Tag = currentEsetItem;
                }

                this.originalPaths[item] = targetFile;
                this.fileListView.Items.Add(item);
                ++count;
            }

            // リストビュー更新
            this.UpdateFileListView();

            // 個数カウント更新
            this.UpdateFileNum();

            // 出力先コンボボックスの一覧更新
            this.destinationComboBox.Items.AddRange(
                OptionStore.RootOptions.FileEvent.ExportFolderPathList.ToArray());
            if (this.destinationComboBox.Items.Count > 0)
            {
                this.destinationComboBox.SelectedIndex = 0;
            }

            // アイテムのチェック状態変更時のハンドラを追加
            this.fileListView.ItemChecked += this.OnChecked;

            this.destinationComboBox.PromptText = Properties.Resources.ExportDialogDestinationComboBoxComment;
        }

        /// <summary>
        /// 保存するエミッタセットのリスト
        /// </summary>
        public List<FileCopyInfo> EsetSaveList { get; set; }

        /// <summary>
        /// 保存するプレビューのリスト
        /// </summary>
        public List<FileCopyInfo> PrevSaveList { get; set; }

        /// <summary>
        /// コピーするアセットのリスト
        /// </summary>
        public List<FileCopyInfo> FileCopyList { get; set; }

        /// <summary>
        /// ダイアログクローズ時のフック処理
        /// </summary>
        /// <param name="e">イベント</param>
        protected override void OnClosing(CancelEventArgs e)
        {
            // 結果コード別ハンドラ呼び出し
            switch (this.DialogResult)
            {
                case DialogResult.OK:
                    e.Cancel = !this.OnResultOk();
                    break;
                case DialogResult.Cancel:
                    e.Cancel = !this.OnResultCancel();
                    break;
                default:
                    e.Cancel = !this.OnResultCancel();
                    break;
            }

            base.OnClosing(e);
        }

        /// <summary>
        /// esetのチェック状態を変更した際のアセット項目表示非表示切り替え
        /// </summary>
        /// <param name="clickedItemIndex">クリックされたアイテムのインデックス</param>
        private void UpdateAssetVisibility(int clickedItemIndex)
        {
            var item = this.fileListView.Items[clickedItemIndex];
            if (item.SubItems[1].Text == EsetStr)
            {
                var assetItems = (List<ListViewItem>)item.Tag;

                if (item.Checked)
                {
                    // チェックがついたesetに関連するアセットをリストに復帰
                    assetItems.Reverse();
                    foreach (var assetItem in assetItems)
                    {
                        this.fileListView.Items.Add(assetItem);
                    }

                    this.fileListView.Sort();
                    assetItems.Clear();
                }
                else
                {
                    // チェックがついたesetに関連するアセットを非表示
                    foreach (var assetItem in this.fileListView.Items.OfType<ListViewItem>())
                    {
                        if (assetItem.SubItems[1].Text == FtxbStr ||
                            assetItem.SubItems[1].Text == FmdbStr ||
                            assetItem.SubItems[1].Text == EcmbStr)
                        {
                            if (assetItem.Tag == item)
                            {
                                assetItems.Add(assetItem);
                            }
                        }
                    }

                    foreach (var assetItem in assetItems)
                    {
                        this.fileListView.Items.Remove(assetItem);
                    }
                }
            }
        }

        /// <summary>
        /// ファイルリストビューを更新します。
        /// </summary>
        private void UpdateFileListView()
        {
            foreach (var item in this.fileListView.Items.OfType<ListViewItem>())
            {
                if (item.Checked)
                {
                    var basePath = this.destinationComboBox.Text;
                    if (item.SubItems[1].Text == FtxbStr)
                    {
                        if (this.textureCheckBox.Checked)
                        {
                            basePath = Path.Combine(basePath, IOConstants.TextureFolderName);
                        }
                    }
                    else if (item.SubItems[1].Text == FmdbStr)
                    {
                        if (this.primitiveCheckBox.Checked)
                        {
                            basePath = Path.Combine(basePath, IOConstants.PrimitiveFolderName);
                        }
                    }
                    else if (item.SubItems[1].Text == EcmbStr)
                    {
                        if (this.combinerCheckBox.Checked)
                        {
                            basePath = Path.Combine(basePath, IOConstants.CombinerShaderFolderName);
                        }
                    }

                    item.SubItems[3].Text = Path.Combine(basePath, item.SubItems[0].Text);

                    if (File.Exists(item.SubItems[3].Text))
                    {
                        var timestamp = File.GetCreationTime(item.SubItems[3].Text);
                        item.SubItems[2].ForeColor = Color.Red;
                        item.SubItems[2].Text =
                            timestamp.ToShortDateString() + " " + timestamp.ToLongTimeString();
                    }
                    else
                    {
                        item.SubItems[2].Text = string.Empty;
                    }
                }
                else
                {
                    if (item.SubItems[1].Text == FtxbStr ||
                        item.SubItems[1].Text == FmdbStr ||
                        item.SubItems[1].Text == EcmbStr)
                    {
                        var srcPath = this.originalPaths[item];
                        var timestamp = File.GetLastWriteTime(srcPath);
                        item.SubItems[2].ForeColor = Color.Gray;
                        item.SubItems[2].Text =
                            timestamp.ToShortDateString() + " " + timestamp.ToLongTimeString();
                        item.SubItems[3].Text = srcPath;
                    }
                    else
                    {
                        item.SubItems[2].ForeColor = Color.Gray;
                        item.SubItems[2].Text = string.Empty;
                        item.SubItems[3].Text = string.Empty;
                    }
                }
            }
        }

        /// <summary>
        /// ファイルの総数とチェックされた個数を更新します。
        /// </summary>
        private void UpdateFileNum()
        {
            int esetNum = 0, esetChecked = 0;
            int prevNum = 0, prevChecked = 0;
            int ftxbNum = 0, ftxbChecked = 0;
            int fmdbNum = 0, fmdbChecked = 0;
            int ecmbNum = 0, ecmbChecked = 0;
            int totalNum = 0, totalChecked = 0;

            foreach (var menuItem in this.typeCheckButton.DropDownMenu.Items.OfType<ToolStripItem>())
            {
                menuItem.Enabled = false;
            }

            foreach (var menuItem in this.typeClearButton.DropDownMenu.Items.OfType<ToolStripItem>())
            {
                menuItem.Enabled = false;
            }

            foreach (var item in this.fileListView.Items.OfType<ListViewItem>())
            {
                switch (item.SubItems[1].Text)
                {
                    case EsetStr:
                        this.typeCheckButton.DropDownMenu.Items[1].Enabled = true;
                        this.typeClearButton.DropDownMenu.Items[1].Enabled = true;
                        ++esetNum;
                        if (item.Checked)
                        {
                            ++esetChecked;
                        }

                        break;
                    case PrevStr:
                        this.typeCheckButton.DropDownMenu.Items[2].Enabled = true;
                        this.typeClearButton.DropDownMenu.Items[2].Enabled = true;
                        ++prevNum;
                        if (item.Checked)
                        {
                            ++prevChecked;
                        }

                        break;
                    case FtxbStr:
                        this.typeCheckButton.DropDownMenu.Items[3].Enabled = true;
                        this.typeClearButton.DropDownMenu.Items[3].Enabled = true;
                        ++ftxbNum;
                        if (item.Checked)
                        {
                            ++ftxbChecked;
                        }

                        break;
                    case FmdbStr:
                        this.typeCheckButton.DropDownMenu.Items[4].Enabled = true;
                        this.typeClearButton.DropDownMenu.Items[4].Enabled = true;
                        ++fmdbNum;
                        if (item.Checked)
                        {
                            ++fmdbChecked;
                        }

                        break;
                    case EcmbStr:
                        if (this.enableCombiner)
                        {
                            this.typeCheckButton.DropDownMenu.Items[5].Enabled = true;
                            this.typeClearButton.DropDownMenu.Items[5].Enabled = true;
                            ++ecmbNum;
                            if (item.Checked)
                            {
                                ++ecmbChecked;
                            }
                        }

                        break;
                }
            }

            totalChecked = esetChecked + prevChecked + ftxbChecked + fmdbChecked + ecmbChecked;
            totalNum = esetNum + prevNum + ftxbNum + fmdbNum + ecmbNum;

            if (totalNum > 0)
            {
                this.typeCheckButton.DropDownMenu.Items[0].Enabled = true;
                this.typeClearButton.DropDownMenu.Items[0].Enabled = true;
            }

            this.fileCountView.Items[0].SubItems[1].Text = esetChecked.ToString("D");
            this.fileCountView.Items[0].SubItems[3].Text = esetNum.ToString("D");
            this.fileCountView.Items[1].SubItems[1].Text = prevChecked.ToString("D");
            this.fileCountView.Items[1].SubItems[3].Text = prevNum.ToString("D");
            this.fileCountView.Items[2].SubItems[1].Text = ftxbChecked.ToString("D");
            this.fileCountView.Items[2].SubItems[3].Text = ftxbNum.ToString("D");
            this.fileCountView.Items[3].SubItems[1].Text = fmdbChecked.ToString("D");
            this.fileCountView.Items[3].SubItems[3].Text = fmdbNum.ToString("D");
            if (this.enableCombiner)
            {
                this.fileCountView.Items[4].SubItems[1].Text = ecmbChecked.ToString("D");
                this.fileCountView.Items[4].SubItems[3].Text = ecmbNum.ToString("D");
                this.fileCountView.Items[5].SubItems[1].Text = totalChecked.ToString("D");
                this.fileCountView.Items[5].SubItems[3].Text = totalNum.ToString("D");
            }
            else
            {
                this.fileCountView.Items[4].SubItems[1].Text = totalChecked.ToString("D");
                this.fileCountView.Items[4].SubItems[3].Text = totalNum.ToString("D");
            }
        }

        /// <summary>
        /// OKボタンを押した時の処理.
        /// </summary>
        /// <returns>そのままダイアログをクローズする場合はtrue,それでは困る場合はfalse.</returns>
        private bool OnResultOk()
        {
            if (Directory.Exists(this.destinationComboBox.Text))
            {
                this.EsetSaveList.Clear();
                this.PrevSaveList.Clear();
                this.FileCopyList.Clear();

                foreach (var item in this.fileListView.Items.OfType<ListViewItem>())
                {
                    if (item.Checked)
                    {
                        var srcPath = this.originalPaths[item];
                        var dstPath = item.SubItems[3].Text;
                        var pathPair = new FileCopyInfo(srcPath, dstPath);

                        switch (item.SubItems[1].Text)
                        {
                            case EsetStr:
                                this.EsetSaveList.Add(pathPair);
                                break;
                            case PrevStr:
                                this.PrevSaveList.Add(pathPair);
                                break;
                            case FtxbStr:
                            case FmdbStr:
                            case EcmbStr:
                                var result = this.CheckAssetOverwriting(pathPair.SrcPath, pathPair.DstPath);
                                switch (result)
                                {
                                    case DialogResult.Yes:
                                        this.FileCopyList.Add(pathPair);
                                        break;
                                    case DialogResult.Cancel:
                                        return false;
                                }

                                break;
                        }
                    }
                }

                OptionStore.RootOptions.FileEvent.AddExportFolderPath(this.destinationComboBox.Text);
            }
            else
            {
                ThreadSafeMsgBox.Show(
                    Resources.ExportDialogWaringPathNotFound,
                    Resources.ExportDialogWaringCaption,
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Warning);

                return false;
            }

            return true;
        }

        /// <summary>
        /// キャンセルボタンを押した時の処理.
        /// </summary>
        /// <returns>そのままダイアログをクローズする場合はtrue,それでは困る場合はfalse.</returns>
        private bool OnResultCancel()
        {
            return true;
        }

        /// <summary>
        /// タイムスタンプチェックと上書き確認ダイアログのダブルチェック
        /// </summary>
        /// <param name="srcPath">コピー元パス</param>
        /// <param name="dstPath">コピー先パス</param>
        /// <returns>コピーが許可された場合はYes, 許可しない場合はNo, キャンセルされたらCancel.</returns>
        private DialogResult CheckAssetOverwriting(string srcPath, string dstPath)
        {
            if (!File.Exists(dstPath))
            {
                return DialogResult.Yes;
            }

            var srcTimestamp = File.GetLastWriteTime(srcPath);
            var dstTimestamp = File.GetLastWriteTime(dstPath);

            if (srcTimestamp.CompareTo(dstTimestamp) == 0)
            {
                return DialogResult.Yes;
            }

            return ThreadSafeOverwriteAssetDialog.Show(srcPath, srcTimestamp, dstPath, dstTimestamp);
        }

        /// <summary>
        /// テクスチャイメージの取得
        /// </summary>
        /// <param name="fileFullPath">ファイルパス</param>
        /// <returns>取得したイメージ(失敗したらnull)</returns>
        private Image LoadTexture(string fileFullPath)
        {
            var resultWithData = BusinessLogic.Manager.TextureManager.Instance.LoadTexture(fileFullPath, true);

            if (resultWithData.ResultCode != LoadTextureResultCode.Success)
            {
                return null;
            }

            var combinedImage = new Bitmap(Constants.IconSize * 2, Constants.IconSize, PixelFormat.Format32bppArgb);
            {
                System.Diagnostics.Debug.Assert(resultWithData.TextureData is FtxTextureData, "ftxbが読めていない。");
                var ftxTarget = resultWithData.TextureData as FtxTextureData;
                var rgbMatrix = RenderUtility.CreateRgbColorMatrix(ftxTarget.CompSel, ftxTarget.PixelFormat);
                var alphaMatrix = RenderUtility.CreateAlphaColorMatrix(ftxTarget.CompSel, ftxTarget.PixelFormat);

                var iconSize = new Size(Constants.IconSize, Constants.IconSize);
                var srcImage = ftxTarget.GeneratePreviewBitmap(iconSize);
                var rgbImage = TextureData.ResizeBitmap(srcImage, iconSize, rgbMatrix);
                rgbImage = ColorUtility.ConvertBitmapWithGamma(
                    rgbImage, rgbMatrix, true, resultWithData.TextureData.PixelFormat.IsSrgb());
                var alphaImage = TextureData.ResizeBitmap(srcImage, iconSize);
                alphaImage = ColorUtility.ConvertBitmapWithGamma(
                    alphaImage, alphaMatrix, false, resultWithData.TextureData.PixelFormat.IsSrgb());

                using (var g = Graphics.FromImage(combinedImage))
                {
                    g.DrawImage(rgbImage, 0, 0);
                    g.DrawImage(alphaImage, Constants.IconSize, 0);
                }
            }

            return combinedImage;
        }

        /// <summary>
        /// ドロップイベント
        /// </summary>
        /// <param name="sender">sender</param>
        /// <param name="e">event</param>
        private void OnFolderDragDrop(object sender, DragEventArgs e)
        {
            var paths = (string[])e.Data.GetData(DataFormats.FileDrop, false);
            if (paths == null || paths.Length == 0)
            {
                return;
            }

            string path = paths[0];

            if (Directory.Exists(path) == true)
            {
                this.destinationComboBox.Text = path;
            }
            else if (File.Exists(path) == true)
            {
                this.destinationComboBox.Text = Path.GetDirectoryName(path);
            }
        }

        /// <summary>
        /// ドラッグ中イベント
        /// </summary>
        /// <param name="sender">sender</param>
        /// <param name="e">event</param>
        private void OnFolderDragEnter(object sender, DragEventArgs e)
        {
            e.Effect = e.Data.GetDataPresent(DataFormats.FileDrop) ?
                DragDropEffects.Copy : DragDropEffects.None;
        }

        /// <summary>
        /// パス参照ボタンクリック
        /// </summary>
        /// <param name="sender">sender</param>
        /// <param name="e">event</param>
        private void OnPathBrowseButtonClick(object sender, EventArgs e)
        {
            using (var fbd = new FolderBrowserDialog
            {
                SelectedPath = this.destinationComboBox.Text,
                RootFolder = Environment.SpecialFolder.Desktop,
                ShowNewFolderButton = true,
            })
            {
                if (PathUtility.DirectoryExists(this.destinationComboBox.Text))
                {
                    fbd.SelectedPath = this.destinationComboBox.Text;
                }

                if (fbd.ShowDialog() == DialogResult.OK)
                {
                    this.destinationComboBox.Text = fbd.SelectedPath;
                }
            }
        }

        /// <summary>
        /// テクスチャ保存先オプションが変更された場合の処理
        /// </summary>
        /// <param name="sender">sender</param>
        /// <param name="e">event</param>
        private void OnTextureCheckedChanged(object sender, EventArgs e)
        {
            foreach (var item in this.fileListView.Items.OfType<ListViewItem>())
            {
                if (item.SubItems[1].Text == FtxbStr)
                {
                    var basePath = this.destinationComboBox.Text;
                    if (this.textureCheckBox.Checked)
                    {
                        basePath = Path.Combine(basePath, IOConstants.TextureFolderName);
                    }

                    item.SubItems[3].Text = Path.Combine(basePath, item.SubItems[0].Text);
                }
            }

            this.UpdateFileListView();
        }

        /// <summary>
        /// プリミティブの保存先オプションが変更された場合の処理
        /// </summary>
        /// <param name="sender">sender</param>
        /// <param name="e">event</param>
        private void OnPrimitiveCheckedChanged(object sender, EventArgs e)
        {
            foreach (var item in this.fileListView.Items.OfType<ListViewItem>())
            {
                if (item.SubItems[1].Text == FmdbStr)
                {
                    var basePath = this.destinationComboBox.Text;
                    if (this.primitiveCheckBox.Checked)
                    {
                        basePath = Path.Combine(basePath, IOConstants.PrimitiveFolderName);
                    }

                    item.SubItems[3].Text = Path.Combine(basePath, item.SubItems[0].Text);
                }
            }

            this.UpdateFileListView();
        }

        /// <summary>
        /// コンバイナシェーダの保存先オプションが変更された場合の処理
        /// </summary>
        /// <param name="sender">sender</param>
        /// <param name="e">event</param>
        private void OnCombinerCheckedChanged(object sender, EventArgs e)
        {
            foreach (var item in this.fileListView.Items.OfType<ListViewItem>())
            {
                if (item.SubItems[1].Text == EcmbStr)
                {
                    var basePath = this.destinationComboBox.Text;
                    if (this.combinerCheckBox.Checked)
                    {
                        basePath = Path.Combine(basePath, IOConstants.CombinerShaderFolderName);
                    }

                    item.SubItems[3].Text = Path.Combine(basePath, item.SubItems[0].Text);
                }
            }

            this.UpdateFileListView();
        }

        /// <summary>
        /// 保存先パスが変更された場合の処理
        /// </summary>
        /// <param name="sender">sender</param>
        /// <param name="e">event</param>
        private void OnDestinationComboBoxValueChanged(object sender, EventArgs e)
        {
            this.UpdateFileListView();
        }

        /// <summary>
        /// ドロップダウンボタンにメニューアイテムを追加します
        /// </summary>
        /// <param name="dropDownButton">ドロップダウンボタン</param>
        /// <param name="text">ラベル</param>
        /// <param name="icon">アイコン</param>
        /// <param name="id">ID</param>
        /// <param name="check">設定するチェック状態</param>
        private void AddDropDown(DropDownButton dropDownButton, string text, Image icon, int id, bool check)
        {
            dropDownButton.DropDownMenu.Items.Add(text, icon);
            dropDownButton.DropDownMenu.Items[id].Click += (s, e) =>
            {
                foreach (var item in this.fileListView.Items.OfType<ListViewItem>())
                {
                    if (id == 0 || item.SubItems[1].Text == text)
                    {
                        item.Checked = check;
                    }
                }

                foreach (var item in this.fileListView.Items.OfType<ListViewItem>())
                {
                    if (id == 0 || item.SubItems[1].Text == text)
                    {
                        item.Checked = check;
                    }
                }
            };
        }

        /// <summary>
        /// リストビューの内容を描画します。
        /// </summary>
        /// <param name="s">使用しません</param>
        /// <param name="e">描画イベント引数</param>
        private void OnDrawSubItem(object s, DrawListViewSubItemEventArgs e)
        {
            e.DrawDefault = false;

            // Get the bound for the sub item
            Rectangle rect = e.Bounds;
            e.Graphics.FillRectangle(Brushes.White, rect);

            // Draw background color
            if (e.Item.Selected)
            {
                using (var brush = new SolidBrush(Color.FromArgb(255, 240, 240, 240)))
                {
                    e.Graphics.FillRectangle(this.fileListView.Focused ? SystemBrushes.Highlight : brush, rect);
                }
            }

            if (e.ColumnIndex == 0)
            {
                // チェックボックスの描画
                var state = e.Item.Checked ? CheckBoxState.CheckedNormal : CheckBoxState.UncheckedNormal;
                CheckBoxRenderer.DrawCheckBox(e.Graphics, new Point(rect.X, rect.Y + 2), state);

                // チェックボックスぶんシフト
                rect.X += 16;

                // アイテムの種類に応じてアイコンを選択
                string itemType = e.Item.SubItems[1].Text;
                Image icon = null;

                switch (itemType)
                {
                    case EsetStr:
                        icon = Resources.Icon_EmitterSet;
                        break;
                    case PrevStr:
                        icon = Resources.Icon_PreviewSetting;
                        break;
                    case FmdbStr:
                        icon = Resources.Icon_Primitives;
                        break;
                    case FtxbStr:
                        icon = this.texImages[e.Item];
                        break;
                    case EcmbStr:
                        icon = Resources.Icon_Effect_Combiner;
                        break;
                }

                // アイコンを描画
                var iconRect = new Rectangle(rect.X, rect.Y, icon.Width, icon.Height);
                e.Graphics.DrawImage(icon, iconRect);

                // アイコンぶんシフト
                rect.X += icon.Width;
            }

            // Setup string format
            var format = new StringFormat(StringFormatFlags.NoWrap)
            {
                Alignment = StringAlignment.Near,
                LineAlignment = StringAlignment.Center
            };

            // Render the text
            using (var brush = new SolidBrush(e.SubItem.ForeColor))
            {
                e.Graphics.DrawString(
                    e.SubItem.Text,
                    e.SubItem.Font,
                    e.Item.Selected && this.fileListView.Focused ? SystemBrushes.HighlightText : brush,
                    rect,
                    format);
            }
        }

        /// <summary>
        /// アイテムチェック時のイベント
        /// </summary>
        /// <param name="sender">使用しません</param>
        /// <param name="args">チェック状態を変更したアイテム情報</param>
        private void OnChecked(object sender, ItemCheckedEventArgs args)
        {
            this.fileListView.ItemChecked -= this.OnChecked;

            if (args.Item.SubItems[1].Text == EsetStr)
            {
                this.UpdateAssetVisibility(this.fileListView.Items.IndexOf(args.Item));
            }

            this.UpdateFileListView();
            this.UpdateFileNum();

            this.fileListView.ItemChecked += this.OnChecked;
        }
    }
}
