﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Windows.Forms;
using EffectMaker.Foundation.Debugging.Profiling;
using EffectMaker.UIControls.EffectBrowser.Utilities;
using EffectMaker.UIControls.Input;
using WeifenLuo.WinFormsUI.Docking;
using EffectMaker.Foundation.Log;

namespace EffectMaker.UIDialogs.EffectBrowserDialog
{
    /// <summary>
    /// エフェクトブラウザダイアログ
    /// </summary>
    public partial class EffectBrowserDialog : WeifenLuo.WinFormsUI.Docking.DockContent
    {
        /// <summary>
        /// エフェクトブラウザパネル
        /// 作成に時間がかかるのでダイアログのVisibleがtrueになるまで作成を遅らせる
        /// </summary>
        private UIControls.EffectBrowser.EffectBrowserPanel ebpPanel;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public EffectBrowserDialog()
        {
            this.InitializeComponent();

            this.lblInitializing.Text = Properties.Resources.EffectBrowserDialog_Initializing;
            this.Text = Properties.Resources.EffectBrowserTitle;
        }

        /// <summary>
        /// TextureSelected.
        /// </summary>
        public event EventHandler<TextureSelectedEventArgs> TextureSelected
        {
            add
            {
                // パネルが作られていないときは一時的にローカルに記録
                if (this.ebpPanel == null)
                {
                    this.LocalTextureSelected += value;
                }
                else
                {
                    this.ebpPanel.TextureSelected += value;
                }
            }

            remove
            {
                // パネルが作られていないときは一時的にローカルに記録
                if (this.ebpPanel == null)
                {
                    this.LocalTextureSelected -= value;
                }
                else
                {
                    this.ebpPanel.TextureSelected -= value;
                }
            }
        }

        /// <summary>
        /// EsetEditing.
        /// </summary>
        public event EventHandler<EsetEditingEventArgs> EsetEditing
        {
            add
            {
                // パネルが作られていないときは一時的にローカルに記録
                if (this.ebpPanel == null)
                {
                    this.LocalEsetEditing += value;
                }
                else
                {
                    this.ebpPanel.EsetEditing += value;
                }
            }

            remove
            {
                // パネルが作られていないときは一時的にローカルに記録
                if (this.ebpPanel == null)
                {
                    this.LocalEsetEditing -= value;
                }
                else
                {
                    this.ebpPanel.EsetEditing -= value;
                }
            }
        }

        /// <summary>
        /// EsetExporting.
        /// </summary>
        public event EventHandler<EsetExportingEventArgs> EsetExporting
        {
            add
            {
                // パネルが作られていないときは一時的にローカルに記録
                if (this.ebpPanel == null)
                {
                    this.LocalEsetExporting += value;
                }
                else
                {
                    this.ebpPanel.EsetExporting += value;
                }
            }

            remove
            {
                // パネルが作られていないときは一時的にローカルに記録
                if (this.ebpPanel == null)
                {
                    this.LocalEsetExporting -= value;
                }
                else
                {
                    this.ebpPanel.EsetExporting -= value;
                }
            }
        }

        /// <summary>
        /// PrevEditing.
        /// </summary>
        public event EventHandler<PrevEditingEventArgs> PrevEditing
        {
            add
            {
                // パネルが作られていないときは一時的にローカルに記録
                if (this.ebpPanel == null)
                {
                    this.LocalPrevEditing += value;
                }
                else
                {
                    this.ebpPanel.PrevEditing += value;
                }
            }

            remove
            {
                // パネルが作られていないときは一時的にローカルに記録
                if (this.ebpPanel == null)
                {
                    this.LocalPrevEditing -= value;
                }
                else
                {
                    this.ebpPanel.PrevEditing -= value;
                }
            }
        }

        /// <summary>
        /// Workspace opening.
        /// </summary>
        public event EventHandler<WorkspaceOpeningEventArgs> WorkspaceOpening
        {
            add
            {
                // パネルが作られていないときは一時的にローカルに記録
                if (this.ebpPanel == null)
                {
                    this.LocalWorkspaceOpening += value;
                }
                else
                {
                    this.ebpPanel.WorkspaceOpening += value;
                }
            }

            remove
            {
                // パネルが作られていないときは一時的にローカルに記録
                if (this.ebpPanel == null)
                {
                    this.LocalWorkspaceOpening -= value;
                }
                else
                {
                    this.ebpPanel.WorkspaceOpening -= value;
                }
            }
        }

        /// <summary>
        /// The eset editing.
        /// パネルを作るまでの一時保存領域.
        /// </summary>
        private event EventHandler<EsetEditingEventArgs> LocalEsetEditing;

        /// <summary>
        /// The eset exporting.
        /// パネルを作るまでの一時保存領域.
        /// </summary>
        private event EventHandler<EsetExportingEventArgs> LocalEsetExporting;

        /// <summary>
        /// The prev editing.
        /// パネルを作るまでの一時保存領域.
        /// </summary>
        private event EventHandler<PrevEditingEventArgs> LocalPrevEditing;

        /// <summary>
        /// The workspace opening.
        /// パネルを作るまでの一時保存領域.
        /// </summary>
        private event EventHandler<WorkspaceOpeningEventArgs> LocalWorkspaceOpening;

        /// <summary>
        /// The texture selected.
        /// パネルを作るまでの一時保存領域.
        /// </summary>
        private event EventHandler<TextureSelectedEventArgs> LocalTextureSelected;

        /// <summary>
        /// コンフィグファイルの保存
        /// </summary>
        public void SaveConfig()
        {
            if (this.ebpPanel != null)
            {
                this.ebpPanel.SaveConfig();
            }
        }

        /// <summary>
        /// ダイアログが初めて表示されたときの処理を行います。
        /// </summary>
        /// <param name="e">イベント情報</param>
        protected override void OnShown(EventArgs e)
        {
            Logger.Log(LogLevels.Debug, "Beginning EffectBrowserDialog.OnShown...");
            using (new ProfileTimer("EffectBrowserDialog.OnShown"))
            {
                base.OnShown(e);

                // 初めてダイアログを表示するときパネルを作る
                if (this.ebpPanel == null)
                {
                    // パネルを作成する前に描画処理を行い、初期化メッセージを表示する
                    Application.DoEvents();

                    // エフェクトブラウザパネルを作成
                    this.ebpPanel = new EffectMaker.UIControls.EffectBrowser.EffectBrowserPanel
                    {
                        Dock = System.Windows.Forms.DockStyle.Fill,
                        Font = new System.Drawing.Font("Tahoma", 9F),
                        Location = new System.Drawing.Point(0, 0),
                        Name = "ebpPanel",
                        Size = new System.Drawing.Size(784, 562),
                        TabIndex = 0
                    };

                    // ローカルに記録したイベントハンドラを設定
                    this.ebpPanel.TextureSelected += this.LocalTextureSelected;
                    this.ebpPanel.EsetEditing += this.LocalEsetEditing;
                    this.ebpPanel.EsetExporting += this.LocalEsetExporting;
                    this.ebpPanel.PrevEditing += this.LocalPrevEditing;
                    this.ebpPanel.WorkspaceOpening += this.LocalWorkspaceOpening;

                    this.Controls.Add(this.ebpPanel);

                    this.lblInitializing.Visible = false;
                }
            }
        }

        /// <summary>
        /// フローティングウィンドウ時のショートカットキーハンドリング
        /// </summary>
        /// <param name="msg">メッセージ</param>
        /// <param name="keyData">キーデータ</param>
        /// <returns>処理済みならtrueを返す。</returns>
        protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
        {
            if (this.DockState == DockState.Float)
            {
                GlobalKeyEventHandler.Instance.ProcessShortcut(keyData);
            }

            return base.ProcessCmdKey(ref msg, keyData);
        }

        /// <summary>
        /// フォームが閉じる前に呼び出される.
        /// </summary>
        /// <param name="sender">sender.</param>
        /// <param name="e">event.</param>
        private void OnFormClosing(object sender, FormClosingEventArgs e)
        {
            // フォームはクローズしないで、非表示にする.
            this.Hide();

            e.Cancel = true; // キャンセルしないとダイアログが閉じてしまう.
        }
    }
}
