﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Runtime.InteropServices;
using System.Text;
using System.Windows.Forms;
using EffectMaker.BusinessLogic.CurveEditorParameters;
using EffectMaker.BusinessLogic.ViewerMessages;
using EffectMaker.DataModel.AnimationTable;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Render.Layout;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.Extensions;
using EffectMaker.UIControls.Input;
using EffectMaker.UIControls.Layout;
using EffectMaker.UIControls.Specifics.CurveEditor;
using EffectMaker.UIDialogs.Properties;
using EffectMaker.UILogic.ViewModels;
using WeifenLuo.WinFormsUI.Docking;
using Orientation = EffectMaker.UIControls.Layout.Orientation;

namespace EffectMaker.UIDialogs.CurveEditorDialog
{
    /// <summary>
    /// Class for the command history dialog.
    /// </summary>
    /// WeifenLuo.WinFormsUI.Docking.DockContent
    public partial class CurveEditorDialog : WeifenLuo.WinFormsUI.Docking.DockContent
    {
        /// <summary>
        /// コンテキストテーブル
        /// </summary>
        private readonly Dictionary<int, EmitterAnimationEditorParameter> contextTable
            = new Dictionary<int, EmitterAnimationEditorParameter>();

        /// <summary>
        /// アニメ名テーブル
        /// </summary>
        private readonly Dictionary<string, int> nameTable = new Dictionary<string, int>();

        /// <summary>
        /// エディタが選択しているエミッタ
        /// </summary>
        private EmitterViewModel selectedEmitter = null;

        /// <summary>
        /// 選択しているチャンネル
        /// </summary>
        private int selectedChannel = 0;

        /// <summary>
        /// Initializes a new instance of the <see cref="CurveEditorDialog"/> class.
        /// </summary>
        public CurveEditorDialog()
        {
            LayoutEngineBase.SuspendLayout();

            this.InitializeComponent();

            // エディタの初期値
            this.curveEditor.EditorMode = CurveEditor.EditorModes.EmitterTimeAnim;
            this.curveEditor.Enabled = false;
            this.curveListPanel.Enabled = false;
            this.curveEditor.LabelDigit = 2;
            this.curveEditor.ZeroPinMode = -1;

            // バインド設定
            this.curveEditor.AddBinding("Values", "AnimationTable");
            this.curveEditor.ValuesTargetPropertyName = "AnimationTable";
            this.curveEditor.AddBinding("LoopMode", "LoopMode");
            this.curveEditor.LoopModeTargetPropertyName = "LoopMode";
            this.curveEditor.AddBinding("InterMode", "InterpolationMode");
            this.curveEditor.InterModeTargetPropertyName = "InterpolationMode";
            this.curveEditor.AddBinding("ValueChangedExecutable", "OnValueChangedExecutable");
            this.curveEditor.AddBinding("EnableX", "EnableX");
            this.curveEditor.AddBinding("EnableY", "EnableY");
            this.curveEditor.AddBinding("EnableZ", "EnableZ");
            this.curveEditor.AddBinding("EnableW", "EnableW");
            this.curveEditor.AddBinding("EnableOtherKeySnap", "EnableOtherKeySnap");
            this.curveEditor.AddBinding("SnapTimeLevel", "SnapTime");
            this.curveEditor.AddBinding("SnapValueLevel", "SnapValue");
            this.curveEditor.AddBinding("ZeroPinMode", "ZeroPinMode");
            this.curveEditor.AddBinding("ResetExecutable", "OnResetExecutable");

            // エミッタ時間アニメ名とIDのテーブル
            this.nameTable["EmitterShapeScale"] = 1;
            this.nameTable["EmitterTransformPosition"] = 2;
            this.nameTable["EmitterTransformRotation"] = 3;
            this.nameTable["EmitterTransformScale"] = 4;
            this.nameTable["EmitterEmissionRate"] = 5;
            this.nameTable["EmitterEmissionOmniVelocity"] = 6;
            this.nameTable["EmitterEmissionDirectiveVelocity"] = 7;
            this.nameTable["EmitterEmissionGravity"] = 8;
            this.nameTable["EmitterParticleLife"] = 9;
            this.nameTable["EmitterScaleBasic"] = 10;

            EmitterAnimationViewModel.OnEndBlocking = () => this.curveListPanel.Refresh();

            this.curveListPanel.SetCallbacks(
                this.OnButtonClick,
                this.OnCheckClick,
                (i, t) => this.Select(i, t),
                () => this.selectedChannel);

            // リスト作成
            this.curveListPanel.AddSection(Resources.CurveEditorDialogEmitter);
            this.curveListPanel.AddChannel(Resources.CurveEditorDialogEmitterShape, Resources.CurveEditorDialogScaleElement, 1);
            this.curveListPanel.AddChannel(Resources.CurveEditorDialogEmitterTransform, Resources.CurveEditorDialogPosition, 2);
            this.curveListPanel.AddChannel(Resources.CurveEditorDialogEmitterTransform, Resources.CurveEditorDialogRotation, 3);
            this.curveListPanel.AddChannel(Resources.CurveEditorDialogEmitterTransform, Resources.CurveEditorDialogScaleElement, 4);
            this.curveListPanel.AddSection(Resources.CurveEditorDialogEmission);
            this.curveListPanel.AddChannel(Resources.CurveEditorDialogEmissionTiming, Resources.CurveEditorDialogEmissionRate, 5);
            this.curveListPanel.AddChannel(Resources.CurveEditorDialogInitVelocity, Resources.CurveEditorDialogOmniVelocity, 6);
            this.curveListPanel.AddChannel(Resources.CurveEditorDialogInitVelocity, Resources.CurveEditorDialogDirectionVelocity, 7);
            this.curveListPanel.AddChannel(Resources.CurveEditorDialogOtherSettings, Resources.CurveEditorDialogGravity, 8);
            this.curveListPanel.AddSection(Resources.CurveEditorDialogParticle);
            this.curveListPanel.AddChannel(Resources.CurveEditorDialogParticleLife, Resources.CurveEditorDialogLife, 9);
            this.curveListPanel.AddSection(Resources.CurveEditorDialogScale);
            this.curveListPanel.AddChannel(Resources.CurveEditorDialogBasicScale, Resources.CurveEditorDialogBasis, 10);

            this.curveListPanel.SizeChanged += (s, e) =>
            {
                foreach (var panel in this.curveListPanel.Controls.OfType<StackPanel>())
                {
                    var subPanel = panel.Controls.FirstOrDefault(c => c is StackPanel) as StackPanel;
                    if (subPanel != null)
                    {
                        panel.MinimumSize = new Size(this.curveListPanel.Width -2, 26);
                        subPanel.MinimumSize = new Size(this.curveListPanel.Width - 2 - 52 - 12, 26);
                        subPanel.Margin = new Padding(0, 0, 0, 3);
                    }
                }
            };

            this.KeyDown += (s, e) =>
            {
                if (e.KeyCode == Keys.Up)
                {
                    int next = this.selectedChannel - 1;
                    if (next < 0)
                    {
                        next = this.nameTable.Values.Max();
                    }

                    this.Select(next, true);
                }
                else if (e.KeyCode == Keys.Down)
                {
                    int next = this.selectedChannel + 1;
                    if (next > this.nameTable.Values.Max())
                    {
                        next = 0;
                    }

                    this.Select(next, true);
                }
            };

            LayoutEngineBase.ResumeLayout();
        }

        /// <summary>
        /// The set data context.
        /// </summary>
        /// <param name="dataContext">
        /// The data context.
        /// </param>
        public void AddDataContext(object dataContext)
        {
            var param = dataContext as EmitterAnimationEditorParameter;
            if (param == null)
            {
                throw new InvalidDataException();
            }

            // DataContextがnullで渡されてきたアニメは無効化する
            if (param.DataContext == null)
            {
                this.curveListPanel.SetButtonState(this.nameTable[param.AnimationName], false, false);
                return;
            }

            int itemHash = param.DataContext.GetHashCode();
            if (this.contextTable.ContainsKey(itemHash) == false)
            {
                this.contextTable.Add(itemHash, param);
            }

            this.curveListPanel.SetButtonState(
                this.nameTable[param.AnimationName],
                IsEnabledAnimation((ViewModelBase)param.DataContext));

            if ((EmitterAnimationViewModel.IsInitializing || ExportableViewModel.IsPasting)
                && this.selectedChannel != this.nameTable[param.AnimationName])
            {
                return;
            }

            this.Select(this.nameTable[param.AnimationName], false);
        }

        /// <summary>
        /// The set data context.
        /// </summary>
        /// <param name="dataContext">
        /// The data context.
        /// </param>
        public void RemoveDataContext(object dataContext)
        {
            var param = dataContext as EmitterAnimationEditorParameter;
            if (param == null)
            {
                throw new InvalidDataException();
            }
            else if (param.DataContext == null)
            {
                return;
            }

            int itemHash = param.DataContext.GetHashCode();
            if (!this.contextTable.ContainsKey(itemHash))
            {
                return;
            }

            this.curveListPanel.SetButtonState(this.nameTable[param.AnimationName], false);

            if (this.selectedChannel == this.nameTable[param.AnimationName] ||
                this.selectedEmitter == null)
            {
                this.UpdateChannelList();
                this.Deselect();
                this.ClearEditor();
            }

            if (this.selectedEmitter == null)
            {
                this.curveEditor.UnderlayCurveDrawer.ClearUnderlayCurve(true);
            }

            this.contextTable.Remove(itemHash);
        }

        /// <summary>
        /// 選択されたエミッタを通知する
        /// </summary>
        /// <param name="dataContext">選択されたエミッタのViewModel</param>
        public void NotifyEmitter(object dataContext)
        {
            if (dataContext is EmitterViewModel)
            {
                this.selectedEmitter = dataContext as EmitterViewModel;
            }
            else if (dataContext is FieldViewModel)
            {
                var field = dataContext as FieldViewModel;
                if (field.Parent is EmitterViewModel)
                {
                    this.selectedEmitter = field.Parent as EmitterViewModel;
                }
            }
            else if (dataContext is ReservedShaderNodeViewModel)
            {
                var vm = dataContext as ReservedShaderNodeViewModel;
                if (vm.Parent is EmitterViewModel)
                {
                    this.selectedEmitter = vm.Parent as EmitterViewModel;
                }
            }
            else if (dataContext is CustomActionViewModel)
            {
                var field = dataContext as CustomActionViewModel;
                if (field.Parent is EmitterViewModel)
                {
                    this.selectedEmitter = field.Parent as EmitterViewModel;
                }
            }
            else
            {
                this.selectedEmitter = null;
                this.curveEditor.UnderlayCurveDrawer.ClearUnderlayCurve(true);
                this.UpdateChannelList();
                this.Deselect();
                this.ClearEditor();

                return;
            }

            this.contextTable.Clear();

            using (new ControlUpdateBlocker(this))
            {
                this.selectedEmitter.UpdateEmitterAnimation();
            }

            this.UpdateChannelList();
            this.NotifyTabPage(null);
        }

        /// <summary>
        /// タブの選択に併せてリストの選択状態を更新
        /// </summary>
        /// <param name="dataContext">データコンテキスト</param>
        public void NotifyTabPage(object dataContext)
        {
            if (dataContext != null)
            {
                int itemHash = dataContext.GetHashCode();
                if (this.contextTable.ContainsKey(itemHash))
                {
                    var param = this.contextTable[itemHash];
                    if (this.nameTable.ContainsKey(param.AnimationName))
                    {
                        this.Select(this.nameTable[param.AnimationName], false);
                        return;
                    }
                }
            }

            var selected = this.selectedEmitter.SelectedPropertyPage;
            if (selected == this.selectedEmitter.EmitterEmitterViewModel)
            {
                // 既にエミッタタブ内のアニメを選択済みなら変更しない
                if (1 <= this.selectedChannel && this.selectedChannel <= 4)
                {
                    return;
                }

                // エミッタタブ内のエミッタ時間アニメを末尾からチェックし、有効なものを選択対象とする
                int selectTarget = -1;
                for (int i = 4; i >= 1; --i)
                {
                    if (this.curveListPanel.GetButtonState(i))
                    {
                        selectTarget = i;
                    }
                }

                // 有効なアニメがなかったら、タブ内の先頭要素を選択対象とする
                this.Select(selectTarget != -1 ? selectTarget : 1, false);
            }
            else if (selected == this.selectedEmitter.EmitterEmissionViewModel)
            {
                // 同じく、既に放出タブ内のアニメを選択済みなら変更しない
                if (5 <= this.selectedChannel && this.selectedChannel <= 8)
                {
                    return;
                }

                // 放出タブ内のエミッタ時間アニメを末尾からチェックし、有効なものを選択対象とする
                int selectTarget = -1;
                for (int i = 8; i >= 5; --i)
                {
                    if (this.curveListPanel.GetButtonState(i))
                    {
                        selectTarget = i;
                    }
                }

                // 有効なアニメがなかったら、タブ内の先頭要素を選択対象とする
                this.Select(selectTarget != -1 ? selectTarget : 5, false);
            }
            else if (selected == this.selectedEmitter.EmitterParticleViewModel)
            {
                this.Select(9, false);
            }
            else if (selected == this.selectedEmitter.EmitterScaleViewModel)
            {
                this.Select(10, false);
            }
            else
            {
                this.Deselect();
                this.curveListPanel.Invalidate();
            }
        }

        /// <summary>
        /// The on load.
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        protected override void OnLoad(EventArgs e)
        {
            this.Parent.Parent.ClientSize = new Size(960, 480);
            this.Parent.ClientSize = new Size(960, 480);
            this.ClientSize = new Size(960, 480);
            this.curveEditor.ResetViewport();

            base.OnLoad(e);
        }

        /// <summary>
        /// The on visible changed.
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        protected override void OnVisibleChanged(EventArgs e)
        {
            base.OnVisibleChanged(e);

            if (this.selectedChannel == 0)
            {
                this.ClearEditor();
                this.curveEditor.ResetViewport();
            }
            else
            {
                this.curveEditor.AdjustViewport(false);
            }
        }

        /// <summary>
        /// コマンドキーが押されたときの処理を処理を行います。
        /// </summary>
        /// <param name="msg">ウィンドウメッセージ</param>
        /// <param name="keyData">キー値</param>
        /// <returns>コマンドキーが処理された場合はtrue、それ以外の場合はfalseを返します。</returns>
        protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
        {
            // フローティングウィンドウ時にグローバルなショートカットキー処理を行う
            if (this.DockState == DockState.Float)
            {
                GlobalKeyEventHandler.Instance.ProcessShortcut(keyData);
            }

            return base.ProcessCmdKey(ref msg, keyData);
        }

        /// <summary>
        /// フォームを閉じるときの前処理を行います。
        /// </summary>
        /// <param name="e">イベント情報</param>
        protected override void OnFormClosing(FormClosingEventArgs e)
        {
            base.OnFormClosing(e);

            // フォームを再利用するため、フォーム非表示にしてクローズ処理をキャンセルする
            this.Hide();
            e.Cancel = true;
        }

        /// <summary>
        /// 編集対象のカーブを切り替え
        /// </summary>
        /// <param name="param">
        /// The data context.
        /// </param>
        /// <param name="noResize">スケールのリサイズをしたくない場合はtrue.</param>
        private void SwitchEditor(EmitterAnimationEditorParameter param, bool noResize = false)
        {
            if (this.curveEditor.LogicalTreeElementExtender.DataContext == param.DataContext)
            {
                return;
            }

            bool oldBlockingToggleProperty = this.curveEditor.IsBlockingToggleProperty;
            this.curveEditor.IsBlockingToggleProperty = true;

            this.curveEditor.Enabled = true;
            this.curveEditor.IntegerOnly = param.IntegerOnly;

            this.curveEditor.NormalizeAt = param.NormalizeAt;
            this.curveEditor.LabelDigit = param.LabelDigit;
            this.curveEditor.LabelPrefix = param.LabelPrefix;
            this.curveEditor.MaxLimit = param.MaxLimit;
            this.curveEditor.MinLimit = param.MinLimit;
            this.curveEditor.DefaultValue = param.DefaultValue;
            this.curveEditor.Channels = param.Channels.Select(channelName => new CurveChannel { Name = channelName }).ToList();

            this.curveEditor.LogicalTreeElementExtender.DataContext = param.DataContext;
            this.curveEditor.Initialize(noResize);
            this.curveEditor.DefaultZeroPin = param.DefaultZeroPin;

            this.curveEditor.IsBlockingToggleProperty = oldBlockingToggleProperty;

            if (!noResize)
            {
                this.curveEditor.AdjustViewport(true);
            }
        }

        /// <summary>
        /// エディタの内容をクリアする
        /// </summary>
        private void ClearEditor()
        {
            bool oldBlockingToggleProperty = this.curveEditor.IsBlockingToggleProperty;
            this.curveEditor.IsBlockingToggleProperty = true;

            this.curveEditor.Enabled = false;
            this.curveEditor.LogicalTreeElementExtender.ClearDataContext();
            this.curveEditor.Channels.Clear();
            this.curveEditor.ZeroPinMode = -1;
            this.curveEditor.LoopMode = 0;
            this.curveEditor.InterMode = 0;
            this.curveEditor.Initialize(true);
            this.curveEditor.Values = new AnimationTableData(0, 0.5f, 0.5f, 0.5f, 0.5f);

            this.curveEditor.IsBlockingToggleProperty = oldBlockingToggleProperty;
        }

        /// <summary>
        /// カーブを選択した際の処理
        /// </summary>
        /// <param name="id">選択しようとしているチャンネル</param>
        /// <param name="toggle">既に選択されている場合は解除するか否か</param>
        /// <param name="keepingScale">スケールを維持する切り替えか否か</param>
        private void Select(int id, bool toggle, bool keepingScale = true)
        {
            if (this.selectedChannel == id && toggle)
            {
                id = 0;
            }

            bool onlyRequestDraw = this.selectedChannel == id && id != 0;

            using (new ControlUpdateBlocker(this, () =>
            {
                this.curveListPanel.Refresh();
                if (onlyRequestDraw)
                {
                    this.curveEditor.RequestDraw(true);
                }
                else
                {
                    this.curveEditor.Refresh();
                }
            }))
            {
                foreach (var stp in this.curveListPanel.Controls.OfType<StackPanel>())
                {
                    var panel = stp.Controls.OfType<StackPanel>().First();
                    panel.BackColor = id == (int)panel.Tag ?
                        Color.FromArgb(176, 196, 222) : Color.Transparent;
                }

                this.selectedChannel = id;

                if (this.selectedEmitter != null)
                {
                    this.SyncTabPage();
                }

                var contextFilter = this.contextTable.Where(
                    x => this.nameTable[x.Value.AnimationName] == this.selectedChannel);
                contextFilter = contextFilter.Where(
                    x => GetEmitterViewModel(x.Value) == this.selectedEmitter);
                foreach (var param in contextFilter)
                {
                    // 現在選択されているカーブに対する再選択(アンドゥ・リドゥ時)は切り替え処理を省いて再描画のみとする
                    if (!onlyRequestDraw)
                    {
                        this.SwitchEditor(param.Value, keepingScale);
                    }

                    this.curveEditor.UpdateResetButton();
                    this.UpdateVisibleCheck();
                    return;
                }

                this.ClearEditor();
                this.UpdateVisibleCheck();
            }
        }

        /// <summary>
        /// カーブの選択解除
        /// </summary>
        private void Deselect()
        {
            foreach (var stp in this.curveListPanel.Controls.OfType<StackPanel>())
            {
                var panel = stp.Controls.OfType<StackPanel>().First();
                panel.BackColor = Color.Transparent;
            }

            this.selectedChannel = 0;
        }

        /// <summary>
        /// カーブの選択に併せてタブページの選択を変更します.
        /// </summary>
        private void SyncTabPage()
        {
            // タブの切り替えに連動したイベントをいったん潰しておく
            var tmp = WorkspaceRootViewModel.Instance.Dialogs.OnNotifyTabPageToCurveEditorExecutable;
            WorkspaceRootViewModel.Instance.Dialogs.OnNotifyTabPageToCurveEditorExecutable = null;

            switch (this.selectedChannel)
            {
                case 1:
                case 2:
                case 3:
                case 4:
                    this.selectedEmitter.SelectedPropertyPage
                        = this.selectedEmitter.EmitterEmitterViewModel;
                    break;
                case 5:
                case 6:
                case 7:
                case 8:
                    this.selectedEmitter.SelectedPropertyPage
                        = this.selectedEmitter.EmitterEmissionViewModel;
                    break;
                case 9:
                    this.selectedEmitter.SelectedPropertyPage
                        = this.selectedEmitter.EmitterParticleViewModel;
                    break;
                case 10:
                    this.selectedEmitter.SelectedPropertyPage
                        = this.selectedEmitter.EmitterScaleViewModel;
                    break;
            }

            WorkspaceRootViewModel.Instance.Dialogs.OnNotifyTabPageToCurveEditorExecutable = tmp;
        }

        /// <summary>
        /// 現在のエミッタにあわせてリストの内容を更新する
        /// </summary>
        private void UpdateChannelList()
        {
            var stackPanels = this.curveListPanel.Controls.OfType<StackPanel>();
            foreach (var subCtl in stackPanels.SelectMany(ctl => ctl.Controls).OfType<UIImageButton>())
            {
                subCtl.Checked = false;
            }

            this.Deselect();

            if (this.selectedEmitter == null)
            {
                this.curveListPanel.Enabled = false;
                return;
            }

            this.curveListPanel.Enabled = true;

            var updatedList = new List<int>();
            foreach (var context in this.contextTable)
            {
                var name = context.Value.AnimationName;
                var evm = GetEmitterViewModel(context.Value);
                if (this.selectedEmitter == evm)
                {
                    updatedList.Add(this.nameTable[name]);
                    this.curveListPanel.SetButtonState(
                        this.nameTable[name],
                        IsEnabledAnimation((ViewModelBase)context.Value.DataContext));
                }
            }

            // 上のループで処理されなかった項目は無効化されているのでこちらで処理する
            for (int i = this.nameTable.Values.Min(); i <= this.nameTable.Values.Max(); ++i)
            {
                if (updatedList.Contains(i))
                {
                    continue;
                }

                this.curveListPanel.SetButtonState(i, false, false);
            }
        }

        /// <summary>
        /// チェックボックスの状態に基いて薄く表示するカーブを更新します
        /// </summary>
        private void UpdateVisibleCheck()
        {
            this.curveEditor.UnderlayCurveDrawer.ClearUnderlayCurve(true);
            var panels = this.curveListPanel.Controls.OfType<StackPanel>();
            foreach (var panel in panels)
            {
                foreach (var check in panel.Controls.OfType<UICheckBox>())
                {
                    this.OnCheckClick(check, (int)check.Tag);
                }
            }
        }

        /// <summary>
        /// チェックボックスがクリックされた際の処理
        /// </summary>
        /// <param name="checkBox">クリックされたチェックボックス</param>
        /// <param name="id">クリックされたチャンネルID</param>
        private void OnCheckClick(UICheckBox checkBox, int id)
        {
            using (new ControlUpdateBlocker(this, () => this.curveEditor.RequestDraw()))
            {
                var contextFilter = this.contextTable.Where(
                    pair => GetEmitterViewModel(pair.Value) == this.selectedEmitter);
                var keyValuePairs = contextFilter as IList<KeyValuePair<int, EmitterAnimationEditorParameter>> ?? contextFilter.ToArray();
                if (!keyValuePairs.Any())
                {
                    return;
                }

                var target = keyValuePairs.FirstOrDefault(
                    pair => pair.Value.AnimationName == this.nameTable.First(x => x.Value == id).Key);
                if (target.Value == null)
                {
                    return;
                }

                var vm = (ViewModelBase)target.Value.DataContext;
                var table = vm.GetDataModelValue<AnimationTableData>("AnimationTable");
                if (checkBox.Checked && this.selectedChannel != id)
                {
                    this.curveEditor.UnderlayCurveDrawer.AddUnderlayCurve(
                        vm.GetHashCode(),
                        table,
                        target.Value.Channels.Count,
                        vm.GetDataModelValue<int>("InterpolationMode"));
                }
                else
                {
                    this.curveEditor.UnderlayCurveDrawer.RemoveUnderlayCurve(vm.GetHashCode());
                }

                if (id == this.selectedChannel)
                {
                    if (checkBox.Checked)
                    {
                        this.SwitchEditor(target.Value, true);
                    }
                    else
                    {
                        this.ClearEditor();
                    }
                }
            }
        }

        /// <summary>
        /// アニメON/OFFボタンクリック時の処理
        /// </summary>
        /// <param name="check">パネル内のチェックボックス</param>
        /// <param name="button">押したボタン</param>
        /// <param name="id">押したボタンのチャンネルID</param>
        private void OnButtonClick(UICheckBox check, UIImageButton button, int id)
        {
            using (new ControlUpdateBlocker(this, () =>
            {
                this.curveListPanel.Refresh();
                this.curveEditor.Refresh();
            }))
            {
                switch (id)
                {
                    case 1:
                        this.selectedEmitter.EmitterEmitterViewModel.EmitterEmitterShapeViewModel
                            .EnableShapeScaleAnimation = button.Checked;
                        break;
                    case 2:
                        this.selectedEmitter.EmitterEmitterViewModel.EmitterEmitterTransformViewModel
                            .EnablePositionAnimation = button.Checked;
                        break;
                    case 3:
                        this.selectedEmitter.EmitterEmitterViewModel.EmitterEmitterTransformViewModel
                            .EnableRotationAnimation = button.Checked;
                        break;
                    case 4:
                        this.selectedEmitter.EmitterEmitterViewModel.EmitterEmitterTransformViewModel
                            .EnableScaleAnimation = button.Checked;
                        break;
                    case 5:
                        this.selectedEmitter.EmitterEmissionViewModel.EmitterEmissionTimingViewModel
                            .EnableEmitTimeRateAnimation = button.Checked;
                        break;
                    case 6:
                        this.selectedEmitter.EmitterEmissionViewModel.EmitterEmissionVelocityViewModel
                            .EnableOmniVelocityAnimation = button.Checked;
                        break;
                    case 7:
                        this.selectedEmitter.EmitterEmissionViewModel.EmitterEmissionVelocityViewModel
                            .EnableOrientedDirectionalVelocityAnimation = button.Checked;
                        break;
                    case 8:
                        this.selectedEmitter.EmitterEmissionViewModel.EmitterEmissionGravityViewModel
                            .EnableGravityAnimation = button.Checked;
                        break;
                    case 9:
                        this.selectedEmitter.EmitterParticleViewModel.EmitterParticleLifeViewModel
                            .EnableLifeAnimation = button.Checked;
                        break;
                    case 10:
                        this.selectedEmitter.EmitterScaleViewModel.EmitterScaleBasicViewModel
                            .EnableAnimation = button.Checked;
                        break;
                }

                check.Checked = button.Checked;
            }
        }

        /// <summary>
        /// 全てのカーブを平行移動させるダイアログを表示します。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void OnShiftAllCurvesButtonClicked(object sender, EventArgs e)
        {
            var dialog = new CurveShiftDialog();

            // ダイアログを表示
            DialogResult dialogResult = dialog.ShowDialog(this);

            if (dialogResult != DialogResult.OK)
            {
                return;
            }

            int shiftX = dialog.ShiftX;

            // カーブを平行移動させる
            foreach (var item in this.contextTable)
            {
                EmitterAnimationViewModel animVM = item.Value.DataContext as EmitterAnimationViewModel;

                if (animVM == null)
                {
                    continue;
                }

                if (animVM.AnimationTable == null)
                {
                    continue;
                }

                AnimationTableData animation = new AnimationTableData();

                var keys = animVM.AnimationTable.ToArray();

                foreach (KeyFrameData key in keys)
                {
                    animation.AddKeyFrame(key.Frame + shiftX, key.Value.X, key.Value.Y, key.Value.Z, key.Value.W, false);
                }

                // ViewModel のデータを変更することで変更通知が発生する
                // TODO: カーブA、B、CがあってAがアクティブな場合、シフト後にUndoするとカーブAの
                // 履歴が全部なくなるまで B、C の Undo の順番が回ってこない件の対応
                animVM.AnimationTable = animation;
            }

            // Select() 内で Draw 処理も行っているためチャンネルを再セレクトして画面を更新する
            // 選択中のチャンネルのカーブは更新されるが、未選択の薄く表示しているカーブが更新されないのでその対処
            {
                int selectedChannnel = this.selectedChannel;
                this.Deselect();
                this.Select(0, true);
                this.Select(selectedChannnel, true);
            }
        }

        /// <summary>
        /// エミッタアニメの親エミッタViewModelを得る
        /// </summary>
        /// <param name="param">アニメーションパラメータ</param>
        /// <returns>親エミッタViewModel</returns>
        private static EmitterViewModel GetEmitterViewModel(EmitterAnimationEditorParameter param)
        {
            if (param.DataContext == null)
            {
                return null;
            }

            var eavm = (HierarchyViewModel)param.DataContext;
            return eavm.Parent.Parent.Parent as EmitterViewModel;
        }

        /// <summary>
        /// アニメーションが有効になっているViewModelかどうかを判定する
        /// </summary>
        /// <param name="viewModel">エミッタ時間アニメを持つViewModel</param>
        /// <returns>有効ならtrue</returns>
        private static bool IsEnabledAnimation(ViewModelBase viewModel)
        {
            var flag = false;
            var model = viewModel as EmitterAnimationViewModel;
            if (model != null)
            {
                flag = model.EnableAnimation;
            }

            return flag;
        }
    }
}
