﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Globalization;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using EffectMaker.Foundation.Interfaces;

namespace EffectMaker.UIControls.ValueConverters
{
    /// <summary>
    /// An IValueConverter that converts CheckState to boolean back and forth.
    /// </summary>
    public class UncheckedCheckStateValueConverter : IValueConverter
    {
        /// <summary>
        /// Convert boolean to CheckState, true means Checked and false means Unchecked.
        /// </summary>
        /// <param name="value">The boolean value to convert.
        /// If the value is not a boolean, null is returned.</param>
        /// <param name="targetType">The type to convert to.</param>
        /// <param name="parameter">A custom parameter.</param>
        /// <returns>Returns Checked if the value is true, Unchecked if the value is false,
        /// and null if the value is not of type boolean.</returns>
        public object Convert(
            object value,
            Type targetType,
            object parameter)
        {
            if ((value is bool) == false)
            {
                return null;
            }

            var v = (bool)value;
            return v ? CheckState.Checked : CheckState.Unchecked;
        }

        /// <summary>
        /// Convert CheckState to boolean, Checked means true and
        /// Indeterminate or Unchecked means false.
        /// </summary>
        /// <param name="value">The CheckState value to convert.
        /// If the value is not a CheckState enum, null is returned.</param>
        /// <param name="targetType">The type to convert to.</param>
        /// <param name="parameter">A custom parameter.</param>
        /// <returns>Returns true if the value is Checked,
        /// false if the value is Indeterminate or Unchecked,
        /// and null if the value is not of type CheckState num.</returns>
        public object ConvertBack(
            object value,
            Type targetType,
            object parameter)
        {
            if (Enum.IsDefined(typeof(CheckState), value) == false)
            {
                return null;
            }

            var v = (CheckState)Enum.ToObject(typeof(CheckState), value);

            return v == CheckState.Checked;
        }
    }
}
