﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.Layout;

namespace EffectMaker.UIControls
{
    /// <summary>
    /// An extended collection of IControl that wraps TreeNode collection.
    /// </summary>
    public class TreeNodeCollectionWrapper : IIndexableCollection<ILogicalTreeElement>
    {
        /// <summary>
        /// Stores the number of IControl among the child tree nodes.
        /// </summary>
        private int count;

        /// <summary>
        /// Stores a reference to the wrapped TreeNode.
        /// </summary>
        private TreeNode node;

        /// <summary>
        /// Stores a reference to the wrapped TreeView.
        /// </summary>
        private TreeView tree;

        /// <summary>
        /// The elements of tree view.
        /// </summary>
        private List<ILogicalTreeElement> subElements = new List<ILogicalTreeElement>();

        /// <summary>
        /// Initializes the ControlCollectionWrapper instance.
        /// </summary>
        /// <param name="node">The control to wrap the TreeNodeCollection.</param>
        public TreeNodeCollectionWrapper(TreeNode node)
        {
            if (node == null)
            {
                throw new ArgumentNullException("node");
            }

            this.node = node;
            this.count = node.Nodes.OfType<IItemsControl>().Count();
        }

        /// <summary>
        /// Initializes the ControlCollectionWrapper instance.
        /// </summary>
        /// <param name="tree">The tree to wrap the TreeNodeCollection.</param>
        public TreeNodeCollectionWrapper(TreeView tree)
        {
            if (tree == null)
            {
                throw new ArgumentNullException("tree");
            }

            this.tree = tree;
            this.count = tree.Nodes.OfType<IItemsControl>().Count();
        }

        /// <summary>
        /// Gets the number of IControl contained in the child nodes.
        /// </summary>
        public int Count
        {
            get { return this.count; }
        }

        /// <summary>
        /// Gets whether the list is readonly or not.
        /// </summary>
        public bool IsReadOnly
        {
            get
            {
                if (this.node != null)
                {
                    return this.node.Nodes.IsReadOnly;
                }
                else
                {
                    return this.tree.Nodes.IsReadOnly;
                }
            }
        }

        /// <summary>
        /// Gets a control from it's index in the children list.
        /// Warning: the setter is publicly available but not supported.
        /// </summary>
        /// <param name="index">The index of the control to retrieve.</param>
        /// <returns>Returns the control at the given index.</returns>
        public ILogicalTreeElement this[int index]
        {
            get
            {
                if (this.node != null)
                {
                    return this.node.Nodes[index] as ILogicalTreeElement;
                }
                else
                {
                    return this.tree.Nodes[index] as ILogicalTreeElement;
                }
            }
        }

        /// <summary>
        /// Add an IControl to the child nodes.
        /// </summary>
        /// <param name="item">The child control to add.</param>
        public void Add(ILogicalTreeElement item)
        {
            var ctrl = item as TreeNode;

            if (ctrl == null)
            {
                this.subElements.Add(item);
                return;
            }

            if (this.node != null)
            {
                this.node.Nodes.Add(ctrl);
                if (this.node is UITreeNode)
                {
                    ((UITreeNode)this.node).OnChildNodeAdded(new TreeViewEventArgs(ctrl));
                }
            }
            else
            {
                this.tree.Nodes.Add(ctrl);
                if (this.tree is UITreeView)
                {
                    ((UITreeView)this.tree).OnChildNodeAdded(new TreeViewEventArgs(ctrl));
                }
            }

            this.count++;
        }

        /// <summary>
        /// Clears the nodes.
        /// </summary>
        public void Clear()
        {
            //// TODO: Clears sub elements.
            //// ツリーノードの更新などでこの関数を呼んだとき、this.node.Nodesの再登録は行っているが
            //// subElementsの再登録は行っていないのでClearしっぱなしになってしまう。
            //// 一時的な対処として、subElementsをClearしないように変更。
            //// this.subElements.Clear();

            LayoutEngineBase.SuspendLayout();

            if (this.node != null)
            {
                if (this.node is UITreeNode)
                {
                    var n = (UITreeNode)this.node;
                    foreach (TreeNode tn in n.Nodes)
                    {
                        n.OnChildNodeRemoved(new TreeViewEventArgs(tn));
                    }
                }

                this.node.Nodes.Clear();
            }
            else
            {
                if (this.tree is UITreeView)
                {
                    var n = (UITreeView)this.tree;
                    foreach (TreeNode tn in n.Nodes)
                    {
                        n.OnChildNodeRemoved(new TreeViewEventArgs(tn));
                    }
                }

                this.tree.Nodes.Clear();
            }

            this.count = 0;

            LayoutEngineBase.ResumeLayout();
        }

        /// <summary>
        /// Checks whether an item is contained in the nodes.
        /// </summary>
        /// <param name="item">The item to check.</param>
        /// <returns>Returns true if the item exists in the child nodes,
        /// false otherwise.</returns>
        public bool Contains(ILogicalTreeElement item)
        {
            if (this.subElements.Contains(item))
            {
                return true;
            }

            if (this.node != null)
            {
                return this.node.Nodes.Contains(item as TreeNode);
            }
            else
            {
                return this.tree.Nodes.Contains(item as TreeNode);
            }
        }

        /// <summary>
        /// Not implemented.
        /// </summary>
        /// <param name="array">An array of controls.</param>
        /// <param name="arrayIndex">The start index.</param>
        public void CopyTo(ILogicalTreeElement[] array, int arrayIndex)
        {
            throw new NotImplementedException();
        }

        /// <summary>
        /// Removes a child control.
        /// </summary>
        /// <param name="item">The child control to remove.</param>
        /// <returns>Returns true if the control could be removed, false otherwise.</returns>
        public bool Remove(ILogicalTreeElement item)
        {
            try
            {
                LayoutEngineBase.SuspendLayout();

                var ctrl = item as TreeNode;

                if (ctrl == null)
                {
                    if (this.subElements.Contains(item))
                    {
                        this.subElements.Remove(item);
                        return true;
                    }

                    return false;
                }

                if ((this.node != null && this.node.Nodes.Contains(ctrl)) ||
                    (this.tree != null && this.tree.Nodes.Contains(ctrl)))
                {
                    if (this.node != null)
                    {
                        if (this.node is UITreeNode)
                        {
                            ((UITreeNode)this.node).OnChildNodeRemoved(new TreeViewEventArgs(ctrl));
                        }

                        this.node.Nodes.Remove(ctrl);
                    }
                    else
                    {
                        if (this.tree is UITreeView)
                        {
                            ((UITreeView)this.tree).OnChildNodeRemoved(new TreeViewEventArgs(ctrl));
                        }

                        this.tree.Nodes.Remove(ctrl);
                    }

                    this.count--;
                    return true;
                }

                return false;
            }
            finally
            {
                LayoutEngineBase.ResumeLayout();
            }
        }

        /// <summary>
        /// Provide an enumerator that iterate among the IControl children.
        /// </summary>
        /// <returns>Returns an IControl children enumerator.</returns>
        public IEnumerator<ILogicalTreeElement> GetEnumerator()
        {
            if (this.node != null)
            {
                foreach (var t in this.node.Nodes.OfType<ILogicalTreeElement>())
                {
                    yield return t;
                }

                foreach (var t in this.subElements)
                {
                    yield return t;
                }
            }
            else
            {
                foreach (var t in this.tree.Nodes.OfType<ILogicalTreeElement>())
                {
                    yield return t;
                }

                foreach (var t in this.subElements)
                {
                    yield return t;
                }
            }
        }

        /// <summary>
        /// Provide an enumerator that iterate among the IControl children.
        /// </summary>
        /// <returns>Returns an IControl children enumerator.</returns>
        System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
        {
            return (IEnumerator<ILogicalTreeElement>)this.GetEnumerator();
        }
    }
}
