﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using EffectMaker.Foundation.Interfaces;

namespace EffectMaker.UIControls
{
    /// <summary>
    /// Collection for ToolStripDropDownItem control.
    /// </summary>
    public class ToolStripDropDownItemCollectionWrapper : IndexableCollectionBase<ILogicalTreeElement>
    {
        /// <summary>
        /// Stores the node being processed.
        /// </summary>
        private ToolStripDropDownItem node;

        /// <summary>
        /// Initialized the ToolStripDropDownItemCollectionWrapper instance.
        /// </summary>
        /// <param name="node">The node to wrap the collection.</param>
        public ToolStripDropDownItemCollectionWrapper(ToolStripDropDownItem node)
        {
            this.node = node;
        }

        /// <summary>
        /// Gets the number of item in the collection.
        /// </summary>
        public override int Count
        {
            get { return this.node.DropDownItems.Count; }
        }

        /// <summary>
        /// Gets an item at a given index.
        /// </summary>
        /// <param name="index">The index of the item to get.</param>
        /// <returns>Returns the item as an ILogicalTreeElement, null otherwise.</returns>
        public override ILogicalTreeElement this[int index]
        {
            get { return this.node.DropDownItems[index] as ILogicalTreeElement; }
        }

        /// <summary>
        /// Adds an item.
        /// </summary>
        /// <param name="item">Item to add. Must be of type ToolStripItem.</param>
        public override void Add(ILogicalTreeElement item)
        {
            var ctrl = item as ToolStripItem;

            if (ctrl == null)
            {
                throw new ArgumentException("item");
            }

            this.node.DropDownItems.Add(ctrl);
        }

        /// <summary>
        /// Clears the collection.
        /// </summary>
        public override void Clear()
        {
            this.node.DropDownItems.Clear();
        }

        /// <summary>
        /// Check whether the item is contained in the collection.
        /// </summary>
        /// <param name="item">The item to check.</param>
        /// <returns>Returns true if the item is contained in the collection,
        /// false otherwise.</returns>
        public override bool Contains(ILogicalTreeElement item)
        {
            return this.node.DropDownItems.Contains(item as ToolStripItem);
        }

        /// <summary>
        /// Removes an item from the collection.
        /// </summary>
        /// <param name="item">Item to remove.</param>
        /// <returns>Returns true if the item is removed/removable, false otherwise.</returns>
        public override bool Remove(ILogicalTreeElement item)
        {
            var ctrl = item as ToolStripItem;

            if (ctrl == null)
            {
                return false;
            }

            this.node.DropDownItems.Remove(ctrl);
            return true;
        }

        /// <summary>
        /// Gets an enumerator on the sequence of logical tree children.
        /// </summary>
        /// <returns>Returns an enumerator on the sequence of logical tree children.</returns>
        public override IEnumerator<ILogicalTreeElement> GetEnumerator()
        {
            return this.node.DropDownItems.OfType<ILogicalTreeElement>().GetEnumerator();
        }
    }

    /// <summary>
    /// Collection for ContextMenuStrip control.
    /// </summary>
    public class ContextMenuStripCollectionWrapper : IndexableCollectionBase<ILogicalTreeElement>
    {
        /// <summary>
        /// Stores the node being processed.
        /// </summary>
        private ContextMenuStrip node;

        /// <summary>
        /// Initialized the ContextMenuStripCollectionWrapper instance.
        /// </summary>
        /// <param name="node">The node to wrap the collection.</param>
        public ContextMenuStripCollectionWrapper(ContextMenuStrip node)
        {
            if (node == null)
            {
                throw new ArgumentNullException("node");
            }

            this.node = node;
        }

        /// <summary>
        /// Gets the number of item in the collection.
        /// </summary>
        public override int Count
        {
            get { return this.node.Items.Count; }
        }

        /// <summary>
        /// Gets an item at a given index.
        /// </summary>
        /// <param name="index">The index of the item to get.</param>
        /// <returns>Returns the item as an ILogicalTreeElement, null otherwise.</returns>
        public override ILogicalTreeElement this[int index]
        {
            get { return this.node.Items[index] as ILogicalTreeElement; }
        }

        /// <summary>
        /// Adds an item.
        /// </summary>
        /// <param name="item">Item to add. Must be of type ToolStripItem.</param>
        public override void Add(ILogicalTreeElement item)
        {
            var ctrl = item as ToolStripItem;

            if (ctrl == null)
            {
                throw new ArgumentException("item");
            }

            this.node.Items.Add(ctrl);
        }

        /// <summary>
        /// Clears the collection.
        /// </summary>
        public override void Clear()
        {
            this.node.Items.Clear();
        }

        /// <summary>
        /// Check whether the item is contained in the collection.
        /// </summary>
        /// <param name="item">The item to check.</param>
        /// <returns>Returns true if the item is contained in the collection,
        /// false otherwise.</returns>
        public override bool Contains(ILogicalTreeElement item)
        {
            return this.node.Items.Contains(item as ToolStripItem);
        }

        /// <summary>
        /// Removes an item from the collection.
        /// </summary>
        /// <param name="item">Item to remove.</param>
        /// <returns>Returns true if the item is removed/removable, false otherwise.</returns>
        public override bool Remove(ILogicalTreeElement item)
        {
            this.node.Items.Remove(item as ToolStripItem);
            return true;
        }

        /// <summary>
        /// Gets an enumerator on the sequence of logical tree children.
        /// </summary>
        /// <returns>Returns an enumerator on the sequence of logical tree children.</returns>
        public override IEnumerator<ILogicalTreeElement> GetEnumerator()
        {
            return this.node.Items.OfType<ILogicalTreeElement>().GetEnumerator();
        }
    }

    /// <summary>
    /// Collection for MenuStrip control.
    /// </summary>
    public class MenuStripCollectionWrapper : IndexableCollectionBase<ILogicalTreeElement>
    {
        /// <summary>
        /// Stores the node being processed.
        /// </summary>
        private MenuStrip node;

        /// <summary>
        /// Initialized the MenuStripCollectionWrapper instance.
        /// </summary>
        /// <param name="node">The node to wrap the collection.</param>
        public MenuStripCollectionWrapper(MenuStrip node)
        {
            if (node == null)
            {
                throw new ArgumentNullException("node");
            }

            this.node = node;
        }

        /// <summary>
        /// Gets the number of item in the collection.
        /// </summary>
        public override int Count
        {
            get { return this.node.Items.Count; }
        }

        /// <summary>
        /// Gets an item at a given index.
        /// </summary>
        /// <param name="index">The index of the item to get.</param>
        /// <returns>Returns the item as an ILogicalTreeElement, null otherwise.</returns>
        public override ILogicalTreeElement this[int index]
        {
            get { return this.node.Items[index] as ILogicalTreeElement; }
        }

        /// <summary>
        /// Adds an item.
        /// </summary>
        /// <param name="item">Item to add. Must be of type ToolStripItem.</param>
        public override void Add(ILogicalTreeElement item)
        {
            var ctrl = item as ToolStripItem;

            if (ctrl == null)
            {
                throw new ArgumentException("item");
            }

            this.node.Items.Add(ctrl);
        }

        /// <summary>
        /// Clears the collection.
        /// </summary>
        public override void Clear()
        {
            this.node.Items.Clear();
        }

        /// <summary>
        /// Check whether the item is contained in the collection.
        /// </summary>
        /// <param name="item">The item to check.</param>
        /// <returns>Returns true if the item is contained in the collection,
        /// false otherwise.</returns>
        public override bool Contains(ILogicalTreeElement item)
        {
            return this.node.Items.Contains(item as ToolStripItem);
        }

        /// <summary>
        /// Removes an item from the collection.
        /// </summary>
        /// <param name="item">Item to remove.</param>
        /// <returns>Returns true if the item is removed/removable, false otherwise.</returns>
        public override bool Remove(ILogicalTreeElement item)
        {
            this.node.Items.Remove(item as ToolStripItem);
            return true;
        }

        /// <summary>
        /// Gets an enumerator on the sequence of logical tree children.
        /// </summary>
        /// <returns>Returns an enumerator on the sequence of logical tree children.</returns>
        public override IEnumerator<ILogicalTreeElement> GetEnumerator()
        {
            return this.node.Items.OfType<ILogicalTreeElement>().GetEnumerator();
        }
    }

    /// <summary>
    /// Collection for ToolStrip control.
    /// </summary>
    public class ToolStripCollectionWrapper : IndexableCollectionBase<ILogicalTreeElement>
    {
        /// <summary>
        /// Stores the node being processed.
        /// </summary>
        private ToolStrip node;

        /// <summary>
        /// Initialized the MenuStripCollectionWrapper instance.
        /// </summary>
        /// <param name="node">The node to wrap the collection.</param>
        public ToolStripCollectionWrapper(ToolStrip node)
        {
            if (node == null)
            {
                throw new ArgumentNullException("node");
            }

            this.node = node;
        }

        /// <summary>
        /// Gets the number of item in the collection.
        /// </summary>
        public override int Count
        {
            get { return this.node.Items.Count; }
        }

        /// <summary>
        /// Gets an item at a given index.
        /// </summary>
        /// <param name="index">The index of the item to get.</param>
        /// <returns>Returns the item as an ILogicalTreeElement, null otherwise.</returns>
        public override ILogicalTreeElement this[int index]
        {
            get { return this.node.Items[index] as ILogicalTreeElement; }
        }

        /// <summary>
        /// Adds an item.
        /// </summary>
        /// <param name="item">Item to add. Must be of type ToolStripItem.</param>
        public override void Add(ILogicalTreeElement item)
        {
            var ctrl = item as ToolStripItem;

            if (ctrl == null)
            {
                throw new ArgumentException("item");
            }

            this.node.Items.Add(ctrl);
        }

        /// <summary>
        /// Clears the collection.
        /// </summary>
        public override void Clear()
        {
            this.node.Items.Clear();
        }

        /// <summary>
        /// Check whether the item is contained in the collection.
        /// </summary>
        /// <param name="item">The item to check.</param>
        /// <returns>Returns true if the item is contained in the collection,
        /// false otherwise.</returns>
        public override bool Contains(ILogicalTreeElement item)
        {
            return this.node.Items.Contains(item as ToolStripItem);
        }

        /// <summary>
        /// Removes an item from the collection.
        /// </summary>
        /// <param name="item">Item to remove.</param>
        /// <returns>Returns true if the item is removed/removable, false otherwise.</returns>
        public override bool Remove(ILogicalTreeElement item)
        {
            this.node.Items.Remove(item as ToolStripItem);
            return true;
        }

        /// <summary>
        /// Gets an enumerator on the sequence of logical tree children.
        /// </summary>
        /// <returns>Returns an enumerator on the sequence of logical tree children.</returns>
        public override IEnumerator<ILogicalTreeElement> GetEnumerator()
        {
            return this.node.Items.OfType<ILogicalTreeElement>().GetEnumerator();
        }
    }
}
