﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

using EffectMaker.UIControls.Extensions;

namespace EffectMaker.UIControls.Focus
{
    /// <summary>
    /// A class that collects all the child controls in the owner in their tab order.
    /// </summary>
    public class TabOrderProcessor
    {
        /// <summary>The owner control.</summary>
        private Control owner;

        /// <summary>
        /// The child controls of the owner that can receive tab focus.
        /// コントロールの親子関係に依存する。
        /// 子A[1002],子A孫[2009,2003],子B[1001],子B孫[2005,2004]とあった場合
        /// 1001,2004,2005,1002,2003,2009の順になるため、TabIndexの割り付け方に注意。
        /// </summary>
        private List<Control> focusableChildControls = new List<Control>();

        /// <summary>
        /// All the child controls in the owner control and the pointer to
        /// the previous and next control that can receive tab focus.
        /// </summary>
        private Dictionary<Control, Tuple<int, int>> childControls =
            new Dictionary<Control, Tuple<int, int>>();

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="owner">The owner control.</param>
        public TabOrderProcessor(Control owner)
        {
            this.owner = owner;

            this.CollectFocusableChildControls(this.owner);
        }

        /// <summary>
        /// Get the first focusable child control.
        /// </summary>
        /// <param name="forward">False to get the last focusable child control.</param>
        /// <returns>The first or last child control depending on the assigned argument.</returns>
        public Control GetFirstControl(bool forward)
        {
            int count = this.focusableChildControls.Count;
            if (count <= 0)
            {
                return null;
            }

            if (forward == true)
            {
                return this.focusableChildControls[0];
            }
            else
            {
                return this.focusableChildControls[count - 1];
            }
        }

        /// <summary>
        /// Get the next child control that can receive tab focus.
        /// </summary>
        /// <param name="ctrl">The current control.</param>
        /// <param name="forward">The search direction.</param>
        /// <returns>The next child control that receives focus.</returns>
        public Control GetNextControl(Control ctrl, bool forward)
        {
            Tuple<int, int> linkData;
            if (this.childControls.TryGetValue(ctrl, out linkData) == false)
            {
                return null;
            }

            int index = -1;
            if (forward == true)
            {
                index = linkData.Item2;
                if (index >= this.focusableChildControls.Count)
                {
                    index = 0;
                }
            }
            else
            {
                index = linkData.Item1;
                if (index < 0)
                {
                    index = this.focusableChildControls.Count - 1;
                }
            }

            if (index < 0 || index >= this.focusableChildControls.Count)
            {
                return null;
            }
            else
            {
                return this.focusableChildControls[index];
            }
        }

        /// <summary>
        /// Collect child controls that can receive tab focus.
        /// </summary>
        /// <param name="ctrl">The parent control.</param>
        private void CollectFocusableChildControls(Control ctrl)
        {
            int index = this.focusableChildControls.Count;

            if (ctrl.Controls.Count <= 0)
            {
                if (ctrl.CanReceiveFocus(true) == true)
                {
                    this.focusableChildControls.Add(ctrl);
                    this.childControls.Add(ctrl, Tuple.Create(index - 1, index + 1));
                }
                else
                {
                    this.childControls.Add(ctrl, Tuple.Create(index - 1, index));
                }
            }
            else
            {
                this.childControls.Add(ctrl, Tuple.Create(index - 1, index));

                foreach (Control child in this.SortChildControlsByTabIndex(ctrl))
                {
                    this.CollectFocusableChildControls(child);
                }
            }
        }

        /// <summary>
        /// Sort child controls by their tab index.
        /// </summary>
        /// <param name="ctrl">The parent control.</param>
        /// <returns>The child controls sorted by the tab index.</returns>
        private IEnumerable<Control> SortChildControlsByTabIndex(Control ctrl)
        {
            return from child in ctrl.Controls.Cast<Control>()
                   orderby child.TabIndex
                   select child;
        }
    }
}
