﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Linq;

namespace EffectMaker.UIControls.Extenders
{
    /// <summary>
    /// This class extends the capabilities of an ILogicalTreeElement.
    /// </summary>
    public class LogicalTreeElementExtender : ExtenderBase
    {
        /// <summary>
        /// Backing field for the Extender property.
        /// </summary>
        private ILogicalTreeElement extendedElement;

        /// <summary>
        /// Flag that tells whether the DataContext is set or cleared.
        /// </summary>
        private bool isDataContextSet;

        /// <summary>
        /// Flag that tells whether the DataContext has been set through data binding.
        /// </summary>
        private bool isDataContextSetThroughBinding;

        /// <summary>
        /// Backing field for DataContext property.
        /// </summary>
        private object dataContext;

        /// <summary>
        /// Initializes the ControlExtender.
        /// </summary>
        /// <param name="extendedElement">Logical tree element to extend the capabilities.</param>
        public LogicalTreeElementExtender(ILogicalTreeElement extendedElement)
            : base(extendedElement)
        {
            this.extendedElement = extendedElement;
        }

        /// <summary>
        /// Gets a value indicating whether is updating data context.
        /// </summary>
        public static bool IsUpdatingDataContext { get; private set; }

        /// <summary>
        /// Gets or sets the data context.
        /// This is the source of the binding applied on the control.
        /// DataContext raises special change notification and
        /// inherit its value from parents DataContext.
        /// </summary>
        /// <remarks>
        /// DO NOT access this property directly from outside the extending control.
        /// You should ALWAYS access the DataContext property of the extending control
        /// in case the extending control has it's own implementation other than calling
        /// this DataContext property.
        /// Eg.
        /// <code>
        /// class MyControl : ILogicalTreeElement
        /// {
        ///     private LogicalTreeElementExtender controlExtender = null;
        ///
        ///     public LogicalTreeElementExtender LogicalTreeElementExtender
        ///     {
        ///         get { return this.controlExtender; }
        ///     }
        ///
        ///     [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        ///     public object DataContext
        ///     {
        ///         get { return this.controlExtender.DataContext; }
        ///         set
        ///         {
        ///             this.controlExtender.DataContext = value;
        ///             ++this.dataContextSetCounter; // NOT EXECUTED!!!!
        ///         }
        ///     }
        /// }
        /// </code>
        /// If you do something like this:
        /// <code>
        /// var control = new MyControl();
        /// control.LogicalTreeElementExtender.DataContext = new object();
        /// </code>
        /// The logic in MyControl.DataContext.set will not execute anymore.
        /// </remarks>
        public object DataContext
        {
            get
            {
                if (this.isDataContextSet || this.isDataContextSetThroughBinding)
                {
                    return this.dataContext;
                }

                var parent = this.extendedElement.Parent;
                if (parent != null)
                {
                    return parent.DataContext;
                }

                return null;
            }

            set
            {
                if (object.Equals(this.dataContext, value) == false || this.isDataContextSet == false)
                {
                    this.dataContext = value;
                    this.isDataContextSet = true;
                    this.isDataContextSetThroughBinding = false;
                    this.NotifyDataContextChanged();
                }
            }
        }

        /// <summary>
        /// Clears the DataContext.
        /// Clearing the DataContext and setting it to null are two different actions.
        /// Setting it to null means the DataContext is null and become the
        /// root of all its children where the DataContext is not set.
        /// Clearing the DataContext let the value propagation pass through
        /// the control to reach the DataContext of an upper parent.
        /// </summary>
        /// <remarks>
        /// DO NOT call this method directly from outside the extending control.
        /// You should ALWAYS call ClearDataContext() method of the extending control
        /// in case the extending control has it's own implementation other than calling
        /// this ClearDataContext().
        /// Eg.
        /// <code>
        /// class MyControl : ILogicalTreeElement
        /// {
        ///     private LogicalTreeElementExtender controlExtender = null;
        ///
        ///     public LogicalTreeElementExtender LogicalTreeElementExtender
        ///     {
        ///         get { return this.controlExtender; }
        ///     }
        ///
        ///     public void ClearDataContext()
        ///     {
        ///         this.controlExtender.ClearDataContext();
        ///         ++this.clearDataContextCounter; // NOT EXECUTED!!!!
        ///     }
        /// }
        /// </code>
        /// If you do something like this:
        /// <code>
        /// var control = new MyControl();
        /// control.LogicalTreeElementExtender.ClearDataContext();
        /// </code>
        /// The logic in MyControl.ClearDataContext() will not execute anymore.
        /// </remarks>
        public void ClearDataContext()
        {
            this.isDataContextSet = false;
            this.isDataContextSetThroughBinding = false;

            var inheritedDataContext = this.DataContext;
            if (object.Equals(inheritedDataContext, this.dataContext))
            {
                this.NotifyDataContextChanged();
            }

            this.dataContext = null;
        }

        /// <summary>
        /// Set the DataContext with special binding setup flag.
        /// </summary>
        /// <param name="dataContext">The new data context to set.</param>
        internal void SetDataContextThroughBinding(object dataContext)
        {
            if (object.Equals(this.dataContext, dataContext) == false)
            {
                this.dataContext = dataContext;
                this.isDataContextSet = false;
                this.isDataContextSetThroughBinding = true;
                this.NotifyDataContextChanged();
            }
        }

        /// <summary>
        /// Raise the PropertyChanged event on the current control and on
        /// all the child controls that do not have a DataContext set.
        /// </summary>
        internal void NotifyDataContextChanged()
        {
            LogicalTreeElementExtender.IsUpdatingDataContext = true;
            try
            {
                this.NotifyPropertyChanged(propertyName: "DataContext");
                foreach (var child in this.extendedElement.Children.OfType<ILogicalTreeElement>())
                {
                    if (child.LogicalTreeElementExtender.isDataContextSet == false)
                    {
                        if (child.LogicalTreeElementExtender.isDataContextSetThroughBinding)
                        {
                            // Do not call ClearDataContext() directly, for the method triggers
                            // PropertyChanged event, which redundantly makes the binders to
                            // update values from the data source.
                            child.LogicalTreeElementExtender.isDataContextSet = false;
                            child.LogicalTreeElementExtender.isDataContextSetThroughBinding = false;
                            child.LogicalTreeElementExtender.dataContext = null;
                        }

                        child.LogicalTreeElementExtender.NotifyDataContextChanged();
                    }
                }
            }
            finally
            {
                LogicalTreeElementExtender.IsUpdatingDataContext = false;
            }
        }
    }
}
