﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.ComponentModel;
using System.Runtime.CompilerServices;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.EventArguments;
using EffectMaker.UIControls.Extenders;

namespace EffectMaker.UIControls.Core
{
    /// <summary>
    /// IBindable を実装した最小限のクラスです。
    /// </summary>
    public class SimpleBindable : IBindable
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public SimpleBindable()
        {
            this.Bindings = new BindingContainer(this);
            this.LogicalTreeElementExtender = new LogicalTreeElementExtender(this);
            this.Children = new NullIndexableCollection<ILogicalTreeElement>();
        }

        #region IBindable members

        /// <summary>
        /// バインダのコレクションを取得します。
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        #endregion

        #region ILogicalTreeElement members

        /// <summary>
        /// LogicalTreeElementExtender を取得します。
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender { get; private set; }

        /// <summary>
        /// データコンテキストを取得または設定します。
        /// </summary>
        public object DataContext
        {
            get { return this.LogicalTreeElementExtender.DataContext; }
            set { this.LogicalTreeElementExtender.DataContext = value; }
        }

        /// <summary>
        /// 親コントロールを取得または設定します。
        /// </summary>
        public ILogicalTreeElement Parent { get; set; }

        /// <summary>
        /// チャイルドコントロールのコレクションを取得します。
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children { get; private set; }

        /// <summary>
        /// データコンテキストをクリアします。
        /// </summary>
        public void ClearDataContext()
        {
            this.LogicalTreeElementExtender.ClearDataContext();
        }

        #endregion

        #region INotifyPropertyChanged members

        /// <summary>
        /// プロパティの値が変わったときに発生するイベントです。
        /// </summary>
        public event PropertyChangedEventHandler PropertyChanged;

        /// <summary>
        /// PropertyChanged イベントを発生させます。
        /// </summary>
        /// <param name="updateType">どのような方法でプロパティが変更されたか</param>
        /// <param name="propertyName">変更されたプロパティの名前</param>
        public void NotifyPropertyChanged(
            BindingUpdateType updateType = BindingUpdateType.PropertyChanged,
            [CallerMemberName]string propertyName = null)
        {
            this.PropertyChanged?.Invoke(this, new PropertyChangedExEventArgs(propertyName, updateType));
        }

        #endregion
    }
}
