﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UIControls.Behaviors;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;
using EffectMaker.UIControls.Extensions;

namespace EffectMaker.UIControls.BaseControls
{
    /// <summary>
    /// An extended TreeNode class.
    /// </summary>
    public class UITreeNode : TreeNode, IItemsControl
    {
        /// <summary>
        /// Backing field for the Extender property.
        /// </summary>
        private LogicalTreeElementExtender controlExtender;

        /// <summary>
        /// Stores the ItemsControlExtender instance.
        /// </summary>
        private ItemsControlExtender itemsControlExtender;

        /// <summary>
        /// Backing field for Controls property.
        /// </summary>
        private IIndexableCollection<ILogicalTreeElement> nodesWrapper;

        /// <summary>
        /// Backing field for the Resources property.
        /// </summary>
        private IDictionary<string, object> resources = new Dictionary<string, object>();

        /// <summary>
        /// Backing field for the SwapNodesExecutable property.
        /// </summary>
        private IExecutable swapNodesExecutable;

        /// <summary>
        /// 複数選択フラグ
        /// </summary>
        private bool isMultiSelected = false;

        /// <summary>
        /// Initializes the UITreeNode instance.
        /// </summary>
        public UITreeNode()
        {
            this.controlExtender = new LogicalTreeElementExtender(this);
            this.itemsControlExtender = new ItemsControlExtender(this);
            this.Bindings = new BindingContainer(this);
            this.Behaviors = new BehaviorCollection(this);

            this.AddBinding("IsExpanded", "IsExpanded");
            this.AddBinding("IsSelected", "IsSelected");
            this.AddBinding("IsMultiSelected", "IsMultiSelected");

            var binder = new Binder(this, "SwapNodesExecutable", "SwapNodesExecutable")
            {
                Mode = BindingMode.OneWay,
            };
            this.Bindings.Add(binder);
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// Raised when the current node get selected or unselected.
        /// </summary>
        public event EventHandler<ValueChangedEventArgs<bool>> SelectionChanged;

        /// <summary>
        /// Raised when the mouse pointer is over the control and a mouse button is pressed.
        /// </summary>
        public event EventHandler<MouseEventArgs> MouseDown;

        /// <summary>
        /// Raised when the mouse pointer is over the control and a mouse button is released.
        /// </summary>
        public event EventHandler<MouseEventArgs> MouseUp;

        /// <summary>
        /// Raised when the control is clicked by the mouse.
        /// </summary>
        public event EventHandler<MouseEventArgs> MouseClick;

        /// <summary>
        /// Raised when the control is double clicked by the mouse.
        /// </summary>
        public event EventHandler<MouseEventArgs> MouseDoubleClick;

        /// <summary>
        /// Executable called from UI to notify the ViewModel layer of a node swapping.
        /// </summary>
        public IExecutable SwapNodesExecutable
        {
            get { return this.swapNodesExecutable; }
            set { this.controlExtender.SetValue(ref this.swapNodesExecutable, value); }
        }

        /// <summary>
        /// Gets or sets whether the current node is selected or not.
        /// </summary>
        public new bool IsSelected
        {
            get
            {
                if (this.TreeView != null)
                {
                    return this.TreeView.SelectedNode == this;
                }

                return false;
            }

            set
            {
                if (this.TreeView != null)
                {
                    this.TreeView.SelectedNode = this;
                }
            }
        }

        /// <summary>
        /// 複数選択されているかどうかを表すフラグを取得または設定します。
        /// </summary>
        public bool IsMultiSelected
        {
            get
            {
                return this.isMultiSelected;
            }

            set
            {
                if (this.isMultiSelected == value)
                {
                    return;
                }

                this.LogicalTreeElementExtender.SetValue(ref this.isMultiSelected, value);
                if (this.TreeView != null)
                {
                    this.TreeView.Invalidate();
                }
            }
        }

        /// <summary>
        /// Gets the resources.
        /// </summary>
        public IDictionary<string, object> Resources
        {
            get { return this.resources; }
        }

        /// <summary>
        /// Gets the region that displays the open/close symbol of the node.
        /// </summary>
        public virtual Rectangle ExpanderRectangle
        {
            get { return Rectangle.Empty; }
        }

        /// <summary>
        /// Gets the region that dislpays the content of the node.
        /// </summary>
        public virtual Rectangle ContentRectangle
        {
            get { return Rectangle.Empty; }
        }

        /// <summary>
        /// Gets the region that dislpays the name of the node.
        /// </summary>
        public virtual Rectangle NameRectangle
        {
            get { return Rectangle.Empty; }
        }

        /// <summary>
        /// Gets or sets whether this node can be renamed.
        /// </summary>
        public virtual bool CanRename { get; set; }

        /// <summary>
        /// Gets whether the node is being dragged or not.
        /// </summary>
        public bool IsDragging
        {
            get
            {
                var treeview = this.TreeView as UITreeView;

                if (treeview == null)
                {
                    return false;
                }

                return treeview.DraggingNode == this;
            }
        }

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get
            {
                object parent = base.Parent;

                if (parent == null)
                {
                    parent = this.TreeView;
                }

                return parent as ILogicalTreeElement;
            }
        }

        /// <summary>
        /// Gets the collection of child controls.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Controls
        {
            get
            {
                if (this.nodesWrapper == null)
                {
                    this.nodesWrapper = new TreeNodeCollectionWrapper(this);
                }

                return this.nodesWrapper;
            }
        }

        /// <summary>
        /// Gets the collection of logical tree elements.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                return this.Controls;
            }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.controlExtender; }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Gets the behaviors collection.
        /// </summary>
        public BehaviorCollection Behaviors { get; private set; }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See ControlExtender for more information.
        /// <see cref="ControlExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get { return this.controlExtender.DataContext; }
            set { this.controlExtender.DataContext = value; }
        }

        /// <summary>
        /// Gets or sets the children items source.
        /// </summary>
        public IEnumerable ItemsSource
        {
            get { return this.itemsControlExtender.ItemsSource; }
            set { this.itemsControlExtender.ItemsSource = value; }
        }

        /// <summary>
        /// Gets or sets the type of item container to construct.
        /// If this is null, the ItemContainerSelector is used.
        /// </summary>
        public Type ItemContainerType
        {
            get { return this.itemsControlExtender.ItemContainerType; }
            set { this.itemsControlExtender.ItemContainerType = value; }
        }

        /// <summary>
        /// Gets or sets the item container selector.
        /// If this is null, no item container is instanced.
        /// </summary>
        public IItemContainerSelector ItemContainerSelector
        {
            get { return this.itemsControlExtender.ItemContainerSelector; }
            set { this.itemsControlExtender.ItemContainerSelector = value; }
        }

        /// <summary>
        /// 階層を辿って全てのツリーノードを取得します.
        /// </summary>
        /// <returns>全てのツリーノードを返します.</returns>
        public IEnumerable<UITreeNode> GetAllNodes()
        {
            foreach (UITreeNode node in this.Nodes)
            {
                yield return node;
                foreach (UITreeNode sub in node.GetAllNodes())
                {
                    yield return sub;
                }
            }
        }

        /// <summary>
        /// Compares two child controls for sorting.
        /// </summary>
        /// <param name="a">First control.</param>
        /// <param name="b">Second control.</param>
        /// <returns>Returns 0 if equal, 1 or -1 if different depending on order.</returns>
        public virtual int CompareControls(IControl a, IControl b)
        {
            return 0;
        }

        /// <summary>
        /// Retrieve the data item corresponding to the container.
        /// </summary>
        /// <param name="container">The container to look for its data item.</param>
        /// <returns>Returns the data item, or null if not found.</returns>
        public object GetDataItemFromContainer(IControl container)
        {
            return this.itemsControlExtender.GetDataItemFromContainer(container);
        }

        /// <summary>
        /// Retrieve the container corresponding to the data item.
        /// </summary>
        /// <param name="dataItem">The data item to look for its data.</param>
        /// <returns>Returns the container, or null if not found.</returns>
        public IControl GetContainerFromDataItem(object dataItem)
        {
            return this.itemsControlExtender.GetContainerFromDataItem(dataItem);
        }

        /// <summary>
        /// Clears the DataContext.
        /// See ControlExtender for more details.
        /// <see cref="ControlExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.controlExtender.ClearDataContext();
        }

        /// <summary>
        /// Called when the mouse pointer is over the control and a mouse button is pressed.
        /// </summary>
        /// <param name="e">Event argument.</param>
        public virtual void OnMouseDown(MouseEventArgs e)
        {
            var handler = this.MouseDown;
            if (handler != null)
            {
                handler(this, e);
            }
        }

        /// <summary>
        /// Called when the mouse pointer is over the control and a mouse button is released.
        /// </summary>
        /// <param name="e">Event argument.</param>
        public virtual void OnMouseUp(MouseEventArgs e)
        {
            var handler = this.MouseUp;
            if (handler != null)
            {
                handler(this, e);
            }
        }

        /// <summary>
        /// Called when the control is clicked by the mouse.
        /// </summary>
        /// <param name="e">Event argument.</param>
        public virtual void OnMouseClick(MouseEventArgs e)
        {
            var handler = this.MouseClick;
            if (handler != null)
            {
                handler(this, e);
            }
        }

        /// <summary>
        /// Called when the control is double clicked by the mouse.
        /// </summary>
        /// <param name="e">Event argument.</param>
        public virtual void OnMouseDoubleClick(MouseEventArgs e)
        {
            var handler = this.MouseDoubleClick;
            if (handler != null)
            {
                handler(this, e);
            }
        }

        /// <summary>
        /// 複数選択をトグルします。
        /// </summary>
        /// <returns>トグル操作の結果</returns>
        public virtual bool ToggleMultiSelection()
        {
            this.IsMultiSelected = !this.IsMultiSelected;
            return this.IsMultiSelected;
        }

        /// <summary>
        /// Called just after the control have been instanced.
        /// It is the responsibility of the UITreeNode instancer to call this method.
        /// </summary>
        protected internal virtual void OnInitialized()
        {
        }

        /// <summary>
        /// Called when the current node get selected of unselected.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected internal virtual void OnSelectionChanged(ValueChangedEventArgs<bool> e)
        {
            var handler = this.SelectionChanged;
            if (handler != null)
            {
                handler(this, e);
            }
        }

        /// <summary>
        /// When overridden, perform self rendering.
        /// </summary>
        /// <param name="bounds">The rectangular region in whic to render.</param>
        /// <param name="state">The current node state.</param>
        /// <param name="gr">The graphics instance to render to.</param>
        /// <returns>Returns true if it self render, or false to tall parent
        /// TreeView to perform default node rendering.</returns>
        protected internal virtual bool Render(Rectangle bounds, TreeNodeStates state, Graphics gr)
        {
            // The tree node might already be removed between the draw message is dispatched
            // and the actual rendering.
            // (the message dispatches, and the rendering happens on the next application loop)
            if (((TreeNode)this).Parent == null &&
                (this.TreeView == null || this.TreeView.Nodes.Contains(this) == false))
            {
                return true;
            }

            return false;
        }

        /// <summary>
        /// Called when a child node is added.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected internal virtual void OnChildNodeAdded(TreeViewEventArgs e)
        {
        }

        /// <summary>
        /// Called when a child node is removed.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected internal virtual void OnChildNodeRemoved(TreeViewEventArgs e)
        {
        }
    }
}
