﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UIControls.Behaviors;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;
using EffectMaker.UIControls.Input;

namespace EffectMaker.UIControls
{
    /// <summary>
    /// Extended MenuItem class.
    /// </summary>
    public class UIToolStripMenuItem : ToolStripMenuItem, IControl, IExecutionSource
    {
        /// <summary>
        /// Backing field for the LogicalTreeElementExtender property.
        /// </summary>
        private LogicalTreeElementExtender logicalTreeElementExtender;

        /// <summary>
        /// Stores the ExecutionSourceExtender instance.
        /// </summary>
        private ExecutionSourceExtender executionSourceExtender;

        /// <summary>
        /// Backing field for the Bindings property.
        /// </summary>
        private BindingContainer bindings;

        /// <summary>
        /// Backing field for the Behaviors property.
        /// </summary>
        private BehaviorCollection behaviors;

        /// <summary>
        /// Backing field for the Children property.
        /// </summary>
        private ToolStripDropDownItemCollectionWrapper children;

        /// <summary>
        /// Backing field for the Resources property.
        /// </summary>
        private IDictionary<string, object> resources = new Dictionary<string, object>();

        /// <summary>
        /// Initializes the UIToolStripMenuItem instance.
        /// </summary>
        public UIToolStripMenuItem()
        {
            this.logicalTreeElementExtender = new LogicalTreeElementExtender(this);
            this.executionSourceExtender = new ExecutionSourceExtender(this);
            this.bindings = new BindingContainer(this);
            this.behaviors = new BehaviorCollection(this);
            this.children = new ToolStripDropDownItemCollectionWrapper(this);
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning disable 67

        /// <summary>
        /// Fired before Execute method of the IExecutionSource is called.
        /// </summary>
        public event EventHandler BeforeExecute
        {
            add { this.executionSourceExtender.BeforeExecute += value; }
            remove { this.executionSourceExtender.BeforeExecute -= value; }
        }

        /// <summary>
        /// Fired after Execute method of the IExecutionSource is called.
        /// </summary>
        public event EventHandler AfterExecute
        {
            add { this.executionSourceExtender.AfterExecute += value; }
            remove { this.executionSourceExtender.AfterExecute -= value; }
        }

        /// <summary>
        /// Gets the resources.
        /// </summary>
        public IDictionary<string, object> Resources
        {
            get { return this.resources; }
        }

        /// <summary>
        /// Gets the behavior collection.
        /// </summary>
        public BehaviorCollection Behaviors
        {
            get { return this.behaviors; }
        }

        /// <summary>
        /// Gets the control collection.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Controls
        {
            get { return this.children; }
        }

        /// <summary>
        /// Gets the binding collection.
        /// </summary>
        public BindingContainer Bindings
        {
            get { return this.bindings; }
        }

        /// <summary>
        /// Gets the LogicalTreeElementExtender instance.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.logicalTreeElementExtender; }
        }

        /// <summary>
        /// Gets or sets the data context.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get
            {
                return this.logicalTreeElementExtender.DataContext;
            }

            set
            {
                this.logicalTreeElementExtender.DataContext = value;
            }
        }

        /// <summary>
        /// Gets the logical tree parent.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get
            {
                if (base.Parent != null)
                {
                    return base.Parent as ILogicalTreeElement;
                }

                if (this.OwnerItem != null)
                {
                    return this.OwnerItem as ILogicalTreeElement;
                }

                if (this.Owner != null)
                {
                    return this.Owner as ILogicalTreeElement;
                }

                return null;
            }
        }

        /// <summary>
        /// Gets the logical tree children.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get { return this.Controls; }
        }

        /// <summary>
        /// Gets or sets the Executable instance.
        /// </summary>
        public IExecutable Executable
        {
            get
            {
                return this.executionSourceExtender.Executable;
            }

            set
            {
                this.executionSourceExtender.Executable = value;
            }
        }

        /// <summary>
        /// Gets or sets the custom parameter of the Execution instance.
        /// </summary>
        public object ExecutionParameter
        {
            get
            {
                return this.executionSourceExtender.ExecutionParameter;
            }

            set
            {
                this.executionSourceExtender.ExecutionParameter = value;
            }
        }

        /// <summary>
        /// Clears the data context.
        /// </summary>
        public void ClearDataContext()
        {
            this.logicalTreeElementExtender.ClearDataContext();
        }

        /// <summary>
        /// Called when the menu is clicked.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected override void OnClick(EventArgs e)
        {
            base.OnClick(e);

            this.executionSourceExtender.ExecuteOnEvent(this, e, true);
        }
    }
}
