﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.ComponentModel;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UIControls.Behaviors;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;
using EffectMaker.UIControls.Input;
using EffectMaker.UIControls.Layout;

namespace EffectMaker.UIControls.BaseControls
{
    /// <summary>
    /// An extended ToolStripButton class.
    /// </summary>
    public class UIToolStripComboBox : ToolStripComboBox, IControl
    {
        /// <summary>
        /// アイテム一覧
        /// </summary>
        private IEnumerable<KeyValuePair<string, object>> availableItems;

        /// <summary>
        /// サイズを自動調整するフラグ.
        /// </summary>
        private bool autoAdjustWidth;

        /// <summary>
        /// Backing field for the Extender property.
        /// </summary>
        private LogicalTreeElementExtender controlExtender;

        /// <summary>
        /// Backing field for Controls property.
        /// </summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>
        /// Backing field for the Resources property.
        /// </summary>
        private IDictionary<string, object> resources = new Dictionary<string, object>();

        /// <summary>
        /// 外部からのプロパティ変更時に設定サイズを保持する変数.
        /// </summary>
        private int savedWidth;

        /// <summary>
        /// Constructor.
        /// </summary>
        public UIToolStripComboBox()
        {
            //// デフォルトで自動調整は有効
            this.autoAdjustWidth = true;
            this.savedWidth = 0;

            this.controlExtender = new LogicalTreeElementExtender(this);
            this.Bindings = new BindingContainer(this);
            this.Behaviors = new BehaviorCollection(this);
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// Gets the resources.
        /// </summary>
        public IDictionary<string, object> Resources
        {
            get { return this.resources; }
        }

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get { return base.Parent as ILogicalTreeElement; }
        }

        /// <summary>
        /// Gets the collection of child controls.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Controls
        {
            get
            {
                if (this.controlsWrapper == null)
                {
                    this.controlsWrapper = new NullIndexableCollection<ILogicalTreeElement>();
                }

                return this.controlsWrapper;
            }
        }

        /// <summary>
        /// Gets the collection of logical tree elements.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                return this.Controls;
            }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.controlExtender; }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Gets the behaviors collection.
        /// </summary>
        public BehaviorCollection Behaviors { get; private set; }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See ControlExtender for more information.
        /// <see cref="ControlExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get { return this.controlExtender.DataContext; }
            set { this.controlExtender.DataContext = value; }
        }

        /// <summary>
        /// 選択中のアイテム
        /// </summary>
        public new object SelectedItem
        {
            get
            {
                if ((this.SelectedIndex >= 0) &&
                    (this.SelectedIndex < this.Items.Count))
                {
                    var item = this.Items[this.SelectedIndex] as ComboBoxItem;
                    return item.Value;
                }
                else
                {
                    return null;
                }
            }

            set
            {
                int index = this.Items
                    .OfType<ComboBoxItem>()
                    .FindIndex(cbi => object.Equals(cbi.Value, value));

                if (index < 0 || index >= this.Items.Count)
                {
                    this.SelectedIndex = -1;
                    return;
                }

                if (this.SelectedIndex != index)
                {
                    this.SelectedIndex = index;
                    this.controlExtender.NotifyPropertyChanged(propertyName: "SelectedIndex");
                    this.controlExtender.NotifyPropertyChanged(propertyName: "SelectedItem");
                }
            }
        }

        /// <summary>
        /// アイテム一覧
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> AvailableItems
        {
            get
            {
                return this.availableItems;
            }

            set
            {
                IEnumerable oldValue = this.availableItems;

                if (this.controlExtender.SetValue(ref this.availableItems, value))
                {
                    // 古いアイテムにイベントを登録している場合、解除する。
                    if (oldValue != null && oldValue is INotifyCollectionChanged)
                    {
                        ((INotifyCollectionChanged)oldValue).CollectionChanged -=
                            this.OnCollectionChanged;
                    }

                    // INotifyPropertyChangedを継承しているインスタンスの場合、
                    // イベントを登録する。
                    if (value is INotifyCollectionChanged)
                    {
                        ((INotifyCollectionChanged)value).CollectionChanged +=
                            this.OnCollectionChanged;
                    }

                    // アイテムをクリアする。
                    Items.Clear();

                    // ComboBoxItemを登録する。
                    foreach (KeyValuePair<string, object> item in this.availableItems)
                    {
                        Items.Add(new ComboBoxItem(item.Key, item.Value));
                    }

                    // サイズの自動調整を実行する。
                    this.AdjustWidth();
                }
            }
        }

        /// <summary>
        /// サイズ自動調整フラグ
        /// </summary>
        public bool AutoAdjustWidth
        {
            get
            {
                return this.autoAdjustWidth;
            }

            set
            {
                this.autoAdjustWidth = value;
                this.AdjustWidth();
            }
        }

        /// <summary>
        /// サイズプロパティ
        /// </summary>
        public new int Width
        {
            get
            {
                return base.Width;
            }

            set
            {
                base.Width = value;
                this.savedWidth = value;
            }
        }

        /// <summary>
        /// Clears the DataContext.
        /// See ControlExtender for more details.
        /// <see cref="ControlExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.controlExtender.ClearDataContext();
        }

        /// <summary>
        /// Find the index of the combo box item with the specified name and value.
        /// </summary>
        /// <param name="item">A key value pair with the name and value to find.</param>
        /// <returns>The index of the combo box item or -1 if the item is not found.</returns>
        public int IndexOfComboBoxItem(KeyValuePair<string, object> item)
        {
            for (int i = 0; i < this.Items.Count; ++i)
            {
                var comboBoxItem = this.Items[i] as ComboBoxItem;
                if ((comboBoxItem != null) &&
                    (comboBoxItem.Name == item.Key) &&
                    (comboBoxItem.Value.Equals(item.Value) == true))
                {
                    // The item is found, return the index.
                    return i;
                }
            }

            return -1;
        }

        /// <summary>
        /// Find the combo box item with the specified name and value.
        /// </summary>
        /// <param name="item">A key value pair with the name and value to find.</param>
        /// <returns>The combo box item or null if the item is not found.</returns>
        public ComboBoxItem FindComboBoxItem(KeyValuePair<string, object> item)
        {
            for (int i = 0; i < this.Items.Count; ++i)
            {
                var comboBoxItem = this.Items[i] as ComboBoxItem;
                if ((comboBoxItem != null) &&
                    (comboBoxItem.Name == item.Key) &&
                    (comboBoxItem.Value.Equals(item.Value) == true))
                {
                    // The item is found, return the index.
                    return comboBoxItem;
                }
            }

            return null;
        }

        /// <summary>
        /// Find the index of the combo box item with the specified value.
        /// </summary>
        /// <param name="value">The value to find.</param>
        /// <returns>The index of the combo box item or -1 if the item is not found.</returns>
        public int IndexOfComboBoxItemByValue(object value)
        {
            for (int i = 0; i < this.Items.Count; ++i)
            {
                var comboBoxItem = this.Items[i] as ComboBoxItem;
                if ((comboBoxItem != null) &&
                    (comboBoxItem.Value.Equals(value) == true))
                {
                    // The item is found, return the index.
                    return i;
                }
            }

            return -1;
        }

        /// <summary>
        /// Find the combo box item with the specified value.
        /// </summary>
        /// <param name="value">The value to find.</param>
        /// <returns>The combo box item or null if the item is not found.</returns>
        public ComboBoxItem FindComboBoxItemByValue(object value)
        {
            for (int i = 0; i < this.Items.Count; ++i)
            {
                var comboBoxItem = this.Items[i] as ComboBoxItem;
                if ((comboBoxItem != null) &&
                    (comboBoxItem.Value.Equals(value) == true))
                {
                    // The item is found, return the index.
                    return comboBoxItem;
                }
            }

            return null;
        }

        /// <summary>
        /// DropDownStyleを変更した際に発生するイベントです.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected override void OnDropDownStyleChanged(EventArgs e)
        {
            base.OnDropDownStyleChanged(e);
        }

        /// <summary>
        /// Called when the user changes the selected item from the UI.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected override void OnSelectedIndexChanged(EventArgs e)
        {
            base.OnSelectedIndexChanged(e);

            this.controlExtender.NotifyPropertyChanged(propertyName: "SelectedIndex");
            this.controlExtender.NotifyPropertyChanged(propertyName: "SelectedItem");
        }

        /// <summary>
        /// コンボボックスの幅を自動調整するメソッドです.
        /// </summary>
        protected void AdjustWidth()
        {
            if (this.Items.Count <= 0)
            {
                return;
            }

            // サイズ自動調整を実行する場合
            if (this.autoAdjustWidth)
            {
                System.Drawing.Graphics g = this.ComboBox.CreateGraphics();

                int maxLength = 0;
                foreach (ComboBoxItem item in this.Items)
                {
                    maxLength = (int)Math.Max(maxLength, g.MeasureString(item.Name, this.Font).Width);
                }

                base.Width = maxLength + 30;
            }
            else
            {
                // 固定サイズを設定
                base.Width = this.savedWidth;
            }
        }

        /// <summary>
        /// コンボボックスのアイテムを描画する時に呼び出されるイベントです.
        /// </summary>
        /// <param name="sender">sender</param>
        /// <param name="e">event</param>
        private void OnDrawItem(object sender, DrawItemEventArgs e)
        {
            if (e.Index < 0)
            {
                return;
            }

            // 背景を描画
            if (this.Enabled == false)
            {
                DrawItemEventArgs disableArgs = new DrawItemEventArgs(
                    e.Graphics,
                    e.Font,
                    e.Bounds,
                    e.Index,
                    e.State,
                    e.ForeColor,
                    SystemColors.Control);

                disableArgs.DrawBackground();
            }
            else
            {
                e.DrawBackground();
            }

            // 文字列を描画
            string text = this.Items[e.Index].ToString();
            using (Brush brush = new SolidBrush(e.ForeColor))
            {
                Font font = this.Font;
                float textY = e.Bounds.Y + ((e.Bounds.Height -
                              e.Graphics.MeasureString(text, font).Height) / 2);

                e.Graphics.DrawString(text, font, brush, e.Bounds.X, textY);

                // フォーカスを示す矩形を描画
                e.DrawFocusRectangle();
            }
        }

        /// <summary>
        /// コレクションが変更された時に呼び出されるイベントです.
        /// </summary>
        /// <param name="sender">sender</param>
        /// <param name="e">event</param>
        private void OnCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            this.BeginUpdate();

            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                    {
                        // The items have to be KeyValuePairs.
                        var newItems = e.NewItems.Cast<KeyValuePair<string, object>>();
                        if (newItems == null)
                        {
                            this.EndUpdate();
                            return;
                        }

                        // Create the combo box items.
                        ComboBoxItem[] itemsToAdd =
                            newItems.Select(item => new ComboBoxItem(item.Key, item.Value)).ToArray();

                        // Add the items.
                        if (e.NewStartingIndex >= 0)
                        {
                            int index = e.NewStartingIndex;
                            foreach (ComboBoxItem item in itemsToAdd)
                            {
                                this.Items.Insert(index, item);
                                ++index;
                            }
                        }
                        else
                        {
                            this.Items.AddRange(itemsToAdd);
                        }

                        break;
                    }

                case NotifyCollectionChangedAction.Remove:
                    {
                        // The items have to be KeyValuePairs.
                        var items = e.OldItems.Cast<KeyValuePair<string, object>>();
                        if (items == null)
                        {
                            this.EndUpdate();
                            return;
                        }

                        // Remove the items.
                        foreach (KeyValuePair<string, object> item in items)
                        {
                            // Find the item from the combo box items and remove it.
                            int index = this.IndexOfComboBoxItem(item);
                            if ((index >= 0) && (index < this.Items.Count))
                            {
                                this.Items.RemoveAt(index);
                            }
                        }

                        break;
                    }

                case NotifyCollectionChangedAction.Replace:
                    {
                        // The items have to be KeyValuePairs.
                        var oldItems = e.OldItems.Cast<KeyValuePair<string, object>>().ToList();
                        var newItems = e.NewItems.Cast<KeyValuePair<string, object>>().ToList();
                        if (oldItems == null || newItems == null)
                        {
                            this.EndUpdate();
                            return;
                        }

                        // Replace the items.
                        for (int i = 0; (i < oldItems.Count) && (i < newItems.Count); ++i)
                        {
                            // Find the item from the combo box items and replace it.
                            int index = this.IndexOfComboBoxItem(oldItems[i]);
                            if ((index >= 0) && (index < this.Items.Count))
                            {
                                // The item is found, replace it.
                                this.Items[index] =
                                    new ComboBoxItem(newItems[i].Key, newItems[i].Value);
                            }
                        }

                        break;
                    }

                case NotifyCollectionChangedAction.Move:
                    {
                        if ((e.OldStartingIndex < 0) ||
                            (e.OldStartingIndex >= this.Items.Count))
                        {
                            this.EndUpdate();
                            return;
                        }

                        // Save the item first, it will be removed from the list.
                        var item = this.Items[e.OldStartingIndex];

                        // アイテムを削除する。
                        this.Items.RemoveAt(e.OldStartingIndex);

                        // アイテムを追加する。
                        if ((e.NewStartingIndex < 0) ||
                            (e.NewStartingIndex >= this.Items.Count))
                        {
                            this.Items.Add(item);
                        }
                        else
                        {
                            this.Items.Insert(e.NewStartingIndex, item);
                        }

                        break;
                    }

                case NotifyCollectionChangedAction.Reset:
                    {
                        this.SelectedIndex = -1;
                        Items.Clear();

                        break;
                    }
            }

            this.EndUpdate();

            // サイズの自動調整を実行する。
            this.AdjustWidth();
        }

        /// <summary>
        /// コンボボックスに実際に登録されるアイテム
        /// コンボボックスに表示される値は、インスタンスのtoString()の
        /// 値となり、意味のない文字列の可能性もあるため、nameに
        /// セットした値を表示する。
        /// </summary>
        public class ComboBoxItem
        {
            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="name">コンボボックスに表示する名前</param>
            /// <param name="value">The value of the combo box item.</param>
            public ComboBoxItem(string name, object value)
            {
                this.Name = name;
                this.Value = value;
            }

            /// <summary>
            /// コンボボックスに表示する名前
            /// </summary>
            public string Name { get; set; }

            /// <summary>
            /// The value of the combo box item.
            /// </summary>
            public object Value { get; private set; }

            /// <summary>
            /// コンボボックスに表示する名前を返す
            /// </summary>
            /// <returns>コンボボックスに表示する名前</returns>
            public override string ToString()
            {
                return this.Name;
            }
        }
    }
}
