﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.ComponentModel;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Render.Layout;
using EffectMaker.UIControls.Behaviors;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;
using EffectMaker.UIControls.Layout;

namespace EffectMaker.UIControls.BaseControls
{
    /// <summary>
    /// An extended list box control class.
    /// </summary>
    public class UIListBox : ListBox, IControl, ILayoutElement
    {
        /// <summary>
        /// アイテム一覧
        /// </summary>
        private IEnumerable<KeyValuePair<string, object>> availableItems;

        /// <summary>
        /// Backing field for the Extender property.
        /// </summary>
        private LogicalTreeElementExtender controlExtender;

        /// <summary>
        /// ILayoutElement extender.
        /// </summary>
        private LayoutElementExtender layoutElementExtender;

        /// <summary>
        /// Backing field for Controls property.
        /// </summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>
        /// Backing field for the Resources property.
        /// </summary>
        private IDictionary<string, object> resources = new Dictionary<string, object>();

        /// <summary>
        /// Constructor.
        /// </summary>
        public UIListBox()
        {
            this.controlExtender = new LogicalTreeElementExtender(this);
            this.layoutElementExtender = new LayoutElementExtender(this);
            this.Bindings = new BindingContainer(this);
            this.Behaviors = new BehaviorCollection(this);
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// VisibilityがCollapsedになった時に元のサイズを覚えておくためのプロパティ
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Size OriginalSize { get; set; }

        /// <summary>
        /// Gets the resources.
        /// </summary>
        public IDictionary<string, object> Resources
        {
            get { return this.resources; }
        }

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get { return base.Parent as ILogicalTreeElement; }
        }

        /// <summary>
        /// Gets the collection of child controls.
        /// </summary>
        public new IIndexableCollection<ILogicalTreeElement> Controls
        {
            get
            {
                if (this.controlsWrapper == null)
                {
                    this.controlsWrapper = new ControlCollectionWrapper(this);
                }

                return this.controlsWrapper;
            }
        }

        /// <summary>
        /// Gets the collection of logical tree elements.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                return this.Controls;
            }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.controlExtender; }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Gets the behaviors collection.
        /// </summary>
        public BehaviorCollection Behaviors { get; private set; }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See ControlExtender for more information.
        /// <see cref="ControlExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get { return this.controlExtender.DataContext; }
            set { this.controlExtender.DataContext = value; }
        }

        #region Layout

        /// <summary>
        /// Gets the layout element extender instance of this control.
        /// </summary>
        public LayoutElementExtender LayoutElementExtender
        {
            get { return this.layoutElementExtender; }
        }

        /// <summary>
        /// Gets or sets the visibility.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Visibility Visibility
        {
            get { return this.layoutElementExtender.Visibility; }
            set { this.layoutElementExtender.Visibility = value; }
        }

        /// <summary>
        /// Gets or sets the horizontal alignment.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public HAlignment HorizontalAlignment
        {
            get { return this.layoutElementExtender.HorizontalAlignment; }
            set { this.layoutElementExtender.HorizontalAlignment = value; }
        }

        /// <summary>
        /// Gets or sets the vertical alignment.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public VAlignment VerticalAlignment
        {
            get { return this.layoutElementExtender.VerticalAlignment; }
            set { this.layoutElementExtender.VerticalAlignment = value; }
        }

        #endregion // Layout

        /// <summary>
        /// 選択中のアイテム
        /// </summary>
        public new object SelectedItem
        {
            get
            {
                if ((this.SelectedIndex >= 0) &&
                    (this.SelectedIndex < this.Items.Count))
                {
                    var item = this.Items[this.SelectedIndex] as ListBoxItem;
                    return item.Value;
                }
                else
                {
                    return null;
                }
            }

            set
            {
                int index = this.Items
                    .OfType<ListBoxItem>()
                    .FindIndex(item => object.Equals(item.Value, value));

                if (index < 0 || index >= this.Items.Count)
                {
                    this.SelectedIndex = -1;
                    return;
                }

                if (this.SelectedIndex != index)
                {
                    this.SelectedIndex = index;
                    this.controlExtender.NotifyPropertyChanged(propertyName: "SelectedIndex");
                    this.controlExtender.NotifyPropertyChanged(propertyName: "SelectedItem");
                }
            }
        }

        /// <summary>
        /// アイテム一覧
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> AvailableItems
        {
            get
            {
                return this.availableItems;
            }

            set
            {
                IEnumerable oldValue = this.availableItems;

                if (this.controlExtender.SetValue(ref this.availableItems, value))
                {
                    // 古いアイテムにイベントを登録している場合、解除する。
                    if (oldValue != null && oldValue is INotifyCollectionChanged)
                    {
                        ((INotifyCollectionChanged)oldValue).CollectionChanged -=
                            this.OnCollectionChanged;
                    }

                    // INotifyPropertyChangedを継承しているインスタンスの場合、
                    // イベントを登録する。
                    if (value is INotifyCollectionChanged)
                    {
                        ((INotifyCollectionChanged)value).CollectionChanged +=
                            this.OnCollectionChanged;
                    }

                    // アイテムをクリアする。
                    Items.Clear();

                    // ListBoxItemを登録する。
                    foreach (KeyValuePair<string, object> item in this.availableItems)
                    {
                        Items.Add(new ListBoxItem(item.Key, item.Value));
                    }
                }
            }
        }

        /// <summary>
        /// Clears the DataContext.
        /// See ControlExtender for more details.
        /// <see cref="ControlExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.controlExtender.ClearDataContext();
        }

        /// <summary>
        /// Find the index of the list box item with the specified name and value.
        /// </summary>
        /// <param name="item">A key value pair with the name and value to find.</param>
        /// <returns>The index of the list box item or -1 if the item is not found.</returns>
        public int IndexOfListBoxItem(KeyValuePair<string, object> item)
        {
            for (int i = 0; i < this.Items.Count; ++i)
            {
                var listBoxItem = this.Items[i] as ListBoxItem;
                if ((listBoxItem != null) &&
                    (listBoxItem.Name == item.Key) &&
                    (listBoxItem.Value.Equals(item.Value) == true))
                {
                    // The item is found, return the index.
                    return i;
                }
            }

            return -1;
        }

        /// <summary>
        /// Find the list box item with the specified name and value.
        /// </summary>
        /// <param name="item">A key value pair with the name and value to find.</param>
        /// <returns>The list box item or null if the item is not found.</returns>
        public ListBoxItem FindListBoxItem(KeyValuePair<string, object> item)
        {
            for (int i = 0; i < this.Items.Count; ++i)
            {
                var listBoxItem = this.Items[i] as ListBoxItem;
                if ((listBoxItem != null) &&
                    (listBoxItem.Name == item.Key) &&
                    (listBoxItem.Value.Equals(item.Value) == true))
                {
                    // The item is found, return the index.
                    return listBoxItem;
                }
            }

            return null;
        }

        /// <summary>
        /// Find the index of the list box item with the specified value.
        /// </summary>
        /// <param name="value">The value to find.</param>
        /// <returns>The index of the list box item or -1 if the item is not found.</returns>
        public int IndexOfListBoxItemByValue(object value)
        {
            for (int i = 0; i < this.Items.Count; ++i)
            {
                var listBoxItem = this.Items[i] as ListBoxItem;
                if ((listBoxItem != null) &&
                    (listBoxItem.Value.Equals(value) == true))
                {
                    // The item is found, return the index.
                    return i;
                }
            }

            return -1;
        }

        /// <summary>
        /// Find the list box item with the specified value.
        /// </summary>
        /// <param name="value">The value to find.</param>
        /// <returns>The list box item or null if the item is not found.</returns>
        public ListBoxItem FindListBoxItemByValue(object value)
        {
            for (int i = 0; i < this.Items.Count; ++i)
            {
                var listBoxItem = this.Items[i] as ListBoxItem;
                if ((listBoxItem != null) &&
                    (listBoxItem.Value.Equals(value) == true))
                {
                    // The item is found, return the index.
                    return listBoxItem;
                }
            }

            return null;
        }

        /// <summary>
        /// Called when the user changes the selected item from the UI.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected override void OnSelectedIndexChanged(EventArgs e)
        {
            base.OnSelectedIndexChanged(e);

            this.controlExtender.NotifyPropertyChanged(propertyName: "SelectedIndex");
            this.controlExtender.NotifyPropertyChanged(propertyName: "SelectedItem");
        }

        /// <summary>
        /// コレクションが変更された時に呼び出されるイベントです.
        /// </summary>
        /// <param name="sender">sender</param>
        /// <param name="e">event</param>
        private void OnCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            this.BeginUpdate();

            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                    {
                        // The items have to be KeyValuePairs.
                        var newItems = e.NewItems.Cast<KeyValuePair<string, object>>();
                        if (newItems == null)
                        {
                            this.EndUpdate();
                            return;
                        }

                        // Create the list box items.
                        ListBoxItem[] itemsToAdd =
                            newItems.Select(item => new ListBoxItem(item.Key, item.Value)).ToArray();

                        // Add the items.
                        if (e.NewStartingIndex >= 0)
                        {
                            int index = e.NewStartingIndex;
                            foreach (ListBoxItem item in itemsToAdd)
                            {
                                this.Items.Insert(index, item);
                                ++index;
                            }
                        }
                        else
                        {
                            this.Items.AddRange(itemsToAdd);
                        }

                        break;
                    }

                case NotifyCollectionChangedAction.Remove:
                    {
                        // The items have to be KeyValuePairs.
                        var items = e.OldItems.Cast<KeyValuePair<string, object>>();
                        if (items == null)
                        {
                            this.EndUpdate();
                            return;
                        }

                        // Remove the items.
                        foreach (KeyValuePair<string, object> item in items)
                        {
                            // Find the item from the list box items and remove it.
                            int index = this.IndexOfListBoxItem(item);
                            if ((index >= 0) && (index < this.Items.Count))
                            {
                                this.Items.RemoveAt(index);
                            }
                        }

                        break;
                    }

                case NotifyCollectionChangedAction.Replace:
                    {
                        // The items have to be KeyValuePairs.
                        var oldItems = e.OldItems.Cast<KeyValuePair<string, object>>().ToList();
                        var newItems = e.NewItems.Cast<KeyValuePair<string, object>>().ToList();
                        if (oldItems == null || newItems == null)
                        {
                            this.EndUpdate();
                            return;
                        }

                        // Replace the items.
                        for (int i = 0; (i < oldItems.Count) && (i < newItems.Count); ++i)
                        {
                            // Find the item from the list box items and replace it.
                            int index = this.IndexOfListBoxItem(oldItems[i]);
                            if ((index >= 0) && (index < this.Items.Count))
                            {
                                // The item is found, replace it.
                                this.Items[index] =
                                    new ListBoxItem(newItems[i].Key, newItems[i].Value);
                            }
                        }

                        break;
                    }

                case NotifyCollectionChangedAction.Move:
                    {
                        if ((e.OldStartingIndex < 0) ||
                            (e.OldStartingIndex >= this.Items.Count))
                        {
                            this.EndUpdate();
                            return;
                        }

                        // Save the item first, it will be removed from the list.
                        var item = this.Items[e.OldStartingIndex];

                        // アイテムを削除する。
                        this.Items.RemoveAt(e.OldStartingIndex);

                        // アイテムを追加する。
                        if ((e.NewStartingIndex < 0) ||
                            (e.NewStartingIndex >= this.Items.Count))
                        {
                            this.Items.Add(item);
                        }
                        else
                        {
                            this.Items.Insert(e.NewStartingIndex, item);
                        }

                        break;
                    }

                case NotifyCollectionChangedAction.Reset:
                    {
                        this.SelectedIndex = -1;
                        Items.Clear();

                        break;
                    }
            }

            this.EndUpdate();
        }

        /// <summary>
        /// コンボボックスに実際に登録されるアイテム
        /// コンボボックスに表示される値は、インスタンスのtoString()の
        /// 値となり、意味のない文字列の可能性もあるため、nameに
        /// セットした値を表示する。
        /// </summary>
        public class ListBoxItem
        {
            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="name">コンボボックスに表示する名前</param>
            /// <param name="value">The value of the list box item.</param>
            public ListBoxItem(string name, object value)
            {
                this.Name = name;
                this.Value = value;
            }

            /// <summary>
            /// コンボボックスに表示する名前
            /// </summary>
            public string Name { get; set; }

            /// <summary>
            /// The value of the list box item.
            /// </summary>
            public object Value { get; private set; }

            /// <summary>
            /// コンボボックスに表示する名前を返す
            /// </summary>
            /// <returns>コンボボックスに表示する名前</returns>
            public override string ToString()
            {
                return this.Name;
            }
        }
    }
}
