﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Windows.Forms;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.Specifics.TreeNodes;

namespace EffectMaker.UIControls.Specifics
{
    /// <summary>
    /// ワークスペースを表示するためのツリービューです.
    /// </summary>
    public class WorkspaceTreeView : UITreeView
    {
        /// <summary>
        /// マウスの位置です.
        /// OnBeforeSelect()の判定に使います.
        /// </summary>
        private Point mousePos;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public WorkspaceTreeView()
        {
            // リネーム用のUIを作成.
            this.RenameEditBox = new RenameEditBox();

            // イベントハンドラを登録.
            this.MouseDoubleClick += this.OnMouseDoubleClick;
        }

        /// <summary>
        /// リネーム用のUIです.
        /// WorkspaceTreeViewでリネームを可能にするには親コンポーネントのControlsに
        /// このRenameEditBoxをAddする必要があります.
        /// </summary>
        public RenameEditBox RenameEditBox { get; set; }

        /// <summary>
        /// Delキーで行う処理を取得または設定します.
        /// </summary>
        public Action DeleteAction { get; set; }

        /// <summary>
        /// リネームを開始します.
        /// </summary>
        public void BeginRename()
        {
            var node = this.SelectedNode as UITreeNode;
            if (node == null || node.CanRename == false)
            {
                return;
            }

            this.RenameEditBox.BeginRename(node);
        }

        /// <summary>
        /// マウスを移動したときの処理を行います.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            base.OnMouseMove(e);

            // OnBeforeSelect()の判定に使うためマウスの位置を記録する.
            this.mousePos = e.Location;
        }

        /// <summary>
        /// 選択ノードを変更する直前の処理を行います.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected override void OnBeforeSelect(TreeViewCancelEventArgs e)
        {
            // 表示アイコンをクリックしたときは選択ノードの変更をキャンセルする.
            var node = e.Node as ProjectTreeNodeBase;
            if (node != null)
            {
                if (node.ShowButtonRectangle.Contains(this.mousePos))
                {
                    e.Cancel = true;
                }
            }

            base.OnBeforeSelect(e);
        }

        /// <summary>
        /// Deleteキーハンドリング用
        /// </summary>
        /// <param name="e">イベント情報</param>
        protected override void OnPreviewKeyDown(PreviewKeyDownEventArgs e)
        {
            if (e.KeyData == Keys.Delete)
            {
                if (this.DeleteAction != null)
                {
                    this.DeleteAction();
                }
            }

            base.OnPreviewKeyDown(e);
        }

        /// <summary>
        /// マウスをダブルクリックしたときの処理を行います。
        /// </summary>
        /// <param name="sender">The sender object.</param>
        /// <param name="e">Event arguments.</param>
        private void OnMouseDoubleClick(object sender, MouseEventArgs e)
        {
            // 選択中のノードを取得
            var node = this.SelectedNode as UITreeNode;
            if (node == null || node.CanRename == false)
            {
                return;
            }

            // ノード名をダブルクリックしたときリネームを開始
            if (node.NameRectangle.Contains(e.Location))
            {
                this.BeginRename();
            }
        }
    }
}
