﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using EffectMaker.BusinessLogic.Protocol;
using EffectMaker.BusinessLogic.ViewerMessages;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.Extensions;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UIControls.Specifics.TreeNodes
{
    /// <summary>
    /// An extended ProjectTreeNodeBase for preview implementation.
    /// </summary>
    public class PreviewTreeNode : ProjectTreeNodeBase
    {
        /// <summary>
        /// Initializes the PreviewTreeNode instance.
        /// </summary>
        public PreviewTreeNode()
        {
            this.CanRename = true;

            this.IsDrawModifyMark = true;

            this.AddBinding("Text", "Name");
            this.AddBinding("IsDisplayed", "Displayed");
        }

        /// <summary>
        /// Render the node of viewer.
        /// </summary>
        /// <param name="bounds">The rectangular region in whic to render.</param>
        /// <param name="state">The current node state.</param>
        /// <param name="gr">The graphics instance to render to.</param>
        /// <returns>Returns true if it self render, or false to tall parent
        /// TreeView to perform default node rendering.</returns>
        protected internal override bool Render(Rectangle bounds, TreeNodeStates state, Graphics gr)
        {
            // The tree node might already be removed between the draw message is dispatched
            // and the actual rendering.
            // (the message dispatches, and the rendering happens on the next application loop)
            if (((TreeNode)this).Parent == null &&
                (this.TreeView == null || this.TreeView.Nodes.Contains(this) == false))
            {
                return true;
            }

            // ノード作成時に無効な領域の描画呼び出しが発生する
            if (bounds.Width == 0 || bounds.Height == 0)
            {
                return true;
            }

            // 背景を描画
            this.DrawBackground(bounds, state, gr);

            // 目のアイコンを描画
            this.DrawDisplayedIcon(bounds, state, gr);

            // 階層情報の表示位置を取得
            int locationX = ProjectTreeNodeBase.ShowButtonWidth +
                            ProjectTreeNodeBase.ShowButtonMarginRight +
                            bounds.Left;

            // 階層情報を描画
            locationX = this.DrawHierarchy(bounds, state, gr, locationX);

            // ノードアイコンを描画
            locationX = this.DrawNodeTypeIcon(bounds, state, gr, locationX, Properties.Resources.Icon_PreviewSetting);

            // 編集マークを描画
            locationX = this.DrawModifyMark(bounds, state, gr, locationX);

            // テキストを描画
            this.DrawText(bounds, state, gr, locationX, this.Text, this.NodeFont);

            return true;
        }

        /// <summary>
        /// コンテキストメニューのセットアップ.
        /// </summary>
        protected override void SetupContextMenu()
        {
            var docMenu = (UIContextMenuStrip)this.ContextMenuStrip;

            WorkspaceRootViewModel workspaceRootViewModel = null;
            var dataContext = this.DataContext as IHierarchyObject;
            if (dataContext == null)
            {
                Logger.Log(
                    LogLevels.Error,
                    "EmitterSetTreeNode.SetupContextMenu : can not to get WorkspaceRootViewModel.");
                return;
            }

            workspaceRootViewModel = dataContext.FindFarthestParentOfType<WorkspaceRootViewModel>();
            WorkspaceController wc = this.GetWorkspaceController();

            // プレビューを追加.
            var addPreviewMenu = new UIToolStripMenuItem()
            {
                Text = Properties.Resources.MenuItemNameAddPreview,
                Image = Properties.Resources.Icon_PreviewSetting,
                ImageTransparentColor = Color.White,
            };
            addPreviewMenu.Bindings.Add(new UIControls.DataBinding.Binder(
                addPreviewMenu, "Enabled", "CanCreateNewPreview"));
            addPreviewMenu.DataContext = workspaceRootViewModel;
            addPreviewMenu.Click += (s, e) => wc.CreateNewPreview();

            // コピー
            var copyMenu = new UIToolStripMenuItem
            {
                Text = Properties.Resources.MenuCopyNode,
                ShortcutKeys = Keys.Control | Keys.C,
                Image = Properties.Resources.Icon_Copy,
            };

            // ペースト
            var pasteMenu = new UIToolStripMenuItem
            {
                Text = Properties.Resources.MenuPasteNodeContent,
                ShortcutKeys = Keys.Control | Keys.V,
                Image = Properties.Resources.Icon_Paste,
            };

            // 複製
            var duplicateMenu = new UIToolStripMenuItem
            {
                Text = Properties.Resources.MenuItemNameDuplicate,
                ShortcutKeys = Keys.Control | Keys.D,
                Image = Properties.Resources.Icon_DuplicateNode,
            };
            duplicateMenu.Bindings.Add(
                new UIControls.DataBinding.Binder(
                    duplicateMenu, "Enabled", "CanDuplicateSelectedNode"));
            duplicateMenu.DataContext = WorkspaceRootViewModel.Instance;
            duplicateMenu.Click += (s, e) => WorkspaceRootViewModel.Instance.Controller.DuplicateSelectedNode();

            // 名前を変更
            var renameMenu = new UIToolStripMenuItem()
            {
                Text = Properties.Resources.MenuItemNameRename,
                ShortcutKeys = Keys.F2,
                Image = Properties.Resources.Icon_ChangeName,
                ImageTransparentColor = Color.White,
            };
            renameMenu.Click += (s, e) =>
            {
                var treeView = this.TreeView as WorkspaceTreeView;
                if (treeView != null)
                {
                    treeView.BeginRename();
                }
            };

            // 保存
            var saveMenu = new UIToolStripMenuItem
            {
                Text = Properties.Resources.MenuItemNameSave,
                ShortcutKeys = Keys.Control | Keys.S,
                Image = Properties.Resources.Icon_FileSave,
            };

            saveMenu.Click += (ss, ee) => workspaceRootViewModel.OnFileSave(false);

            saveMenu.Bindings.Add(
                new UIControls.DataBinding.Binder(
                    saveMenu, "Enabled", "CanSaveSelectedNode"));
            saveMenu.DataContext = workspaceRootViewModel;

            // 別名で保存
            var saveAsMenu = new UIToolStripMenuItem
            {
                Text = Properties.Resources.MenuItemNameSaveAs,
                ShortcutKeys = Keys.Control | Keys.Shift | Keys.S,
                Image = Properties.Resources.Icon_SaveAs,
            };

            saveAsMenu.Click += (ss, ee) => workspaceRootViewModel.OnFileSave(true);

            saveAsMenu.Bindings.Add(
                new UIControls.DataBinding.Binder(
                    saveAsMenu, "Enabled", "CanSaveAsSelectedNode"));
            saveAsMenu.DataContext = workspaceRootViewModel;

            var deleteMenu = new UIToolStripMenuItem
            {
                Text = Properties.Resources.MenuCloseNode,
                Image = Properties.Resources.Icon_Delete,
                ShortcutKeys = Keys.Control | Keys.W,
            };

            deleteMenu.Bindings.Add(new UIControls.DataBinding.Binder(
                deleteMenu,
                "Enabled",
                "CanRemoveSelectedNode"));
            deleteMenu.DataContext = WorkspaceRootViewModel.Instance;
            deleteMenu.Click += (ss, ee) => WorkspaceRootViewModel.Instance.Controller.Remove();

            // --セパレータ

            // エクスプローラで開く.
            var openExplorerMenu = new UIToolStripMenuItem
            {
                Text = Properties.Resources.MenuItemNameOpenExplorer,
                ShortcutKeys = Keys.Control | Keys.E,
                Image = Properties.Resources.Icon_Explorer,
            };

            // コンテキストメニューをセット.
            docMenu.Items.AddRange(
                new ToolStripItem[]
                {
                    addPreviewMenu,
                    new ToolStripSeparator(),
                    copyMenu,
                    pasteMenu,
                    duplicateMenu,
                    renameMenu,
                    saveMenu,
                    saveAsMenu,
                    deleteMenu,
                    new ToolStripSeparator(),
                    openExplorerMenu,
                });

            copyMenu
                .AddBinding("Executable", "NodeCopyExecutable")
                .UpdateElement();

            pasteMenu
                .AddBinding("Executable", "NodePasteExecutable")
                .UpdateElement();

            ////duplicateMenu
            ////    .AddBinding("Executable", "NodeDuplicateExecutable")
            ////    .UpdateElement();

            openExplorerMenu
                .AddBinding("Executable", "NodeOpenExlorerExecutable")
                .UpdateElement();

            docMenu
                .AddBinding("ContextMenuOpeningExecutable", "EvaluateMenuItemsExecutable")
                .UpdateElement();
        }
    }
}
