﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using EffectMaker.Foundation.Collections.Generic;
using EffectMaker.Foundation.Log;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.Extensions;

namespace EffectMaker.UIControls.Specifics.TexturePattern
{
    /// <summary>
    /// テクスチャパターンテーブルのUIです.
    /// </summary>
    public partial class TexturePatternTable : UIUserControl
    {
        /// <summary>
        /// テーブルの長さです.
        /// </summary>
        private const int TableLength = 32;

        /// <summary>
        /// テーブルデータです.
        /// </summary>
        private ArrayCollection<int> table = new ArrayCollection<int>(TableLength);

        /// <summary>
        /// 使用テーブル数です.
        /// TableCountのset処理が走るようにあえて初期値を入れない.
        /// </summary>
        private int tableCount = 0;

        /// <summary>
        /// 簡易入力UIです.
        /// </summary>
        private UITextBox simpleTextBox;

        /// <summary>
        /// コメントラベルです.
        /// </summary>
        private UIBorderedLabel commentLabel;

        /// <summary>
        /// 配列入力UIです.
        /// </summary>
        private UITextBox[] arrayTextBox;

        /// <summary>
        /// 配列入力UIのラベルです.
        /// </summary>
        private UILabel[] arrayTextLabel;

        /// <summary>
        /// 配列入力UIの位置です.
        /// </summary>
        private int arrayBoxPosX = 6;

        /// <summary>
        /// 配列入力UIのマージンです.
        /// </summary>
        private Padding arrayBoxMargin = new Padding(6, 0, 1, 0);

        /// <summary>
        /// 配列入力UIの横幅です.
        /// </summary>
        private int arrayBoxWidth = 17;

        /// <summary>
        /// 配列入力UIの高さです.
        /// </summary>
        private int arrayBoxHeight = 18;

        /// <summary>
        /// 配列入力UIの左右間隔です.
        /// </summary>
        private int arrayBoxWidthInterval = 1;

        /// <summary>
        /// 配列入力UIの上下間隔です.
        /// </summary>
        private int arrayBoxHeightInterval = 11;

        /// <summary>
        /// 配列入力UIのラベルの位置です.
        /// </summary>
        private int arrayBoxLabelPosX = -2;

        /// <summary>
        /// 簡易入力UIの更新フラグです.
        /// 簡易入力UIの書き換えによりTableを変更するときはfalseにします.
        /// </summary>
        private bool updateSimpleTable = true;

        /// <summary>
        /// 配列入力UIの更新フラグです.
        /// 配列入力UIの書き換えによりTableを変更するときはfalseにします.
        /// </summary>
        private bool updateArrayTable = true;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public TexturePatternTable()
        {
            this.InitializeComponent();

            // 簡易入力UIを作成
            this.simpleTextBox = new UITextBox()
            {
                Text = "0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0",
                HorizontalAlignment = Foundation.Render.Layout.HAlignment.Stretch,
                Margin = new Padding(4, 0, 0, 3),
            };
            this.simpleInput.Controls.Add(this.simpleTextBox);

            this.simpleTextBox.KeyDown += this.OnSimpleInputUiKeyDown;
            this.simpleTextBox.Leave += this.OnSimpleInputUiLeave;

            // コメントラベルを作成
            this.commentLabel = new UIBorderedLabel()
            {
                Text = string.Empty,
                TextAlign = ContentAlignment.MiddleLeft,
                BorderColor = Color.FromArgb(0xA9, 0xA9, 0xA9),
                BackColor = Color.FromArgb(0xF0, 0xF0, 0xFF),
////                MinimumSize = new Size(298, 0),
////                MaxWidth = 330,
                Height = 18,
                Margin = new Padding(4, 3, 0, 8),
                HorizontalAlignment = Foundation.Render.Layout.HAlignment.Stretch
            };
            this.simpleInput.Controls.Add(this.commentLabel);

            // 配列入力UIを作成
            this.arrayTextBox = new UITextBox[32];

            for (int i = 0; i < 32; ++i)
            {
                int x = i % 16;
                int y = i / 16;

                this.arrayTextBox[i] = new UITextBox()
                {
                    Text = "0",
                    TextAlign = System.Windows.Forms.HorizontalAlignment.Right,
                    Width = this.ArrayBoxWidth,
                    Height = this.ArrayBoxHeight,
                    Location = new Point(
                        this.arrayBoxMargin.Left + (x * (this.ArrayBoxWidth + this.ArrayBoxWidthInterval)),
                        y * (this.ArrayBoxHeight + this.ArrayBoxHeightInterval)),
                };

                this.Controls.Add(this.arrayTextBox[i]);

                this.arrayTextBox[i].KeyDown += this.OnArrayInputUiKeyDown;
                this.arrayTextBox[i].Leave += this.OnArrayInputUiLeave;
            }

            // 配列入力UIのラベルを作成
            this.arrayTextLabel = new UILabel[32];
            Font labelFont = new Font("Arial", 6);

            for (int i = 0; i < 32; ++i)
            {
                int x = i % 16;
                int y = i / 16;

                this.arrayTextLabel[i] = new UILabel()
                {
                    Text = i.ToString(),
                    Width = this.ArrayBoxWidth,
                    Font = labelFont,
                    Location = new Point(
                        this.arrayBoxMargin.Left + this.arrayBoxLabelPosX + (x * (this.ArrayBoxWidth + this.ArrayBoxWidthInterval)),
                        (y * (this.ArrayBoxHeight + this.ArrayBoxHeightInterval)) + this.ArrayBoxHeight),
                };

                this.Controls.Add(this.arrayTextLabel[i]);
            }

            // TableCountを設定して配列入力UIのenable設定を走らす
            this.TableCount = 2;
        }

        /// <summary>
        /// 配列入力UIの横幅を取得または設定します.
        /// </summary>
        [DefaultValue(17)]
        public int ArrayBoxWidth
        {
            get
            {
                return this.arrayBoxWidth;
            }

            set
            {
                if (value == this.arrayBoxWidth)
                {
                    return;
                }

                this.arrayBoxWidth = value;
            }
        }

        /// <summary>
        /// 配列入力UIの高さを取得または設定します.
        /// </summary>
        [DefaultValue(18)]
        public int ArrayBoxHeight
        {
            get
            {
                return this.arrayBoxHeight;
            }

            set
            {
                if (value == this.arrayBoxHeight)
                {
                    return;
                }

                this.arrayBoxHeight = value;
            }
        }

        /// <summary>
        /// 配列入力UIの左右間隔を取得または設定します.
        /// </summary>
        [DefaultValue(2)]
        public int ArrayBoxWidthInterval
        {
            get
            {
                return this.arrayBoxWidthInterval;
            }

            set
            {
                if (value == this.arrayBoxWidthInterval)
                {
                    return;
                }

                this.arrayBoxWidthInterval = value;
            }
        }

        /// <summary>
        /// 配列入力UIの上下間隔を取得または設定します.
        /// </summary>
        [DefaultValue(11)]
        public int ArrayBoxHeightInterval
        {
            get
            {
                return this.arrayBoxHeightInterval;
            }

            set
            {
                if (value == this.arrayBoxHeightInterval)
                {
                    return;
                }

                this.arrayBoxHeightInterval = value;
            }
        }

        /// <summary>
        /// ラベルに表示するテキストを取得または設定します.
        /// </summary>
        [DefaultValue("")]
        public string LabelText
        {
            get
            {
                return this.simpleInput.Text;
            }

            set
            {
                this.simpleInput.Text = value;
            }
        }

        /// <summary>
        /// コメントに表示するテキストを取得または設定します.
        /// </summary>
        [DefaultValue("")]
        public string CommentText
        {
            get
            {
                return this.commentLabel.Text;
            }

            set
            {
                this.commentLabel.Text = value;
            }
        }

        /// <summary>
        /// パターンテーブルを取得または設定します.
        /// </summary>
        public ArrayCollection<int> Table
        {
            get
            {
                return this.table;
            }

            set
            {
                if (value == null || value.Equals(this.table))
                {
                    return;
                }

                this.table = (ArrayCollection<int>)value.Clone();

                if (this.updateSimpleTable)
                {
                    this.UpdateSimpleTable();
                }

                if (this.updateArrayTable)
                {
                    this.UpdateArrayTable();
                }

                this.LogicalTreeElementExtender.NotifyPropertyChanged(propertyName: "Table");
            }
        }

        /// <summary>
        /// Get or set the data binding target for the modification flags.
        /// (the data binding target for the caption label.)
        /// </summary>
        public string ModificationFlagTarget
        {
            get
            {
                return this.simpleInput.ModificationFlagTarget;
            }

            set
            {
                this.simpleInput.ModificationFlagTarget = value;
            }
        }

        /// <summary>
        /// 使用テーブル数を取得または設定します.
        /// </summary>
        [DefaultValue(2)]
        public int TableCount
        {
            get
            {
                return this.tableCount;
            }

            set
            {
                if (value > TableLength)
                {
                    value = TableLength;
                }

                if (value == this.tableCount)
                {
                    return;
                }

                this.tableCount = value;

                for (int i = 0; i < this.tableCount; ++i)
                {
                    this.arrayTextBox[i].Enabled = true;
                }

                for (int i = this.tableCount; i < TableLength; ++i)
                {
                    this.arrayTextBox[i].Enabled = false;
                }
            }
        }

        /// <summary>
        /// ヘルプとしてリンクするページのIDを取得または設定します。
        /// </summary>
        public string DocumentId
        {
            get { return simpleInput.DocumentId; }
            set { this.simpleInput.DocumentId = value; }
        }

        /// <summary>
        /// Called when a parent request the desired size.
        /// </summary>
        /// <param name="proposedSize">The available parent size.</param>
        /// <returns>Returns the desired sife of the control.</returns>
        public override Size GetPreferredSize(Size proposedSize)
        {
            if (this.IsSelfOrParentCollapsed() == true)
            {
                return Size.Empty;
            }

            Size preferredSize = new Size(0, 0);  // 推薦サイズ
            Size availableSize = proposedSize;    // レイアウト可能なサイズ

            // パディングを反映
            preferredSize.Width += this.Padding.Horizontal;
            availableSize.Width -= this.Padding.Horizontal;

            preferredSize.Height += this.Padding.Vertical;
            availableSize.Height -= this.Padding.Vertical;

            // 簡易入力UIのサイズを反映
            Size simpleInputSize = this.simpleInput
                .GetElementDisplaySize(availableSize);

            preferredSize.Height += simpleInputSize.Height;
            availableSize.Height -= simpleInputSize.Height;

            // 配列入力UIのサイズを反映
            preferredSize.Height += 2 * (this.arrayBoxHeight + this.arrayBoxHeightInterval);
            availableSize.Height -= 2 * (this.arrayBoxHeight + this.arrayBoxHeightInterval);

            preferredSize.Width += this.arrayBoxMargin.Left + (16 * (this.arrayBoxWidth + this.arrayBoxWidthInterval)) + this.arrayBoxMargin.Right;
            availableSize.Width -= this.arrayBoxMargin.Left + (16 * (this.arrayBoxWidth + this.arrayBoxWidthInterval)) + this.arrayBoxMargin.Right;

            return preferredSize;
        }

        /// <summary>
        /// Called when a layout process is running.
        /// </summary>
        /// <param name="e">The event argument.</param>
        protected override void OnLayout(LayoutEventArgs e)
        {
            if (this.IsSelfOrParentCollapsed() == true)
            {
                return;
            }

            base.OnLayout(e);

            Size availableSize = this.ClientSize;  // レイアウト可能なサイズ
            Point location = new Point(0, 0);      // レイアウト位置

            // パディングを反映
            availableSize.Width -= this.Padding.Horizontal;
            availableSize.Height -= this.Padding.Vertical;

            location.X += this.Padding.Left;
            location.Y += this.Padding.Top;

            // 簡易入力UIの位置を設定
            this.simpleInput.Location = location;

            // 簡易入力UIのサイズを設定
            Size simpleInputSize = this.simpleInput
                .GetElementDisplaySize(availableSize);

            if (simpleInputSize.Width < availableSize.Width)
            {
                simpleInputSize.Width = availableSize.Width;
            }

            this.simpleInput.SetElementDisplaySize(simpleInputSize);

            availableSize.Height -= simpleInputSize.Height;
            location.Y += simpleInputSize.Height;

            // コンストラクタより前にこの処理が呼ばれることがある
            if (this.arrayTextBox == null)
            {
                return;
            }

            // 配列入力UIの位置を設定
            for (int i = 0; i < 32; ++i)
            {
                int x = i % 16;
                int y = i / 16;

                if (this.arrayTextBox[i] == null)
                {
                    break;
                }

                this.arrayTextBox[i].Location = new Point(
                    this.arrayBoxPosX + location.X + (x * (this.arrayBoxWidth + this.arrayBoxWidthInterval)),
                    location.Y + (y * (this.arrayBoxHeight + this.arrayBoxHeightInterval)));

                this.arrayTextBox[i].Width = this.arrayBoxWidth;
                this.arrayTextBox[i].Height = this.arrayBoxHeight;
            }

            // コンストラクタより前にこの処理が呼ばれることがある
            if (this.arrayTextLabel == null)
            {
                return;
            }

            // 配列入力UIラベルの位置を設定
            for (int i = 0; i < 32; ++i)
            {
                int x = i % 16;
                int y = i / 16;

                if (this.arrayTextLabel[i] == null)
                {
                    break;
                }

                this.arrayTextLabel[i].Location = new Point(
                    this.arrayBoxPosX + this.arrayBoxLabelPosX + location.X + (x * (this.arrayBoxWidth + this.arrayBoxWidthInterval)),
                    location.Y + (y * (this.arrayBoxHeight + this.arrayBoxHeightInterval)) + this.arrayBoxHeight);
            }
        }

        /// <summary>
        /// 簡易入力UIからパターンテーブルを入力します.
        /// </summary>
        private void LoadSimpleTable()
        {
            ArrayCollection<int> newTable = (ArrayCollection<int>)this.Table.Clone();

            // テキストをコンマで区切られたトークンに分離
            string[] tokens = this.simpleTextBox.Text.Split(',');

            int arrayIndex = 0;  // 番号を記録する配列インデックス
            int tokenIndex = 0;  // 処理対象のトークンインデックス

            // トークンごとに処理を行う
            while (tokenIndex < tokens.Length && arrayIndex < TableLength)
            {
                string token = tokens[tokenIndex];

                // 連番記号'-'を検索
                int hyphenIndex = token.IndexOf('-');

                if (hyphenIndex >= 0)
                {
                    // 連番記号が見つかったときの処理

                    // 連番の開始番号を取得
                    int numBegin;
                    string strNumBegin = token.Substring(0, hyphenIndex);
                    if (int.TryParse(strNumBegin, out numBegin) == false)
                    {
                        newTable[arrayIndex] = 0;
                        ++tokenIndex;
                        ++arrayIndex;
                        continue;
                    }

                    // 連番の終了番号を取得
                    int numEnd;
                    string strNumEnd = token.Substring(hyphenIndex + 1, token.Length - hyphenIndex - 1);
                    if (int.TryParse(strNumEnd, out numEnd) == false)
                    {
                        newTable[arrayIndex] = 0;
                        ++tokenIndex;
                        ++arrayIndex;
                        continue;
                    }

                    // 連番を配列に記録
                    for (int num = numBegin; num <= numEnd; ++num)
                    {
                        if (arrayIndex >= TableLength)
                        {
                            break;
                        }

                        newTable[arrayIndex] = num;
                        ++arrayIndex;
                    }

                    ++tokenIndex;
                }
                else
                {
                    // 連番記号が見つからなかったときの処理

                    // 番号を取得
                    int num;
                    if (int.TryParse(token, out num) == false)
                    {
                        num = 0;
                    }

                    // 番号を配列に記録
                    newTable[arrayIndex] = num;
                    ++tokenIndex;
                    ++arrayIndex;
                }
            }

            // 一時的に簡易入力UIの更新を止めてパターンテーブルを更新する
            this.updateSimpleTable = false;

            this.Table = newTable;

            this.updateSimpleTable = true;
        }

        /// <summary>
        /// 配列入力UIからパターンテーブルを入力します.
        /// </summary>
        /// <param name="index">変更されたUIの配列インデックス.</param>
        private void LoadArrayTable(int index)
        {
            int num;

            // 変更されたUIの値を取得
            if (int.TryParse(this.arrayTextBox[index].Text, out num) == false)
            {
                // 不正な数値が入力されたとき元に戻す
                this.arrayTextBox[index].Text = this.table[index].ToString();
                return;
            }

            // 一時的に配列入力UIの更新を止めてパターンテーブルを更新する
            this.updateArrayTable = false;

            var newTable = (ArrayCollection<int>)this.table.Clone();
            newTable[index] = num;
            this.Table = newTable;

            this.updateArrayTable = true;
        }

        /// <summary>
        /// 簡易入力UIの内容を更新します.
        /// </summary>
        private void UpdateSimpleTable()
        {
            string strArray = string.Empty;

            // パターンテーブルを文字列に変換
            for (int i = 0; i < TableLength;)
            {
                if (i > 0)
                {
                    strArray += ",";
                }

                // 連番の長さを取得
                int j;  // 連番の長さ

                for (j = 1; i + j < TableLength; ++j)
                {
                    if (this.table[i + j] != this.table[i] + j)
                    {
                        break;
                    }
                }

                if (j >= 3)
                {
                    // 3つ以上の連番のとき、連番記号を使った文字列に変換
                    strArray += this.table[i] + "-" + this.table[i + j - 1];
                    i += j;
                }
                else
                {
                    // 連番でないとき、そのまま変換
                    strArray += this.table[i];
                    ++i;
                }
            }

            // 簡易入力UIを更新
            this.simpleTextBox.Text = strArray;
        }

        /// <summary>
        /// 配列入力UIの内容を更新します.
        /// </summary>
        private void UpdateArrayTable()
        {
            for (int i = 0; i < TableLength; ++i)
            {
                this.arrayTextBox[i].Text = this.table[i].ToString();
            }
        }

        #region Event handlers

        /// <summary>
        /// 簡易入力UIでキーが押されたときの処理を行います.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The event arguments.</param>
        private void OnSimpleInputUiKeyDown(object sender, KeyEventArgs e)
        {
/*
            if (this.simpleTextBox.ReadOnly)
            {
                return;
            }
 */

            if (e.KeyCode == Keys.Enter)
            {
                this.LoadSimpleTable();
            }
        }

        /// <summary>
        /// 簡易入力UIがフォーカスを失ったときの処理を行います.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The event arguments.</param>
        private void OnSimpleInputUiLeave(object sender, EventArgs e)
        {
/*
            if (this.simpleTextBox.ReadOnly)
            {
                return;
            }
 */

            this.LoadSimpleTable();
        }

        /// <summary>
        /// 配列入力UIでキーが押されたときの処理を行います.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The event arguments.</param>
        private void OnArrayInputUiKeyDown(object sender, KeyEventArgs e)
        {
            int index = Array.IndexOf(this.arrayTextBox, sender);

            if (e.KeyCode == Keys.Enter)
            {
                this.LoadArrayTable(index);
            }
        }

        /// <summary>
        /// 配列入力UIがフォーカスを失ったときの処理を行います.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The event arguments.</param>
        private void OnArrayInputUiLeave(object sender, EventArgs e)
        {
            int index = Array.IndexOf(this.arrayTextBox, sender);

            if (index > 0)
            {
                this.LoadArrayTable(index);
            }
        }

        #endregion
    }
}
