﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.ComponentModel;
using System.Drawing;
using System.IO;
using System.Windows.Forms;
using EffectMaker.Foundation;
using EffectMaker.Foundation.Texture;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.Layout;
using EffectMaker.UILogic.Manager;

namespace EffectMaker.UIControls.Specifics
{
    /// <summary>
    /// テクスチャ詳細ダイアログ
    /// </summary>
    public partial class TextureDetailPanel : UIUserControl
    {
        /// <summary>
        /// 対象ファイルパス
        /// </summary>
        private string targetFilePath = string.Empty;

        /// <summary>
        /// イメージビュー
        /// </summary>
        private TextureImageView imageView;

        /// <summary>
        /// 要求されたチャンネル
        /// </summary>
        private ColorChannelFlags requestColorChannel = ColorChannelFlags.None;

        /// <summary>
        ///  コンストラクタ
        /// </summary>
        public TextureDetailPanel()
        {
            this.InitializeComponent();
            this.tsiShowHelp.ToolTipText = Properties.Resources.TextureDeailPanelToolTipHelp;
            this.tsiMagnify.ToolTipText = Properties.Resources.TextureDeailPanelToolTipMagnification;
            this.tsiMagnifyOptions.ToolTipText = Properties.Resources.TextureDeailPanelToolTipChangeMagnification;
            this.tsiMagnifyMinus.ToolTipText = Properties.Resources.TextureDeailPanelToolTipZoomOut;
            this.tsiMagnifyPlus.ToolTipText = Properties.Resources.TextureDeailPanelToolTipZoomIn;
            this.tsiChannelRGB.ToolTipText = Properties.Resources.TextureDeailPanelToolTipRGBChannel;
            this.tsiChannelR.ToolTipText = Properties.Resources.TextureDeailPanelToolTipRChannel;
            this.tsiChannelG.ToolTipText = Properties.Resources.TextureDeailPanelToolTipGChannel;
            this.tsiChannelB.ToolTipText = Properties.Resources.TextureDeailPanelToolTipBChannel;
            this.tsiChannelA.ToolTipText = Properties.Resources.TextureDeailPanelToolTipAChannel;
            this.cbxDepth.ToolTipText = Properties.Resources.TextureDeailPanelToolTipDepth;
            this.tsiShowMipmap.ToolTipText = Properties.Resources.TextureDeailPanelToolTipShowMipmaps;
            this.tsiMipmapOrientation.ToolTipText = Properties.Resources.TextureDeailPanelToolTipRotateMipmaps;

            this.UpdateImage();

            // 表示深さコンボボックス
            {
                cbxDepth.Items.Clear();

                // todo:未実装
#if false
                if ((target_ == null) || (target_.IsArray == false))
                {
                    cbxDepth.Enabled = false;
                }
                else
                {
                    cbxDepth.Items.Add(Strings.TextureImageViewPanel_AllView);

                    var maxDepth = target_.IsCube ? target_.DepthCount / 6 : target_.DepthCount;
                    for(int i = 0;i != maxDepth;++ i)
                    {
                        cbxDepth.Items.Add(i.ToString());
                    }

                    cbxDepth.SelectedIndex = 0;

                    cbxDepth.Enabled = true;
                }
#endif
            }

            //-----------------------------------------------------------------
            // ツールバー初期設定
            //-----------------------------------------------------------------
            // 表示倍率選択項目初期設定
            foreach (ToolStripMenuItem item in tsiMagnifyOptions.DropDownItems)
            {
                item.Click += this.TsiMagnifyOption_Click;
            }

            this.UpdateMagnifyState();
            this.UpdateMipmapOrientation();
        }

        /// <summary>
        /// グリップを表示するか
        /// </summary>
        [Browsable(true)]
        public bool SizingGrip
        {
            get { return sspStatus.SizingGrip; }

            set { sspStatus.SizingGrip = value; }
        }

        /// <summary>
        /// 対象ファイルパス
        /// </summary>
        public string TargetFilePath
        {
            get
            {
                return this.targetFilePath;
            }

            set
            {
                if (this.targetFilePath == value)
                {
                    ////return;
                }

                this.targetFilePath = value;

                this.UpdateImage();
            }
        }

        /// <summary>
        /// チャンネル
        /// </summary>
        public ColorChannelFlags ColorChannel
        {
            get
            {
                if (this.imageView.IsTargetEnabled)
                {
                    return this.imageView.Channel;
                }
                else
                {
                    return ColorChannelFlags.None;
                }
            }

            set
            {
                this.requestColorChannel = value;

                if (this.imageView.IsTargetEnabled == false)
                {
                    return;
                }

                // アルファが指定されてもアルファがなければアルファを削除
                if ((value == ColorChannelFlags.A) && (this.imageView.Target.HasAlpha == false))
                {
                    value = ColorChannelFlags.Rgb;
                }

                this.imageView.Channel = value;
            }
        }

        /// <summary>
        /// Gets a value indicating whether is image enabled.
        /// </summary>
        public bool IsImageEnabled
        {
            get { return this.imageView.IsTargetEnabled; }
        }

        /// <summary>
        /// チャンネルを全て含んでいるか
        /// </summary>
        /// <param name="srcChannel">チャンネル</param>
        /// <returns>含んでいるか</returns>
        public bool Contains(ColorChannelFlags srcChannel)
        {
            return (srcChannel & ~this.ColorChannel) == 0;
        }

        /// <summary>
        /// 倍率とミップマップの表示設定を初期化します。
        /// </summary>
        public void ResetSettings()
        {
            if (this.imageView == null || !this.imageView.IsTargetEnabled)
            {
                return;
            }

            this.imageView.Magnify = 1.0f;
            this.UpdateMagnifyState();

            // ミップマップ表示切替
            if (this.imageView.Target.MipmapCount == 1)
            {
                tsiShowMipmap.Enabled = false;
                tsiMipmapOrientation.Enabled = false;
            }
            else
            {
                tsiShowMipmap.Enabled = true;
                this.imageView.ShowMipmap = false;
                this.imageView.MipmapOrientation = TextureImageView.MipmapOrientationType.Horizontal;
            }

            tsiShowMipmap.Checked = false;
            tsiMipmapOrientation.Enabled = tsiShowMipmap.Checked;

            this.UpdateMipmapOrientation();
        }

        /// <summary>
        /// 中心にフィットさせる
        /// </summary>
        public void FitCenter()
        {
            this.imageView.FitCenter();
        }

        /// <summary>
        /// コマンド キーを処理します。
        /// </summary>
        /// <returns>
        /// キーストロークがコントロールによって処理および使用された場合は true。キーストロークをさらに処理できるようにする場合は false。
        /// </returns>
        /// <param name="msg">処理する Win32 メッセージを表す、参照渡しされた <see cref="T:System.Windows.Forms.Message"/>。</param><param name="keyData">処理するキーを表す <see cref="T:System.Windows.Forms.Keys"/> 値の 1 つ。</param>
        protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
        {
            if ((keyData == Keys.Return || keyData == Keys.Enter) && tsiMagnify.Focused)
            {
                // 倍率テキストボックス確定
                this.OnEnterMagnifyInput();
                return true;
            }
            else if (keyData == (Keys.Control | Keys.Add))
            {
                // 拡大
                this.imageView.ZoomIn();
                return true;
            }
            else if (keyData == (Keys.Control | Keys.Subtract))
            {
                // 縮小
                this.imageView.ZoomOut();
                return true;
            }
            else if (keyData == (Keys.Control | Keys.D1))
            {
                // ＲＧＢチャンネル
                if (tsiChannelRGB.Enabled)
                {
                    this.imageView.Channel = ColorChannelFlags.Rgb;
                }

                return true;
            }
            else if (keyData == (Keys.Control | Keys.D2))
            {
                // Ｒチャンネル
                if (tsiChannelR.Enabled)
                {
                    this.imageView.Channel = ColorChannelFlags.R;
                }

                return true;
            }
            else if (keyData == (Keys.Control | Keys.D3))
            {
                // Ｇチャンネル
                if (tsiChannelG.Enabled)
                {
                    this.imageView.Channel = ColorChannelFlags.G;
                }

                return true;
            }
            else if (keyData == (Keys.Control | Keys.D4))
            {
                // Ｂチャンネル
                if (tsiChannelB.Enabled)
                {
                    this.imageView.Channel = ColorChannelFlags.B;
                }

                return true;
            }
            else if (keyData == (Keys.Control | Keys.D5))
            {
                // Ａチャンネル
                if (tsiChannelA.Enabled)
                {
                    this.imageView.Channel = ColorChannelFlags.A;
                }

                return true;
            }
            else if (keyData == (Keys.Control | Keys.M))
            {
                // ミップマップ表示
                if (tsiShowMipmap.Enabled)
                {
                    this.imageView.ShowMipmap = !this.imageView.ShowMipmap;
                    tsiShowMipmap.Checked = this.imageView.ShowMipmap;
                }

                return true;
            }
            else if (keyData == (Keys.Control | Keys.D))
            {
                // ミップマップ方向
                if (tsiMipmapOrientation.Enabled)
                {
                    this.imageView.MipmapOrientation = (this.imageView.MipmapOrientation == TextureImageView.MipmapOrientationType.Horizontal) ?
                                                        TextureImageView.MipmapOrientationType.Vertical :
                                                        TextureImageView.MipmapOrientationType.Horizontal;
                    this.UpdateMipmapOrientation();
                }

                return true;
            }

            return base.ProcessCmdKey(ref msg, keyData);
        }

        /// <summary>
        /// 画像更新
        /// </summary>
        private void UpdateImage()
        {
            // ウィンドウタイトル
            this.Text = string.Format(EffectMaker.UIControls.Specifics.Properties.Resources.TextureViewerDialog_TitleCaption, this.TargetFilePath);

            var oldImageView = this.imageView;

            if (string.IsNullOrEmpty(this.targetFilePath) == false &&
                (this.targetFilePath.EndsWith(".fmdb", StringComparison.OrdinalIgnoreCase) ||
                 this.TargetFilePath.EndsWith(".eset", StringComparison.OrdinalIgnoreCase)))
            {
                // 拡張子が eset か fmdb ならサムネイルイメージとして生成
                this.CreateBitmapView();

                this.tsiChannelR.Visible = false;
                this.tsiChannelG.Visible = false;
                this.tsiChannelB.Visible = false;
                this.tsiChannelA.Visible = false;
                this.tsiChannelRGB.Visible = false;
                this.tsiShowMipmap.Visible = false;
                this.tsiMipmapOrientation.Visible = false;
                this.toolStripSeparator1.Visible = false;
                this.toolStripSeparator2.Visible = false;
                this.toolStripSeparator3.Visible = false;
                this.toolStripSeparator4.Visible = false;
                this.tsiShowHelp.Visible = false;
                this.sspStatus.Visible = false;
            }
            else
            {
                // それ以外なら普通にテクスチャイメージとして生成
                this.imageView =
                    new TextureImageView(
                        this.targetFilePath,
                        (cbxDepth == null) ? -1 : Math.Max(0, cbxDepth.SelectedIndex) - 1)
                    {
                        Dock = DockStyle.Fill,
                        Parent = pnlImageView,
                    };

                this.tsiChannelR.Visible = true;
                this.tsiChannelG.Visible = true;
                this.tsiChannelB.Visible = true;
                this.tsiChannelA.Visible = true;
                this.tsiChannelRGB.Visible = true;
                this.tsiShowMipmap.Visible = true;
                this.tsiMipmapOrientation.Visible = true;
                this.toolStripSeparator1.Visible = true;
                this.toolStripSeparator2.Visible = true;
                this.toolStripSeparator3.Visible = true;
                this.toolStripSeparator4.Visible = true;
                this.tsiShowHelp.Visible = true;
                this.sspStatus.Visible = true;
            }

            // 初期表示チャンネルを作る
            {
                if (this.requestColorChannel != ColorChannelFlags.None)
                {
                    this.imageView.Channel = this.requestColorChannel;
                }
                else
                {
                    if (this.imageView.IsTargetEnabled)
                    {
                        this.imageView.Channel = ColorChannelFlags.Rgba;
                    }
                }

                // Alpha指定だけど、Alphaなしファイルであれば強制でRGBに設定する
                if (this.imageView.IsTargetEnabled)
                {
                    if (this.imageView.Target.HasAlpha == false)
                    {
                        this.imageView.Channel = ColorChannelFlags.Rgb;
                    }
                }
            }

            this.imageView.MagnifyChanged += (s, e) => this.UpdateMagnifyState();
            this.imageView.ChannelChanged += (s, e) => this.UpdateChannelState();
            this.imageView.PixelHint      += this.ImageView_PixelHint;

            //-----------------------------------------------------------------
            // ツールバー初期設定
            //-----------------------------------------------------------------
            // 表示倍率選択項目初期設定
            foreach (ToolStripMenuItem item in tsiMagnifyOptions.DropDownItems)
            {
                item.Click += this.TsiMagnifyOption_Click;
            }

            if (oldImageView != null)
            {
                this.imageView.Magnify = oldImageView.Magnify;
            }

            // 表示倍率状態
            this.UpdateMagnifyState();
            this.UpdateChannelState();

            // UIの状態を作る
            tsiChannelRGB.Enabled        = this.imageView.IsTargetEnabled;
            tsiChannelR.Enabled          = this.imageView.IsTargetEnabled;
            tsiChannelG.Enabled          = this.imageView.IsTargetEnabled;
            tsiChannelB.Enabled          = this.imageView.IsTargetEnabled;
            tsiChannelA.Enabled          = this.imageView.IsTargetEnabled;
            tsiMagnify.Enabled           = this.imageView.IsTargetEnabled;
            tsiMagnifyOptions.Enabled    = this.imageView.IsTargetEnabled;
            tsiMagnifyPlus.Enabled       = this.imageView.IsTargetEnabled;
            tsiMagnifyMinus.Enabled      = this.imageView.IsTargetEnabled;
            tsiShowMipmap.Enabled        = this.imageView.IsTargetEnabled;
            tsiMipmapOrientation.Enabled = this.imageView.IsTargetEnabled;

            if (this.imageView.IsTargetEnabled)
            {
                tsiChannelA.Enabled = this.imageView.Target.HasAlpha;

                // ミップマップ表示切替
                if (this.imageView.Target.MipmapCount == 1)
                {
                    tsiShowMipmap.Enabled = false;
                    tsiMipmapOrientation.Enabled = false;
                }
                else
                {
                    tsiShowMipmap.Enabled = true;
                    if (oldImageView != null && oldImageView.IsTargetEnabled)
                    {
                        this.imageView.ShowMipmap = oldImageView.ShowMipmap;
                        this.imageView.MipmapOrientation = oldImageView.MipmapOrientation;
                    }
                    else
                    {
                        this.imageView.MipmapOrientation = TextureImageView.MipmapOrientationType.Horizontal;
                    }
                }

                this.imageView.ShowMipmap = tsiShowMipmap.Checked;
                tsiMipmapOrientation.Enabled = tsiShowMipmap.Checked;
            }

            this.UpdateMipmapOrientation();

            //-----------------------------------------------------------------
            // ステータスバー初期設定
            //-----------------------------------------------------------------
            {
                var ftxTarget = this.imageView.Target as FtxTextureData;
                if (ftxTarget == null)
                {
                    tsiInfo0.Text = string.Empty;
                    tsiInfo1.Text = string.Empty;
                    tsiInfo2.Text = string.Empty;
                    tsiInfo3.Text = string.Empty;
                    tsiInfo4.Text = string.Empty;
                }
                else
                {
                    tsiInfo0.Text = ftxTarget.PixelFormat.ToString();
                    tsiInfo1.Text = string.Format("{0} x {1}", ftxTarget.Width, ftxTarget.Height);
                    tsiInfo2.Text = (ftxTarget.MipmapCount == 0) ? EffectMaker.UIControls.Specifics.Properties.Resources.TextureViewerDialog_MipmapNone : ftxTarget.MipmapCount.ToString();
                    tsiInfo3.Text = MathUtility.XBytesText(ftxTarget.NativeImageStream.Length);
                    tsiInfo4.Text = ftxTarget.ConverterName;
                }
            }

            // ピクセルヒント
            lblPixelHint0.Text = string.Empty;
            lblPixelHint1.Text = string.Empty;

            // 前回のイメージを開放する
            if (oldImageView != null)
            {
                oldImageView.Dispose();
            }
        }

        /// <summary>
        /// エミッタセットファイル、プリミティブファイルと一緒に用意されているサムネイルをビューにロードします。
        /// </summary>
        private void CreateBitmapView()
        {
            string thumPath = string.Empty;

            if (File.Exists(this.targetFilePath))
            {
                string dirPath = Path.GetDirectoryName(this.targetFilePath);
                string noExtName = Path.GetFileNameWithoutExtension(this.targetFilePath);
                string fileName = Path.GetFileName(this.targetFilePath);

                string jpgPath = Path.Combine(dirPath, noExtName + ".jpg");
                string pngPath = Path.Combine(dirPath, noExtName + ".png");

                string nwJpgPath = Path.Combine(dirPath, ".NwThumbs", fileName + ".jpg");
                string nwPngPath = Path.Combine(dirPath, ".NwThumbs", fileName + ".png");

                if (this.targetFilePath.EndsWith(".fmdb", StringComparison.OrdinalIgnoreCase))
                {
                    if (File.Exists(jpgPath))
                    {
                        thumPath = jpgPath;
                    }
                    else if (File.Exists(pngPath))
                    {
                        thumPath = pngPath;
                    }
                }

                if (string.IsNullOrEmpty(thumPath))
                {
                    if (File.Exists(nwJpgPath))
                    {
                        thumPath = nwJpgPath;
                    }
                    else if (File.Exists(nwPngPath))
                    {
                        thumPath = nwPngPath;
                    }
                }
            }

            Bitmap bitmap = null;
            if (!string.IsNullOrEmpty(thumPath))
            {
                using (var fs = new FileStream(thumPath, FileMode.Open, FileAccess.Read))
                {
                    bitmap = new Bitmap(fs);
                    fs.Close();
                }
            }

            this.imageView = new TextureImageView(bitmap)
            {
                Dock = DockStyle.Fill,
                Parent = pnlImageView,
            };
        }

        /// <summary>
        /// 倍率リーブ時
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void TsiMagnify_Leave(object sender, EventArgs e)
        {
            this.OnEnterMagnifyInput();
        }

        /// <summary>
        /// 倍率入力の確定処理。
        /// </summary>
        private void OnEnterMagnifyInput()
        {
            // 文字列から'%'を省く
            string text = tsiMagnify.Text;
            text = text.TrimEnd('%');

            // 倍率に変換
            var magnify = 1.0f;
            try
            {
                magnify = float.Parse(text) / 100.0f;
            }
            catch (Exception)
            {
                magnify = this.imageView.Magnify;
            }

            // 倍率変更
            // 変更されない場合もあるので常に UpdateMagnifyState() を呼んでおく
            this.imageView.ChangeMagnify(magnify);
            this.UpdateMagnifyState();

            // フォーカスをイメージビューに移す
            if (tsiMagnify.Focused)
            {
                // Leave イベントを発生させない
                ////using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
                {
                    this.imageView.Focus();
                }
            }
        }

        /// <summary>
        /// 倍率オプションクリック時
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void TsiMagnifyOption_Click(object sender, EventArgs e)
        {
            System.Diagnostics.Debug.Assert(sender is ToolStripMenuItem, "sender is ToolStripMenuItem");

            var item = sender as ToolStripMenuItem;
            this.imageView.ChangeMagnify(float.Parse((string)item.Tag));
        }

        /// <summary>
        /// マイナス倍率クリック時
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void TsiMagnifyMinus_Click(object sender, EventArgs e)
        {
            this.imageView.ZoomOut();
        }

        /// <summary>
        /// プラス倍率クリック時
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void TsiMagnifyPlus_Click(object sender, EventArgs e)
        {
            this.imageView.ZoomIn();
        }

        /// <summary>
        /// 表示倍率状態更新。
        /// </summary>
        private void UpdateMagnifyState()
        {
            // using (var block = new UIControlEventSuppressBlock())
            {
                tsiMagnify.Text = string.Format("{0}%", (int)(this.imageView.Magnify * 100));
            }

            // 倍率選択メニュー
            foreach (ToolStripMenuItem item in tsiMagnifyOptions.DropDownItems)
            {
                var magnify = float.Parse((string)item.Tag);

                item.Checked = magnify == this.imageView.Magnify;
            }
        }

        /// <summary>
        /// チャンネル状態更新。
        /// </summary>
        private void UpdateChannelState()
        {
            var isTargetEnabled = this.imageView.IsTargetEnabled;

            tsiChannelRGB.Checked = isTargetEnabled && this.imageView.Contains(ColorChannelFlags.Rgb);
            tsiChannelR.Checked   = isTargetEnabled && this.imageView.Contains(ColorChannelFlags.R);
            tsiChannelG.Checked   = isTargetEnabled && this.imageView.Contains(ColorChannelFlags.G);
            tsiChannelB.Checked   = isTargetEnabled && this.imageView.Contains(ColorChannelFlags.B);
            tsiChannelA.Checked   = isTargetEnabled && this.imageView.Contains(ColorChannelFlags.A);
        }

        /// <summary>
        /// イメージビューのピクセルヒント変更時
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void ImageView_PixelHint(object sender, TextureImageView.TpBitmapImagePixelHintEventArgs e)
        {
            // ヒントに応じて表示内容を変える
            switch (e.PixelHint)
            {
                case TextureImageView.TpBitmapImagePixelHint.PixelColor:
                {
                    lblPixelHint0.Text = string.Format("X: {0,4}  Y: {1,4}", e.PixelLocation.X, e.PixelLocation.Y);
                    lblPixelHint1.Text = string.Format("R: {0,3}  G: {1,3}  B: {2,3}  A: {3,3}", e.Color.R, e.Color.G, e.Color.B, e.Color.A);
                    break;
                }

                default:
                {
                    lblPixelHint0.Text = string.Empty;
                    lblPixelHint1.Text = string.Empty;
                    break;
                }
            }
        }

        /// <summary>
        /// ミップマップ表示クリック時
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void TsiShowMipmap_Click(object sender, EventArgs e)
        {
            this.imageView.ShowMipmap = tsiShowMipmap.Checked;
            this.tsiMipmapOrientation.Enabled = tsiShowMipmap.Checked;
        }

        /// <summary>
        /// ミップマップ方向クリック時
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void TsiMipmapOrientation_Click(object sender, EventArgs e)
        {
            this.imageView.MipmapOrientation = (this.imageView.MipmapOrientation == TextureImageView.MipmapOrientationType.Horizontal) ?
                                      TextureImageView.MipmapOrientationType.Vertical :
                                      TextureImageView.MipmapOrientationType.Horizontal;

            this.UpdateMipmapOrientation();
        }

        /// <summary>
        /// ミップマップ方向更新
        /// </summary>
        private void UpdateMipmapOrientation()
        {
            tsiMipmapOrientation.Image = (this.imageView.MipmapOrientation == TextureImageView.MipmapOrientationType.Horizontal) ?
                EffectMaker.UIControls.Specifics.Properties.Resources.TextureDetailsDialog_MipmapOrientationH :
                EffectMaker.UIControls.Specifics.Properties.Resources.TextureDetailsDialog_MipmapOrientationV;
        }

        /// <summary>
        /// チャンネルRGBクリック時
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void TsiChannelRGB_Click(object sender, EventArgs e)
        {
            this.imageView.ChangeChannel(ColorChannelFlags.Rgb);
        }

        /// <summary>
        /// チャンネルRクリック時
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void TsiChannelR_Click(object sender, EventArgs e)
        {
            this.imageView.ChangeChannel(ColorChannelFlags.R);
        }

        /// <summary>
        /// チャンネルGクリック時
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void TsiChannelG_Click(object sender, EventArgs e)
        {
            this.imageView.ChangeChannel(ColorChannelFlags.G);
        }

        /// <summary>
        /// チャンネルBクリック時
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void TsiChannelB_Click(object sender, EventArgs e)
        {
            this.imageView.ChangeChannel(ColorChannelFlags.B);
        }

        /// <summary>
        /// チャンネルアルファクリック時
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void TsiChannelA_Click(object sender, EventArgs e)
        {
            this.imageView.ChangeChannel(ColorChannelFlags.A);
        }

        /// <summary>
        /// ヘルプボタン表示クリック時
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void TsiShowHelp_Click(object sender, EventArgs e)
        {
            HelpManager.ShowHelp("92312495");
        }

        /// <summary>
        /// ステータスバーリサイズ処理
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void SspStatus_Resize(object sender, EventArgs e)
        {
            var width = (this.sspStatus.ClientSize.Width - (this.SizingGrip ? 24 : 14)) / 5;

            this.tsiInfo0.Width = width;
            this.tsiInfo1.Width = width;
            this.tsiInfo2.Width = width;
            this.tsiInfo3.Width = width;
            this.tsiInfo4.Width = width;
        }
    }
}
