﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;
using System.Windows.Forms;
using System.Xml;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.DataModel.Manager;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.Layout;
using EffectMaker.UIControls.Xaml;
using EffectMaker.UILogic.Manager;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UIControls.Specifics.TabPages
{
    /// <summary>
    /// An IItemContainerSelector that instances property pages. (TabPage)
    /// </summary>
    public class PropertyPageItemContainerSelector : IItemContainerSelector
    {
        /// <summary>
        /// Regular expression used to detect the translation identifier
        /// in a Translate markup extension expression.
        /// This looks for the text "{Translate XXX}" and gives access to the XXX identifier.
        /// </summary>
        private static Regex translateMarkupExtensionRegex =
            new Regex(@"\{\s*Translate\s+(?<id>[A-Za-z0-9_]+)\s*\}", RegexOptions.Compiled);

        /// <summary>
        /// Stores the mapping of view and view model.
        /// </summary>
        private readonly IDictionary<Type, ResourceInfo> viewViewModelMapping = new Dictionary<Type, ResourceInfo>()
        {
            { typeof(ViewerCameraViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.Viewer.ViewerCameraView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(ViewerBackgroundViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.Viewer.ViewerBackgroundView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(ViewerBasicViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.Viewer.ViewerBasicView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(ModelBasicViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.Model.ModelBasicView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(EmitterSetBasicViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.EmitterSet.EmitterSetBasicView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(EmitterSetUserViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.EmitterSet.EmitterSetUserView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(EmitterBasicViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.Emitter.EmitterBasicView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(EmitterEmitterViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.Emitter.EmitterEmitterView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(EmitterEmissionViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.Emitter.EmitterEmissionView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(EmitterParticleViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.Emitter.EmitterParticleView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(EmitterCombinerViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.Emitter.EmitterCombinerView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(EmitterColorPageViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.Emitter.EmitterColorView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(EmitterTextureGroupViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.Emitter.EmitterTextureView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(EmitterScaleViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.Emitter.EmitterScaleView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(EmitterRotationViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.Emitter.EmitterRotationView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(EmitterCustomShaderViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.Emitter.EmitterCustomShaderView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(PreviewBasicViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.Preview.PreviewBasicView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(PreviewMatrixViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.Preview.PreviewMatrixView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(PreviewAutoMoveViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.Preview.PreviewAutoMoveView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(PreviewColorViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.Preview.PreviewColorView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(PreviewEmitterViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.Preview.PreviewEmitterView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(PreviewEmissionViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.Preview.PreviewParticleEmissionView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(PreviewParticleScaleViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.Preview.PreviewParticleScaleView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(PreviewParticleControlViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.Preview.PreviewParticleControlView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(RandomBasicViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.Field.FieldRandomView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(RandomFe1BasicViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.Field.FieldRandomFe1View.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(MagnetBasicViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.Field.FieldMagnetView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(SpinBasicViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.Field.FieldSpinView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(ConvergeBasicViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.Field.FieldConvergeView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(AddLocationBasicViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.Field.FieldAddLocationView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(CollisionBasicViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.Field.FieldCollisionView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(CurlNoiseBasicViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.Field.FieldCurlNoiseView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(CustomActionUserDataViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.CustomAction.CustomActionView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(CustomBasicViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.Field.FieldCustomView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
            { typeof(EmitterExtParamsUserDataViewModel), new ResourceInfo { ResourceName = "XamlControls.PropertyPages.EmitterExtParams.EmitterExtParamsView.xaml", Kind = ResourceInfo.ResourceKind.Embedded } },
        };

        /// <summary>
        /// Stores the cache of created controls.
        /// </summary>
        private Dictionary<Type, PropertyTabPageBase> controlCache =
            new Dictionary<Type, PropertyTabPageBase>();

        /// <summary>
        /// データソースの変更フラグが変わったときのイベントです。
        /// </summary>
        public event EventHandler DataSourceModifiedChanged;

        /// <summary>
        /// Gets the display text of the root element from the XML content.
        /// </summary>
        /// <param name="xamlContent">XML content.</param>
        /// <returns>Returns the display text of the root element from the XML content.</returns>
        public static string GetTabName(Stream xamlContent)
        {
            const string ErrorText = "<ERROR>";

            string text = GetRootControlTextValue(xamlContent);

            if (text == null)
            {
                return ErrorText;
            }

            Match m = translateMarkupExtensionRegex.Match(text);
            if (m.Success)
            {
                text = m.Groups["id"].Value;
            }

            Assembly assembly = Assembly.GetCallingAssembly();

            string value = null;
            if (ResourceUtility.GetStringValue(assembly, text, out value) == false)
            {
                return text;
            }

            return value;
        }

        /// <summary>
        /// Xamlのページをプリロードします。
        /// </summary>
        public void PreLoadPages()
        {
            LayoutEngineBase.SuspendLayout();

            foreach (var info in this.viewViewModelMapping)
            {
                string text = null;

                using (Stream stream = ResourceUtility.Load(info.Value))
                {
                    text = GetTabName(stream);
                }

                var ctrl = new PropertyTabPageBase(info.Value) { Text = text };
                ctrl.OnPreload();
                this.controlCache.Add(info.Key, ctrl);
            }

            LayoutEngineBase.ResumeLayout();
            LayoutEngineBase.DelayLayout(1);
        }

        /// <summary>
        /// Instances a UITabPage control.
        /// </summary>
        /// <param name="dataItem">The data item for which to create the page.</param>
        /// <returns>Returns an instance of a UITabPage, or null if data item is null.</returns>
        public IControl SelectItemContainer(object dataItem)
        {
            var viewModel = dataItem as PropertyPageViewModel;

            if (viewModel == null)
            {
                return null;
            }

            try
            {
                // The key of the cache dictionary. (the type of the view model)
                Type cacheKey;
                GetTypeKeyForCache(viewModel, out cacheKey);

                PropertyTabPageBase ctrl = null;
                if (this.controlCache.TryGetValue(cacheKey, out ctrl) == false)
                {
                    ctrl = this.ConstructTabPage(viewModel);
                    if (ctrl != null)
                    {
                        ctrl.DataSourceModifiedChanged += this.OnDataSourceModifiedChanged;
                    }

                    this.controlCache.Add(cacheKey, ctrl);
                }

                return ctrl;
            }
            catch (Exception e)
            {
                Logger.Log(e.Message + "@PropertPageItemContainerSelector.SelectItemContainer");
                return null;
            }
        }

        /// <summary>
        /// Called to resolve the resource name when ConstructTabPage method is called.
        /// </summary>
        /// <param name="viewModel">The view model to create the page for.</param>
        /// <returns>Returns the absolute name of the XAML resource.</returns>
        private static string ResolveResourceName(ViewModelBase viewModel)
        {
            var typeName = viewModel.GetType().Name;

            if (typeName.EndsWith("ViewModel"))
            {
                string viewName = typeName.Substring(0, typeName.Length - 5);
                return string.Format("XamlControls.PropertyPages.{0}.xaml", viewName);
            }

            return null;
        }

        /// <summary>
        /// Get the type of the view model for caching the generated property page control.
        /// </summary>
        /// <param name="viewModel">The view model.</param>
        /// <param name="cacheKey">The type as the key of the cache dictionary.</param>
        /// <returns>True if the view model is a user data.</returns>
        private static bool GetTypeKeyForCache(ViewModelBase viewModel, out Type cacheKey)
        {
            bool isUserDataViewModel =
                (viewModel is PreviewUserDataViewModel) ||
                (viewModel is CustomActionUserDataViewModel) ||
                (viewModel is StripeUserDataViewModel) ||
                (viewModel is UserPageViewModel);

            if (isUserDataViewModel == false)
            {
                cacheKey = viewModel.GetType();
                return false;
            }

            var userPageViewModel = viewModel as UserPageViewModel;
            if (userPageViewModel != null)
            {
                cacheKey = userPageViewModel.Contents.Proxy.DataModel.GetType();
            }
            else
            {
                cacheKey = viewModel.Proxy.DataModel.GetType();
            }

            return true;
        }

        /// <summary>
        /// Fast retrive of the Text attribute on the first XML element,
        /// without reading all the XML content.
        /// </summary>
        /// <param name="xamlContent">The XML content to lookup.</param>
        /// <returns>Returns the value of the Text attribute of the first XML element.</returns>
        private static string GetRootControlTextValue(Stream xamlContent)
        {
            using (XmlReader reader = new XmlTextReader(xamlContent))
            {
                reader.Read();
                if (reader.NodeType != XmlNodeType.Element || reader.HasAttributes == false)
                {
                    return null;
                }

                reader.MoveToAttribute("Text");
                if (reader.NodeType != XmlNodeType.Attribute || reader.HasValue == false)
                {
                    return null;
                }

                string text = reader.Value;
                if (string.IsNullOrWhiteSpace(text))
                {
                    return null;
                }

                xamlContent.Position = 0;

                return text;
            }
        }

        /// <summary>
        /// データソースの変更フラグが変わったときのイベントを処理します。
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnDataSourceModifiedChanged(object sender, EventArgs e)
        {
            if (this.DataSourceModifiedChanged == null)
            {
                return;
            }

            this.DataSourceModifiedChanged(this, EventArgs.Empty);
        }

        /// <summary>
        /// Construct a PropertyTabPageBase instance based on a view model.
        /// </summary>
        /// <param name="viewModel">The view model to create the page for.</param>
        /// <returns>Returns an instance of PropertyTabPageBase, or null otherwise.</returns>
        private PropertyTabPageBase ConstructTabPage(ViewModelBase viewModel)
        {
            Type viewModelType = viewModel.GetType();

            ResourceInfo resourceInfo;

            // try to find XAML resource name from custom mapping
            if (this.viewViewModelMapping.TryGetValue(viewModelType, out resourceInfo) == false)
            {
                Type userDataType;
                if (GetTypeKeyForCache(viewModel, out userDataType) == true)
                {
                    string xamlPath = UserDataManager.GetUserDataXamlPath(userDataType);

                    resourceInfo = new ResourceInfo()
                    {
                        ResourceName = xamlPath,
                        Kind = ResourceInfo.ResourceKind.External,
                    };
                }

                if (userDataType != null && (resourceInfo == null))
                {
                    // not found, so try to resolve manually
                    resourceInfo = new ResourceInfo
                    {
                        ResourceName = ResolveResourceName(viewModel),
                        Kind = ResourceInfo.ResourceKind.Embedded
                    };
                }
            }

            if ((resourceInfo == null) || string.IsNullOrWhiteSpace(resourceInfo.ResourceName))
            {
                return null;
            }

            try
            {
                string text = null;

                using (Stream stream = ResourceUtility.Load(resourceInfo))
                {
                    text = GetTabName(stream);
                }

                return new PropertyTabPageBase(resourceInfo) { Text = text };
            }
            catch
            {
                // TODO: Log the error
                return null;
            }
        }
    }
}
