﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using EffectMaker.DataModel.RandomColor;
using EffectMaker.Foundation.Editting;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.DataBinding;

namespace EffectMaker.UIControls.Specifics.RandomColorEditor
{
    /// <summary>
    /// ランダムカラーエディタ
    /// </summary>
    public partial class RandomColorEditor : UIUserControl
    {
        /// <summary>
        /// エレメント
        /// </summary>
        private readonly List<RandomColorElement> elementControls = new List<RandomColorElement>();

        /// <summary>
        /// 値
        /// </summary>
        private RandomColorTable value = new RandomColorTable();

        /// <summary>
        /// 履歴記録用の差分
        /// </summary>
        private RandomColorTable prevValue = new RandomColorTable();

        /// <summary>
        /// RGBの編集を有効にするか
        /// </summary>
        private bool rgbEditEnabled;

        /// <summary>
        /// アルファの編集を有効にするか
        /// </summary>
        private bool alphaEditEnabled;

        /// <summary>
        /// 時間の編集を有効にするか
        /// </summary>
        private bool timeEditEnabled;

        /// <summary>
        /// 値変更イベントの実行中か
        /// </summary>
        private bool changing = false;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public RandomColorEditor()
        {
            this.InitializeComponent();
        }

        /// <summary>
        /// 値
        /// </summary>
        public RandomColorTable Value
        {
            get
            {
                return this.value;
            }

            set
            {
                if (value == null)
                {
                    return;
                }

                this.SetTableInstance(value);

                if (!this.changing)
                {
                    this.prevValue.Set(value);

                    if (!this.IsUpdatingDataContext)
                    {
                        this.NotifyValueChanged();
                    }
                }
            }
        }

        /// <summary>
        /// 色編集情報
        /// </summary>
        public EditingColorInfo EditingColorInfo
        {
            get
            {
                return new EditingColorInfo
                {
                    Control = (this.Value.SelectedRandomColor == null) ? null : this,
                    Color = (this.Value.SelectedRandomColor == null) ? new ColorRgba(1.0f, 1.0f, 1.0f, 1.0f) : this.Value.SelectedRandomColor.Color,
                    RgbEditEnabled = this.RgbEditEnabled,
                    AlphaEditEnabled = this.AlphaEditEnabled,
                    TimeEditEnabled = this.TimeEditEnabled,
                };
            }

            set
            {
                if (value.RefreshRequest)
                {
                    if (this.TargetName == value.RefreshTargetName)
                    {
                        if (this.Value.SelectedRandomColorIndex != -1)
                        {
                            if (this.Focused)
                            {
                                // コントロールに反映させる
                                this.elementControls[this.Value.SelectedRandomColorIndex].Color = value.Color;
                            }

                            // 現在アクティブであればカラーピッカー更新リクエスト
                            value.RefreshColorPickerRequest = this.Focused;
                        }
                    }
                }
                else
                {
                    if (value.Control == this)
                    {
                        if (this.Value.SelectedRandomColor != null)
                        {
                            this.Value.SelectedRandomColor.Color = value.Color;
                            if (!this.IsUpdatingDataContext)
                            {
                                this.LogicalTreeElementExtender.NotifyPropertyChanged(
                                    BindingUpdateType.PropertyChanged, "Value");
                                if (!value.IsEditing)
                                {
                                    this.NotifyValueChanged();
                                }
                            }

                            // コントロールに反映させる
                            this.elementControls[this.Value.SelectedRandomColorIndex].Color = value.Color;
                        }
                    }
                    else
                    {
                        // 他のコントロールがアクティブなので全て未選択にする
                        this.elementControls.ForEach(x => x.IsActive = false);
                        Enumerable.Range(0, this.Value.Count).Select(x => this.Value[x]).ForEach(x => x.IsSelected = false);
                    }
                }
            }
        }

        /// <summary>
        /// RGBの編集を有効にするか
        /// </summary>
        public bool RgbEditEnabled
        {
            get
            {
                return this.rgbEditEnabled;
            }

            set
            {
                this.rgbEditEnabled = value;
                this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "RgbEditEnabled");
            }
        }

        /// <summary>
        /// アルファの編集を有効にするか
        /// </summary>
        public bool AlphaEditEnabled
        {
            get
            {
                return this.alphaEditEnabled;
            }

            set
            {
                this.alphaEditEnabled = value;
                this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "AlphaEditEnabled");
            }
        }

        /// <summary>
        /// 時間の編集を有効にするか
        /// </summary>
        public bool TimeEditEnabled
        {
            get
            {
                return this.timeEditEnabled;
            }

            set
            {
                this.timeEditEnabled = value;
                this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "TimeEditEnabled");
            }
        }

        /// <summary>
        /// 編集対象名
        /// </summary>
        public string TargetName { get; set; }

        /// <summary>
        /// Gets or sets the IExecutable to run when the Value property changes.
        /// </summary>
        public IExecutable ValueChangedExecutable { get; set; }

        /// <summary>
        /// リサイズ自処理
        /// </summary>
        /// <param name="e">イベント引数</param>
        protected override void OnResize(EventArgs e)
        {
            base.OnResize(e);

            this.UpdateElements();
        }

        /// <summary>
        /// Called when the Value property changes.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected virtual void OnValueChanged(ValueChangedExEventArgs e)
        {
            IExecutable executable = this.ValueChangedExecutable;
            if (executable != null && executable.CanExecute(e))
            {
                executable.Execute(e);
            }
        }

        /// <summary>
        /// カラーチェンジイベントハンドラ
        /// </summary>
        /// <param name="s">sender</param>
        /// <param name="e">event</param>
        private void ElementColorChanged(object s, EventArgs e)
        {
            System.Diagnostics.Debug.Assert(s is RandomColorElement, "s is RandomColorElement");
            var sender = s as RandomColorElement;

            this.Value[sender.Index].Color = sender.Color;
            this.LogicalTreeElementExtender.NotifyPropertyChanged(
                BindingUpdateType.PropertyChanged, "Value");
            if (!this.IsUpdatingDataContext)
            {
                this.LogicalTreeElementExtender.NotifyPropertyChanged(
                    BindingUpdateType.PropertyChanged, "EditingColorInfo");
            }
        }

        /// <summary>
        /// チェックチェンジイベントハンドラ
        /// </summary>
        /// <param name="s">sender</param>
        /// <param name="e">event</param>
        private void ElementIsUsedChanged(object s, EventArgs e)
        {
            System.Diagnostics.Debug.Assert(s is RandomColorElement, "s is RandomColorElement");
            var sender = s as RandomColorElement;

            this.Value[sender.Index].Enabled = sender.IsUsed;
            this.LogicalTreeElementExtender.NotifyPropertyChanged(
                BindingUpdateType.PropertyChanged, "Value");
            if (!this.IsUpdatingDataContext)
            {
                this.NotifyValueChanged();
                this.LogicalTreeElementExtender.NotifyPropertyChanged(
                    BindingUpdateType.PropertyChanged, "EditingColorInfo");
            }
        }

        /// <summary>
        /// エレメントの更新
        /// </summary>
        private void UpdateElements()
        {
            this.Controls.Clear();

            while (this.elementControls.Count() < this.Value.Count)
            {
                var elem = new RandomColorElement
                {
                    Index = this.elementControls.Count(),
                    ColorPasted = this.NotifyValueChanged,
                };

                elem.MouseDown += (s, e) =>
                {
                    System.Diagnostics.Debug.Assert(s is RandomColorElement, "s is RandomColorElement");
                    var sender = s as RandomColorElement;

                    this.Focus();

                    this.elementControls.ForEach(x => x.IsActive = false);

                    this.Value[sender.Index].IsSelected = true;

                    EditingColorInfo.Color = this.Value[sender.Index].Color;
                    this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "Value");
                    this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "EditingColorInfo");
                };

                elem.ColorChanged += this.ElementColorChanged;
                elem.IsUsedChanged += this.ElementIsUsedChanged;

                this.elementControls.Add(elem);
            }

            for (var i = 0; i != this.Value.Count; ++i)
            {
                // コントロールを追加する
                this.Controls.Add(this.elementControls[i]);
            }

            // 位置調整
            this.UpdateElementsPosition();
        }

        /// <summary>
        /// 位置調整
        /// </summary>
        private void UpdateElementsPosition()
        {
            if (this.Value.Count == 0)
            {
                return;
            }

            if (this.Value.Count == 1)
            {
                this.elementControls[0].Location = new Point { X = 0, Y = 0 };
                return;
            }

            var width = this.Width - this.elementControls[0].Width;

            for (var i = 0; i != Math.Min(this.Value.Count, this.elementControls.Count()); ++i)
            {
                var x = i * width / (this.Value.Count - 1);

                this.elementControls[i].Location = new Point { X = x, Y = 0 };
            }
        }

        /// <summary>
        /// テーブルをコントロールにセットします。
        /// </summary>
        /// <param name="table">ランダムテーブル</param>
        private void SetTableInstance(RandomColorTable table)
        {
            this.value.Set(table);

            // コントロールに反映させる
            for (var i = 0; i != this.Value.Count; ++i)
            {
                this.elementControls[i].ColorChanged -= this.ElementColorChanged;
                this.elementControls[i].IsUsedChanged -= this.ElementIsUsedChanged;

                this.elementControls[i].IsUsed = table[i].Enabled;
                this.elementControls[i].Color = table[i].Color;
                this.elementControls[i].IsActive = false;

                this.elementControls[i].IsUsedChanged += this.ElementIsUsedChanged;
                this.elementControls[i].ColorChanged += this.ElementColorChanged;
            }

            if (this.value.SelectedRandomColorIndex != -1)
            {
                this.elementControls[this.value.SelectedRandomColorIndex].IsActive = true;
            }

            this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "Value");
            if (!this.IsUpdatingDataContext)
            {
                this.LogicalTreeElementExtender.NotifyPropertyChanged(
                    BindingUpdateType.PropertyChanged, "EditingColorInfo");
            }
        }

        /// <summary>
        /// OnValueChangedを呼び出して、履歴に積む値を確定します。
        /// </summary>
        private void NotifyValueChanged()
        {
            this.changing = true;

            this.LogicalTreeElementExtender.NotifyPropertyChanged(
                BindingUpdateType.Validation, propertyName: "Value");

            this.OnValueChanged(new ValueChangedExEventArgs(
                this.prevValue,
                this.value,
                false,
                this.TargetName));

            this.prevValue.Set(this.value);

            this.changing = false;
        }
    }
}
