﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Windows.Forms;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.UIControls.Specifics.GradationEditor
{
    /// <summary>
    /// グラデーションキー
    /// </summary>
    public class GradationKey
    {
        /// <summary>
        /// つまみサイズ
        /// </summary>
        public const int KnobSize = 6;

        /// <summary>
        /// 水平パディング
        /// </summary>
        private const int PaddingX = 2;

        /// <summary>
        /// 垂直パディング
        /// </summary>
        private const int PaddingY = 2;

        /// <summary>
        /// 削除ボタンビットマップ
        /// </summary>
        private static readonly Bitmap RemoveButtonBitmap = Properties.Resources.GradationEditor_Del;

        /// <summary>
        /// オーナー
        /// </summary>
        private readonly GradationEditor owner;

        /// <summary>
        /// 時間
        /// </summary>
        private int time;

        /// <summary>
        /// ドラッグタイプ
        /// </summary>
        private DragTypes dragTypes;

        /// <summary>
        /// ドラッグ開始マウス位置
        /// </summary>
        private Point drawStartMousePos;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="owner">オーナー</param>
        public GradationKey(GradationEditor owner)
        {
            System.Diagnostics.Debug.Assert(owner != null, "owner != null");

            this.owner = owner;
        }

        /// <summary>
        /// ドラッグタイプ
        /// </summary>
        private enum DragTypes
        {
            /// <summary>
            /// ドラッグでない
            /// </summary>
            Nothing,

            /// <summary>
            /// 削除ボタン
            /// </summary>
            RemoveButton,

            /// <summary>
            /// つまみ
            /// </summary>
            Knob
        }

        /// <summary>
        /// 削除ボタン高さ
        /// </summary>
        public static int RemoveButtonHeight
        {
            get { return RemoveButtonBitmap.Height; }
        }

        /// <summary>
        /// 削除ボタン幅
        /// </summary>
        public static int RemoveButtonWidth
        {
            get { return RemoveButtonBitmap.Width; }
        }

        /// <summary>
        /// ドラッグされたか？
        /// </summary>
        public bool IsDraged
        {
            get;
            private set;
        }

        /// <summary>
        /// 色
        /// </summary>
        public ColorRgba Color { get; set; }

        /// <summary>
        /// 時間
        /// </summary>
        public int Time
        {
            get
            {
                return this.time;
            }

            set
            {
                this.time = Math.Min(Math.Max(value, 0), 100);
            }
        }

        /// <summary>
        /// インデックス
        /// </summary>
        public int Index { get; set; }

        /// <summary>
        /// 基本X座表
        /// </summary>
        public int DrawBasePosX
        {
            get
            {
                return (int)((this.owner.ColorBarRectangle.Width - KnobSize - (PaddingX * 2)) * this.Time * 0.01);
            }
        }

        /// <summary>
        /// 描画
        /// </summary>
        /// <param name="g">描画オブジェクト</param>
        /// <param name="info">描画情報</param>
        public void Draw(Graphics g, DrawInfo info)
        {
            this.DrawKnob(g, info);

            if (this.owner.CanAddRemoveKeys && this.owner.IsEnabledRemoveButton)
            {
                this.DrawRemoveButtton(g);
            }
        }

        /// <summary>
        /// マウスダウンイベント
        /// </summary>
        /// <param name="e">イベント引数</param>
        public void OnMouseDown(MouseEventArgs e)
        {
            var handled = false;
            var isSelected = false;

            this.OnMouseDown(e, ref handled, ref isSelected);
        }

        /// <summary>
        /// マウスダウンイベント
        /// </summary>
        /// <param name="e">イベント引数</param>
        /// <param name="handled">用済みか</param>
        /// <param name="isSelected">選択されたか</param>
        public void OnMouseDown(MouseEventArgs e, ref bool handled, ref bool isSelected)
        {
            var colorBarRect = this.owner.ColorBarRectangle;
            var x = this.DrawBasePosX;

            this.dragTypes = DragTypes.Nothing;
            this.IsDraged = false;

            // 削除ボタン
            if (this.owner.CanAddRemoveKeys && this.owner.IsEnabledRemoveButton)
            {
                var rect = new Rectangle((int)colorBarRect.Left + x, GradationEditor.KeyRemoveButtonPosY, RemoveButtonWidth, RemoveButtonHeight);

                if ((e.X >= rect.Left) && (e.X <= rect.Right) &&
                    (e.Y >= rect.Top)  && (e.Y <= rect.Bottom))
                {
                    isSelected = true;
                    this.dragTypes = DragTypes.RemoveButton;
                    this.drawStartMousePos = new Point(e.X, e.Y);
                    handled = true;
                }
            }

            // つまみ
            {
                var rect = new Rectangle((int)colorBarRect.Left + x, (int)colorBarRect.Top, KnobSize + (PaddingX * 2), (int)colorBarRect.Height);

                if ((e.X >= rect.Left) && (e.X <= rect.Right) &&
                    (e.Y >= rect.Top)  && (e.Y <= rect.Bottom))
                {
                    isSelected = true;
                    this.dragTypes = DragTypes.Knob;
                    this.drawStartMousePos = new Point(e.X, e.Y);
                    handled = true;
                }
            }
        }

        /// <summary>
        /// <see cref="E:System.Windows.Forms.Control.MouseMove"/> イベントを発生させます。
        /// </summary>
        /// <param name="e">イベント データを格納している <see cref="T:System.Windows.Forms.MouseEventArgs"/>。</param>
        /// <param name="isMouseOver">マウスオーバー中か？</param>
        public void OnMouseMove(MouseEventArgs e, ref bool isMouseOver)
        {
            var colorBarRect = this.owner.ColorBarRectangle;
            var x = this.DrawBasePosX;

            // 削除ボタン
            if (this.owner.CanAddRemoveKeys && this.owner.IsEnabledRemoveButton)
            {
                var rect = new Rectangle((int)colorBarRect.Left + x, GradationEditor.KeyRemoveButtonPosY, RemoveButtonWidth, RemoveButtonHeight);

                if ((e.X >= rect.Left) && (e.X <= rect.Right) &&
                    (e.Y >= rect.Top)  && (e.Y <= rect.Bottom))
                {
                    isMouseOver = true;
                }
            }

            // つまみ
            {
                var rect = new Rectangle((int)colorBarRect.Left + x, (int)colorBarRect.Top, KnobSize + (PaddingX * 2), (int)colorBarRect.Height);

                if ((e.X >= rect.Left) && (e.X <= rect.Right) &&
                    (e.Y >= rect.Top)  && (e.Y <= rect.Bottom))
                {
                    isMouseOver = true;
                }
            }
        }

        /// <summary>
        /// マウスドラッグイベント
        /// </summary>
        /// <param name="e">イベント引数</param>
        public void OnMouseDrag(MouseEventArgs e)
        {
            if (this.dragTypes != DragTypes.RemoveButton)
            {
                var oldKeyTime  = this.Time;
                var nextKeyTime = this.owner.MakeTime(e.X, e.Y);

                nextKeyTime = Math.Min(Math.Max(nextKeyTime, 0), this.owner.MaximumFrame);

                // ドラッグされたかを作る
                this.IsDraged = this.IsDraged || (oldKeyTime != nextKeyTime);

                if (this.owner.CanKeysOverlap == false)
                {
                    if (this.owner.IsAnyKeyTime(nextKeyTime) == false)
                    {
                        this.Time = nextKeyTime;
                    }
                }
                else
                {
                    this.Time = nextKeyTime;
                }
            }
        }

        /// <summary>
        /// マウスアップイベント
        /// </summary>
        /// <param name="e">イベント引数</param>
        /// <param name="isReqRemove">削除を要求するか</param>
        public void OnMouseUp(MouseEventArgs e, ref bool isReqRemove)
        {
            if (this.dragTypes == DragTypes.RemoveButton)
            {
                var diffX = Math.Abs(e.X - this.drawStartMousePos.X);
                var diffY = Math.Abs(e.Y - this.drawStartMousePos.Y);

                isReqRemove = (diffX < 6) && (diffY < 6);
            }

            this.dragTypes = DragTypes.Nothing;
        }

        /// <summary>
        /// つまみ描画
        /// </summary>
        /// <param name="g">グラフィックオブジェクト</param>
        /// <param name="info">描画情報</param>
        private void DrawKnob(Graphics g, DrawInfo info)
        {
            var colorBarRect = this.owner.ColorBarRectangle;
            var x = this.DrawBasePosX;
            var rect = new RectangleF(colorBarRect.X + x + PaddingX, colorBarRect.Y + PaddingY, KnobSize, colorBarRect.Height - (PaddingY * 2));

            using (var path = RenderUtility.MakeRoundRectangleGraphicsPath(rect, 1.0f))
            using (var penOut = new Pen(System.Drawing.Color.Black, 3))
            using (var penIn = new Pen(info.Color, 2))
            {
                // 縁
                g.DrawPath(penOut, path);
                g.DrawPath(penIn,  path);
            }
        }

        /// <summary>
        /// 削除ボタン描画
        /// </summary>
        /// <param name="g">グラフィックオブジェクト</param>
        private void DrawRemoveButtton(Graphics g)
        {
            var colorBarRect = this.owner.ColorBarRectangle;

            var x = this.DrawBasePosX - ((RemoveButtonWidth - KnobSize) / 2);
            var y = GradationEditor.KeyRemoveButtonPosY;

            g.DrawImage(RemoveButtonBitmap, colorBarRect.X + x + PaddingX, y);
        }

        /// <summary>
        /// 描画情報
        /// </summary>
        public class DrawInfo
        {
            /// <summary>
            /// 状態
            /// </summary>
            public enum StateKind
            {
                /// <summary>
                /// 通常
                /// </summary>
                Normal,

                /// <summary>
                /// ドラッグ中
                /// </summary>
                Dragging,

                /// <summary>
                /// マウスオーバー中
                /// </summary>
                MouseOver,

                /// <summary>
                /// 選択中
                /// </summary>
                Selected
            }

            /// <summary>
            /// 状態
            /// </summary>
            public StateKind State { get; set; }

            /// <summary>
            /// 状態ごとの色
            /// </summary>
            public Color Color
            {
                get
                {
                    return
                        (this.State == StateKind.Dragging)  ? Color.OrangeRed :
                        (this.State == StateKind.MouseOver) ? Color.Orange :
                        (this.State == StateKind.Selected)  ? Color.CornflowerBlue :
                                                              Color.White;
                }
            }
        }
    }
}
