﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Drawing;
using System.Windows.Forms;
using EffectMaker.Foundation.Render.Layout;
using EffectMaker.Foundation.Render.Renderable;

namespace EffectMaker.UIControls.Specifics.CurveEditor.RenderingElements
{
    /// <summary>
    /// フレーム数をポップアップ表示する描画要素です。
    /// </summary>
    internal class FramePopup : IRenderingElement
    {
        /// <summary>
        /// 選択しているノードの時刻をポップアップ
        /// </summary>
        private readonly TextBlock frameNumber;

        /// <summary>
        /// ノードフレームポップアップの背景
        /// </summary>
        private readonly RoundedRectangleShape popupBack;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public FramePopup()
        {
            this.frameNumber = new TextBlock
            {
                Font = new Font(FontFamily.GenericSansSerif, 9.0f),
                FillColor = Color.Transparent,
                BorderColor = Color.Transparent,
                TextColor = Color.White,
                BorderThickness = 0,
                AutoSize = false,
                Padding = new Padding(0),
                HAlignment = HAlignment.Left,
                VAlignment = VAlignment.Top,
            };

            this.popupBack = new RoundedRectangleShape
            {
                FillColor = Color.FromArgb(224, 87, 105, 138),
                BorderThickness = 0,
                BottomLeftCornerRadius = 2,
                BottomRightCornerRadius = 2,
                TopLeftCornerRadius = 2,
                TopRightCornerRadius = 2,
            };
        }

        /// <summary>
        /// 数値の描画に使用するフォントを取得します。
        /// </summary>
        public Font TextFont
        {
            get { return this.frameNumber.Font; }
        }

        /// <summary>
        /// 描画するテキストを取得または設定します。
        /// </summary>
        public string Text
        {
            get { return this.frameNumber.Text; }
            set { this.frameNumber.Text = value; }
        }

        /// <summary>
        /// 描画領域を取得または設定します。
        /// </summary>
        public RectangleF Bounds
        {
            get
            {
                return this.popupBack.Bounds;
            }

            set
            {
                this.popupBack.Bounds = value;
                this.frameNumber.Bounds = new RectangleF(
                    value.X + 2,
                    value.Y,
                    value.Width,
                    value.Height);
            }
        }

        /// <summary>
        /// 文字列と基準点を設定してサイズを調整します。
        /// </summary>
        /// <param name="viewports">ビューポート</param>
        /// <param name="text">表示したい文字列</param>
        /// <param name="argPos">基準点</param>
        public void Update(LayeredViewports viewports, string text, PointF argPos)
        {
            var txtSize = TextRenderer.MeasureText(text, this.TextFont);
            var tempSize = new SizeF(
                txtSize.Width,
                16.0f);
            var tempPos = argPos;
            tempPos.X += 16.0f;
            tempPos.Y -= 4.0f;

            // 右端の折り返しを処理
            if (tempPos.X + tempSize.Width > viewports.Overlay.Width)
            {
                tempPos.X -= 24.0f + tempSize.Width;
            }

            // 上下端への張り付きを処理
            if (tempPos.Y < 0)
            {
                tempPos.Y = 0.0f;
            }

            if (viewports.Overlay.Height < tempPos.Y + tempSize.Height)
            {
                tempPos.Y = viewports.Overlay.Height - tempSize.Height;
            }

            this.Text = text;
            this.Bounds = new RectangleF(tempPos, tempSize);
        }

        /// <summary>
        /// 表示します。
        /// 既に表示済みにビューポートならば一旦削除して描画順を最後にします。
        /// </summary>
        /// <param name="viewports">表示するビューポート</param>
        public void Show(LayeredViewports viewports)
        {
            this.Hide(viewports);
            viewports.Overlay.AddRenderable(this.popupBack);
            viewports.Overlay.AddRenderable(this.frameNumber);
        }

        /// <summary>
        /// 非表示にします。
        /// </summary>
        /// <param name="viewports">非表示にするビューポート</param>
        public void Hide(LayeredViewports viewports)
        {
            viewports.Overlay.RemoveRenderable(this.frameNumber);
            viewports.Overlay.RemoveRenderable(this.popupBack);
        }
    }
}
