﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Layout;
using EffectMaker.UIControls.Specifics.TabPages;
using EffectMaker.UILogic.Manager;
using MouseEventArgs = System.Windows.Forms.MouseEventArgs;

namespace EffectMaker.UIControls.Specifics.CurveEditor
{
    /// <summary>
    /// カーブエディタのイベントハンドラとオーバーライドを集約します。
    /// </summary>
    public partial class CurveEditor
    {
        #region ハンドラ用フィールド

        /// <summary>
        /// 値の更新中フラグ
        /// </summary>
        private bool changing = false;

        /// <summary>
        /// スクロール中フラグ
        /// </summary>
        private bool scrolling = false;

        /// <summary>
        /// スクロールベロシティ
        /// </summary>
        private PointF scrollVec;

        #endregion

        #region オーバーライド

        /// <summary>
        /// Loadイベント
        /// </summary>
        /// <param name="e">イベント引数</param>
        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);
            this.Initialize();

            if (this.EditorMode == EditorModes.EmitterTimeAnim)
            {
                this.AdjustViewport(false);
            }
        }

        /// <summary>
        /// OnVisibleChangedイベント
        /// </summary>
        /// <param name="e">イベント引数</param>
        protected override void OnVisibleChanged(EventArgs e)
        {
            base.OnVisibleChanged(e);
            this.ResetViewport();
            if (this.values.Count > 1)
            {
                this.AdjustViewport(true);
            }

            while (!this.viewports.Main.IsUpdateAndRenderingSuspended)
            {
                this.viewports.Main.SuspendUpdateAndRendering();
            }

            while (!this.viewports.Underlay.IsUpdateAndRenderingSuspended)
            {
                this.viewports.Underlay.SuspendUpdateAndRendering();
            }

            while (!this.viewports.Overlay.IsUpdateAndRenderingSuspended)
            {
                this.viewports.Overlay.SuspendUpdateAndRendering();
            }

            this.UpdateResetButton();
        }

        /// <summary>
        /// MouseDownイベント
        /// </summary>
        /// <param name="e">イベント引数</param>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            if (!this.Focused)
            {
                if (this.targetNode == null)
                {
                    this.Focus();
                }
                else
                {
                    this.text_FocusDummy.Focus();
                }
            }

            if (e.Button == MouseButtons.Right)
            {
                // 選択状態の更新
                this.UpdateSelection(e.Location, true);

                this.EndDrag(e.Location);
                if (this.selectedNodes.Count != 0)
                {
                    // 右マウスで選択していたら削除
                    this.RemoveNode();
                }
            }
            else if (e.Button == MouseButtons.Middle)
            {
                this.EndDrag(e.Location);
                this.scrolling = true;
                this.viewports.PrevScrollingPosition = e.Location;
            }
            else
            {
                // 選択状態の更新
                this.UpdateSelection(e.Location, true);

                if (this.selectedNodes.Count != 0)
                {
                    // 左マウスで選択していたらドラッグ
                    this.dragging = true;
                    this.dragStarted = false;
                    if ((Control.ModifierKeys & Keys.Shift) == Keys.Shift)
                    {
                        this.BeginLock();
                    }
                }
                else
                {
                    // 選択していなかったら追加
                    this.AddNode(e.Location);
                }

                this.viewports.DragStartPosition = e.Location;
            }

            if (this.entering)
            {
                this.FocusWithoutScroll();
            }

            this.RequestDraw();
            this.UpdateResetButton();
        }

        /// <summary>
        /// MouseUpイベント
        /// </summary>
        /// <param name="e">イベント引数</param>
        protected override void OnMouseUp(MouseEventArgs e)
        {
            // マウスアップでドラッグ終了
            if (e.Button == MouseButtons.Left)
            {
                this.EndDrag(e.Location);

                this.scrollVec.X = this.scrollVec.Y = 0.0f;
                this.timer_DragScroll.Enabled = false;
            }
            else if (e.Button == MouseButtons.Middle)
            {
                this.scrolling = false;
            }
        }

        /// <summary>
        /// MouseWheelイベント
        /// </summary>
        /// <param name="e">イベント引数</param>
        protected void Handle_MouseWheel(MouseEventArgs e)
        {
            this.ZoomViewport(e);
            this.RequestDraw();
            this.UpdateResetButton();
        }

        /// <summary>
        /// 右クリックとホイールイベントをフックするためのプロシージャ
        /// </summary>
        /// <param name="m">メッセージ</param>
        protected override void WndProc(ref Message m)
        {
            if (!this.entering)
            {
                base.WndProc(ref m);
                return;
            }

            var p = new Point(
                EffectMaker.Foundation.Win32.Utility.LOWORD(m.LParam),
                EffectMaker.Foundation.Win32.Utility.HIWORD(m.LParam));

            if (m.Msg == (int)EffectMaker.Foundation.Win32.WM.WM_MOUSEWHEEL)
            {
                int d = EffectMaker.Foundation.Win32.Utility.HIWORD(m.WParam);
                d = d < 32768 ? 120 : -120;

                p = this.PointToClient(p);

                this.Handle_MouseWheel(new MouseEventArgs(MouseButtons.None, 0, p.X, p.Y, d));
            }
            else if (m.Msg == (int)EffectMaker.Foundation.Win32.WM.WM_RBUTTONDOWN)
            {
                this.OnMouseDown(new MouseEventArgs(MouseButtons.Right, 1, p.X, p.Y, 0));
            }
            else if (m.Msg == (int)EffectMaker.Foundation.Win32.WM.WM_RBUTTONUP)
            {
                return;
            }
            else
            {
                base.WndProc(ref m);
            }
        }

        /// <summary>
        /// MouseMoveイベント
        /// </summary>
        /// <param name="e">イベント引数</param>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            base.OnMouseMove(e);
            var nowMouse = this.PointToClient(Cursor.Position);

            // マウスの移動によりホイールズームの中心位置を記憶する処理を解除します
            this.viewports.IsZooming = false;

            var shiftPos = new Point(
                nowMouse.X - (int)this.viewports.Main.Location.X,
                nowMouse.Y - (int)this.viewports.Main.Location.Y);

            this.visualCursor.Location = shiftPos;

            if (this.viewports.IsInnerMouseSpace(nowMouse))
            {
                this.OnMouseEnterViewport();

                if (this.dragging)
                {
                    // ドラッグ判定にマージンを持たせる
                    int diffX = Math.Abs(e.Location.X - (int)this.viewports.DragStartPosition.X);
                    int diffY = Math.Abs(e.Location.Y - (int)this.viewports.DragStartPosition.Y);
                    if (dragStarted || diffX >= 3 || diffY >= 3)
                    {
                        // 一度マージンを超えたらこの処理へはフリーパス
                        this.dragStarted = true;

                        // 左ドラッグ中なら移動処理
                        this.MoveNode(this.CalcLockPos(nowMouse));

                        // ドラッグスクロールorスケーリング停止
                        this.timer_DragScroll.Enabled = false;
                        this.scrollVec.X = this.scrollVec.Y = 0.0f;

                        this.SyncData(true);
                    }
                }
                else if (this.scrolling)
                {
                    // センタードラッグならスクロール処理
                    this.ScrollViewport(nowMouse);
                }
                else
                {
                    // 左かセンターのドラッグ中でなければ選択状態の更新
                    this.UpdateSelection(nowMouse, false);
                }
            }
            else
            {
                if (this.dragging)
                {
                    var lockedPos = this.CalcLockPos(this.viewports.GetClampedMousePosition());

                    if (shiftPos.X < 0)
                    {
                        this.scrollVec.X = -shiftPos.X / 5.0f;
                    }
                    else if (shiftPos.X >= this.viewports.Main.Width)
                    {
                        this.scrollVec.X = -(shiftPos.X - this.viewports.Main.Width) / 5.0f;
                    }
                    else
                    {
                        this.scrollVec.X = 0.0f;
                    }

                    if (shiftPos.Y < 0)
                    {
                        this.scrollVec.Y = -shiftPos.Y / 5.0f;
                    }
                    else if (shiftPos.Y >= this.viewports.Main.Height)
                    {
                        this.scrollVec.Y = -(shiftPos.Y - this.viewports.Main.Height) / 5.0f;
                    }
                    else
                    {
                        this.scrollVec.Y = 0.0f;
                    }

                    this.MoveNode(lockedPos);
                    this.timer_DragScroll.Enabled = true;
                }
                else
                {
                    this.scrollVec.X = this.scrollVec.Y = 0.0f;
                    this.timer_DragScroll.Enabled = false;
                }

                this.OnMouseLeaveViewport();
            }

            if (this.entering || this.dragging)
            {
                this.visualCursor.Update(
                    this.viewports,
                    nowMouse,
                    this.NormalizeAt,
                    this.invalidArea.Pick(nowMouse),
                    this.LabelPrefix,
                    this.EditorMode == EditorModes.ParticleTimeAnim ? "%" : string.Empty);
                this.UpdateSelectedFrame();
            }

            if (!this.ClientRectangle.Contains(this.PointToClient(Control.MousePosition)))
            {
                this.ShowCursor();
            }

            this.RequestDraw();

            this.UpdateResetButton();
        }

        /// <summary>
        /// SizeChangedイベント
        /// </summary>
        /// <param name="e">イベント引数</param>
        protected override void OnSizeChanged(EventArgs e)
        {
            this.ResizeViewport();

            if (this.viewports != null)
            {
                this.ResetViewport();
            }

            base.OnSizeChanged(e);
        }

        /// <summary>
        /// Paintイベント
        /// </summary>
        /// <param name="e">イベント引数</param>
        protected override void OnPaint(PaintEventArgs e)
        {
            if (this.viewports == null)
            {
                return;
            }

            e.Graphics.SmoothingMode = System.Drawing.Drawing2D.SmoothingMode.HighQuality;
            e.Graphics.TextRenderingHint = System.Drawing.Text.TextRenderingHint.ClearTypeGridFit;
            this.viewports.Draw(e.Graphics);
        }

        /// <summary>
        /// MouseLeaveイベント
        /// </summary>
        /// <param name="e">イベント引数</param>
        protected override void OnMouseLeave(EventArgs e)
        {
            this.OnMouseLeaveViewport();
            base.OnMouseLeave(e);
        }

        /// <summary>
        /// 値の変更イベント処理をトリガーします。
        /// </summary>
        /// <param name="e">イベント引数</param>
        protected virtual void OnValueChanged(ValueChangedExEventArgs e)
        {
            if (this.changing)
            {
                return;
            }

            this.changing = true;

            IExecutable executable = this.ValueChangedExecutable;
            if (executable != null && executable.CanExecute(e))
            {
                executable.Execute(e);
            }

            this.changing = false;
        }

        #endregion

        #region イベントハンドラ

        /// <summary>
        /// カーブエディタ上でのキーハンドリング
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="e">イベント引数</param>
        private void Handle_CurveEditor_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.A)
            {
                this.AdjustViewport(false);
            }
            else if (e.KeyCode == Keys.F)
            {
                this.AdjustViewport(true);
            }
            else if (e.KeyCode == Keys.Delete &&
                (sender != this.text_LoopEnd && sender != this.text_Frame && sender != this.text_Value))
            {
                if (this.targetNode != null)
                {
                    if (this.selectedNodes.Count == 0)
                    {
                        this.selectedNodes.Add(this.targetNode);
                    }

                    this.RemoveNode();
                }
            }
            else if (e.KeyCode == Keys.ShiftKey)
            {
                this.BeginLock();
            }

            this.RequestDraw();
            this.UpdateResetButton();
        }

        /// <summary>
        /// シフトキーリリース時の軸ロック終了処理をトリガーします。
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="e">イベント引数</param>
        private void Handle_CurveEditor_KeyUp(object sender, System.Windows.Forms.KeyEventArgs e)
        {
            if (e.KeyCode == Keys.ShiftKey)
            {
                this.EndLock();
            }
        }

        /// <summary>
        /// フレーム数フォーカスイベント
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="e">イベント引数</param>
        private void Handle_text_Frame_Enter(object sender, EventArgs e)
        {
            this.textFrameBackup = this.text_Frame.Text;
        }

        /// <summary>
        /// フレーム数ロストフォーカスイベント
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="e">イベント引数</param>
        private void Handle_text_Frame_Leave(object sender, EventArgs e)
        {
            this.ApplyTextFrame();

            this.UpdateResetButton();
        }

        /// <summary>
        /// フレーム数キー入力イベント
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="e">イベント引数</param>
        private void Handle_text_Frame_KeyDown(object sender, System.Windows.Forms.KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter)
            {
                this.ApplyTextFrame();
                this.text_Frame.SelectAll();
            }
            else if (e.KeyCode == Keys.Escape)
            {
                this.text_Frame.Text = this.textFrameBackup;
            }
        }

        /// <summary>
        /// フレーム数キー入力フィルタ
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="e">イベント引数</param>
        private void Handle_text_Frame_KeyPress(object sender, KeyPressEventArgs e)
        {
            if ((Control.ModifierKeys & Keys.Control) == Keys.Control)
            {
                return;
            }

            // 0～9と、バックスペース以外の時は、イベントをキャンセルする
            if ((e.KeyChar < '0' || '9' < e.KeyChar) && e.KeyChar != '\b')
            {
                e.Handled = true;
            }
        }

        /// <summary>
        /// 値フォーカスイベント
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="e">イベント引数</param>
        private void Handle_text_Value_Enter(object sender, EventArgs e)
        {
            this.textValueBackup = this.text_Value.Text;
            this.text_Value.SelectAll();
        }

        /// <summary>
        /// 値ロストフォーカスイベント
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="e">イベント引数</param>
        private void Handle_text_Value_Leave(object sender, EventArgs e)
        {
            this.ApplyTextValue();

            this.UpdateResetButton();
        }

        /// <summary>
        /// 値キー入力イベント
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="e">イベント引数</param>
        private void Handle_text_Value_KeyDown(object sender, System.Windows.Forms.KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter)
            {
                this.ApplyTextValue();
                this.text_Value.SelectAll();
            }
            else if (e.KeyCode == Keys.Escape)
            {
                this.text_Value.Text = this.textValueBackup;
            }
        }

        /// <summary>
        /// 値キー入力フィルタ
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="e">イベント引数</param>
        private void Handle_text_Value_KeyPress(object sender, KeyPressEventArgs e)
        {
            if ((Control.ModifierKeys & Keys.Control) == Keys.Control)
            {
                return;
            }

            // 0～9と、ピリオド、ハイフン、バックスペース以外の時は、イベントをキャンセルする
            if ((e.KeyChar < '0' || '9' < e.KeyChar) && e.KeyChar != '\b' && e.KeyChar != '.' && e.KeyChar != '-')
            {
                e.Handled = true;
            }
        }

        /// <summary>
        /// ループ終端フォーカスイベント
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="e">イベント引数</param>
        private void Handle_text_LoopEnd_Enter(object sender, EventArgs e)
        {
            this.textLoopEndBackup = this.text_LoopEnd.Text;
        }

        /// <summary>
        /// ループ終端ロストフォーカスイベント
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="e">イベント引数</param>
        private void Handle_text_LoopEnd_Leave(object sender, EventArgs e)
        {
            this.ApplyLoopEnd();

            this.UpdateResetButton();
        }

        /// <summary>
        /// ループ終端変更通知イベント
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="e">イベント引数</param>
        private void Handle_text_LoopEnd_TextChanged(object sender, EventArgs e)
        {
            this.ApplyLoopEnd(true);
        }

        /// <summary>
        /// ループ終端キー入力イベント
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="e">イベント引数</param>
        private void Handle_text_LoopEnd_KeyDown(object sender, System.Windows.Forms.KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter)
            {
                this.ApplyLoopEnd();
                this.text_LoopEnd.SelectAll();
            }
            else if (e.KeyCode == Keys.Escape)
            {
                this.text_LoopEnd.Text = this.textLoopEndBackup;
            }
        }

        /// <summary>
        /// ループ終端キー入力フィルタ
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="e">イベント引数</param>
        private void Handle_text_LoopEnd_KeyPress(object sender, KeyPressEventArgs e)
        {
            if ((Control.ModifierKeys & Keys.Control) == Keys.Control)
            {
                return;
            }

            // 0～9とバックスペース以外の時は、イベントをキャンセルする
            if ((e.KeyChar < '0' || '9' < e.KeyChar) && e.KeyChar != '\b')
            {
                e.Handled = true;
            }
        }

        /// <summary>
        /// リセットボタンイベント
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="e">イベント引数</param>
        private void Handle_button_Reset_Click(object sender, EventArgs e)
        {
            this.ResetNode();
        }

        /// <summary>
        /// ループモード変更イベント
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="e">イベント引数</param>
        private void Handle_button_LoopMode_Click(object sender, EventArgs e)
        {
            this.LoopMode = this.LoopMode == 0 ? 1 : 0;

            this.FocusWithoutScroll();

            this.UpdateResetButton();
        }

        /// <summary>
        /// 他のキーに値スナップモード変更イベント
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="e">イベント引数</param>
        private void Handle_button_SnapKeyMode_Click(object sender, EventArgs e)
        {
            this.otherKeySnapping = !this.otherKeySnapping;
            this.button_SnapKeyMode.BackgroundImage =
                this.otherKeySnapping ?
                Properties.Resources.Icon_Curve__KeySnap_On :
                Properties.Resources.Icon_Curve__KeySnap_Off;

            this.LogicalTreeElementExtender.NotifyPropertyChanged(
                BindingUpdateType.PropertyChanged,
                propertyName: "EnableOtherKeySnap");

            this.UpdateResetButton();
        }

        /// <summary>
        /// ヘルプを開く
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="e">イベント引数</param>
        private void Handle_button_PanelHelp_Click(object sender, EventArgs e)
        {
            if (this.EditorMode == EditorModes.ParticleTimeAnim)
            {
                // パーティクル時間アニメの場合
                HelpManager.ShowHelp("96066185", "");
            }
            else
            {
                // エミッタ時間アニメの場合
                HelpManager.ShowHelp("92116294", "");
            }
        }

        /// <summary>
        /// Xチェックボックス変更イベント
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="e">イベント引数</param>
        private void Handle_checkbox_X_CheckedChanged(object sender, EventArgs e)
        {
            this.CheckDeselection();
            this.LogicalTreeElementExtender.NotifyPropertyChanged(
                BindingUpdateType.PropertyChanged,
                propertyName: "EnableX");

            this.RequestDraw();
            this.UpdateResetButton();
        }

        /// <summary>
        /// Yチェックボックス変更イベント
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="e">イベント引数</param>
        private void Handle_checkbox_Y_CheckedChanged(object sender, EventArgs e)
        {
            this.CheckDeselection();
            this.LogicalTreeElementExtender.NotifyPropertyChanged(
                BindingUpdateType.PropertyChanged,
                propertyName: "EnableY");

            this.RequestDraw();
            this.UpdateResetButton();
        }

        /// <summary>
        /// Zチェックボックス変更イベント
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="e">イベント引数</param>
        private void Handle_checkbox_Z_CheckedChanged(object sender, EventArgs e)
        {
            this.CheckDeselection();
            this.LogicalTreeElementExtender.NotifyPropertyChanged(
                BindingUpdateType.PropertyChanged,
                propertyName: "EnableZ");

            this.RequestDraw();
            this.UpdateResetButton();
        }

        /// <summary>
        /// Wチェックボックス変更イベント
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="e">イベント引数</param>
        private void Handle_checkbox_W_CheckedChanged(object sender, EventArgs e)
        {
            this.CheckDeselection();
            this.LogicalTreeElementExtender.NotifyPropertyChanged(
                BindingUpdateType.PropertyChanged,
                propertyName: "EnableW");

            this.RequestDraw();
            this.UpdateResetButton();
        }

        /// <summary>
        /// 時間スナップ変更イベント
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="e">イベント引数</param>
        private void Handle_combobox_SnapFrame_SelectedIndexChanged(object sender, EventArgs e)
        {
            this.FocusWithoutScroll();

            this.LogicalTreeElementExtender.NotifyPropertyChanged(
                BindingUpdateType.PropertyChanged,
                propertyName: "SnapTimeLevel");

            this.UpdateResetButton();
        }

        /// <summary>
        /// 値スナップ変更イベント
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="e">イベント引数</param>
        private void Handle_combobox_SnapValue_SelectedIndexChanged(object sender, EventArgs e)
        {
            this.FocusWithoutScroll();

            if (this.IntegerOnly)
            {
                if (this.combobox_SnapValue.SelectedIndex < 4)
                {
                    this.combobox_SnapValue.SelectedIndex = 4;
                }
            }

            this.LogicalTreeElementExtender.NotifyPropertyChanged(
                BindingUpdateType.PropertyChanged,
                propertyName: "SnapValueLevel");

            this.UpdateResetButton();
        }

        /// <summary>
        /// 開始位置ランダム変更イベント
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="e">イベント引数</param>
        private void Handle_checkbox_Random_CheckedChanged(object sender, EventArgs e)
        {
            this.Randomize = this.checkbox_Random.Checked ? 1 : 0;

            this.UpdateResetButton();
        }

        /// <summary>
        /// 補間タイプ変更イベント
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="e">イベント引数</param>
        private void Handle_combobox_InterMode_SelectedIndexChanged(object sender, EventArgs e)
        {
            this.FocusWithoutScroll();

            this.InterMode = this.combobox_InterMode.SelectedIndex;
            if (this.manipurator != null)
            {
                this.manipurator.Reconnect(this.viewports);
                this.manipurator.SortCurves(this.viewports, this.InterMode, this.checkboxList);
                this.UpdateTarget();
                this.RequestDraw();
            }

            this.UpdateResetButton();
        }

        /// <summary>
        /// 0固定モード変更イベント
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="e">イベント引数</param>
        private void Handle_combobox_ZeroPin_SelectedIndexChanged(object sender, EventArgs e)
        {
            this.FocusWithoutScroll();

            this.LogicalTreeElementExtender.NotifyPropertyChanged(
                BindingUpdateType.PropertyChanged,
                propertyName: "ZeroPinMode");

            if (this.viewports != null)
            {
                if (this.prevZeroPin != this.ZeroPinMode && -1 <= this.ZeroPinMode && this.ZeroPinMode <= 2 &&
                    !(this.prevZeroPin == -1 && this.ZeroPinMode == 2) &&
                    !(this.prevZeroPin == 2 && this.ZeroPinMode == -1))
                {
                    // 0固定モードが変化した時はビューポートのサイズを調整する
                    // 固定なし(3)への変更時と、非表示(-1)と下辺(2)の相互変更時は省略
                    this.AdjustViewport(false, 1);
                }

                this.LockTranslation();
                this.Rescale();

                if (this.values.Count > 0)
                {
                    this.manipurator.Reconnect(this.viewports);
                    this.manipurator.SortCurves(this.viewports, this.InterMode, this.checkboxList);
                }

                this.UpdateTarget();
                this.RequestDraw();
            }

            this.prevZeroPin = this.ZeroPinMode;
            this.UpdateResetButton();
        }

        /// <summary>
        /// タイマーによるドラッグハンドラ
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="e">イベント引数</param>
        private void Handle_timer_DragScroll_Tick(object sender, EventArgs e)
        {
            bool type0 = this.ZeroPinMode == 0;
            bool type1 = this.ZeroPinMode == 1;
            bool type2 = this.ZeroPinMode == 2 || this.ZeroPinMode == -1;

            bool dragPlus = this.scrollVec.Y < -0.0001f;
            bool dragMinus = this.scrollVec.Y > 0.0001f;

            if (this.EditorMode == EditorModes.EmitterTimeAnim)
            {
                if (this.scrollVec.X < -0.0001f)
                {
                    this.MoveNode(this.CalcLockPos(this.PointToClient(Cursor.Position)));
                    this.AdjustViewport(true, 2);
                    this.LockTranslation();
                    this.Rescale();
                    this.SyncData(true);

                    return;
                }
            }

            if ((type0 && dragMinus) || (type2 && dragPlus))
            {
                return;
            }

            if ((type0 && dragPlus) || (type1 && (dragPlus || dragMinus)) || (type2 && dragMinus))
            {
                this.MoveNode(this.CalcLockPos(this.PointToClient(Cursor.Position)));
                this.AdjustViewport(false, 1);
                this.LockTranslation();
                this.Rescale();
                this.SyncData(true);
            }
            else
            {
                var now = this.viewports.Main.Translation;

                if (this.lockDirection != 2 && this.ValidScale.X > this.MinimumScaleX)
                {
                    now.X += this.scrollVec.X;
                }

                if (this.lockDirection != 1 && this.ValidScale.Y > this.MinimumScaleY)
                {
                    if (this.scrollVec.Y > 0.0f || this.invalidArea.BottomVisible)
                    {
                        now.Y += this.scrollVec.Y / this.ValidScale.Y;
                    }
                }

                this.viewports.Main.Translation = now;
                this.LockTranslation();
                this.Rescale();
                this.MoveNode(this.CalcLockPos(this.viewports.GetClampedMousePosition()));

                this.SyncData(true);

                this.UpdateInvalidArea();
                this.RequestDraw();
            }
        }

        /// <summary>
        /// カーブエディタの先頭フォーカスハンドラ
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="args">イベント引数</param>
        private void Handle_text_FocusSwitcher0_GotFocus(object sender, EventArgs args)
        {
            if ((Control.ModifierKeys & Keys.Shift) == Keys.Shift)
            {
                // 逆送り
                int ch = -1;
                int idx = this.manipurator.GetIndicesByNode(out ch, this.targetNode);

                var selectables = this.EnumSelectableChannel(idx);
                int selectableIndex = selectables.IndexOf(ch);
                if (selectableIndex - 1 < 0)
                {
                    if (idx == 0)
                    {
                        // 前のコントロールをフォーカス
                        Control test = this.Parent;
                        while (test != null)
                        {
                            var tabPage = test as PropertyTabPageBase;
                            if (tabPage != null)
                            {
                                tabPage.SelectNextChildControl(false);
                                break;
                            }

                            test = test.Parent;
                        }

                        return;
                    }

                    --idx;
                    var nextSelectables = this.EnumSelectableChannel(idx);
                    this.targetNode = null;
                    if (nextSelectables.Count != 0)
                    {
                        try
                        {
                            this.targetNode = this.manipurator.GetNodeByIndices(nextSelectables.Last(), idx);
                        }
                        catch
                        {
                            // 例外が起きたらnullを採用する
                        }
                    }
                }
                else
                {
                    try
                    {
                        this.targetNode = this.manipurator.GetNodeByIndices(selectableIndex - 1, idx);
                    }
                    catch
                    {
                        // 例外が起きたらnullを採用する
                        this.targetNode = null;
                    }
                }

                this.UpdateTarget();
                this.CheckDeselection();
                this.text_Value.Select();
                this.text_Value.SelectAll();
            }
            else
            {
                // 順送り
                var nextSelectables = this.EnumSelectableChannel(0);
                this.targetNode = null;
                if (nextSelectables.Count != 0)
                {
                    try
                    {
                        this.targetNode = this.manipurator.GetNodeByIndices(nextSelectables[0], 0);
                    }
                    catch
                    {
                        // 例外が起きたらnullを採用する
                    }
                }

                this.UpdateTarget();
                this.CheckDeselection();
                this.text_FocusDummy.Select();

            }

            this.RequestDraw();
        }

        /// <summary>
        /// カーブエディタの中間フォーカスハンドラ
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="args">イベント引数</param>
        private void Handle_text_FocusSwitcher1_GotFocus(object sender, EventArgs args)
        {
            if ((Control.ModifierKeys & Keys.Shift) == Keys.Shift)
            {
                // 逆送り
                this.text_FocusDummy.Select();
            }
            else
            {
                // 順送り
                this.text_Frame.Select();
            }

            this.RequestDraw();
        }

        /// <summary>
        /// カーブエディタの終端フォーカスハンドラ
        /// </summary>
        /// <param name="sender">イベントを発生させたオブジェクト</param>
        /// <param name="args">イベント引数</param>
        private void Handle_text_FocusSwitcher2_GotFocus(object sender, EventArgs args)
        {
            if ((Control.ModifierKeys & Keys.Shift) == Keys.Shift)
            {
                // 逆送り
                var nextSelectables = this.EnumSelectableChannel(this.values.Count - 1);
                this.targetNode = null;
                if (nextSelectables.Count != 0)
                {
                    try
                    {
                        this.targetNode = this.manipurator.GetNodeByIndices(nextSelectables.Last(), this.values.Count - 1);
                    }
                    catch
                    {
                        // 例外が起きたらnullを採用する
                    }
                }

                this.UpdateTarget();
                this.CheckDeselection();
                this.text_Value.Select();
                this.text_Value.SelectAll();
            }
            else
            {
                int ch = -1;
                int idx = this.manipurator.GetIndicesByNode(out ch, this.targetNode);

                // 順送り
                var selectables = this.EnumSelectableChannel(idx);
                int selectableIndex = selectables.IndexOf(ch);
                if (selectableIndex + 1 >= selectables.Count)
                {
                    if (idx == this.values.Count - 1)
                    {
                        // 末端のキーならページ全体のタブオーダーへ移る
                        Control test = this.Parent;
                        while (test != null)
                        {
                            var tabPage = test as PropertyTabPageBase;
                            if (tabPage != null)
                            {
                                tabPage.SelectNextChildControl(true);
                                break;
                            }

                            test = test.Parent;
                        }

                        return;
                    }

                    ++idx;
                    var nextSelectables = this.EnumSelectableChannel(idx);
                    try
                    {
                        this.targetNode = nextSelectables.Count != 0 ?
                            this.manipurator.GetNodeByIndices(nextSelectables[0], idx) : null;
                    }
                    catch
                    {
                        this.targetNode = null;
                    }
                }
                else
                {
                    try
                    {
                        this.targetNode = this.manipurator.GetNodeByIndices(selectables[selectableIndex + 1], idx);
                    }
                    catch
                    {
                        this.targetNode = null;
                    }
                }

                this.UpdateTarget();
                this.CheckDeselection();
                this.text_FocusDummy.Select();

            }

            this.RequestDraw();
        }

        #endregion
    }
}
