﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.ComponentModel;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.Behaviors;
using EffectMaker.UIControls.Extensions;

namespace EffectMaker.UIControls.Specifics.Behaviors
{
    /// <summary>
    /// Behavior to create copy/paste context menu for tab pages.
    /// </summary>
    public class TabPageCopyPasteContextMenuBehavior : Behavior<UITabPage>, INotifyPropertyChanged
    {
        /// <summary>
        /// Initialize the behavior when attached to the tab page.
        /// </summary>
        protected override void OnAttached()
        {
            // Create the context menu for the tab header.
            var contextMenu = new UIContextMenuStrip();

            var copyPageMenuItem = new UIToolStripMenuItem();
            var pastePageMenuItem = new UIToolStripMenuItem();
            var resetPageDefaultValuesMenuItem = new UIToolStripMenuItem();

            copyPageMenuItem.Text = Properties.Resources.MenuCopyPropertyPage;
            pastePageMenuItem.Text = Properties.Resources.MenuPastePropertyPage;
            resetPageDefaultValuesMenuItem.Text = Properties.Resources.MenuResetPageDefaultValues;

            copyPageMenuItem.AddBinding("Executable", "CopyPropertyPageExecutable");
            pastePageMenuItem.AddBinding("Executable", "PastePropertyPageExecutable");
            resetPageDefaultValuesMenuItem.AddBinding("Executable", "ResetDefaultValueExecutable");

            contextMenu.Opening += (s, e) =>
            {
                // Check if the clipboard contents can be pasted to this page.
                if (pastePageMenuItem.Executable != null)
                {
                    this.AssociatedObject.Focus();
                    pastePageMenuItem.Enabled = pastePageMenuItem.Executable.CanExecute(
                        pastePageMenuItem.ExecutionParameter);
                }
            };

            contextMenu.Items.Add(copyPageMenuItem);
            contextMenu.Items.Add(pastePageMenuItem);
            contextMenu.Items.Add(resetPageDefaultValuesMenuItem);

            this.AssociatedObject.PropertyChanged += (s, e) =>
            {
                if (e.PropertyName == "DataContext")
                {
                    copyPageMenuItem.DataContext = this.DataContext;
                    pastePageMenuItem.DataContext = this.DataContext;
                    resetPageDefaultValuesMenuItem.DataContext = this.DataContext;
                }
            };

            this.AssociatedObject.TabHeaderContextMenuStrip = contextMenu;
        }

        /// <summary>
        /// Deinitialize the behavior when detached from the tab page.
        /// </summary>
        protected override void OnDetaching()
        {
            // Do nothing.
        }
    }
}
