﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Reflection;
using System.Xml;
using System.Xml.Schema;
using System.Xml.Serialization;

namespace EffectMaker.ObsoleteUserDataConverter.ObsoleteUserData
{
    /// <summary>
    /// The template definition serializer for .udd/.usd files.
    /// </summary>
    public class TemplateDefinition : IXmlSerializable
    {
        /// <summary>The name map of the assigned template values.</summary>
        private Dictionary<string, string> templateValues = new Dictionary<string, string>();

        /// <summary>
        /// Default constructor.
        /// </summary>
        public TemplateDefinition() :
            base()
        {
        }

        /// <summary>
        /// Apply template value to the specified property if the template value exists.
        /// </summary>
        /// <param name="def">The definition to apply the template to.</param>
        /// <param name="propertyName">The name of the property to apply template.</param>
        /// <returns>
        /// True on success; False if the template value is not set
        /// or the value cannot be converted to the property type.
        /// </returns>
        public bool ApplyTemplateValue(UserDefinitionBase def, string propertyName)
        {
            if (def == null)
            {
                return false;
            }

            // The template value only applies if the property value
            // has not been explicitly assigned.
            if (def.IsPropertyAssigned(propertyName) == true)
            {
                return false;
            }

            // Get the property info, so we can find out the property type.
            PropertyInfo property = def.GetType().GetProperty(propertyName);
            if (property == null)
            {
                return false;
            }

            string name = propertyName;

            var attr = property.GetCustomAttribute<XmlAttributeAttribute>(false);
            if (attr != null)
            {
                name = attr.AttributeName;
            }

            // Find the template value with the property name.
            string value;
            if (this.templateValues.TryGetValue(name, out value) == false)
            {
                return false;
            }

            // Get the type converter, so we can convert the string value to its type.
            TypeConverter converter = TypeDescriptor.GetConverter(property.PropertyType);
            if (converter == null)
            {
                return false;
            }

            object convertedValue = converter.ConvertFromInvariantString(value);

            // Set the converted template value to the property.
            property.SetValue(def, convertedValue);

            return true;
        }

        /// <summary>
        /// Generates an object from its XML representation.
        /// </summary>
        /// <param name="reader">The XmlReader stream from which the object is deserialized.</param>
        public void ReadXml(XmlReader reader)
        {
            string myLocalName = reader.LocalName;

            this.templateValues.Clear();

            if (reader.HasAttributes == true)
            {
                while (reader.MoveToNextAttribute() == true)
                {
                    this.templateValues.Add(reader.Name, reader.Value);
                }
            }
        }

        /// <summary>
        /// Converts an object into its XML representation.
        /// </summary>
        /// <param name="writer">The XmlWriter stream to which the object is serialized.</param>
        public void WriteXml(XmlWriter writer)
        {
            throw new NotSupportedException("The template definition does not support serializing to Xml.");
        }

        /// <summary>
        /// This method is reserved and should not be used.
        /// When implementing the IXmlSerializable interface,
        /// you should return null from this method, and
        /// instead, if specifying a custom schema is required,
        /// apply the XmlSchemaProviderAttribute to the class.
        /// </summary>
        /// <returns>
        /// An XmlSchema that describes the XML representation
        /// of the object that is produced by the WriteXml
        /// method and consumed by the ReadXml method.
        /// </returns>
        public XmlSchema GetSchema()
        {
            return null;
        }

        /// <summary>
        /// This method should be called after deserialized from Xml
        /// so that the definition can prepare its data before it can
        /// be used.
        /// </summary>
        public virtual void PostLoad()
        {
            // The method does not do anything by default.
        }
    }
}
