﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.ToolDevelopmentKit
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;

    /// <summary>
    /// 列挙型のユーティリティです。
    /// </summary>
    /// <typeparam name="TEnum">テンプレート列挙型です。</typeparam>
    public static class EnumUtility<TEnum>
        where TEnum : struct
    {
        /// <summary>
        /// 静的コンストラクタ
        /// </summary>
        static EnumUtility()
        {
            Debug.Assert(typeof(TEnum).IsSubclassOf(typeof(Enum)), "Unexpected case!");
        }

        //-----------------------------------------------------------------
        // 情報の取得
        //-----------------------------------------------------------------

        /// <summary>
        /// 列挙型で定義されている値を取得します。
        /// </summary>
        /// <param name="list">取得した列挙値を格納するリストです。</param>
        public static void GetValues(IList<TEnum> list)
        {
            Ensure.Argument.NotNull(list);

            foreach (TEnum value in Enum.GetValues(typeof(TEnum)))
            {
                list.Add(value);
            }
        }

        /// <summary>
        /// フラグをサポートする列挙型かどうかを取得します。
        /// </summary>
        /// <returns>フラグをサポートするのであれば true です。</returns>
        public static bool IsFlags()
        {
            object[] attributes = typeof(TEnum).GetCustomAttributes(
                typeof(FlagsAttribute), false);
            return attributes.Length != 0;
        }

        /// <summary>
        /// 列挙型の長さを取得します。
        /// </summary>
        /// <returns>列挙型の長さです。</returns>
        public static int GetLength()
        {
            return Enum.GetValues(typeof(TEnum)).Length;
        }

        //-----------------------------------------------------------------
        // 型変換
        //-----------------------------------------------------------------

        /// <summary>
        /// 列挙型に変換します。
        /// </summary>
        /// <param name="value">変換する値です。</param>
        /// <returns>列挙型に変換した値です。</returns>
        public static TEnum ToEnum(int value)
        {
            return (TEnum)Enum.ToObject(typeof(TEnum), value);
        }

        /// <summary>
        /// 列挙型に変換します。
        /// </summary>
        /// <param name="value">変換する値です。</param>
        /// <returns>列挙型に変換した値です。</returns>
        public static TEnum ToEnum(float value)
        {
            return (TEnum)Enum.ToObject(typeof(TEnum), (int)value);
        }

        /// <summary>
        /// 列挙型に変換します。
        /// </summary>
        /// <param name="value">変換する値です。</param>
        /// <returns>列挙型に変換した値です。</returns>
        public static TEnum ToEnum(object value)
        {
            return (TEnum)Enum.ToObject(typeof(TEnum), value);
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// int 型に変換します。
        /// </summary>
        /// <param name="value">変換する列挙値です。</param>
        /// <returns>変換した int 値です。</returns>
        public static int ToInt(TEnum value)
        {
            // テンプレートだとボックス処理が省けない
            return (int)(object)value;
        }
    }
}
