﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.ToolDevelopmentKit
{
    /// <summary>
    /// 親を持つオブジェクト関連のユーティリティです。
    /// </summary>
    public static class ChildObjectUtility
    {
        /// <summary>
        /// 循環参照のチェックを行います。
        /// </summary>
        /// <typeparam name="TParent">親を持つ型です。</typeparam>
        /// <param name="parent">チェックを行う親です。</param>
        /// <param name="child">チェックを行う子です。</param>
        /// <returns>循環参照する場合は true を返します。</returns>
        public static bool CheckCircularReference<TParent>(
            IChildObject<TParent> parent,
            IChildObject<TParent> child)
            where TParent : class, IChildObject<TParent>
        {
            IChildObject<TParent> current = parent;
            while (current != null)
            {
                if (current == child)
                {
                    return true;
                }

                current = current.Parent;
            }

            return false;
        }

        /// <summary>
        /// 現在の子のチェック行った上で新しい子を返します。
        /// </summary>
        /// <typeparam name="TParent">親の型です。</typeparam>
        /// <param name="parent">親のオブジェクトです。</param>
        /// <param name="currentChild">現在の子のオブジェクトです。</param>
        /// <param name="newChild">新しい子のオブジェクトです。</param>
        /// <returns>新しくセットする子のオブジェクト</returns>
        public static IChildObject<TParent> RegisterToParent<TParent>(
            TParent parent,
            IChildObject<TParent> currentChild,
            IChildObject<TParent> newChild)
            where TParent : class
        {
            if (newChild == null)
            {
                if (currentChild != null)
                {
                    currentChild.Parent = null;
                }

                currentChild = null;
                return currentChild;
            }

            Ensure.Argument.True(newChild.Parent == null);

            if (currentChild != null)
            {
                currentChild.Parent = null;
            }

            newChild.Parent = parent;
            currentChild = newChild;
            return currentChild;
        }

        /// <summary>
        /// 親が設定できるかチェックを行います。
        /// </summary>
        /// <param name="parent">チェックを行う親です。</param>
        /// <param name="child">チェックを行う子です。</param>
        /// <returns>設定できれば true を返します。</returns>
        public static bool IsReadyToSetParent(object parent, IChildObject child)
        {
            return (child.Parent == null) || (parent == null);
        }
    }
}
