﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.ToolDevelopmentKit.Conversion
{
    using System;

    /// <summary>
    /// 自動関連付け動作の種類です。
    /// </summary>
    public enum CoupledTypeAutoMode
    {
        /// <summary>
        /// 処理しません。
        /// </summary>
        None,

        /// <summary>
        /// 派生クラスを関連付けます。
        /// </summary>
        DerivedType,
    }

    /// <summary>
    /// 型の関連付けを記述するアトリビュートです。
    /// <para>
    /// 関連型は、指定されたコンバートハンドラで変換されます。
    /// </para>
    /// <para>
    /// ハンドラ指定を省略した場合には、名前マッチによる変換処理を行う
    /// デフォルトハンドラが指定されます（DefaultConverters.ConvertClass）。
    /// </para>
    /// </summary>
    public class CoupledTypeAttribute : Attribute
    {
        private readonly ConvertHandler forwardConvertHandler;
        private readonly ConvertHandler backwardConvertHandler;
        private readonly CoupledTypeAutoMode coupledTypeAutoMode;

        private Type[] coupledTypes;

        //----------------------------------------------------------
        // コンストラクタ
        //----------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="coupledType">関連付けられた型です。</param>
        public CoupledTypeAttribute(Type coupledType)
            : this(
            new Type[] { coupledType },
            DefaultConverters.ConvertClass,
            DefaultConverters.ConvertClass,
            CoupledTypeAutoMode.None)
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="coupledType">関連付けられた型です。</param>
        /// <param name="handlerType">変換ハンドラを定義している型です。</param>
        /// <param name="handlerName">順・逆変換を行うハンドラ名です(静的メソッドの必要があります)。</param>
        public CoupledTypeAttribute(Type coupledType, Type handlerType, string handlerName)
            : this(coupledType, handlerType, handlerName, handlerName)
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="coupledType">関連付けられた型です。</param>
        /// <param name="handlerType">変換ハンドラを定義している型です。</param>
        /// <param name="forwardHandlerName">順変換ハンドラ名です(静的メソッドの必要があります)。</param>
        /// <param name="backWordHandlerName">逆変換ハンドラ名です(静的メソッドの必要があります)。</param>
        public CoupledTypeAttribute(
            Type coupledType,
            Type handlerType,
            string forwardHandlerName,
            string backWordHandlerName)
            : this(
            new Type[] { coupledType },
            handlerType,
            forwardHandlerName,
            backWordHandlerName,
            CoupledTypeAutoMode.None)
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="coupledType">関連付けられた型です。</param>
        /// <param name="handlerType">変換ハンドラを定義している型です。</param>
        /// <param name="forwardHandlerName">順変換ハンドラ名です(静的メソッドの必要があります)。</param>
        /// <param name="backWordHandlerName">逆変換ハンドラ名です(静的メソッドの必要があります)。</param>
        /// <param name="coupledTypeAutoMode">自動関連付け処理の種類です。</param>
        public CoupledTypeAttribute(
            Type coupledType,
            Type handlerType,
            string forwardHandlerName,
            string backWordHandlerName,
            CoupledTypeAutoMode coupledTypeAutoMode)
            : this(
            new Type[] { coupledType },
            handlerType,
            forwardHandlerName,
            backWordHandlerName,
            coupledTypeAutoMode)
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="coupledTypes">関連付けられた型です。</param>
        /// <param name="handlerType">変換ハンドラを定義している型です。</param>
        /// <param name="forwardHandlerName">順変換ハンドラ名です(静的メソッドの必要があります)。</param>
        /// <param name="backWordHandlerName">逆変換ハンドラ名です(静的メソッドの必要があります)。</param>
        public CoupledTypeAttribute(
            Type[] coupledTypes,
            Type handlerType,
            string forwardHandlerName,
            string backWordHandlerName)
            : this(
            coupledTypes,
            handlerType,
            forwardHandlerName,
            backWordHandlerName,
            CoupledTypeAutoMode.None)
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="coupledTypes">関連付けられた型です。</param>
        /// <param name="handlerType">変換ハンドラを定義している型です。</param>
        /// <param name="forwardHandlerName">順変換ハンドラ名です(静的メソッドの必要があります)。</param>
        /// <param name="backWordHandlerName">逆変換ハンドラ名です(静的メソッドの必要があります)。</param>
        /// <param name="coupledTypeAutoMode">自動関連付け処理の種類です。</param>
        public CoupledTypeAttribute(
            Type[] coupledTypes,
            Type handlerType,
            string forwardHandlerName,
            string backWordHandlerName,
            CoupledTypeAutoMode coupledTypeAutoMode)
            : this(
            coupledTypes,
            CreateConvertHandlerByName(handlerType, forwardHandlerName),
            CreateConvertHandlerByName(handlerType, backWordHandlerName),
            coupledTypeAutoMode)
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="coupledTypes">関連付けられた型です。</param>
        /// <param name="forwardConvertHandler">
        /// 順変換(coupledType => 自分)に利用するハンドラです。
        /// </param>
        /// <param name="backwardConvertHandler">
        /// 逆変換(自分 => coupledType)に利用するハンドラです。
        /// </param>
        /// <param name="coupledTypeAutoMode">自動関連付け処理の種類です。</param>
        public CoupledTypeAttribute(
            Type[] coupledTypes,
            ConvertHandler forwardConvertHandler,
            ConvertHandler backwardConvertHandler,
            CoupledTypeAutoMode coupledTypeAutoMode)
        {
            Ensure.Argument.NotNull(coupledTypes);
            Ensure.Argument.NotNull(forwardConvertHandler);
            Ensure.Argument.NotNull(backwardConvertHandler);

            this.coupledTypes = coupledTypes;
            this.forwardConvertHandler = forwardConvertHandler;
            this.backwardConvertHandler = backwardConvertHandler;
            this.coupledTypeAutoMode = coupledTypeAutoMode;
        }

        //----------------------------------------------------------
        // プロパティ
        //----------------------------------------------------------

        /// <summary>
        /// 関連付けられた型を取得、設定します。
        /// </summary>
        public Type[] CoupledTypes
        {
            get
            {
                return this.coupledTypes;
            }

            set
            {
                Ensure.Argument.NotNull(value);
                this.coupledTypes = value;
            }
        }

        /// <summary>
        /// 派生クラスにも関連付けを取得します。
        /// </summary>
        public CoupledTypeAutoMode CoupledTypeAutoMode
        {
            get { return this.coupledTypeAutoMode; }
        }

        /// <summary>
        /// 順変換(coupledType => アトリビュートが設定されたシリアライズ用クラス)
        /// に利用するハンドラを取得します。
        /// </summary>
        public ConvertHandler ForwardConvertHandler
        {
            get
            {
                return this.forwardConvertHandler;
            }
        }

        /// <summary>
        /// 逆変換(アトリビュートが設定されたシリアライズ用クラス => coupledType)
        /// に利用するハンドラを取得します。
        /// </summary>
        public ConvertHandler BackwardConvertHandler
        {
            get
            {
                return this.backwardConvertHandler;
            }
        }

        //----------------------------------------------------------

        /// <summary>
        /// 変換ハンドラを生成します。
        /// </summary>
        /// <param name="type">変換ハンドラを定義している型です。</param>
        /// <param name="handlerName">変換ハンドラ名です。</param>
        /// <returns>変換ハンドラです。</returns>
        private static ConvertHandler CreateConvertHandlerByName(Type type, string handlerName)
        {
            try
            {
                ConvertHandler result = Delegate.CreateDelegate(
                    typeof(ConvertHandler), type, handlerName) as ConvertHandler;

                return result;
            }
            catch
            {
                throw new ArgumentException("CoupledTypeAttribute : Can't find convert functions.");
            }
        }
    }
}
