﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Xml.Serialization;

using NintendoWare.ToolDevelopmentKit.Conversion;
using NintendoWare.ToolDevelopmentKit.Xml;

using NWCore.DataModel.Major_1.Minor_3.Build_0.Revision_0;


namespace NWCore.Serializer.Major_1.Minor_3.Build_0.Revision_0
{
    /// <summary>
    /// 全エミッタ共通データのシリアライズ用クラスです
    /// </summary>
    [CoupledType(typeof(ComplexEmitterData))]
    [XmlType("ComplexEmitterData")]
    public class ComplexEmitterDataXml
    {
        /// <summary>
        /// 編集用データです。
        /// </summary>
        protected EmitterData m_data;

        #region Construtors

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ComplexEmitterDataXml()
        {
            m_data = new EmitterData();
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ComplexEmitterDataXml( EmitterData emitter,
                                      bool bNormalizePath = true )
        {
            m_data = new EmitterData(emitter);

            m_bNormalizePath = bNormalizePath;
        }

        #endregion

        /// <summary>
        /// EmitterData を取得します。
        /// </summary>
        public EmitterData EditData
        {
            get { return m_data; }
        }

        #region Properties - Animation table list

        /// <summary>
        /// The animation tables.
        /// </summary>
        [XmlElement( "animations" )]
        public EmitterAnimTableListXml Animations
        {
            get { return new EmitterAnimTableListXml( m_data.AnimTableList ); }
            set { m_data.AnimTableList.Set( value.AnimationList ); }
        }

        #endregion

        #region Properties - EditData

        /// <summary>
        /// 編集用データ
        /// </summary>
        public EditDataEmitterXml editor_data
        {
            get { return m_data.EditData.CreateEditDataXml(); }
            set { m_data.EditData.Set(value.EditData); }
        }

        #endregion

        #region Properties - SimpleEmitter

        /// <summary>
        /// アドレス解決後の名前
        /// </summary>
        public string name
        {
            get { return m_data.Name; }
            set { m_data.Name = value; }
        }

        /// <summary>
        /// パーティクルを描画する
        /// </summary>
        public bool DispParticle
        {
            get { return m_data.DispParticle; }
            set { m_data.DispParticle = value; }
        }

        /// <summary>
        /// テクスチャー
        /// </summary>
        public string texture
        {
            get
            {
                if (String.IsNullOrEmpty(m_data.TexPatData.UI_texPatFileName) == false)
                {
                    if (m_bNormalizePath == true)
                        return System.IO.Path.GetFileName(m_data.TexPatData.UI_texPatFileName);
                    else
                        return m_data.TexPatData.UI_texPatFileName;
                }
                else
                {
                    return String.Empty;
                }
            }
            set
            {
                m_data.TexPatData.UI_texPatFileName = value;
            }
        }

        /// <summary>
        /// タイプ
        /// </summary>
        public EmitterType type
        {
            get { return m_data.Type; }
            set { m_data.Type = value; }
        }

        /// <summary>
        /// ボリュームのタイプ
        /// </summary>
        public VolumeType volumeType
        {
            get { return m_data.VolumeType; }
            set { m_data.VolumeType = value; }
        }

        /// <summary>
        /// Blend type
        /// </summary>
        public BlendType UI_blendType
        {
            get { return m_data.BlendType; }
            set { m_data.BlendType = value; }
        }

        /// <summary>
        /// 回転タイプ
        /// </summary>
        public PtclRotType UI_rotType
        {
            get { return m_data.UI_rotType; }
            set { m_data.UI_rotType = value; }
        }

        /// <summary>
        /// カラー共通タイプ
        /// </summary>
        public string UI_colorType
        {
            get { return Constances.GetColorTypeName(m_data.UI_colorType); }
            set { m_data.UI_colorType = Constances.SetColorTypeName(value); }
        }

        /// <summary>
        /// カラースケール
        /// </summary>
        public float colorScale
        {
            get { return m_data.UI_colorScale; }
            set { m_data.UI_colorScale = value; }
        }

        /// <summary>
        /// カラー0設定タイプ
        /// </summary>
        public int UI_color0_type
        {
            get { return (int)(m_data.AnimEditData.Color0.ColorType); }
            set { m_data.AnimEditData.Color0.ColorType = (Constances.ColorSettingType)value; }
        }

        /// <summary>
        /// カラー1設定タイプ
        /// </summary>
        public int UI_color1_type
        {
            get { return (int)(m_data.AnimEditData.Color1.ColorType); }
            set { m_data.AnimEditData.Color1.ColorType = (Constances.ColorSettingType)value; }
        }

        /// <summary>
        /// Mesh type
        /// </summary>
        public MeshType meshType
        {
            get { return m_data.MeshType; }
            set { m_data.MeshType = value; }
        }

        /// <summary>
        /// パーティクルタイプ
        /// </summary>
        public BillboardType UI_particleType
        {
            get { return m_data.ParticleType; }
            set { m_data.ParticleType = value; }
        }

        /// <summary>
        /// ストライプタイプ
        /// </summary>
        public StripeType UI_stripeType
        {
            get { return m_data.StripeType; }
            set { m_data.StripeType = value; }
        }

        /// <summary>
        /// プリミティブファイルパス
        /// </summary>
        public string billboardPrimitiveFileSource
        {
            get
            {
                if (String.IsNullOrEmpty(m_data.BillboardPrimitiveFileSource) == false)
                {
                    if (m_bNormalizePath == true)
                        return System.IO.Path.GetFileName(m_data.BillboardPrimitiveFileSource);
                    else
                        return m_data.BillboardPrimitiveFileSource;
                }
                else
                {
                    return String.Empty;
                }
            }
            set { m_data.BillboardPrimitiveFileSource = value; }
        }

        /// <summary>
        /// ビルボードのスクリーンZ方向へのオフセット
        /// </summary>
        public float toCameraOffset
        {
            get { return m_data.BillboardZOffset; }
            set { m_data.BillboardZOffset = value; }
        }

        /// <summary>
        /// 追随タイプ
        /// </summary>
        public FollowType UI_followType
        {
            get { return m_data.UI_followType; }
            set { m_data.UI_followType = value; }
        }

        /// <summary>
        /// テクスチャーアドレス
        /// </summary>
        public TextureAddressing UI_textureAddressing
        {
            get { return m_data.TexPatData.UI_textureAddressing; }
            set { m_data.TexPatData.UI_textureAddressing = value; }
        }

        /// <summary>
        /// テクスチャーアドレス
        /// </summary>
        public TextureAddressing UI_subTextureAddressing
        {
            get { return m_data.SubTexPatData.UI_textureAddressing; }
            set { m_data.SubTexPatData.UI_textureAddressing = value; }
        }

        /// <summary>
        /// 描画パス
        /// </summary>
        public string drawPath
        {
            get { return m_data.UI_drawPath; }
            set { m_data.UI_drawPath = value; }
        }


        /// <summary>
        /// アルファ
        /// </summary>
        public AlphaDst UI_alphaDstType
        {
            get { return m_data.UI_alphaDstType; }
            set { m_data.UI_alphaDstType = value; }
        }

        /// <summary>
        /// Ｚバッファー、αテストタイプ
        /// </summary>
        public ZBufATestType zBufATestType
        {
            get { return m_data.ZBufATestType; }
            set { m_data.ZBufATestType = value; }
        }

        /// <summary>
        /// カリング
        /// </summary>
        public DisplaySideTypes DisplaySide
        {
            get { return m_data.DisplaySide; }
            set { m_data.DisplaySide = value; }
        }

        /// <summary>
        /// フィルタモード
        /// </summary>
        public FilterMode UI_texFilterMode
        {
            get { return m_data.TexPatData.UI_texFilterMode; }
            set { m_data.TexPatData.UI_texFilterMode = value; }
        }

        /// <summary>
        /// 各種フラグ
        /// </summary>
        public uint UI_hostioEmitterFlg
        {
            get { return m_data.UI_hostioEmitterFlg.Flag; }
            set { m_data.UI_hostioEmitterFlg.Flag = value; }
        }

        /// <summary>
        /// フィルタモード
        /// </summary>
        public FilterMode UI_SubTexFilterMode
        {
            get { return m_data.SubTexPatData.UI_texFilterMode; }
            set { m_data.SubTexPatData.UI_texFilterMode = value; }
        }

        /// <summary>
        /// FadeOutType
        /// </summary>
        public FadeOutType UI_fadeOutType
        {
            get { return m_data.UI_fadeOutType; }
            set { m_data.UI_fadeOutType = value; }
        }

        /// <summary>
        /// サブ・テクスチャー
        /// </summary>
        public string subTexture
        {
            get
            {
                if (String.IsNullOrEmpty(m_data.SubTexPatData.UI_texPatFileName) == false)
                {
                    if (m_bNormalizePath == true)
                        return System.IO.Path.GetFileName(m_data.SubTexPatData.UI_texPatFileName);
                    else
                        return m_data.SubTexPatData.UI_texPatFileName;
                }
                else
                {
                    return String.Empty;
                }
            }
            set { m_data.SubTexPatData.UI_texPatFileName = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int UI_randomSeed
        {
            get { return m_data.UI_randomSeed; }
            set { m_data.UI_randomSeed = value; }
        }


        /// <summary>
        /// ボリューム半径
        /// </summary>
        public Vector3Xml volumeRadius
        {
            get { return UtilityXml.ToVector3Xml(m_data.VolumeRadius); }
            set { m_data.VolumeRadius = UtilityXml.ToVector3(value); }
        }

        /// <summary>
        /// ラインの中心点
        /// </summary>
        public float lineCenter
        {
            get { return m_data.LineCenter; }
            set { m_data.LineCenter = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int UI_volumeSweep
        {
            get { return (int)m_data.VolumeSweepParam; }
            set { m_data.VolumeSweepParam = (uint)value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int UI_volumeSweepStart
        {
            get { return (int)m_data.VolumeSweepStart; }
            set { m_data.VolumeSweepStart = (uint)value; }
        }

        /// <summary>
        /// 弧の開始角度ランダム
        /// </summary>
        public int UI_volumeSweepStartRandom
        {
            get { return (m_data.VolumeSweepStartRandom == true ? 1 : 0); }
            set { m_data.VolumeSweepStartRandom = (value == 1 ? true : false); }
        }

        /// <summary>
        /// 空洞率
        /// </summary>
        public int volumeHollow
        {
            get { return m_data.VolumeHollow; }
            set { m_data.VolumeHollow = value; }
        }

        /// <summary>
        ///  テーブルを使うボリューム形状のインデックス
        /// </summary>
        public int volumeTblIndex
        {
            get { return m_data.VolumeTblIndex; }
            set { m_data.VolumeTblIndex = value; }
        }

        /// <summary>
        ///  テーブルを使うボリューム形状のインデックス
        /// </summary>
        public int volume64TblIndex
        {
            get { return m_data.Volume64TblIndex; }
            set { m_data.Volume64TblIndex = value; }
        }

        /// <summary>
        /// フォグ
        /// </summary>
        public int UI_Fog
        {
            get { return m_data.UI_Fog; }
            set { m_data.UI_Fog = value; }
        }

        /// <summary>
        /// エミッタ速度
        /// </summary>
        public float figureVel
        {
            get { return m_data.FigureVelocity; }
            set { m_data.FigureVelocity = value; }
        }

        /// <summary>
        /// エミッタ速度
        /// </summary>
        public float emitterVel
        {
            get { return m_data.EmitterVelocity; }
            set { m_data.EmitterVelocity = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public float UI_initVelRnd
        {
            get { return m_data.UI_initVelRnd; }
            set { m_data.UI_initVelRnd = value; }
        }

        /// <summary>
        /// 拡散ベクトル
        /// </summary>
        public Vector3Xml spreadVec
        {
            get { return UtilityXml.ToVector3Xml(m_data.SpreadVec); }
            set { m_data.SpreadVec = UtilityXml.ToVector3(value); }
        }

        /// <summary>
        /// エミッタ速度の方向
        /// </summary>
        public Vector3Xml emitterVelDir
        {
            get { return UtilityXml.ToVector3Xml(m_data.EmitterVelDir); }
            set { m_data.EmitterVelDir = UtilityXml.ToVector3(value); }
        }

        /// <summary>
        /// 指定方向拡散角度
        /// </summary>
        public float emitterVelDirAngle
        {
            get { return m_data.EmitterVelDirAngle; }
            set { m_data.EmitterVelDirAngle = value; }
        }

        /// <summary>
        /// 放出開始フレーム
        /// </summary>
        public int startFrame
        {
            get { return m_data.StartFrame; }
            set { m_data.StartFrame = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int UI_emitTime
        {
            get { return m_data.UI_emitTime; }
            set { m_data.UI_emitTime = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int UI_isOneTime
        {
            get { return (m_data.UI_isOneTime ? 1 : 0); }
            set { m_data.UI_isOneTime = (value != 0); }
        }

        /// <summary>
        /// ステップ値
        /// </summary>
        public int lifeStep
        {
            get { return m_data.LifeStep; }
            set { m_data.LifeStep = value; }
        }

        /// <summary>
        /// 運動量ランダム
        /// </summary>
        public float dynamicsRandom
        {
            get { return m_data.DynamicsRandom; }
            set { m_data.DynamicsRandom = value; }
        }

        /// <summary>
        /// Emitter放出レート（旧ファイルの対応）
        /// </summary>
        public float emitRate
        {
            get { return m_data.EmitRate; }
            set { m_data.EmitRate = value; }
        }

        /// <summary>
        /// Emitter放出レート
        /// </summary>
        public float UI_emitRate
        {
            get { return m_data.EmitRate; }
            set { m_data.EmitRate = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int UI_life
        {
            get { return m_data.AnimEditData.UI_life; }
            set { m_data.AnimEditData.UI_life = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int UI_lifeRnd
        {
            get { return m_data.AnimEditData.UI_lifeRnd; }
            set { m_data.AnimEditData.UI_lifeRnd = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public uint UI_lifeInfinit
        {
            get { return m_data.AnimEditData.UI_lifeInfinit; }
            set { m_data.AnimEditData.UI_lifeInfinit = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int UI_lifeStepRnd
        {
            get { return m_data.LifeStepRnd; }
            set { m_data.LifeStepRnd = value; }
        }

        /// <summary>
        /// 空気抵抗
        /// </summary>
        public float airRegist
        {
            get { return m_data.AirRegist; }
            set { m_data.AirRegist = value; }
        }

        /// <summary>
        /// 重力方向
        /// </summary>
        public Vector3Xml UI_gravityDir
        {
            get { return UtilityXml.ToVector3Xml(m_data.GravityDir); }
            set { m_data.GravityDir = UtilityXml.ToVector3(value); }
        }

        /// <summary>
        /// 重力の大きさ
        /// </summary>
        public float UI_gravity
        {
            get { return m_data.GravityVolume; }
            set { m_data.GravityVolume = value; }
        }

        /// <summary>
        /// 将来の為にタイプとして持たせる
        /// </summary>
        public int UI_isFieldCalcType
        {
            get { return m_data.IsWorldField ? 1 : 0; }
            set { m_data.IsWorldField = (value != 0); }
        }

        #region Color0

        /// <summary>Color0：カラー１</summary>
        public RgbaColorXml UI_color0_color1
        {
            get { return UtilityXml.ToRgbaColorXml(m_data.AnimEditData.Color0.Color1); }
            set { m_data.AnimEditData.Color0.Color1 = UtilityXml.ToRgbaColor(value); }
        }

        /// <summary>Color0：カラー２</summary>
        public RgbaColorXml UI_color0_color2
        {
            get { return UtilityXml.ToRgbaColorXml(m_data.AnimEditData.Color0.Color2); }
            set { m_data.AnimEditData.Color0.Color2 = UtilityXml.ToRgbaColor(value); }
        }

        /// <summary>Color0：カラー３</summary>
        public RgbaColorXml UI_color0_color3
        {
            get { return UtilityXml.ToRgbaColorXml(m_data.AnimEditData.Color0.Color3); }
            set { m_data.AnimEditData.Color0.Color3 = UtilityXml.ToRgbaColor(value); }
        }

        /// <summary>Color0：タイム１</summary>
        public int UI_color0_time1
        {
            get { return m_data.AnimEditData.Color0.Time1; }
            set { m_data.AnimEditData.Color0.Time1 = value; }
        }

        /// <summary>Color0：タイム２</summary>
        public int UI_color0_time2
        {
            get { return m_data.AnimEditData.Color0.Time2; }
            set { m_data.AnimEditData.Color0.Time2 = value; }
        }

        /// <summary>Color0：タイム３</summary>
        public int UI_color0_time3
        {
            get { return m_data.AnimEditData.Color0.Time3; }
            set { m_data.AnimEditData.Color0.Time3 = value; }
        }

        /// <summary>Color0：リピート</summary>
        public int UI_color0_numColorRepeat
        {
            get { return m_data.AnimEditData.Color0.NumRepeat; }
            set { m_data.AnimEditData.Color0.NumRepeat = value; }
        }

        public int color0RepeatStartRand
        {
            get { return m_data.AnimEditData.Color0.RandInitColor ? 1 : 0; }
            set { m_data.AnimEditData.Color0.RandInitColor = value > 0 ? true : false; }
        }

        #endregion

        #region Color1

        /// <summary>Color1：カラー１</summary>
        public RgbaColorXml UI_color1_color1
        {
            get { return UtilityXml.ToRgbaColorXml(m_data.AnimEditData.Color1.Color1); }
            set { m_data.AnimEditData.Color1.Color1 = UtilityXml.ToRgbaColor(value); }
        }

        /// <summary>Color1：カラー２</summary>
        public RgbaColorXml UI_color1_color2
        {
            get { return UtilityXml.ToRgbaColorXml(m_data.AnimEditData.Color1.Color2); }
            set { m_data.AnimEditData.Color1.Color2 = UtilityXml.ToRgbaColor(value); }
        }

        /// <summary>Color1：カラー３</summary>
        public RgbaColorXml UI_color1_color3
        {
            get { return UtilityXml.ToRgbaColorXml(m_data.AnimEditData.Color1.Color3); }
            set { m_data.AnimEditData.Color1.Color3 = UtilityXml.ToRgbaColor(value); }
        }

        /// <summary>Color1：タイム１</summary>
        public int UI_color1_time1
        {
            get { return m_data.AnimEditData.Color1.Time1; }
            set { m_data.AnimEditData.Color1.Time1 = value; }
        }

        /// <summary>Color1：タイム２</summary>
        public int UI_color1_time2
        {
            get { return m_data.AnimEditData.Color1.Time2; }
            set { m_data.AnimEditData.Color1.Time2 = value; }
        }

        /// <summary>Color1：タイム３</summary>
        public int UI_color1_time3
        {
            get { return m_data.AnimEditData.Color1.Time3; }
            set { m_data.AnimEditData.Color1.Time3 = value; }
        }

        /// <summary>Color1：リピート</summary>
        public int UI_color1_numColorRepeat
        {
            get { return m_data.AnimEditData.Color1.NumRepeat; }
            set { m_data.AnimEditData.Color1.NumRepeat = value; }
        }

        public int color1RepeatStartRand
        {
            get { return m_data.AnimEditData.Color1.RandInitColor ? 1 : 0; }
            set { m_data.AnimEditData.Color1.RandInitColor = value > 0 ? true : false; }
        }

        #endregion

        #region Alpha

        /// <summary>
        ///
        /// </summary>
        public float UI_alpha1
        {
            get { return m_data.AnimEditData.UI_alpha1; }
            set { m_data.AnimEditData.UI_alpha1 = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public float UI_alpha2
        {
            get { return m_data.AnimEditData.UI_alpha2; }
            set { m_data.AnimEditData.UI_alpha2 = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public float UI_alpha3
        {
            get { return m_data.AnimEditData.UI_alpha3; }
            set { m_data.AnimEditData.UI_alpha3 = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int UI_alphaTime1
        {
            get { return m_data.AnimEditData.UI_alphaTime1; }
            set { m_data.AnimEditData.UI_alphaTime1 = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int UI_alphaTime2
        {
            get { return m_data.AnimEditData.UI_alphaTime2; }
            set { m_data.AnimEditData.UI_alphaTime2 = value; }
        }

        #endregion

        #region Scale

        /// <summary>
        /// 初期スケール(%)
        /// </summary>
        public Vector2i UI_scale1
        {
            get { return m_data.AnimEditData.UI_scale1; }
            set { m_data.AnimEditData.UI_scale1 = value; }
        }

        /// <summary>
        /// スケール基準値
        /// </summary>
        public Vector2Xml UI_scale2
        {
            get { return UtilityXml.ToVector2Xml(m_data.AnimEditData.UI_scale2); }
            set { m_data.AnimEditData.UI_scale2 = UtilityXml.ToVector2(value); }
        }

        /// <summary>
        /// 中間スケール(%)
        /// </summary>
        public Vector2i UI_scale2Relative
        {
            get { return m_data.AnimEditData.UI_scale2Relative; }
            set { m_data.AnimEditData.UI_scale2Relative = value; }
        }

        /// <summary>
        /// 終了スケール(%)
        /// </summary>
        public Vector2i UI_scale3
        {
            get { return m_data.AnimEditData.UI_scale3; }
            set { m_data.AnimEditData.UI_scale3 = value; }
        }


        /// <summary>
        /// イン完了タイミング(%)
        /// </summary>
        public int UI_scaleTime1
        {
            get { return m_data.AnimEditData.UI_scaleTime1; }
            set { m_data.AnimEditData.UI_scaleTime1 = value; }
        }

        /// <summary>
        /// アウト開始タイミング(%)
        /// </summary>
        public int UI_scaleTime2
        {
            get { return m_data.AnimEditData.UI_scaleTime2; }
            set { m_data.AnimEditData.UI_scaleTime2 = value; }
        }

        /// <summary>
        /// スケール基準値の乱数
        /// </summary>
        public float UI_scaleRand
        {
            get { return m_data.UI_scaleRand; }
            set { m_data.UI_scaleRand = value; }
        }

        #endregion

        #region Rotate

        /// <summary>
        ///
        /// </summary>
        public Vector3i UI_initRot
        {
            get { return m_data.UI_initRot; }
            set { m_data.UI_initRot.Set(value); }
        }

        /// <summary>
        ///
        /// </summary>
        public Vector3i UI_initRotRand
        {
            get { return m_data.UI_initRotRand; }
            set { m_data.UI_initRotRand.Set(value); }
        }

        /// <summary>
        /// 減衰率
        /// </summary>
        public float rotResist
        {
            get { return m_data.UI_rotResist; }
            set { m_data.UI_rotResist = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public Vector3i UI_rotVel
        {
            get { return m_data.UI_rotVel; }
            set { m_data.UI_rotVel.Set(value); }
        }

        /// <summary>
        ///
        /// </summary>
        public Vector3i UI_rotVelRand
        {
            get { return m_data.UI_rotVelRand; }
            set { m_data.UI_rotVelRand.Set(value); }
        }

        /// <summary>
        ///
        /// </summary>
        public Vector2Xml UI_rotBasis
        {
            get { return UtilityXml.ToVector2Xml(m_data.UI_rotBasis); }
            set { m_data.UI_rotBasis = UtilityXml.ToVector2(value); }
        }

        #endregion

        /// <summary>
        ///
        /// </summary>
        public Vector3Xml UI_transformScale
        {
            get { return UtilityXml.ToVector3Xml(m_data.UI_transformScale); }
            set { m_data.UI_transformScale = UtilityXml.ToVector3(value); }
        }

        /// <summary>
        ///
        /// </summary>
        public Vector3i UI_transformRotRnd
        {
            get { return m_data.UI_transformRotRnd; }
            set { m_data.UI_transformRotRnd = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public Vector3Xml UI_transformTransRnd
        {
            get { return UtilityXml.ToVector3Xml(m_data.UI_transformTransRnd); }
            set { m_data.UI_transformTransRnd = UtilityXml.ToVector3(value); }
        }

        /// <summary>
        ///
        /// </summary>
        public int UI_fadeAlphaFrameInFade
        {
            get { return m_data.UI_fadeAlphaFrameInFade; }
            set { m_data.UI_fadeAlphaFrameInFade = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int UI_texPatDivX
        {
            get { return m_data.TexPatData.UI_texPatDivX; }
            set { m_data.TexPatData.UI_texPatDivX = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int UI_texPatDivY
        {
            get { return m_data.TexPatData.UI_texPatDivY; }
            set { m_data.TexPatData.UI_texPatDivY = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public uint UI_texPatNumPattern
        {
            get { return m_data.TexPatData.UI_texPatNumPattern; }
            set { m_data.TexPatData.UI_texPatNumPattern = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public uint UI_texPatFreqFrame
        {
            get { return m_data.TexPatData.UI_texPatFreqFrame; }
            set { m_data.TexPatData.UI_texPatFreqFrame = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int UI_numUseTexPatTbl
        {
            get { return m_data.TexPatData.UI_numUseTexPatTbl; }
            set { m_data.TexPatData.UI_numUseTexPatTbl = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int UI_texPatAnimMode
        {
            get { return m_data.TexPatData.UI_texPatAnimType; }
            set { m_data.TexPatData.UI_texPatAnimType = value; }
        }

        /// <summary>
        /// 0:アニメなし 1:アニメ有り
        /// </summary>
        public int UI_texPatAnim
        {
            get { return m_data.TexPatData.UI_texPatAnim; }
            set { m_data.TexPatData.UI_texPatAnim = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int UI_texWrapU
        {
            get { return (int)m_data.TexPatData.UI_texWrapU; }
            set { m_data.TexPatData.UI_texWrapU = (TextureWrapMode)value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int UI_texWrapV
        {
            get { return (int)m_data.TexPatData.UI_texWrapV; }
            set { m_data.TexPatData.UI_texWrapV = (TextureWrapMode)value; }
        }

        /// <summary>
        /// 有効なミップレベル
        /// </summary>
        public float UI_texAvailableMipLevel
        {
            get { return m_data.TexPatData.UI_availableMipLevel; }
            set { m_data.TexPatData.UI_availableMipLevel = value; }
        }

        /// <summary>
        /// ミップマップＬＯＤバイアス
        /// </summary>
        public float UI_texMipMapBias
        {
            get { return m_data.TexPatData.UI_mipMapBias; }
            set { m_data.TexPatData.UI_mipMapBias = value; }
        }

        /// <summary>
        /// テクスチャーパターンのランダム
        /// 0:ランダムなし 1:ランダム有り
        /// </summary>
        public int UI_texPatIsRandom
        {
            get { return m_data.TexPatData.UI_texPatIsRandom ? 1 : 0; }
            set { m_data.TexPatData.UI_texPatIsRandom = (value != 0); }
        }

#if true

        #region texPatTbl
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_0
        {
            get { return m_data.TexPatData.UI_texPatTbl[0]; }
            set { m_data.TexPatData.UI_texPatTbl[0] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_1
        {
            get { return m_data.TexPatData.UI_texPatTbl[1]; }
            set { m_data.TexPatData.UI_texPatTbl[1] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_2
        {
            get { return m_data.TexPatData.UI_texPatTbl[2]; }
            set { m_data.TexPatData.UI_texPatTbl[2] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_3
        {
            get { return m_data.TexPatData.UI_texPatTbl[3]; }
            set { m_data.TexPatData.UI_texPatTbl[3] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_4
        {
            get { return m_data.TexPatData.UI_texPatTbl[4]; }
            set { m_data.TexPatData.UI_texPatTbl[4] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_5
        {
            get { return m_data.TexPatData.UI_texPatTbl[5]; }
            set { m_data.TexPatData.UI_texPatTbl[5] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_6
        {
            get { return m_data.TexPatData.UI_texPatTbl[6]; }
            set { m_data.TexPatData.UI_texPatTbl[6] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_7
        {
            get { return m_data.TexPatData.UI_texPatTbl[7]; }
            set { m_data.TexPatData.UI_texPatTbl[7] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_8
        {
            get { return m_data.TexPatData.UI_texPatTbl[8]; }
            set { m_data.TexPatData.UI_texPatTbl[8] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_9
        {
            get { return m_data.TexPatData.UI_texPatTbl[9]; }
            set { m_data.TexPatData.UI_texPatTbl[9] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_10
        {
            get { return m_data.TexPatData.UI_texPatTbl[10]; }
            set { m_data.TexPatData.UI_texPatTbl[10] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_11
        {
            get { return m_data.TexPatData.UI_texPatTbl[11]; }
            set { m_data.TexPatData.UI_texPatTbl[11] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_12
        {
            get { return m_data.TexPatData.UI_texPatTbl[12]; }
            set { m_data.TexPatData.UI_texPatTbl[12] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_13
        {
            get { return m_data.TexPatData.UI_texPatTbl[13]; }
            set { m_data.TexPatData.UI_texPatTbl[13] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_14
        {
            get { return m_data.TexPatData.UI_texPatTbl[14]; }
            set { m_data.TexPatData.UI_texPatTbl[14] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_15
        {
            get { return m_data.TexPatData.UI_texPatTbl[15]; }
            set { m_data.TexPatData.UI_texPatTbl[15] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_16
        {
            get { return m_data.TexPatData.UI_texPatTbl[16]; }
            set { m_data.TexPatData.UI_texPatTbl[16] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_17
        {
            get { return m_data.TexPatData.UI_texPatTbl[17]; }
            set { m_data.TexPatData.UI_texPatTbl[17] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_18
        {
            get { return m_data.TexPatData.UI_texPatTbl[18]; }
            set { m_data.TexPatData.UI_texPatTbl[18] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_19
        {
            get { return m_data.TexPatData.UI_texPatTbl[19]; }
            set { m_data.TexPatData.UI_texPatTbl[19] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_20
        {
            get { return m_data.TexPatData.UI_texPatTbl[20]; }
            set { m_data.TexPatData.UI_texPatTbl[20] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_21
        {
            get { return m_data.TexPatData.UI_texPatTbl[21]; }
            set { m_data.TexPatData.UI_texPatTbl[21] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_22
        {
            get { return m_data.TexPatData.UI_texPatTbl[22]; }
            set { m_data.TexPatData.UI_texPatTbl[22] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_23
        {
            get { return m_data.TexPatData.UI_texPatTbl[23]; }
            set { m_data.TexPatData.UI_texPatTbl[23] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_24
        {
            get { return m_data.TexPatData.UI_texPatTbl[24]; }
            set { m_data.TexPatData.UI_texPatTbl[24] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_25
        {
            get { return m_data.TexPatData.UI_texPatTbl[25]; }
            set { m_data.TexPatData.UI_texPatTbl[25] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_26
        {
            get { return m_data.TexPatData.UI_texPatTbl[26]; }
            set { m_data.TexPatData.UI_texPatTbl[26] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_27
        {
            get { return m_data.TexPatData.UI_texPatTbl[27]; }
            set { m_data.TexPatData.UI_texPatTbl[27] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_28
        {
            get { return m_data.TexPatData.UI_texPatTbl[28]; }
            set { m_data.TexPatData.UI_texPatTbl[28] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_29
        {
            get { return m_data.TexPatData.UI_texPatTbl[29]; }
            set { m_data.TexPatData.UI_texPatTbl[29] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_30
        {
            get { return m_data.TexPatData.UI_texPatTbl[30]; }
            set { m_data.TexPatData.UI_texPatTbl[30] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_texPatTbl_31
        {
            get { return m_data.TexPatData.UI_texPatTbl[31]; }
            set { m_data.TexPatData.UI_texPatTbl[31] = value; }
        }

        #endregion

#else
        /// <summary>
        /// /*cNumTexPatTbl*/
        /// </summary>
        [XmlArrayItem("texPat")]
        public int[] UI_texPatTbl
        {
            get { return m_data.TexPatData.UI_texPatTbl; }
            set { m_data.TexPatData.UI_texPatTbl = value; }
        }
#endif

        #region SubTexture
        /// <summary>
        ///
        /// </summary>
        public int UI_subTexPatDivX
        {
            get { return m_data.SubTexPatData.UI_texPatDivX; }
            set { m_data.SubTexPatData.UI_texPatDivX = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int UI_subTexPatDivY
        {
            get { return m_data.SubTexPatData.UI_texPatDivY; }
            set { m_data.SubTexPatData.UI_texPatDivY = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public uint UI_subTexPatNumPattern
        {
            get { return m_data.SubTexPatData.UI_texPatNumPattern; }
            set { m_data.SubTexPatData.UI_texPatNumPattern = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public uint UI_subTexPatFreqFrame
        {
            get { return m_data.SubTexPatData.UI_texPatFreqFrame; }
            set { m_data.SubTexPatData.UI_texPatFreqFrame = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int UI_numUseSubTexPatTbl
        {
            get { return m_data.SubTexPatData.UI_numUseTexPatTbl; }
            set { m_data.SubTexPatData.UI_numUseTexPatTbl = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int UI_subTexPatAnimMode
        {
            get { return m_data.SubTexPatData.UI_texPatAnimType; }
            set { m_data.SubTexPatData.UI_texPatAnimType = value; }
        }

        /// <summary>
        /// 0:アニメなし 1:アニメ有り
        /// </summary>
        public int UI_subTexPatAnim
        {
            get { return m_data.SubTexPatData.UI_texPatAnim; }
            set { m_data.SubTexPatData.UI_texPatAnim = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int UI_subTexWrapU
        {
            get { return (int)m_data.SubTexPatData.UI_texWrapU; }
            set { m_data.SubTexPatData.UI_texWrapU = (TextureWrapMode)value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int UI_subTexWrapV
        {
            get { return (int)m_data.SubTexPatData.UI_texWrapV; }
            set { m_data.SubTexPatData.UI_texWrapV = (TextureWrapMode)value; }
        }

        /// <summary>
        /// 有効なミップレベル
        /// </summary>
        public float UI_subAvailableMipLevel
        {
            get { return m_data.SubTexPatData.UI_availableMipLevel; }
            set { m_data.SubTexPatData.UI_availableMipLevel = value; }
        }

        /// <summary>
        /// ミップマップＬＯＤバイアス
        /// </summary>
        public float UI_subMipMapBias
        {
            get { return m_data.SubTexPatData.UI_mipMapBias; }
            set { m_data.SubTexPatData.UI_mipMapBias = value; }
        }

        /// <summary>
        /// テクスチャーパターンのランダム
        /// 0:ランダムなし 1:ランダム有り
        /// </summary>
        public int UI_subTexPatIsRandom
        {
            get { return m_data.SubTexPatData.UI_texPatIsRandom ? 1 : 0; }
            set { m_data.SubTexPatData.UI_texPatIsRandom = (value != 0); }
        }

        #region subTexPatTbl
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_0
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[0]; }
            set { m_data.SubTexPatData.UI_texPatTbl[0] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_1
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[1]; }
            set { m_data.SubTexPatData.UI_texPatTbl[1] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_2
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[2]; }
            set { m_data.SubTexPatData.UI_texPatTbl[2] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_3
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[3]; }
            set { m_data.SubTexPatData.UI_texPatTbl[3] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_4
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[4]; }
            set { m_data.SubTexPatData.UI_texPatTbl[4] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_5
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[5]; }
            set { m_data.SubTexPatData.UI_texPatTbl[5] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_6
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[6]; }
            set { m_data.SubTexPatData.UI_texPatTbl[6] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_7
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[7]; }
            set { m_data.SubTexPatData.UI_texPatTbl[7] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_8
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[8]; }
            set { m_data.SubTexPatData.UI_texPatTbl[8] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_9
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[9]; }
            set { m_data.SubTexPatData.UI_texPatTbl[9] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_10
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[10]; }
            set { m_data.SubTexPatData.UI_texPatTbl[10] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_11
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[11]; }
            set { m_data.SubTexPatData.UI_texPatTbl[11] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_12
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[12]; }
            set { m_data.SubTexPatData.UI_texPatTbl[12] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_13
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[13]; }
            set { m_data.SubTexPatData.UI_texPatTbl[13] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_14
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[14]; }
            set { m_data.SubTexPatData.UI_texPatTbl[14] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_15
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[15]; }
            set { m_data.SubTexPatData.UI_texPatTbl[15] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_16
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[16]; }
            set { m_data.SubTexPatData.UI_texPatTbl[16] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_17
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[17]; }
            set { m_data.SubTexPatData.UI_texPatTbl[17] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_18
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[18]; }
            set { m_data.SubTexPatData.UI_texPatTbl[18] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_19
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[19]; }
            set { m_data.SubTexPatData.UI_texPatTbl[19] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_20
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[20]; }
            set { m_data.SubTexPatData.UI_texPatTbl[20] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_21
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[21]; }
            set { m_data.SubTexPatData.UI_texPatTbl[21] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_22
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[22]; }
            set { m_data.SubTexPatData.UI_texPatTbl[22] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_23
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[23]; }
            set { m_data.SubTexPatData.UI_texPatTbl[23] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_24
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[24]; }
            set { m_data.SubTexPatData.UI_texPatTbl[24] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_25
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[25]; }
            set { m_data.SubTexPatData.UI_texPatTbl[25] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_26
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[26]; }
            set { m_data.SubTexPatData.UI_texPatTbl[26] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_27
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[27]; }
            set { m_data.SubTexPatData.UI_texPatTbl[27] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_28
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[28]; }
            set { m_data.SubTexPatData.UI_texPatTbl[28] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_29
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[29]; }
            set { m_data.SubTexPatData.UI_texPatTbl[29] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_30
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[30]; }
            set { m_data.SubTexPatData.UI_texPatTbl[30] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_subTexPatTbl_31
        {
            get { return m_data.SubTexPatData.UI_texPatTbl[31]; }
            set { m_data.SubTexPatData.UI_texPatTbl[31] = value; }
        }

        #endregion
        #endregion

        /// <summary>
        /// ユーザーデータ
        /// </summary>
        public int emitterUserData
        {
            get { return (int)m_data.UserData.Data; }
            set { m_data.UserData.Data = (uint)value; }
        }

        /// <summary>
        /// ユーザーデータ
        /// </summary>
        public int emitterUserData2
        {
            get { return (int)m_data.UserData.Data2; }
            set { m_data.UserData.Data2 = (uint)value; }
        }

        /// <summary>
        /// ユーザーデータ (float)
        /// </summary>
        public float emitterUserDataF0
        {
            get { return m_data.UserData.UserDataF0; }
            set { m_data.UserData.UserDataF0 = value; }
        }

        /// <summary>
        /// ユーザーデータ (float)
        /// </summary>
        public float emitterUserDataF1
        {
            get { return m_data.UserData.UserDataF1; }
            set { m_data.UserData.UserDataF1 = value; }
        }

        /// <summary>
        /// ユーザーデータ (float)
        /// </summary>
        public float emitterUserDataF2
        {
            get { return m_data.UserData.UserDataF2; }
            set { m_data.UserData.UserDataF2 = value; }
        }

        /// <summary>
        /// ユーザーデータ (float)
        /// </summary>
        public float emitterUserDataF3
        {
            get { return m_data.UserData.UserDataF3; }
            set { m_data.UserData.UserDataF3 = value; }
        }

        /// <summary>
        /// ユーザーデータ (float)
        /// </summary>
        public float emitterUserDataF4
        {
            get { return m_data.UserData.UserDataF4; }
            set { m_data.UserData.UserDataF4 = value; }
        }

        /// <summary>
        /// ユーザーデータ (float)
        /// </summary>
        public float emitterUserDataF5
        {
            get { return m_data.UserData.UserDataF5; }
            set { m_data.UserData.UserDataF5 = value; }
        }

        /// <summary>
        /// ユーザーデータ (float)
        /// </summary>
        public float emitterUserDataF6
        {
            get { return m_data.UserData.UserDataF6; }
            set { m_data.UserData.UserDataF6 = value; }
        }

        /// <summary>
        /// ユーザーデータ (float)
        /// </summary>
        public float emitterUserDataF7
        {
            get { return m_data.UserData.UserDataF7; }
            set { m_data.UserData.UserDataF7 = value; }
        }

        /// <summary>
        /// ユーザーコールバックID
        /// </summary>
        public int emitterUserCallbackId
        {
            get { return m_data.UserData.CallbackId; }
            set { m_data.UserData.CallbackId = value; }
        }

        /// <summary>
        /// ソートオーダー
        /// </summary>
        public int order
        {
            get { return (int)m_data.ParticleSortType; }
            set { m_data.ParticleSortType = (NWCore.DataModel.Major_1.Minor_3.Build_0.Revision_0.ParticleSortType)value; }
        }

        /// <summary>
        /// グローバル乗算カラー０
        /// </summary>
        public RgbaColorXml emitterColor0
        {
            get { return UtilityXml.ToRgbaColorXml( m_data.EmitterColor0 ); }
            set { m_data.EmitterColor0 = UtilityXml.ToRgbaColor( value ); }
        }

        /// <summary>
        /// グローバル乗算カラー１
        /// </summary>
        public RgbaColorXml emitterColor1
        {
            get { return UtilityXml.ToRgbaColorXml( m_data.EmitterColor1 ); }
            set { m_data.EmitterColor1 = UtilityXml.ToRgbaColor( value ); }
        }

        /// <summary>
        /// グローバル乗算アルファ
        /// </summary>
        public float emitterAlpha
        {
            get { return m_data.EmitterAlpha; }
            set { m_data.EmitterAlpha = value; }
        }


        /// <summary>
        /// エミッタ形状スケール
        /// </summary>
        public Vector3Xml emitterFormScale
        {
            get { return UtilityXml.ToVector3Xml( m_data.EmitterFormScale ); }
            set { m_data.EmitterFormScale.Set( UtilityXml.ToVector3(value) ); }
        }

        /// <summary>
        /// コンバイナ：サブテクスチャカラーの上段との合成。
        /// </summary>
        public CombinerBlendTypes combinerSubTextureColorBlend
        {
            get { return m_data.UI_combinerSubTextureColorBlend; }
            set { m_data.UI_combinerSubTextureColorBlend = value; }
        }

        /// <summary>
        /// コンバイナ：プリミティブカラーの上段との合成。
        /// </summary>
        public CombinerBlendTypes combinerPrimitiveColorBlend
        {
            get { return m_data.UI_combinerPrimitiveColorBlend; }
            set { m_data.UI_combinerPrimitiveColorBlend = value; }
        }

        /// <summary>
        /// コンバイナ：サブテクスチャアルファの上段との合成。
        /// </summary>
        public CombinerBlendTypes combinerSubTextureAlphaBlend
        {
            get { return m_data.UI_combinerSubTextureAlphaBlend; }
            set { m_data.UI_combinerSubTextureAlphaBlend = value; }
        }

        /// <summary>
        /// コンバイナ：プリミティブアルファの上段との合成。
        /// </summary>
        public CombinerBlendTypes combinerPrimitiveAlphaBlend
        {
            get { return m_data.UI_combinerPrimitiveAlphaBlend; }
            set { m_data.UI_combinerPrimitiveAlphaBlend = value; }
        }

        /// <summary>
        /// コンバイナ：テクスチャカラーをカラー値か1.0で置き換える。
        /// </summary>
        public CombinerColorUsageTypes combinerTextureColorUsage
        {
            get { return m_data.UI_combinerTextureColorUsage; }
            set { m_data.UI_combinerTextureColorUsage = value; }
        }

        /// <summary>
        /// コンバイナ：サブテクスチャカラーをカラー値か1.0で置き換える。
        /// </summary>
        public CombinerColorUsageTypes combinerSubTextureColorUsage
        {
            get { return m_data.UI_combinerSubTextureColorUsage; }
            set { m_data.UI_combinerSubTextureColorUsage = value; }
        }

        /// <summary>
        /// コンバイナ：プリミティブカラーをカラー値か1.0で置き換える。
        /// </summary>
        public CombinerColorUsageTypes combinerPrimitiveColorUsage
        {
            get { return m_data.UI_combinerPrimitiveColorUsage; }
            set { m_data.UI_combinerPrimitiveColorUsage = value; }
        }

        /// <summary>
        /// コンバイナ：テクスチャアルファをアルファ値か1.0で置き換える。
        /// </summary>
        public CombinerColorUsageTypes combinerTextureAlphaUsage
        {
            get { return m_data.UI_combinerTextureAlphaUsage; }
            set { m_data.UI_combinerTextureAlphaUsage = value; }
        }

        /// <summary>
        /// コンバイナ：サブテクスチャアルファをアルファ値か1.0で置き換える。
        /// </summary>
        public CombinerColorUsageTypes combinerSubTextureAlphaUsage
        {
            get { return m_data.UI_combinerSubTextureAlphaUsage; }
            set { m_data.UI_combinerSubTextureAlphaUsage = value; }
        }

        /// <summary>
        /// コンバイナ：プリミティブアルファをアルファ値か1.0で置き換える。
        /// </summary>
        public CombinerColorUsageTypes combinerPrimitiveAlphaUsage
        {
            get { return m_data.UI_combinerPrimitiveAlphaUsage; }
            set { m_data.UI_combinerPrimitiveAlphaUsage = value; }
        }

        #endregion

        #region Properties - Child

        /// <summary>
        /// ブレンドのタイプ
        /// </summary>
        public BlendType UI_childBlendType
        {
            get { return m_data.ChildAssistData.UI_childBlendType; }
            set { m_data.ChildAssistData.UI_childBlendType = value; }
        }

        /// <summary>
        /// Child mesh type
        /// </summary>
        public MeshType childMeshType
        {
            get { return m_data.ChildData.MeshType; }
            set { m_data.ChildData.MeshType = value; }
        }

        /// <summary>
        /// ビルボードタイプ
        /// </summary>
        public BillboardType childBillboardType
        {
            get { return m_data.ChildData.BillboardType; }
            set { m_data.ChildData.BillboardType = value; }
        }

        /// <summary>
        /// プリミティブファイルパス
        /// </summary>
        public string childBillboardPrimitiveFileSource
        {
            get { return m_data.ChildData.BillboardPrimitiveFileSource; }
            set { m_data.ChildData.BillboardPrimitiveFileSource = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public string UI_childColorType
        {
            get
            {
                return Constances.GetColorTypeName(
                    m_data.ChildAssistData.UI_childColorType);
            }
            set
            {
                m_data.ChildAssistData.UI_childColorType =
                    Constances.SetColorTypeName(value);
            }
        }

        /// <summary>
        /// カラースケール継承
        /// </summary>
        public bool isChildColorScaleInherited
        {
            get { return m_data.ChildAssistData.UI_isChildColorScaleInherited; }
            set { m_data.ChildAssistData.UI_isChildColorScaleInherited = value; }
        }

        /// <summary>
        /// カラースケール
        /// </summary>
        public float childColorScale
        {
            get { return m_data.ChildAssistData.UI_childColorScale; }
            set { m_data.ChildAssistData.UI_childColorScale = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public AlphaDst UI_childAlphaDstType
        {
            get { return m_data.ChildAssistData.UI_childAlphaDstType; }
            set { m_data.ChildAssistData.UI_childAlphaDstType = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public PtclRotType UI_childRotType
        {
            get { return m_data.ChildAssistData.UI_childRotType; }
            set { m_data.ChildAssistData.UI_childRotType = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public string UI_childColorInheritType
        {
            get
            {
                return Constances.GetColorInheritTypeNames(
                    m_data.ChildAssistData.UI_childColorInheritType);
            }
            set
            {
                m_data.ChildAssistData.UI_childColorInheritType =
                    Constances.SetColorInheritTypeNames(value);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public TextureAddressing UI_childTextureAddressing
        {
            get { return m_data.ChildTexPatData.UI_textureAddressing; }
            set { m_data.ChildTexPatData.UI_textureAddressing = value; }
        }

        /// <summary>
        /// Ｚバッファー、αテストタイプ
        /// </summary>
        public ZBufATestType childZBufATestType
        {
            get { return m_data.ChildData.ZBufATestType; }
            set { m_data.ChildData.ZBufATestType = value; }
        }

        /// <summary>
        /// カリング
        /// </summary>
        public DisplaySideTypes childDisplaySide
        {
            get { return m_data.ChildData.DisplaySide; }
            set { m_data.ChildData.DisplaySide = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public FilterMode UI_childTexFilterMode
        {
            get { return m_data.ChildTexPatData.UI_texFilterMode; }
            set { m_data.ChildTexPatData.UI_texFilterMode = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public string childTexture
        {
            //TODO:まだ未完成です。
            get
            {
                if (String.IsNullOrEmpty(m_data.ChildTexPatData.UI_texPatFileName) == false)
                {
                    if (m_bNormalizePath == true)
                        return System.IO.Path.GetFileName(m_data.ChildTexPatData.UI_texPatFileName);
                    else
                        return m_data.ChildTexPatData.UI_texPatFileName;
                }
                else
                {
                    return String.Empty;
                }
            }
            set { m_data.ChildTexPatData.UI_texPatFileName = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public uint childFlg
        {
            get { return m_data.ChildFlag.Flag; }
            set { m_data.ChildFlag.Flag = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int UI_childLife
        {
            get { return m_data.ChildAssistData.UI_childLife; }
            set { m_data.ChildAssistData.UI_childLife = value; }
        }

        /// <summary>
        /// 運動量ランダム
        /// </summary>
        public float childDynamicsRandom
        {
            get { return m_data.ChildData.DynamicsRandom; }
            set { m_data.ChildData.DynamicsRandom = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public Vector2Xml UI_childRotBasis
        {
            get { return UtilityXml.ToVector2Xml(m_data.ChildAssistData.UI_childRotBasis); }
            set { m_data.ChildAssistData.UI_childRotBasis = UtilityXml.ToVector2(value); }
        }

        /// <summary>
        ///
        /// </summary>
        public int childEmitRate
        {
            get { return m_data.ChildData.EmitRate; }
            set { m_data.ChildData.EmitRate = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int childEmitTiming
        {
            get { return m_data.ChildData.EmitTiming; }
            set { m_data.ChildData.EmitTiming = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int childEmitStep
        {
            get { return m_data.ChildData.EmitStep; }
            set { m_data.ChildData.EmitStep = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public float childInitPosRand
        {
            get { return m_data.ChildData.InitPosRand; }
            set { m_data.ChildData.InitPosRand = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public float childVelInheritRate
        {
            get { return m_data.ChildData.VelInheritRate; }
            set { m_data.ChildData.VelInheritRate = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public float childFigureVel
        {
            get { return m_data.ChildData.FigureVelocity; }
            set { m_data.ChildData.FigureVelocity = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public Vector3Xml childRandVel
        {
            get { return UtilityXml.ToVector3Xml(m_data.ChildData.RandVel); }
            set { m_data.ChildData.RandVel = UtilityXml.ToVector3(value); }
        }

        /// <summary>
        ///
        /// </summary>
        public float childAirRegist
        {
            get { return m_data.ChildData.AirRegist; }
            set { m_data.ChildData.AirRegist = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int UI_childTexPatDivX
        {
            get { return m_data.ChildTexPatData.UI_texPatDivX; }
            set { m_data.ChildTexPatData.UI_texPatDivX = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int UI_childTexPatDivY
        {
            get { return m_data.ChildTexPatData.UI_texPatDivY; }
            set { m_data.ChildTexPatData.UI_texPatDivY = value; }
        }

        /// <summary>
        /// Number of the patterns used on the texture pattern table
        /// </summary>
        public uint UI_childTexPatNumPattern
        {
            get { return m_data.ChildTexPatData.UI_texPatNumPattern; }
            set { m_data.ChildTexPatData.UI_texPatNumPattern = value; }
        }

        /// <summary>
        /// The number of frames a pattern takes
        /// </summary>
        public uint UI_childTexPatFreqFrame
        {
            get { return m_data.ChildTexPatData.UI_texPatFreqFrame; }
            set { m_data.ChildTexPatData.UI_texPatFreqFrame = value; }
        }

        /// <summary>
        /// Number of the patterns used on the texture pattern table
        /// </summary>
        public int UI_childNumUseTexPatTbl
        {
            get { return m_data.ChildTexPatData.UI_numUseTexPatTbl; }
            set { m_data.ChildTexPatData.UI_numUseTexPatTbl = value; }
        }

        /// <summary>
        /// Texture animation mode.
        /// </summary>
        public int UI_childTexPatAnimMode
        {
            get { return m_data.ChildTexPatData.UI_texPatAnimType; }
            set { m_data.ChildTexPatData.UI_texPatAnimType = value; }
        }

        /// <summary>
        /// 0:アニメなし 1:アニメ有り
        /// </summary>
        public int UI_childTexPatAnim
        {
            get { return m_data.ChildTexPatData.UI_texPatAnim; }
            set { m_data.ChildTexPatData.UI_texPatAnim = value; }
        }

        /// <summary>
        /// Texture pattern table
        /// </summary>
        public int UI_childTexPatTbl_0
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[0]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[0] = value; }
        }
        /// <summary>
        /// Texture pattern table
        /// </summary>
        public int UI_childTexPatTbl_1
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[1]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[1] = value; }
        }
        /// <summary>
        /// Texture pattern table
        /// </summary>
        public int UI_childTexPatTbl_2
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[2]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[2] = value; }
        }
        /// <summary>
        /// Texture pattern table
        /// </summary>
        public int UI_childTexPatTbl_3
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[3]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[3] = value; }
        }
        /// <summary>
        /// Texture pattern table
        /// </summary>
        public int UI_childTexPatTbl_4
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[4]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[4] = value; }
        }
        /// <summary>
        /// Texture pattern table
        /// </summary>
        public int UI_childTexPatTbl_5
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[5]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[5] = value; }
        }
        /// <summary>
        /// Texture pattern table
        /// </summary>
        public int UI_childTexPatTbl_6
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[6]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[6] = value; }
        }
        /// <summary>
        /// Texture pattern table
        /// </summary>
        public int UI_childTexPatTbl_7
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[7]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[7] = value; }
        }
        /// <summary>
        /// Texture pattern table
        /// </summary>
        public int UI_childTexPatTbl_8
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[8]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[8] = value; }
        }
        /// <summary>
        /// Texture pattern table
        /// </summary>
        public int UI_childTexPatTbl_9
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[9]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[9] = value; }
        }
        /// <summary>
        /// Texture pattern table
        /// </summary>
        public int UI_childTexPatTbl_10
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[10]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[10] = value; }
        }
        /// <summary>
        /// Texture pattern table
        /// </summary>
        public int UI_childTexPatTbl_11
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[11]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[11] = value; }
        }
        /// <summary>
        /// Texture pattern table
        /// </summary>
        public int UI_childTexPatTbl_12
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[12]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[12] = value; }
        }
        /// <summary>
        /// Texture pattern table
        /// </summary>
        public int UI_childTexPatTbl_13
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[13]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[13] = value; }
        }
        /// <summary>
        /// Texture pattern table
        /// </summary>
        public int UI_childTexPatTbl_14
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[14]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[14] = value; }
        }
        /// <summary>
        /// NumTexPatTbl
        /// </summary>
        public int UI_childTexPatTbl_15
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[15]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[15] = value; }
        }
        /// <summary>
        /// Texture pattern table
        /// </summary>
        public int UI_childTexPatTbl_16
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[16]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[16] = value; }
        }
        /// <summary>
        /// Texture pattern table
        /// </summary>
        public int UI_childTexPatTbl_17
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[17]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[17] = value; }
        }
        /// <summary>
        /// Texture pattern table
        /// </summary>
        public int UI_childTexPatTbl_18
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[18]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[18] = value; }
        }
        /// <summary>
        /// Texture pattern table
        /// </summary>
        public int UI_childTexPatTbl_19
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[19]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[19] = value; }
        }
        /// <summary>
        /// Texture pattern table
        /// </summary>
        public int UI_childTexPatTbl_20
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[20]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[20] = value; }
        }
        /// <summary>
        /// Texture pattern table
        /// </summary>
        public int UI_childTexPatTbl_21
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[21]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[21] = value; }
        }
        /// <summary>
        /// Texture pattern table
        /// </summary>
        public int UI_childTexPatTbl_22
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[22]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[22] = value; }
        }
        /// <summary>
        /// Texture pattern table
        /// </summary>
        public int UI_childTexPatTbl_23
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[23]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[23] = value; }
        }
        /// <summary>
        /// Texture pattern table
        /// </summary>
        public int UI_childTexPatTbl_24
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[24]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[24] = value; }
        }
        /// <summary>
        /// Texture pattern table
        /// </summary>
        public int UI_childTexPatTbl_25
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[25]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[25] = value; }
        }
        /// <summary>
        /// Texture pattern table
        /// </summary>
        public int UI_childTexPatTbl_26
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[26]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[26] = value; }
        }
        /// <summary>
        /// Texture pattern table
        /// </summary>
        public int UI_childTexPatTbl_27
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[27]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[27] = value; }
        }
        /// <summary>
        /// Texture pattern table
        /// </summary>
        public int UI_childTexPatTbl_28
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[28]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[28] = value; }
        }
        /// <summary>
        /// Texture pattern table
        /// </summary>
        public int UI_childTexPatTbl_29
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[29]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[29] = value; }
        }
        /// <summary>
        /// Texture pattern table
        /// </summary>
        public int UI_childTexPatTbl_30
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[30]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[30] = value; }
        }
        /// <summary>
        /// Texture pattern table
        /// </summary>
        public int UI_childTexPatTbl_31
        {
            get { return m_data.ChildTexPatData.UI_texPatTbl[31]; }
            set { m_data.ChildTexPatData.UI_texPatTbl[31] = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int UI_childTexWrapU
        {
            get { return (int)m_data.ChildTexPatData.UI_texWrapU; }
            set { m_data.ChildTexPatData.UI_texWrapU = (TextureWrapMode)value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int UI_childTexWrapV
        {
            get { return (int)m_data.ChildTexPatData.UI_texWrapV; }
            set { m_data.ChildTexPatData.UI_texWrapV = (TextureWrapMode)value; }
        }

        /// <summary>
        /// 有効なミップレベル
        /// </summary>
        public float UI_childTexAvailableMipLevel
        {
            get { return m_data.ChildTexPatData.UI_availableMipLevel; }
            set { m_data.ChildTexPatData.UI_availableMipLevel = value; }
        }

        /// <summary>
        /// ミップマップＬＯＤバイアス
        /// </summary>
        public float UI_childTexMipMapBias
        {
            get { return m_data.ChildTexPatData.UI_mipMapBias; }
            set { m_data.ChildTexPatData.UI_mipMapBias = value; }
        }

        /// <summary>
        /// テクスチャーパターンのランダム
        /// 0:ランダムなし 1:ランダム有り
        /// </summary>
        public int UI_childTexPatIsRandom
        {
            get { return m_data.ChildTexPatData.UI_texPatIsRandom ? 1 : 0; }
            set { m_data.ChildTexPatData.UI_texPatIsRandom = (value != 0); }
        }

        /// <summary>
        ///
        /// </summary>
        public RgbaColorXml UI_childConstColor0
        {
            get { return UtilityXml.ToRgbaColorXml(m_data.ChildAssistData.UI_childConstColor0); }
            set { m_data.ChildAssistData.UI_childConstColor0 = UtilityXml.ToRgbaColor(value); }
        }

        /// <summary>
        ///
        /// </summary>
        public RgbaColorXml UI_childConstColor1
        {
            get { return UtilityXml.ToRgbaColorXml(m_data.ChildAssistData.UI_childConstColor1); }
            set { m_data.ChildAssistData.UI_childConstColor1 = UtilityXml.ToRgbaColor(value); }
        }

        /// <summary>
        ///
        /// </summary>
        public RgbaColorXml UI_childColor
        {
            get { return UtilityXml.ToRgbaColorXml(m_data.ChildAssistData.UI_childColor); }
            set { m_data.ChildAssistData.UI_childColor = UtilityXml.ToRgbaColor(value); }
        }

        /// <summary>
        ///
        /// </summary>
        public RgbaColorXml UI_childInheritanceColor
        {
            get { return UtilityXml.ToRgbaColorXml(m_data.ChildAssistData.UI_childInheritanceColor); }
            set { m_data.ChildAssistData.UI_childInheritanceColor = UtilityXml.ToRgbaColor(value); }
        }

        /// <summary>
        ///
        /// </summary>
        public float childAlpha
        {
            get { return m_data.ChildData.Alpha; }
            set { m_data.ChildData.Alpha = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public float childAlphaTarget
        {
            get { return m_data.ChildData.AlphaTarget; }
            set { m_data.ChildData.AlphaTarget = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public float childAlphaInit
        {
            get { return m_data.ChildData.AlphaInit; }
            set { m_data.ChildData.AlphaInit = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int UI_childAlphaSec
        {
            get { return m_data.ChildAssistData.UI_childAlphaSec; }
            set { m_data.ChildAssistData.UI_childAlphaSec = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int UI_childAlphaBaseSec
        {
            get { return m_data.ChildAssistData.UI_childAlphaBaseSec; }
            set { m_data.ChildAssistData.UI_childAlphaBaseSec = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public float childScaleInheritRate
        {
            get { return m_data.ChildData.ScaleInheritRate; }
            set { m_data.ChildData.ScaleInheritRate = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public Vector2Xml childScale
        {
            get { return UtilityXml.ToVector2Xml(m_data.ChildData.Scale); }
            set { m_data.ChildData.Scale = UtilityXml.ToVector2(value); }
        }

        /// <summary>
        ///
        /// </summary>
        public float childScaleRand
        {
            get { return m_data.ChildData.RandomScale; }
            set { m_data.ChildData.RandomScale = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int UI_childScaleSec
        {
            get { return m_data.ChildAssistData.UI_childScaleSec; }
            set { m_data.ChildAssistData.UI_childScaleSec = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public Vector2i UI_childScaleTarget
        {
            get { return m_data.ChildAssistData.UI_childScaleTarget; }
            set { m_data.ChildAssistData.UI_childScaleTarget = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public Vector3i UI_childInitRot
        {
            get { return m_data.ChildAssistData.UI_childInitRot; }
            set { m_data.ChildAssistData.UI_childInitRot = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public Vector3i UI_childInitRotRand
        {
            get { return m_data.ChildAssistData.UI_childInitRotRand; }
            set { m_data.ChildAssistData.UI_childInitRotRand = value; }
        }

        /// <summary>
        /// 減衰率
        /// </summary>
        public float childRotResist
        {
            get { return m_data.ChildAssistData.UI_childRotResist; }
            set { m_data.ChildAssistData.UI_childRotResist = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public Vector3i UI_childRotVel
        {
            get { return m_data.ChildAssistData.UI_childRotVel; }
            set { m_data.ChildAssistData.UI_childRotVel = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public Vector3i UI_childRotVelRand
        {
            get { return m_data.ChildAssistData.UI_childRotVelRand; }
            set { m_data.ChildAssistData.UI_childRotVelRand = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public float UI_childGravity
        {
            get { return m_data.ChildAssistData.UI_childGravity; }
            set { m_data.ChildAssistData.UI_childGravity = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public Vector3Xml UI_childGravityDir
        {
            get { return UtilityXml.ToVector3Xml(m_data.ChildAssistData.UI_childGravityDir); }
            set { m_data.ChildAssistData.UI_childGravityDir = UtilityXml.ToVector3(value); }
        }

        /// <summary>
        /// Child soft particle flag.
        /// </summary>
        public int childUseSoftParticle
        {
            get { return (m_data.ChildData.UseSoftParticle == true ? 1 : 0); }
            set { m_data.ChildData.UseSoftParticle = (value == 1 ? true : false); }
        }

        /// <summary>
        /// Child soft particle fade distance.
        /// </summary>
        public float childSoftParticleFadeDistance
        {
            get { return m_data.ChildData.SoftParticleFadeDistance; }
            set { m_data.ChildData.SoftParticleFadeDistance = value; }
        }

        /// <summary>
        /// Child soft particle volume parameter.
        /// </summary>
        public float childSoftParticleVolumeParam
        {
            get { return m_data.ChildData.SoftParticleVolumeParam; }
            set { m_data.ChildData.SoftParticleVolumeParam = value; }
        }

        /// <summary>
        /// Child particle shader type.
        /// </summary>
        public ShaderType childShaderType
        {
            get { return m_data.ChildData.ShaderType; }
            set { m_data.ChildData.ShaderType = value; }
        }

        /// <summary>
        /// Child particle shader setting;
        /// </summary>
        public uint childUserShaderSetting
        {
            get { return m_data.ChildData.ShaderSetting; }
            set { m_data.ChildData.ShaderSetting = value; }
        }

        /// <summary>
        /// Shader parameter 0 for child particle.
        /// </summary>
        public float childShaderParam0
        {
            get { return m_data.ChildData.ShaderParam0; }
            set { m_data.ChildData.ShaderParam0 = value; }
        }

        /// <summary>
        /// Shader parameter 1 for child particle.
        /// </summary>
        public float childShaderParam1
        {
            get { return m_data.ChildData.ShaderParam1; }
            set { m_data.ChildData.ShaderParam1 = value; }
        }

        /// <summary>
        /// Child user shader flags;
        /// </summary>
        public uint childUserShaderFlags
        {
            get { return m_data.ChildData.UserShaderFlags; }
            set { m_data.ChildData.UserShaderFlags = value; }
        }

        /// <summary>
        /// Child user shader flags;
        /// </summary>
        public uint childUserShaderSwitch
        {
            get { return m_data.ChildData.UserShaderSwitch; }
            set { m_data.ChildData.UserShaderSwitch = value; }
        }

        /// <summary>
        /// Child user shader params;
        /// </summary>
        public float[] childUserShaderParams
        {
            get
            {
                float[] rvalue = new float[m_data.ChildData.UserShaderParams.Length];
                for (int i = 0; i < m_data.ChildData.UserShaderParams.Length; ++i)
                {
                    rvalue[i] = m_data.ChildData.UserShaderParams[i];
                }
                return rvalue;
            }
            set
            {
                for (int i = 0; i < m_data.ChildData.UserShaderParams.Length; ++i)
                {
                    m_data.ChildData.UserShaderParams[i] = value[i];
                }
            }
        }

        /// <summary>
        /// コンバイナ：プリミティブカラーの上段との合成。
        /// </summary>
        public CombinerBlendTypes childCombinerPrimitiveColorBlend
        {
            get { return m_data.ChildData.UI_combinerPrimitiveColorBlend; }
            set { m_data.ChildData.UI_combinerPrimitiveColorBlend = value; }
        }

        /// <summary>
        /// コンバイナ：プリミティブアルファの上段との合成。
        /// </summary>
        public CombinerBlendTypes childCombinerPrimitiveAlphaBlend
        {
            get { return m_data.ChildData.UI_combinerPrimitiveAlphaBlend; }
            set { m_data.ChildData.UI_combinerPrimitiveAlphaBlend = value; }
        }

        /// <summary>
        /// コンバイナ：テクスチャカラーをカラー値か1.0で置き換える。
        /// </summary>
        public CombinerColorUsageTypes childCombinerTextureColorUsage
        {
            get { return m_data.ChildData.UI_combinerTextureColorUsage; }
            set { m_data.ChildData.UI_combinerTextureColorUsage = value; }
        }

        /// <summary>
        /// コンバイナ：プリミティブカラーをカラー値か1.0で置き換える。
        /// </summary>
        public CombinerColorUsageTypes childCombinerPrimitiveColorUsage
        {
            get { return m_data.ChildData.UI_combinerPrimitiveColorUsage; }
            set { m_data.ChildData.UI_combinerPrimitiveColorUsage = value; }
        }

        /// <summary>
        /// コンバイナ：テクスチャアルファをアルファ値か1.0で置き換える。
        /// </summary>
        public CombinerColorUsageTypes childCombinerTextureAlphaUsage
        {
            get { return m_data.ChildData.UI_combinerTextureAlphaUsage; }
            set { m_data.ChildData.UI_combinerTextureAlphaUsage = value; }
        }

        /// <summary>
        /// コンバイナ：プリミティブアルファをアルファ値か1.0で置き換える。
        /// </summary>
        public CombinerColorUsageTypes childCombinerPrimitiveAlphaUsage
        {
            get { return m_data.ChildData.UI_combinerPrimitiveAlphaUsage; }
            set { m_data.ChildData.UI_combinerPrimitiveAlphaUsage = value; }
        }

        #endregion

        #region Properties - Field

        /// <summary>
        ///
        /// </summary>
        public uint fieldFlg
        {
            get { return m_data.FieldFlag.Flag; }
            set { m_data.FieldFlag.Flag = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int fieldRandomBlank
        {
            get { return m_data.FieldRandomData.Blank; }
            set { m_data.FieldRandomData.Blank = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public Vector3Xml fieldRandomVelAdd
        {
            get { return UtilityXml.ToVector3Xml(m_data.FieldRandomData.VelocityAddition); }
            set { m_data.FieldRandomData.VelocityAddition = UtilityXml.ToVector3(value); }
        }


        /// <summary>
        ///
        /// </summary>
        public float fieldMagnetPower
        {
            get { return m_data.FieldMagnetData.FieldMagnetPower; }
            set { m_data.FieldMagnetData.FieldMagnetPower = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public Vector3Xml fieldMagnetPos
        {
            get { return UtilityXml.ToVector3Xml(m_data.FieldMagnetData.FieldMagnetPos); }
            set { m_data.FieldMagnetData.FieldMagnetPos = UtilityXml.ToVector3(value); }
        }

        /// <summary>
        ///
        /// </summary>
        public uint fieldMagnetFlg
        {
            get { return m_data.FieldMagnetData.Flag.Flag; }
            set { m_data.FieldMagnetData.Flag.Flag = value; }
        }

        /// <summary>
        /// スピンー回転速度
        /// </summary>
        public int fieldSpinRotate
        {
            get { return m_data.FieldSpinData.FieldSpinRotation; }
            set { m_data.FieldSpinData.FieldSpinRotation = value; }
        }

        /// <summary>
        /// スピンー軸
        /// </summary>
        public int fieldSpinAxis
        {
            get { return m_data.FieldSpinData.FieldSpinAxis; }
            set { m_data.FieldSpinData.FieldSpinAxis = value; }
        }

        /// <summary>
        /// スピンー拡散速度
        /// </summary>
        public float fieldSpinOuter
        {
            get { return m_data.FieldSpinData.FieldSpinOuter; }
            set { m_data.FieldSpinData.FieldSpinOuter = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int fieldCollisionType
        {
            get { return m_data.FieldCollisionData.Type; }
            set { m_data.FieldCollisionData.Type = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int fieldCollisionIsWorld
        {
            get { return m_data.FieldCollisionData.IsWorld ? 1 : 0; }
            set { m_data.FieldCollisionData.IsWorld = (value != 0); }
        }

        /// <summary>
        ///
        /// </summary>
        public float fieldCollisionCoord
        {
            get { return m_data.FieldCollisionData.Coordinate; }
            set { m_data.FieldCollisionData.Coordinate = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public float fieldCollisionCoef
        {
            get { return m_data.FieldCollisionData.Coefficient; }
            set { m_data.FieldCollisionData.Coefficient = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public Vector3Xml fieldConvergencePos
        {
            get { return UtilityXml.ToVector3Xml(m_data.FieldConvergenceData.Position); }
            set { m_data.FieldConvergenceData.Position = UtilityXml.ToVector3(value); }
        }

        /// <summary>
        ///
        /// </summary>
        public float fieldConvergenceRatio
        {
            get { return m_data.FieldConvergenceData.Ratio; }
            set { m_data.FieldConvergenceData.Ratio = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public Vector3Xml fieldPosAdd
        {
            get { return UtilityXml.ToVector3Xml(m_data.FieldPosAddData.Addition); }
            set { m_data.FieldPosAddData.Addition = UtilityXml.ToVector3(value); }
        }

        #endregion

        #region Properties - Fluctuation

        /// <summary>
        /// FluctuationFlag を使用。ビットによって機能が違います。
        /// </summary>
        public uint fluctuationFlg
        {
            get { return m_data.FluctuationFlag.Flag; }
            set { m_data.FluctuationFlag.Flag = value; }
        }

        /// <summary>
        /// 振幅スケール
        /// </summary>
        public float fluctuationScale
        {
            get { return m_data.FluctuationData.Scale; }
            set { m_data.FluctuationData.Scale = value; }
        }

        /// <summary>
        /// 位相ランダム(0 / 1)
        /// </summary>
        public uint fluctuationPhaseRnd
        {
            get { return m_data.FluctuationData.PhaseRange; }
            set { m_data.FluctuationData.PhaseRange = value; }
        }

        /// <summary>
        /// 周期
        /// </summary>
        public int UI_fluctuationFreq
        {
            get { return m_data.FluctuationData.Frequency; }
            set { m_data.FluctuationData.Frequency = value; }
        }

        #endregion

        #region Properties - Stripe

        /// <summary>
        ///
        /// </summary>
        public int stripeFlg
        {
            get { return m_data.StripeFlag; }
            set { m_data.StripeFlag = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int stripeHLType
        {
            get { return m_data.StripeData.HLType; }
            set { m_data.StripeData.HLType = value; }
        }
        /// <summary>
        ///
        /// </summary>
        public int stripeSubDivs
        {
            get { return m_data.StripeData.SubDivs; }
            set { m_data.StripeData.SubDivs = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public StripeComputeType stripeComputeType
        {
            get { return m_data.StripeData.Type; }
            set { m_data.StripeData.Type = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public float stripeStartAlpha
        {
            get { return m_data.StripeData.StartAlpha; }
            set { m_data.StripeData.StartAlpha = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public float stripeEndAlpha
        {
            get { return m_data.StripeData.EndAlpha; }
            set { m_data.StripeData.EndAlpha = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public Vector2Xml stripeUVScroll
        {
            get { return UtilityXml.ToVector2Xml(m_data.StripeData.UVScroll); }
            set { m_data.StripeData.UVScroll = UtilityXml.ToVector2(value); }
        }

        /// <summary>
        ///
        /// </summary>
        public int stripeNumHistory
        {
            get { return m_data.StripeData.NumHistory; }
            set { m_data.StripeData.NumHistory = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int stripeNumDivisions
        {
            get { return m_data.StripeData.NumDivisions; }
            set { m_data.StripeData.NumDivisions = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int UI_stripeHistoryStep
        {
            get { return m_data.StripeData.HistoryStep; }
            set { m_data.StripeData.HistoryStep = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public float stripeDirInterpolate
        {
            get { return m_data.StripeData.DirInterpolation; }
            set { m_data.StripeData.DirInterpolation = value; }
        }

        /// <summary>
        /// 接続タイプ
        /// </summary>
        public StripeConnectTypes stripeConnectType
        {
            get { return m_data.StripeData.ConnectType; }
            set { m_data.StripeData.ConnectType = value; }
        }

        /// <summary>
        /// ストライプテクスチャ座標
        /// </summary>
        public StripeTexCoordTypes TexCoordType
        {
            get { return m_data.StripeData.TexCoordType; }
            set { m_data.StripeData.TexCoordType = value; }
        }

        #endregion

        #region Shader

        /// <summary>
        /// Soft particle flag.
        /// </summary>
        public int useSoftParticle
        {
            get { return (m_data.UseSoftParticle == true ? 1 : 0); }
            set { m_data.UseSoftParticle = (value == 1 ? true : false); }
        }

        /// <summary>
        /// Soft particle fade distance.
        /// </summary>
        public float softParticleFadeDistance
        {
            get { return m_data.SoftParticleFadeDistance; }
            set { m_data.SoftParticleFadeDistance = value; }
        }

        /// <summary>
        /// Soft particle volume parameter.
        /// </summary>
        public float softParticleVolumeParam
        {
            get { return m_data.SoftParticleVolumeParam; }
            set { m_data.SoftParticleVolumeParam = value; }
        }

        /// <summary>
        /// Particle shader type.
        /// </summary>
        public ShaderType shaderType
        {
            get { return m_data.ShaderType; }
            set { m_data.ShaderType = value; }
        }

        /// <summary>
        /// Particle shader setting.
        /// </summary>
        public uint userShaderSetting
        {
            get { return m_data.ShaderSetting; }
            set { m_data.ShaderSetting = value; }
        }

        /// <summary>
        /// シェーダーパラメータ0
        /// </summary>
        public float shaderParam0
        {
            get { return m_data.ShaderParam0; }
            set { m_data.ShaderParam0 = value; }
        }

        /// <summary>er
        /// シェーダーパラメータ1
        /// </summary>
        public float shaderParam1
        {
            get { return m_data.ShaderParam1; }
            set { m_data.ShaderParam1 = value; }
        }

        /// <summary>
        /// ユーザーシェーダフラグ
        /// </summary>
        public uint userShaderFlags
        {
            get { return m_data.UserShaderFlags; }
            set { m_data.UserShaderFlags = value; }
        }

        /// <summary>
        /// ユーザーシェーダフラグ
        /// </summary>
        public uint UserShaderSwitch
        {
            get { return m_data.UserShaderSwitch; }
            set { m_data.UserShaderSwitch = value; }
        }

        /// <summary>
        /// ユーザーシェーダパラメータ
        /// </summary>
        public float[] userShaderParams
        {
            get
            {
                float[] rvalue = new float[m_data.UserShaderParams.Length];
                for (int i = 0; i < m_data.UserShaderParams.Length; ++i)
                {
                    rvalue[i] = m_data.UserShaderParams[i];
                }
                return rvalue;
            }
            set
            {
                for (int i = 0; i < m_data.UserShaderParams.Length; ++i)
                {
                    m_data.UserShaderParams[i] = value[i];
                }
            }
        }

        #endregion

        #region Texture UV animation

        /// <summary>
        /// ＵＶアニメーション
        /// </summary>
        public int uvShiftAnimMode
        {
            get { return m_data.UVShiftAnimMode; }
            set { m_data.UVShiftAnimMode = value; }
        }


        /// <summary>
        /// ＵＶアニメーション詳細設定
        /// </summary>
        public int isUvShiftAnimComplex
        {
            get { return ( m_data.IsUVShiftAnimComplex==true ? 1 : 0 ); }
            set { m_data.IsUVShiftAnimComplex = (value==1); }
        }


        /// <summary>
        /// ＵＶスクロール加算値
        /// </summary>
        public Vector2Xml uvScroll
        {
            get { return UtilityXml.ToVector2Xml( m_data.UVScroll ); }
            set { m_data.UVScroll = UtilityXml.ToVector2( value ); }
        }


        /// <summary>
        /// ＵＶスクロール初期値
        /// </summary>
        public Vector2Xml uvScrollInit
        {
            get { return UtilityXml.ToVector2Xml( m_data.UVScrollInit ); }
            set { m_data.UVScrollInit = UtilityXml.ToVector2( value ); }
        }


        /// <summary>
        /// ＵＶスクロール　ランダム
        /// </summary>
        public Vector2Xml uvScrollInitRand
        {
            get { return UtilityXml.ToVector2Xml( m_data.UVScrollInitRand ); }
            set { m_data.UVScrollInitRand = UtilityXml.ToVector2( value ); }
        }


        /// <summary>
        /// ＵＶスケール加算値
        /// </summary>
        public Vector2Xml uvScale
        {
            get { return UtilityXml.ToVector2Xml( m_data.UVScale ); }
            set { m_data.UVScale = UtilityXml.ToVector2( value ); }
        }


        /// <summary>
        /// ＵＶスケール初期値
        /// </summary>
        public Vector2Xml uvScaleInit
        {
            get { return UtilityXml.ToVector2Xml( m_data.UVScaleInit ); }
            set { m_data.UVScaleInit = UtilityXml.ToVector2( value ); }
        }


        /// <summary>
        /// ＵＶスケール　ランダム
        /// </summary>
        public Vector2Xml uvScaleInitRand
        {
            get { return UtilityXml.ToVector2Xml( m_data.UVScaleInitRand ); }
            set { m_data.UVScaleInitRand = UtilityXml.ToVector2( value ); }
        }


        /// <summary>
        /// ＵＶ回転加算値
        /// </summary>
        public float uvRot
        {
            get { return m_data.UVRot; }
            set { m_data.UVRot = value; }
        }


        /// <summary>
        /// ＵＶ回転初期値
        /// </summary>
        public float uvRotInit
        {
            get { return m_data.UVRotInit; }
            set { m_data.UVRotInit = value; }
        }


        /// <summary>
        /// ＵＶ回転　ランダム
        /// </summary>
        public float uvRotInitRand
        {
            get { return m_data.UVRotInitRand; }
            set { m_data.UVRotInitRand = value; }
        }

        #endregion

        #region SubTexture

        /// <summary>
        /// ＵＶアニメーション
        /// </summary>
        public int subTexUVShiftAnimMode
        {
            get { return m_data.SubTexUVShiftAnimMode; }
            set { m_data.SubTexUVShiftAnimMode = value; }
        }


        /// <summary>
        /// ＵＶアニメーション詳細設定
        /// </summary>
        public int subTexIsUvShiftAnimComplex
        {
            get { return (m_data.SubTexIsUVShiftAnimComplex == true ? 1 : 0); }
            set { m_data.SubTexIsUVShiftAnimComplex = (value == 1); }
        }


        /// <summary>
        /// ＵＶスクロール加算値
        /// </summary>
        public Vector2Xml subTexUVScroll
        {
            get { return UtilityXml.ToVector2Xml(m_data.SubTexUVScroll); }
            set { m_data.SubTexUVScroll = UtilityXml.ToVector2(value); }
        }


        /// <summary>
        /// ＵＶスクロール初期値
        /// </summary>
        public Vector2Xml subTexUVScrollInit
        {
            get { return UtilityXml.ToVector2Xml(m_data.SubTexUVScrollInit); }
            set { m_data.SubTexUVScrollInit = UtilityXml.ToVector2(value); }
        }


        /// <summary>
        /// ＵＶスクロール　ランダム
        /// </summary>
        public Vector2Xml subTexUVScrollInitRand
        {
            get { return UtilityXml.ToVector2Xml(m_data.SubTexUVScrollInitRand); }
            set { m_data.SubTexUVScrollInitRand = UtilityXml.ToVector2(value); }
        }


        /// <summary>
        /// ＵＶスケール加算値
        /// </summary>
        public Vector2Xml subTexUVScale
        {
            get { return UtilityXml.ToVector2Xml(m_data.SubTexUVScale); }
            set { m_data.SubTexUVScale = UtilityXml.ToVector2(value); }
        }


        /// <summary>
        /// ＵＶスケール初期値
        /// </summary>
        public Vector2Xml subTexUVScaleInit
        {
            get { return UtilityXml.ToVector2Xml(m_data.SubTexUVScaleInit); }
            set { m_data.SubTexUVScaleInit = UtilityXml.ToVector2(value); }
        }


        /// <summary>
        /// ＵＶスケール　ランダム
        /// </summary>
        public Vector2Xml subTexUVScaleInitRand
        {
            get { return UtilityXml.ToVector2Xml(m_data.SubTexUVScaleInitRand); }
            set { m_data.SubTexUVScaleInitRand = UtilityXml.ToVector2(value); }
        }


        /// <summary>
        /// ＵＶ回転加算値
        /// </summary>
        public float subTexUVRot
        {
            get { return m_data.SubTexUVRot; }
            set { m_data.SubTexUVRot = value; }
        }


        /// <summary>
        /// ＵＶ回転初期値
        /// </summary>
        public float subTexUVRotInit
        {
            get { return m_data.SubTexUVRotInit; }
            set { m_data.SubTexUVRotInit = value; }
        }


        /// <summary>
        /// ＵＶ回転　ランダム
        /// </summary>
        public float subTexUVRotInitRand
        {
            get { return m_data.SubTexUVRotInitRand; }
            set { m_data.SubTexUVRotInitRand = value; }
        }

        #endregion

        #region Transform

        /// <summary>
        ///
        /// </summary>
        public Vector3Xml UI_transformTrans
        {
            get { return UtilityXml.ToVector3Xml(m_data.UI_transformTrans); }
            set { m_data.UI_transformTrans = UtilityXml.ToVector3(value); }
        }

        /// <summary>
        ///
        /// </summary>
        public Vector3i UI_transformRot
        {
            get { return m_data.UI_transformRot; }
            set { m_data.UI_transformRot = value; }
        }

        #endregion

        #region Property for relative file path

        /// <summary>
        /// Normalize path to file name.
        /// </summary>
        protected bool m_bNormalizePath = true;

        /// <summary>
        /// Get or set the flag indicating whether to use relative path or not.
        /// </summary>
        [System.Xml.Serialization.XmlIgnore]
        public bool UseRelativePath
        {
            get { return m_bNormalizePath; }
            set { m_bNormalizePath = value; }
        }

        #endregion
    }
}
