﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Xml;
using System.Xml.Serialization;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Manager;
using EffectMaker.Foundation.Collections.Generic;
using EffectMaker.Foundation.Utility;
using NintendoWare.ToolDevelopmentKit.Logs;
using NWCore.DataModel;
using NWCore.res;
using NWCore.src.Remake.DataModel.DataModels;
using NWCore.src.Remake.DataModel.DataModels.Emitter;

namespace NWCore.Remake.Updater
{
    /// <summary>
    /// 現行版EffectMakerからRemake版EffectMakerの中間ファイルに変換を行うクラスです.
    /// </summary>
    public class EsetUpdater
    {
        private List<EmitterCustomShaderConvertingData> customShaderTypeList;
        private List<CustomActionConvertingData> customActionTypeList;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public EsetUpdater(List<EmitterCustomShaderConvertingData> argCustomShaderTypeList,
            List<CustomActionConvertingData> argCustomActionTypeList)
        {
            this.customShaderTypeList = argCustomShaderTypeList;
            this.customActionTypeList = argCustomActionTypeList;
        }

        /// <summary>
        /// 角度をインデックスから度数に変換します。
        /// </summary>
        /// <param name="idx">インデックス角です.</param>
        /// <returns>度に変換した値を返却します。</returns>
        public static float IndexToDegree(float idx, bool is180)
        {
            if (idx < 0)
            {
                const uint i_half_round_idx = 0x80000000;
                idx = (float)( (idx * -1) / i_half_round_idx ) * 180.0f;
                if (is180)
                    return idx * -1;
                else
                    return 360.0f - idx;
            }
            else
            {
                const uint i_half_round_idx = 0x80000000;
                return (float)( idx / i_half_round_idx ) * 180.0f;
            }
        }

        /// <summary>
        /// インデックス角からラジアンに変換します.
        /// </summary>
        /// <param name="fAngle">インデックス角です.</param>
        /// <returns>ラジアンに変換した値を返却します.</returns>
        public static float NW4FToRadian(float fAngle)
        {
            double fSrcAngle = (double)fAngle / (double)(0x80000000);
            double fDstAngle = fSrcAngle * System.Math.PI;

            return (float)fDstAngle;
        }

        /// <summary>
        /// ラジアンから度に変換します.
        /// </summary>
        /// <param name="fRadian">ラジアンです.</param>
        /// <returns>度数法に変換した値を返却します.</returns>
        public static float RadianToDegree(float fRadian)
        {
            double fSrcAngle = (double)fRadian;
            double fConvert = (double)180.0 / System.Math.PI;
            double fDstAngle = fSrcAngle * fConvert;

            return (float)fDstAngle;
        }

        /// <summary>
        /// ファイルパスが空でない場合にファイル名のみを取り出して返します。
        /// </summary>
        /// <param name="path">パス</param>
        /// <returns>ファイル名</returns>
        public static string GetFileName(string path)
        {
            return string.IsNullOrEmpty(path) ? string.Empty : Path.GetFileName(path);
        }

        /// <summary>
        /// 古い形式から新しい形式に変換します.
        /// </summary>
        /// <param name="oldEset">旧型式エミッタセット.</param>
        /// <param name="oldEmitterList">旧型式エミッタリスト.</param>
        /// <param name="path">新形式中間ファイル出力ファイルパス</param>
        /// <returns>変換に成功したらtrueを返却します.</returns>
        public bool Convert
        (
            NWCore.DataModel.EmitterSetData oldEset,
            List<NWCore.DataModel.EmitterData> oldEmitterList,
            string path
        )
        {
            // エミッタセットのインスタンスを生成.
            EffectMaker.DataModel.DataModels.EmitterSetData newEset = new EffectMaker.DataModel.DataModels.EmitterSetData();

            // 新形式に変換.
            this.ConvertEset(oldEset, oldEmitterList, newEset);

            try
            {
                // シリアライズ処理.
                using (TextWriter writer = new StreamWriter(path, false, Encoding.UTF8))
                {
                    // 拡張タイプ.
                    Type[] extraTypes = new Type[]
                    {
//                        typeof(EffectMaker.DataModel.DataModels.RandomData),
                        typeof(EffectMaker.DataModel.DataModels.RandomFe1Data),
                        typeof(EffectMaker.DataModel.DataModels.MagnetData),
                        typeof(EffectMaker.DataModel.DataModels.SpinData),
                        typeof(EffectMaker.DataModel.DataModels.ConvergeData),
                        typeof(EffectMaker.DataModel.DataModels.AddLocationData),
                        typeof(EffectMaker.DataModel.DataModels.CollisionData),
                        typeof(EffectMaker.DataModel.DataModels.CurlNoiseData),
                        typeof(EffectMaker.DataModel.DataModels.StripeHistoryData),
                        typeof(EffectMaker.DataModel.DataModels.StripeHistoryUserData),
                        typeof(EffectMaker.DataModel.DataModels.StripeSuperData),
                        typeof(EffectMaker.DataModel.DataModels.StripeSuperUserData)
                    };

                    // シリアライザー生成.
                    var serializer = new XmlSerializer(typeof(EffectMaker.DataModel.DataModels.EmitterSetData), extraTypes);

                    // シリアライズ！
                    serializer.Serialize(writer, newEset);
                }
            }
            catch
            {
                return false;
            }

            // 例外が発生しなければ正常終了.
            return true;
        }

        #region EmitterSet

        /// <summary>
        /// エミッタセットデータを新型式に変換します.
        /// </summary>
        /// <param name="oldEset">旧型式エミッタセット.</param>
        /// <param name="oldEmitterList">旧型式エミッタリスト.</param>
        /// <param name="newEset">新型式エミッタセット.</param>
        private void ConvertEset
        (
            NWCore.DataModel.EmitterSetData oldEset,
            List<NWCore.DataModel.EmitterData> oldEmitterList,
            EffectMaker.DataModel.DataModels.EmitterSetData newEset
        )
        {
            // 基本設定.
            this.ConvertEsetBasicSettingsTab(oldEset, newEset);

            // ユーザーデータ.
            this.ConvertEsetUserDataTab(oldEset, newEset);

            // 旧型式のエミッタセット数だけ処理.
            for (int i = 0; i < oldEmitterList.Count; ++i)
            {
                // エミッタのインスタンスを生成.
                EffectMaker.DataModel.DataModels.EmitterData newEmitter = new EffectMaker.DataModel.DataModels.EmitterData();

                // エミッタをコンバート.
                this.ConvertEmitter(oldEmitterList[i], newEmitter);

                // エミッタリストに追加.
                newEset.EmitterList.Add(newEmitter);
            }
        }

        #region 基本設定

        /// <summary>
        /// 【エミッタセット】基本設定タブの変換処理です.
        /// </summary>
        private void ConvertEsetBasicSettingsTab(NWCore.DataModel.EmitterSetData oldEset, EffectMaker.DataModel.DataModels.EmitterSetData newEset)
        {
            // ファイル名.
            newEset.Name = string.Copy(oldEset.Name);

            // コメント.
            newEset.EmitterSetBasicData.Comment = string.Copy(oldEset.EditData.Comment);

            // カラーラベル.
            newEset.EmitterSetBasicData.LabelColor = oldEset.EditData.ColorLabelIndex;
        }

        #endregion

        #region ユーザーデータ

        /// <summary>
        /// 【エミッタセット】ユーザーデータタブの変換処理です.
        /// </summary>
        private void ConvertEsetUserDataTab(NWCore.DataModel.EmitterSetData oldEset, EffectMaker.DataModel.DataModels.EmitterSetData newEset)
        {
            ushort flags = 0;

            // ビット.
            flags |= (ushort)(oldEset.UserData.UserBit00 ? 1 <<  0 : 0);
            flags |= (ushort)(oldEset.UserData.UserBit01 ? 1 <<  1 : 0);
            flags |= (ushort)(oldEset.UserData.UserBit02 ? 1 <<  2 : 0);
            flags |= (ushort)(oldEset.UserData.UserBit03 ? 1 <<  3 : 0);
            flags |= (ushort)(oldEset.UserData.UserBit04 ? 1 <<  4 : 0);
            flags |= (ushort)(oldEset.UserData.UserBit05 ? 1 <<  5 : 0);
            flags |= (ushort)(oldEset.UserData.UserBit06 ? 1 <<  6 : 0);
            flags |= (ushort)(oldEset.UserData.UserBit07 ? 1 <<  7 : 0);
            flags |= (ushort)(oldEset.UserData.UserBit08 ? 1 <<  8 : 0);
            flags |= (ushort)(oldEset.UserData.UserBit09 ? 1 <<  9 : 0);
            flags |= (ushort)(oldEset.UserData.UserBit10 ? 1 << 10 : 0);
            flags |= (ushort)(oldEset.UserData.UserBit11 ? 1 << 11 : 0);
            flags |= (ushort)(oldEset.UserData.UserBit12 ? 1 << 12 : 0);
            flags |= (ushort)(oldEset.UserData.UserBit13 ? 1 << 13 : 0);
            flags |= (ushort)(oldEset.UserData.UserBit14 ? 1 << 14 : 0);
            flags |= (ushort)(oldEset.UserData.UserBit15 ? 1 << 15 : 0);

            newEset.EmitterSetUserData.EmitterSetUserFlagData.Flags = flags;

            // 数値.
            newEset.EmitterSetUserData.EmitterSetUserValueData.Value1 = (uint)oldEset.UserData.UserSlider0;
            newEset.EmitterSetUserData.EmitterSetUserValueData.Value2 = (uint)oldEset.UserData.UserSlider1;
            // XXX = oldEset.UserData.UserSlider2;
            // XXX = oldEset.UserData.UserSlider3;
            // XXX = oldEset.UserData.UserSlider4;
            // XXX = oldEset.UserData.UserSlider5;

            // XXX = oldEset.UserData.UserSliderF0;
            // XXX = oldEset.UserData.UserSliderF1;
            // XXX = oldEset.UserData.UserSliderF2;
            // XXX = oldEset.UserData.UserSliderF3;
            // XXX = oldEset.UserData.UserSliderF4;
            // XXX = oldEset.UserData.UserSliderF5;
            // XXX = oldEset.UserData.UserSliderF6;
            // XXX = oldEset.UserData.UserSliderF7;

            // コールバックID.
            // XXX = oldEset.userData.CallbackId

        }
        #endregion

        #endregion

        #region Emitter

        /// <summary>
        /// エミッタを新型式に変換します.
        /// </summary>
        /// <param name="oldEmitter">旧型式.</param>
        /// <param name="newEmitter">新型式.</param>
        private void ConvertEmitter(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            // エミッタ名.
            newEmitter.Name = string.Copy(oldEmitter.Name);

            // 基本設定.
            this.ConvertEmitterBasicSettingsTab(oldEmitter, newEmitter);

            // エミッタ.
            this.ConvertEmitterEmitterTab(oldEmitter, newEmitter);

            // 放出.
            this.ConvertEmitterEmissionTab(oldEmitter, newEmitter);

            // パーティクル.
            this.ConvertEmitterParticleTab(oldEmitter, newEmitter);

            // コンバイナ.
            this.ConvertEmitterCombinerTab(oldEmitter, newEmitter);

            // テクスチャ0.
            this.ConvertEmitterTexture0Tab(oldEmitter, newEmitter);

            // テクスチャ1.
            this.ConvertEmitterTexture1Tab(oldEmitter, newEmitter);

            // テクスチャ2.
            this.ConvertEmitterTexture2Tab(oldEmitter, newEmitter);

            // カラー.
            this.ConvertEmitterColorTab(oldEmitter, newEmitter);

            // スケール.
            this.ConvertEmitterScaleTab(oldEmitter, newEmitter);

            // 回転.
            this.ConvertEmitterRotationTab(oldEmitter, newEmitter);

            // 終了処理.
            this.ConvertEmitterEndProcessTab(oldEmitter, newEmitter);

            // カスタムアクション.
            this.ConvertEmitterCustomActionTab(oldEmitter, newEmitter);

            // カスタムシェーダ.
            this.ConvertEmitterCustomShaderTab(oldEmitter, newEmitter);

            // チャイルドを使用する場合は，子供をぶら下げる.
            if ((oldEmitter.Type == DataModel.EmitterType.Complex)
             && (oldEmitter.ChildFlag.Enable == true) )
            {
                // チャイルド用のインスタンスを生成.
                EffectMaker.DataModel.DataModels.EmitterData childData = new EffectMaker.DataModel.DataModels.EmitterData();

                // チャイルドデータを変換.
                this.ConvertChild(oldEmitter, childData);

                // チャイルドデータをリストに追加..
                newEmitter.EmitterList.Add(childData);
            }

            // フィールド.
            {
                // フィールドデータを変換.
                this.ConvertField(oldEmitter, newEmitter);
            }
        }

        #region 基本設定

        /// <summary>
        /// 【エミッタ】基本設定タブの変換処理です.
        /// </summary>
        /// <param name="oldEmitter">旧型式</param>
        /// <param name="newEmitter">新形式</param>
        private void ConvertEmitterBasicSettingsTab(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            //------ 基本 グループ -------
            // パーティクル追従タイプ.
            switch (oldEmitter.UI_followType)
            {
                // 完全追従
                case DataModel.FollowType.All:
                    {
                        newEmitter.EmitterBasicSettingData.EmitterBasicBasicData.ParticleFollowType = 0;
                    }
                    break;

                // 追従しない.
                case DataModel.FollowType.None:
                    {
                        newEmitter.EmitterBasicSettingData.EmitterBasicBasicData.ParticleFollowType = 1;
                    }
                    break;

                // 位置のみ追従.
                case DataModel.FollowType.PosOnly:
                    {
                        newEmitter.EmitterBasicSettingData.EmitterBasicBasicData.ParticleFollowType = 2;
                    }
                    break;
            }

            // GPUエミッタの有効無効
            if ( oldEmitter.FieldFlag.Flag > 0 && oldEmitter.Type == EmitterType.Complex )
            {
                // 高機能モードでフィールドが設定されている場合、CPUに振り替え
                newEmitter.EmitterBasicSettingData.EmitterBasicBasicData.EnableGpuEmitter = 0;
            }
            else
            {
                // それ以外の場合はGPUの設定を維持する
                newEmitter.EmitterBasicSettingData.EmitterBasicBasicData.EnableGpuEmitter = oldEmitter.EnableGpuParticle ? 1 : 0;
            }

            // 乱数タイプ.
            newEmitter.EmitterBasicSettingData.EmitterBasicRandomData.RandomType = (int)oldEmitter.UI_randomType;

            // 固定乱数値.
            newEmitter.EmitterBasicSettingData.EmitterBasicRandomData.FixedRandomSeed = oldEmitter.UI_randomSeed;

            //----- 描画設定グループ ------
            // パーティクルを描画する.
            newEmitter.EmitterBasicSettingData.EmitterBasicRenderData.EnableDispParticle = oldEmitter.DispParticle;

            // 描画パス.
            newEmitter.EmitterBasicSettingData.EmitterBasicRenderData.DrawPath = oldEmitter.UI_drawPathID;

            // 描画パス名.
            newEmitter.EmitterBasicSettingData.EmitterBasicRenderData.DrawPathName = oldEmitter.UI_drawPath;

            //----- パーティクルソートグループ -----
            newEmitter.EmitterBasicSettingData.EmitterBasicRenderData.ParticleSortOrder = (int)oldEmitter.ParticleSortType;
        }

        #endregion

        #region エミッタ

        /// <summary>
        /// 【エミッタ】エミッタタブの変換処理です.
        /// </summary>
        /// <param name="oldEmitter">旧型式.</param>
        /// <param name="newEmitter">新形式.</param>
        private void ConvertEmitterEmitterTab(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            //*********************************
            //  エミッタ形状グループ
            //*********************************
            // エミッタタイプ.
            newEmitter.EmitterEmitterData.EmitterEmitterShapeData.EmitterType = (int)oldEmitter.VolumeType;


            //----- 円サブグループ ------
            // エミッタ半径
            newEmitter.EmitterEmitterData.EmitterEmitterShapeData.EmitterTypeEmitterRadius.X = oldEmitter.VolumeRadius.X;
            newEmitter.EmitterEmitterData.EmitterEmitterShapeData.EmitterTypeEmitterRadius.Y = oldEmitter.VolumeRadius.Y;
            newEmitter.EmitterEmitterData.EmitterEmitterShapeData.EmitterTypeEmitterRadius.Z = oldEmitter.VolumeRadius.Z;

            // 弧の幅(角度)
            {
                // Convert NW4FToRadian
                float radianValue = NW4FToRadian(oldEmitter.VolumeSweepParam);

                // Convert RadianToDegree
                RadianToDegree(radianValue);

                newEmitter.EmitterEmitterData.EmitterEmitterShapeData.EmitterShapeLongitudeArcLength = RadianToDegree(radianValue);
            }

            // 弧の開始(角度)
            newEmitter.EmitterEmitterData.EmitterEmitterShapeData.EmitterShapeStartArc = IndexToDegree(oldEmitter.VolumeSweepStart, false);

            // 弧の開始角度ランダム
            newEmitter.EmitterEmitterData.EmitterEmitterShapeData.EmitterShapeStartArcRandom = oldEmitter.VolumeSweepStartRandom;

            //----- 円(等分割)サブグループ -----
            // ランダム位置.
            newEmitter.EmitterEmitterData.EmitterEmitterShapeData.EmitterShapeRandomPos = oldEmitter.VolumeSurfacePosRand;


            //----- 円(ボリューム)サブグループ -----
            // 空洞率(%).
            newEmitter.EmitterEmitterData.EmitterEmitterShapeData.EmitterShapeHollowRatio = (float)oldEmitter.VolumeHollow;


            //----- 球サブグループ ------
            // 弧の開き方.
            newEmitter.EmitterEmitterData.EmitterEmitterShapeData.EmitterShapeArcType = (int)((oldEmitter.VolumeLatitudeEnabled) ? 1 : 0);


            //----- 緯度サブグループ -----
            // 軸.
            newEmitter.EmitterEmitterData.EmitterEmitterShapeData.EmitterShapeSphereAxis = (int)oldEmitter.VolumeLatitudeDir;

            // 弧の幅(角度)(緯度)
            newEmitter.EmitterEmitterData.EmitterEmitterShapeData.EmitterShapeLatitudeArcLength = oldEmitter.VolumeLatitude;

            //----- 球(等分割)サブグループ -----
            // 分割数.
            newEmitter.EmitterEmitterData.EmitterEmitterShapeData.EmitterShapeSphereAndDivisionDivision = oldEmitter.VolumeTblIndex;

            //----- 球(等分割64)サブグループ -----
            // 分割数.
            newEmitter.EmitterEmitterData.EmitterEmitterShapeData.EmitterShapeSphereAndDivision64Division = (oldEmitter.Volume64TblIndex);

            //----- 球(ボリューム)サブグループ -----
            //----- 円柱サブグループ ------
            //----- 円柱(ボリュームサブグルプ) -----
            //----- 立方体サブグループ -----
            //----- 立方体(ボリューム) サブグループ -----

            //----- ラインサブグループ ----
            // 長さ
            newEmitter.EmitterEmitterData.EmitterEmitterShapeData.EmitterShapeLineLength = oldEmitter.VolumeRadius.Z;

            // 中心位置.
            newEmitter.EmitterEmitterData.EmitterEmitterShapeData.EmitterShapeLineCenter = oldEmitter.LineCenter;

            //----- ライン(等分割)サブグループ -----
            //----- 矩形サブグループ -----

            //----- プリミティブサブグループ -----
            // ファイルパス.
            newEmitter.EmitterEmitterData.EmitterEmitterShapeData.PrimitiveFilePath = GetFileName(oldEmitter.PrimitiveEmitterFilePath);

            // 放出タイプ.
            newEmitter.EmitterEmitterData.EmitterEmitterShapeData.EmitterShapeEmissionType = (int)oldEmitter.VolumePrimEmitType;


            //----- スケールサブグループ -----
            // スケール
            newEmitter.EmitterEmitterData.EmitterEmitterShapeData.EmitterScaleShapeScale.X = oldEmitter.EmitterFormScale.X;
            newEmitter.EmitterEmitterData.EmitterEmitterShapeData.EmitterScaleShapeScale.Y = oldEmitter.EmitterFormScale.Y;
            newEmitter.EmitterEmitterData.EmitterEmitterShapeData.EmitterScaleShapeScale.Z = oldEmitter.EmitterFormScale.Z;

            {
                NWCore.DataModel.AnimTableData emitter_form_scale_x = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.emitter_form_scale_x);
                NWCore.DataModel.AnimTableData emitter_form_scale_y = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.emitter_form_scale_y);
                NWCore.DataModel.AnimTableData emitter_form_scale_z = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.emitter_form_scale_z);

                // スケール - エミッタ時間アニメ
                newEmitter.EmitterEmitterData.EmitterEmitterShapeData.ShapeScaleAnimation.EnableAnimation = emitter_form_scale_x.IsEnabled;

                // スケール - アニメーションテーブル
                newEmitter.EmitterEmitterData.EmitterEmitterShapeData.ShapeScaleAnimation.LoopMode = (emitter_form_scale_x.IsLoop) ? 1 : 0;
                switch (emitter_form_scale_x.InterpolationType)
                {
                    // リニア.
                    case DataModel.AnimInterpolationTypes.Linear:
                        {
                            newEmitter.EmitterEmitterData.EmitterEmitterShapeData.ShapeScaleAnimation.InterpolationMode = 0;
                        }
                        break;

                    // スムース.
                    case DataModel.AnimInterpolationTypes.Smooth:
                        {
                            newEmitter.EmitterEmitterData.EmitterEmitterShapeData.ShapeScaleAnimation.InterpolationMode = 1;
                        }
                        break;
                }

                // テーブルを一回クリア.
                newEmitter.EmitterEmitterData.EmitterEmitterShapeData.ShapeScaleAnimation.AnimationTable.Clear();

                // キーデータをぶっこむ.
                for (int i = 0; i < emitter_form_scale_x.KeyFrames.Count; ++i)
                {
                    NWCore.DataModel.FloatAnimTableKeyFrameData keyX = emitter_form_scale_x.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;
                    NWCore.DataModel.FloatAnimTableKeyFrameData keyY = emitter_form_scale_y.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;
                    NWCore.DataModel.FloatAnimTableKeyFrameData keyZ = emitter_form_scale_z.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;

                    newEmitter.EmitterEmitterData.EmitterEmitterShapeData.ShapeScaleAnimation.AnimationTable.AddKeyFrame(
                        keyX.KeyFrame,
                        keyX.Value,
                        keyY.Value,
                        keyZ.Value,
                        0.0f,
                        false);
                }
            }

            //**********************************
            //  エミッタトランスフォームグループ
            //**********************************
            // スケール.
            newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterScale.X = oldEmitter.UI_transformScale.X;
            newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterScale.Y = oldEmitter.UI_transformScale.Y;
            newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterScale.Z = oldEmitter.UI_transformScale.Z;

            {
                NWCore.DataModel.AnimTableData emitter_scale_x = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.emitter_scale_x);
                NWCore.DataModel.AnimTableData emitter_scale_y = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.emitter_scale_y);
                NWCore.DataModel.AnimTableData emitter_scale_z = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.emitter_scale_z);

                // スケール - エミッタ時間アニメ
                newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterScaleAnimation.EnableAnimation = emitter_scale_x.IsEnabled;

                // スケール - アニメーションテーブル.
                newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterScaleAnimation.LoopMode = (emitter_scale_x.IsLoop) ? 1 : 0;

                switch (emitter_scale_x.InterpolationType)
                {
                    // リニア.
                    case DataModel.AnimInterpolationTypes.Linear:
                        {
                            newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterScaleAnimation.InterpolationMode = 0;
                        }
                        break;

                    // スムース.
                    case DataModel.AnimInterpolationTypes.Smooth:
                        {
                            newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterScaleAnimation.InterpolationMode = 1;
                        }
                        break;
                }

                // テーブルを一回クリア.
                newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterScaleAnimation.AnimationTable.Clear();

                // キーデータをぶっこむ.
                for (int i = 0; i < emitter_scale_x.KeyFrames.Count; ++i)
                {
                    NWCore.DataModel.FloatAnimTableKeyFrameData keyX = emitter_scale_x.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;
                    NWCore.DataModel.FloatAnimTableKeyFrameData keyY = emitter_scale_y.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;
                    NWCore.DataModel.FloatAnimTableKeyFrameData keyZ = emitter_scale_z.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;

                    newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterScaleAnimation.AnimationTable.AddKeyFrame(
                        keyX.KeyFrame,
                        keyX.Value,
                        keyY.Value,
                        keyZ.Value,
                        0.0f,
                        false);
                }
            }


            // 回転.
            newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterRotation.X = IndexToDegree(oldEmitter.UI_transformRot.X, true);
            newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterRotation.Y = IndexToDegree(oldEmitter.UI_transformRot.Y, true);
            newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterRotation.Z = IndexToDegree(oldEmitter.UI_transformRot.Z, true);
            // 回転 - ランダム幅.
            newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterRotationRandom.X = IndexToDegree(oldEmitter.UI_transformRotRnd.X, false);
            newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterRotationRandom.Y = IndexToDegree(oldEmitter.UI_transformRotRnd.Y, false);
            newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterRotationRandom.Z = IndexToDegree(oldEmitter.UI_transformRotRnd.Z, false);

            {
                NWCore.DataModel.AnimTableData emitter_rot_x = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.emitter_rot_x);
                NWCore.DataModel.AnimTableData emitter_rot_y = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.emitter_rot_y);
                NWCore.DataModel.AnimTableData emitter_rot_z = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.emitter_rot_z);

                // 回転 - エミッタ時間アニメ
                newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterRotationAnimation.EnableAnimation = emitter_rot_x.IsEnabled;

                // 回転 - アニメーションテーブル.
                newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterRotationAnimation.LoopMode = (emitter_rot_x.IsLoop) ? 1 : 0;

                switch (emitter_rot_x.InterpolationType)
                {
                    // リニア.
                    case DataModel.AnimInterpolationTypes.Linear:
                        {
                            newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterRotationAnimation.InterpolationMode = 0;
                        }
                        break;

                    // スムース.
                    case DataModel.AnimInterpolationTypes.Smooth:
                        {
                            newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterRotationAnimation.InterpolationMode = 1;
                        }
                        break;
                }

                // テーブルを一回クリア.
                newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterRotationAnimation.AnimationTable.Clear();

                // キーデータをぶっこむ.
                for (int i = 0; i < emitter_rot_x.KeyFrames.Count; ++i)
                {
                    NWCore.DataModel.FloatAnimTableKeyFrameData keyX = emitter_rot_x.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;
                    NWCore.DataModel.FloatAnimTableKeyFrameData keyY = emitter_rot_y.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;
                    NWCore.DataModel.FloatAnimTableKeyFrameData keyZ = emitter_rot_z.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;

                    newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterRotationAnimation.AnimationTable.AddKeyFrame(
                        keyX.KeyFrame,
                        RadianToDegree(keyX.Value),
                        RadianToDegree(keyY.Value),
                        RadianToDegree(keyZ.Value),
                        0.0f,
                        false);
                }
            }

            // 位置.
            newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterPosition.X = oldEmitter.UI_transformTrans.X;
            newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterPosition.Y = oldEmitter.UI_transformTrans.Y;
            newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterPosition.Z = oldEmitter.UI_transformTrans.Z;
            // 位置 - ランダム幅
            newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterPositionRandom.X = oldEmitter.UI_transformTransRnd.X;
            newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterPositionRandom.Y = oldEmitter.UI_transformTransRnd.Y;
            newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterPositionRandom.Z = oldEmitter.UI_transformTransRnd.Z;

            {
                NWCore.DataModel.AnimTableData emitter_trans_x = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.emitter_trans_x);
                NWCore.DataModel.AnimTableData emitter_trans_y = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.emitter_trans_y);
                NWCore.DataModel.AnimTableData emitter_trans_z = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.emitter_trans_z);

                // 位置 - エミッタ時間アニメ
                newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterPositionAnimation.EnableAnimation = emitter_trans_x.IsEnabled;

                // 位置 - アニメーションテーブル.
                newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterPositionAnimation.LoopMode = (emitter_trans_x.IsLoop) ? 1 : 0;

                switch (emitter_trans_x.InterpolationType)
                {
                    // リニア.
                    case DataModel.AnimInterpolationTypes.Linear:
                        {
                            newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterPositionAnimation.InterpolationMode = 0;
                        }
                        break;

                    // スムース.
                    case DataModel.AnimInterpolationTypes.Smooth:
                        {
                            newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterPositionAnimation.InterpolationMode = 1;
                        }
                        break;
                }

                // テーブルを一回クリア.
                newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterPositionAnimation.AnimationTable.Clear();

                // キーデータをぶっこむ.
                for (int i = 0; i < emitter_trans_x.KeyFrames.Count; ++i)
                {
                    NWCore.DataModel.FloatAnimTableKeyFrameData keyX = emitter_trans_x.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;
                    NWCore.DataModel.FloatAnimTableKeyFrameData keyY = emitter_trans_y.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;
                    NWCore.DataModel.FloatAnimTableKeyFrameData keyZ = emitter_trans_z.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;

                    newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EmitterPositionAnimation.AnimationTable.AddKeyFrame(
                        keyX.KeyFrame,
                        keyX.Value,
                        keyY.Value,
                        keyZ.Value,
                        0.0f,
                        false);
                }
            }

            // ランダム幅を放出毎に更新.
            newEmitter.EmitterEmitterData.EmitterEmitterTransformData.EnableUpdateRandom = oldEmitter.UI_UpdateEmitterMatrixByEmit;


            //*********************************
            //  グローバル乗算カラーアルファ
            //*********************************
            // カラー0
            newEmitter.EmitterColorData.EmitterColor.Color0Value.R = oldEmitter.EmitterColor0.R;
            newEmitter.EmitterColorData.EmitterColor.Color0Value.G = oldEmitter.EmitterColor0.G;
            newEmitter.EmitterColorData.EmitterColor.Color0Value.B = oldEmitter.EmitterColor0.B;
            newEmitter.EmitterColorData.EmitterColor.Color0Value.A = oldEmitter.EmitterColor1.A;
            newEmitter.EmitterColorData.EmitterColor.Color0BehaviorType = 0;

            {
                NWCore.DataModel.AnimTableData color0_r = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.color0_r);
                NWCore.DataModel.AnimTableData color0_g = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.color0_g);
                NWCore.DataModel.AnimTableData color0_b = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.color0_b);


                // カラー0 - エミッタ時間アニメ.
                if (color0_r.IsEnabled) newEmitter.EmitterColorData.EmitterColor.Color0BehaviorType = 2;

                newEmitter.EmitterColorData.EmitterColor.Color0Animation.EnableAnimation = color0_r.IsEnabled;

                newEmitter.EmitterColorData.EmitterColor.Color0Animation.LoopMode = (color0_r.IsLoop == true) ? 1 : 0;
                switch (color0_r.InterpolationType)
                {
                    case DataModel.AnimInterpolationTypes.Linear:
                        {
                            newEmitter.EmitterColorData.EmitterColor.Color0Animation.InterpolationMode = 0;
                        }
                        break;

                    case DataModel.AnimInterpolationTypes.Smooth:
                        {
                            newEmitter.EmitterColorData.EmitterColor.Color0Animation.InterpolationMode = 1;
                        }
                        break;
                }

                newEmitter.EmitterColorData.EmitterColor.Color0Animation.AnimationTable.Clear();

                // カラー0 - アニメーションテーブル.
                for (int i = 0; i < color0_r.KeyFrames.Count; ++i)
                {
                    NWCore.DataModel.FloatAnimTableKeyFrameData keyR = color0_r.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;
                    NWCore.DataModel.FloatAnimTableKeyFrameData keyG = color0_g.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;
                    NWCore.DataModel.FloatAnimTableKeyFrameData keyB = color0_b.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;

                    newEmitter.EmitterColorData.EmitterColor.Color0Animation.AnimationTable.AddKeyFrame(
                        keyR.KeyFrame,
                        keyR.Value,
                        keyG.Value,
                        keyB.Value,
                        1.0f,
                        false);
                }
            }

            // カラー1
            newEmitter.EmitterColorData.EmitterColor.Color1Value.R = oldEmitter.EmitterColor1.R;
            newEmitter.EmitterColorData.EmitterColor.Color1Value.G = oldEmitter.EmitterColor1.G;
            newEmitter.EmitterColorData.EmitterColor.Color1Value.B = oldEmitter.EmitterColor1.B;
            newEmitter.EmitterColorData.EmitterColor.Color1Value.A = oldEmitter.EmitterColor1.A;
            newEmitter.EmitterColorData.EmitterColor.Color1BehaviorType = 0;

            {
                NWCore.DataModel.AnimTableData color1_r = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.color1_r);
                NWCore.DataModel.AnimTableData color1_g = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.color1_g);
                NWCore.DataModel.AnimTableData color1_b = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.color1_b);


                // カラー1 - エミッタ時間アニメ.
                if (color1_r.IsEnabled) newEmitter.EmitterColorData.EmitterColor.Color1BehaviorType = 2;

                newEmitter.EmitterColorData.EmitterColor.Color1Animation.EnableAnimation = color1_r.IsEnabled;

                newEmitter.EmitterColorData.EmitterColor.Color1Animation.LoopMode = (color1_r.IsLoop == true) ? 1 : 0;
                switch (color1_r.InterpolationType)
                {
                    case DataModel.AnimInterpolationTypes.Linear:
                        {
                            newEmitter.EmitterColorData.EmitterColor.Color1Animation.InterpolationMode = 0;
                        }
                        break;

                    case DataModel.AnimInterpolationTypes.Smooth:
                        {
                            newEmitter.EmitterColorData.EmitterColor.Color1Animation.InterpolationMode = 1;
                        }
                        break;
                }

                newEmitter.EmitterColorData.EmitterColor.Color1Animation.AnimationTable.Clear();

                // カラー1 - アニメーションテーブル.
                for (int i = 0; i < color1_r.KeyFrames.Count; ++i)
                {
                    NWCore.DataModel.FloatAnimTableKeyFrameData keyR = color1_r.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;
                    NWCore.DataModel.FloatAnimTableKeyFrameData keyG = color1_g.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;
                    NWCore.DataModel.FloatAnimTableKeyFrameData keyB = color1_b.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;

                    newEmitter.EmitterColorData.EmitterColor.Color1Animation.AnimationTable.AddKeyFrame(
                        keyR.KeyFrame,
                        keyR.Value,
                        keyG.Value,
                        keyB.Value,
                        1.0f,
                        false);
                }
            }

            // アルファ.
            /* 対応するもの無し = oldEmitter.EmitterAlpha; */

            {
                NWCore.DataModel.AnimTableData alpha = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.alpha);

                // アルファ - エミッタ時間アニメ.
                if (alpha.IsEnabled)
                {
                    newEmitter.EmitterColorData.EmitterColor.Alpha0BehaviorType = 2;
                    newEmitter.EmitterColorData.EmitterColor.Alpha1BehaviorType = 2;
                }
                newEmitter.EmitterColorData.EmitterColor.Alpha0Animation.EnableAnimation = alpha.IsEnabled;
                newEmitter.EmitterColorData.EmitterColor.Alpha1Animation.EnableAnimation = alpha.IsEnabled;
                newEmitter.EmitterColorData.EmitterColor.Alpha0Animation.LoopMode = (alpha.IsLoop == true) ? 1 : 0;
                newEmitter.EmitterColorData.EmitterColor.Alpha1Animation.LoopMode = (alpha.IsLoop == true) ? 1 : 0;

                switch (alpha.InterpolationType)
                {
                    case DataModel.AnimInterpolationTypes.Linear:
                        {
                            newEmitter.EmitterColorData.EmitterColor.Alpha0Animation.InterpolationMode = 0;
                            newEmitter.EmitterColorData.EmitterColor.Alpha1Animation.InterpolationMode = 0;
                        }
                        break;

                    case DataModel.AnimInterpolationTypes.Smooth:
                        {
                            newEmitter.EmitterColorData.EmitterColor.Alpha0Animation.InterpolationMode = 1;
                            newEmitter.EmitterColorData.EmitterColor.Alpha1Animation.InterpolationMode = 1;
                        }
                        break;
                }

                newEmitter.EmitterColorData.EmitterColor.Alpha0Animation.AnimationTable.Clear();
                newEmitter.EmitterColorData.EmitterColor.Alpha1Animation.AnimationTable.Clear();

                // アルファ - アニメーションテーブル.
                for (int i = 0; i < alpha.KeyFrames.Count; ++i)
                {
                    NWCore.DataModel.FloatAnimTableKeyFrameData keyA = alpha.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;

                    newEmitter.EmitterColorData.EmitterColor.Alpha0Animation.AnimationTable.AddKeyFrame(
                        keyA.KeyFrame,
                        keyA.Value,
                        1.0f,
                        1.0f,
                        1.0f,
                        false);

                    newEmitter.EmitterColorData.EmitterColor.Alpha1Animation.AnimationTable.AddKeyFrame(
                        keyA.KeyFrame,
                        keyA.Value,
                        1.0f,
                        1.0f,
                        1.0f,
                        false);
                }
            }
        }

        #endregion

        #region 放出

        /// <summary>
        /// 【エミッタ】放出タブの変換処理です.
        /// </summary>
        /// <param name="oldEmitter">旧型式.</param>
        /// <param name="newEmitter">新形式.</param>
        private void ConvertEmitterEmissionTab(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            //********************************
            //  放出タイミング
            //********************************
            // ワンタイム.
            newEmitter.EmitterEmissionData.EmitterEmissionTimingData.EnableOnce = oldEmitter.UI_isOneTime;

            // 放出開始フレーム.
            newEmitter.EmitterEmissionData.EmitterEmissionTimingData.EmitStartFrame = (uint)oldEmitter.StartFrame;

            // 放出時間.
            newEmitter.EmitterEmissionData.EmitterEmissionTimingData.EmitDuration = (uint)oldEmitter.UI_emitTime;

            // 放出間隔タイプ.
            newEmitter.EmitterEmissionData.EmitterEmissionTimingData.EmitDistUnitType = (oldEmitter.EmitDistEnabled) ? 1 : 0;

            //----- 放出間隔サブグループ -----
            // 放出レート.
            newEmitter.EmitterEmissionData.EmitterEmissionTimingData.EmitTimeEmissionRate = oldEmitter.EmitRate;

            {
                NWCore.DataModel.AnimTableData emit_rate = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.emit_rate);

                // 放出レート - エミッタ時間アニメ
                newEmitter.EmitterEmissionData.EmitterEmissionTimingData.EmitTimeRateAnimation.EnableAnimation = emit_rate.IsEnabled;

                newEmitter.EmitterEmissionData.EmitterEmissionTimingData.EmitTimeRateAnimation.LoopMode = (emit_rate.IsLoop) ? 1 : 0;

                switch (emit_rate.InterpolationType)
                {
                    // リニア.
                    case DataModel.AnimInterpolationTypes.Linear:
                        {
                            newEmitter.EmitterEmissionData.EmitterEmissionTimingData.EmitTimeRateAnimation.InterpolationMode = 0;
                        }
                        break;

                    // スムース.
                    case DataModel.AnimInterpolationTypes.Smooth:
                        {
                            newEmitter.EmitterEmissionData.EmitterEmissionTimingData.EmitTimeRateAnimation.InterpolationMode = 1;
                        }
                        break;
                }

                // テーブルを一端クリア.
                newEmitter.EmitterEmissionData.EmitterEmissionTimingData.EmitTimeRateAnimation.AnimationTable.Clear();

                // 放出レート - アニメーションテーブル.
                for (int i = 0; i < emit_rate.KeyFrames.Count; ++i)
                {
                    NWCore.DataModel.FloatAnimTableKeyFrameData key = emit_rate.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;

                    newEmitter.EmitterEmissionData.EmitterEmissionTimingData.EmitTimeRateAnimation.AnimationTable.AddKeyFrame(
                        key.KeyFrame,
                        key.Value,
                        0.0f,
                        0.0f,
                        0.0f,
                        false);
                }
            }

            // 放出レートランダム
            newEmitter.EmitterEmissionData.EmitterEmissionTimingData.EmitTimeEmissionRateRandom = oldEmitter.EmitRateRand;


            // 放出間隔.
            newEmitter.EmitterEmissionData.EmitterEmissionTimingData.EmitTimeDistUnit = oldEmitter.LifeStep;

            // 放出間隔（距離放出）
            newEmitter.EmitterEmissionData.EmitterEmissionTimingData.EmitDistDistUnit = oldEmitter.EmitDistUnit;

            // 放出間隔ランダム(加算フレーム)
            newEmitter.EmitterEmissionData.EmitterEmissionTimingData.EmitTimeDistUnitRandom = oldEmitter.LifeStepRnd;

            // 1フレームあたりの最大移動距離.
            newEmitter.EmitterEmissionData.EmitterEmissionTimingData.EmitDistMaxDistPerFrame = oldEmitter.EmitDistMax;

            // 1フレームあたりの最小移動距離.
            newEmitter.EmitterEmissionData.EmitterEmissionTimingData.EmitDistMinDistPerFrame = oldEmitter.EmitDistMin;

            // 移動距離切り捨ての閾値.
            newEmitter.EmitterEmissionData.EmitterEmissionTimingData.EmitDistMargin = oldEmitter.EmitDistMargin;


            //********************************
            //  放出時の位置と初速.
            //********************************
            // ランダム位置.
            newEmitter.EmitterEmitterData.EmitterEmitterShapeData.RandomPosition = oldEmitter.InitPosRand;

            // 全方向初速.
            newEmitter.EmitterEmissionData.EmitterEmissionVelocityData.OmnidirectionalVelocity = oldEmitter.FigureVelocity;

            {
                NWCore.DataModel.AnimTableData all_directional_vel = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.all_directional_vel);

                // 全方向初速 - エミッタ時間アニメ.
                newEmitter.EmitterEmissionData.EmitterEmissionVelocityData.OmnidirectionalVelocityAnimation.EnableAnimation = all_directional_vel.IsEnabled;

                newEmitter.EmitterEmissionData.EmitterEmissionVelocityData.OmnidirectionalVelocityAnimation.LoopMode = (all_directional_vel.IsLoop) ? 1 : 0;

                switch (all_directional_vel.InterpolationType)
                {
                    // リニア.
                    case DataModel.AnimInterpolationTypes.Linear:
                        {
                            newEmitter.EmitterEmissionData.EmitterEmissionVelocityData.OmnidirectionalVelocityAnimation.InterpolationMode = 0;
                        }
                        break;

                    // スムース.
                    case DataModel.AnimInterpolationTypes.Smooth:
                        {
                            newEmitter.EmitterEmissionData.EmitterEmissionVelocityData.OmnidirectionalVelocityAnimation.InterpolationMode = 1;
                        }
                        break;
                }


                newEmitter.EmitterEmissionData.EmitterEmissionVelocityData.OmnidirectionalVelocityAnimation.AnimationTable.Clear();

                // 全方向初速 - アニメーションテーブル.
                for (int i = 0; i < all_directional_vel.KeyFrames.Count; ++i)
                {
                    NWCore.DataModel.FloatAnimTableKeyFrameData key = all_directional_vel.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;

                    newEmitter.EmitterEmissionData.EmitterEmissionVelocityData.OmnidirectionalVelocityAnimation.AnimationTable.AddKeyFrame(
                        key.KeyFrame,
                        key.Value,
                        0.0f,
                        0.0f,
                        0.0f,
                        false);
                }
            }

            // 指定方向初速.
            newEmitter.EmitterEmissionData.EmitterEmissionVelocityData.OrientedVelocity = oldEmitter.EmitterVelocity;

            {
                NWCore.DataModel.AnimTableData directional_vel = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.directional_vel);

                // 指定方向初速 - エミッタ時間アニメ.
                newEmitter.EmitterEmissionData.EmitterEmissionVelocityData.OrientedDirectionVelocityAnimation.EnableAnimation = directional_vel.IsEnabled;

                newEmitter.EmitterEmissionData.EmitterEmissionVelocityData.OrientedDirectionVelocityAnimation.LoopMode = (directional_vel.IsLoop) ? 1 : 0;

                switch (directional_vel.InterpolationType)
                {
                    // リニア.
                    case DataModel.AnimInterpolationTypes.Linear:
                        {
                            newEmitter.EmitterEmissionData.EmitterEmissionVelocityData.OrientedDirectionVelocityAnimation.InterpolationMode = 0;
                        }
                        break;

                    // スムース.
                    case DataModel.AnimInterpolationTypes.Smooth:
                        {
                            newEmitter.EmitterEmissionData.EmitterEmissionVelocityData.OrientedDirectionVelocityAnimation.InterpolationMode = 1;
                        }
                        break;
                }

                newEmitter.EmitterEmissionData.EmitterEmissionVelocityData.OrientedDirectionVelocityAnimation.AnimationTable.Clear();

                // 指定方向初速 - アニメーションテーブル.
                for (int i = 0; i < directional_vel.KeyFrames.Count; ++i)
                {
                    NWCore.DataModel.FloatAnimTableKeyFrameData key = directional_vel.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;

                    newEmitter.EmitterEmissionData.EmitterEmissionVelocityData.OrientedDirectionVelocityAnimation.AnimationTable.AddKeyFrame(
                        key.KeyFrame,
                        key.Value,
                        0.0f,
                        0.0f,
                        0.0f,
                        false);
                }
            }

            // 指定方向.
            newEmitter.EmitterEmissionData.EmitterEmissionVelocityData.OrientedDirection.X = oldEmitter.EmitterVelDir.X;
            newEmitter.EmitterEmissionData.EmitterEmissionVelocityData.OrientedDirection.Y = oldEmitter.EmitterVelDir.Y;
            newEmitter.EmitterEmissionData.EmitterEmissionVelocityData.OrientedDirection.Z = oldEmitter.EmitterVelDir.Z;

            // 指定方向拡散角度.
            newEmitter.EmitterEmissionData.EmitterEmissionVelocityData.OrientedDirectionalDiffusionAngle = oldEmitter.EmitterVelDirAngle;

            // Y軸拡散初速.
            newEmitter.EmitterEmissionData.EmitterEmissionVelocityData.YaxisDiffusionVelocity = oldEmitter.YAxisDiffusionVel;

            // 拡散初速.
            newEmitter.EmitterEmissionData.EmitterEmissionVelocityData.DiffusionVelocity.X = oldEmitter.SpreadVec.X;
            newEmitter.EmitterEmissionData.EmitterEmissionVelocityData.DiffusionVelocity.Y = oldEmitter.SpreadVec.Y;
            newEmitter.EmitterEmissionData.EmitterEmissionVelocityData.DiffusionVelocity.Z = oldEmitter.SpreadVec.Z;

            // 初速度のランダム(%)
            newEmitter.EmitterEmissionData.EmitterEmissionVelocityData.VelocityRandom = oldEmitter.UI_initVelRnd;

            // エミッタ速度継承率
            newEmitter.EmitterEmissionData.EmitterEmissionGravityData.VelocityInheritRate = oldEmitter.UI_emitVelInherit;

            // 空気抵抗.
            newEmitter.EmitterEmissionData.EmitterEmissionGravityData.AirRegist = oldEmitter.AirRegist;


            //********************************
            //  重力.
            //********************************
            // 座標系 [重力をワールド座標系で適用する]
            newEmitter.EmitterEmissionData.EmitterEmissionGravityData.EnableWorldCoordinate = oldEmitter.IsWorldField;

            // 大きさ
            newEmitter.EmitterEmissionData.EmitterEmissionGravityData.GravityVolume = oldEmitter.GravityVolume;

            {
                NWCore.DataModel.AnimTableData gravity = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.gravity);

                // 大きさ - エミッタ時間アニメ
                newEmitter.EmitterEmissionData.EmitterEmissionGravityData.GravityAnimation.EnableAnimation = gravity.IsEnabled;

                newEmitter.EmitterEmissionData.EmitterEmissionGravityData.GravityAnimation.LoopMode = (gravity.IsLoop) ? 1 : 0;

                switch (gravity.InterpolationType)
                {
                    // リニア.
                    case DataModel.AnimInterpolationTypes.Linear:
                        {
                            newEmitter.EmitterEmissionData.EmitterEmissionGravityData.GravityAnimation.InterpolationMode = 0;
                        }
                        break;

                    // スムース.
                    case DataModel.AnimInterpolationTypes.Smooth:
                        {
                            newEmitter.EmitterEmissionData.EmitterEmissionGravityData.GravityAnimation.InterpolationMode = 1;
                        }
                        break;
                }

                // テーブルを一端クリア.
                newEmitter.EmitterEmissionData.EmitterEmissionGravityData.GravityAnimation.AnimationTable.Clear();

                // 放出レート - アニメーションテーブル.
                for (int i = 0; i < gravity.KeyFrames.Count; ++i)
                {
                    NWCore.DataModel.FloatAnimTableKeyFrameData key = gravity.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;

                    newEmitter.EmitterEmissionData.EmitterEmissionGravityData.GravityAnimation.AnimationTable.AddKeyFrame(
                        key.KeyFrame,
                        key.Value,
                        0.0f,
                        0.0f,
                        0.0f,
                        false);
                }
            }

            // 向き.
            newEmitter.EmitterEmissionData.EmitterEmissionGravityData.GravityDir.X = oldEmitter.GravityDir.X;
            newEmitter.EmitterEmissionData.EmitterEmissionGravityData.GravityDir.Y = oldEmitter.GravityDir.Y;
            newEmitter.EmitterEmissionData.EmitterEmissionGravityData.GravityDir.Z = oldEmitter.GravityDir.Z;
        }

        #endregion

        #region パーティクル

        /// <summary>
        /// 【エミッタ】パーティクルタブの変換処理です.
        /// </summary>
        /// <param name="oldEmitter">旧型式.</param>
        /// <param name="newEmitter">新形式.</param>
        private void ConvertEmitterParticleTab(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            //*******************************
            //  寿命設定.
            //*******************************
            // 寿命無限.
            newEmitter.EmitterParticleData.EmitterParticleLifeData.EnableInfinityLife = (oldEmitter.AnimEditData.UI_lifeInfinit == 0) ? false : true;

            // 寿命.
            newEmitter.EmitterParticleData.EmitterParticleLifeData.Life = oldEmitter.AnimEditData.UI_life;

            {
                NWCore.DataModel.AnimTableData ptcl_life = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.ptcl_life);

                // 寿命 - エミッタ時間アニメ
                newEmitter.EmitterParticleData.EmitterParticleLifeData.LifeAnimation.EnableAnimation = ptcl_life.IsEnabled;

                newEmitter.EmitterParticleData.EmitterParticleLifeData.LifeAnimation.LoopMode = (ptcl_life.IsLoop) ? 1 : 0;

                switch (ptcl_life.InterpolationType)
                {
                    case DataModel.AnimInterpolationTypes.Linear:
                        {
                            newEmitter.EmitterParticleData.EmitterParticleLifeData.LifeAnimation.InterpolationMode = 0;
                        }
                        break;

                    case DataModel.AnimInterpolationTypes.Smooth:
                        {
                            newEmitter.EmitterParticleData.EmitterParticleLifeData.LifeAnimation.InterpolationMode = 1;
                        }
                        break;
                }

                newEmitter.EmitterParticleData.EmitterParticleLifeData.LifeAnimation.AnimationTable.Clear();

                // 寿命 - アニメーションテーブル.
                for (int i = 0; i < ptcl_life.KeyFrames.Count; ++i)
                {
                    NWCore.DataModel.IntAnimTableKeyFrameData key = ptcl_life.KeyFrames[i] as NWCore.DataModel.IntAnimTableKeyFrameData;

                    if (key != null)
                    {
                        newEmitter.EmitterParticleData.EmitterParticleLifeData.LifeAnimation.AnimationTable.AddKeyFrame(
                            key.KeyFrame,
                            key.Value,
                            0.0f,
                            0.0f,
                            0.0f,
                            false);
                    }
                }
            }

            // 寿命ランダム(%) (寿命から減らす量)
            newEmitter.EmitterParticleData.EmitterParticleLifeData.LifeRandom = oldEmitter.AnimEditData.UI_lifeRnd;

            //********************************
            // 運動量ランダム.
            //********************************
            // ランダム値.
            newEmitter.EmitterEmissionData.EmitterEmissionGravityData.MomentumRandom =
                MathUtility.Clamp(oldEmitter.DynamicsRandom, 0.0f, 1.0f);

            //********************************
            // パーティクル形状.
            //********************************
            // 形状.
            newEmitter.EmitterParticleData.EmitterParticleShapeData.ShapeType = (oldEmitter.MeshType != DataModel.MeshType.Stripe) ? (int)oldEmitter.MeshType : 0;

            // トリミング.
            newEmitter.EmitterParticleData.EmitterParticleShapeData.EnableTrimming = oldEmitter.TextureTrimming;

            // ファイルパス.
            newEmitter.EmitterParticleData.EmitterParticleShapeData.PrimitiveFilePath = GetFileName(oldEmitter.BillboardPrimitiveFileSource);

            // プリミティブスケールZ
            newEmitter.EmitterParticleData.EmitterParticleShapeData.PrimitiveScaleZType = (int)oldEmitter.PrimitiveScaleType;

            if (oldEmitter.MeshType == DataModel.MeshType.Stripe)
            {
                // ストライプタイプ.
                switch (oldEmitter.StripeType)
                {
                    // 履歴式.
                    case DataModel.StripeType.Stripe:
                        {
                            EffectMaker.DataModel.DataModels.StripeHistoryData stripeData = new EffectMaker.DataModel.DataModels.StripeHistoryData();
                            EffectMaker.DataModel.DataModels.StripeHistoryUserData userData = new EffectMaker.DataModel.DataModels.StripeHistoryUserData();

                            // 計算方式.
                            userData.StripeType = (int)oldEmitter.StripeData.Type;

                            // エミッタに完全追従.
                            userData.FollowEmitter = oldEmitter.StripeData.FollowEmitter;

                            // オプション.
                            userData.Option = oldEmitter.StripeData.HLType;

                            // テクスチャリング.
                            userData.TexCoordType = (int)oldEmitter.StripeData.TexCoordType;

                            // 分割数.
                            userData.NumDivisions = oldEmitter.StripeData.NumDivisions;

                            // 履歴数(ストライプの長さ)
                            userData.StripeLength = oldEmitter.StripeData.NumHistory;

                            // 履歴ポリゴン化の間隔.
                            userData.SliceLength = oldEmitter.StripeData.HistoryStep;

                            // 先端α.
                            userData.StartSliceAlpha = oldEmitter.StripeData.StartAlpha;

                            // 末端α.
                            userData.EndSliceAlpha = oldEmitter.StripeData.EndAlpha;

                            // 履歴補間パラメータ.
                            userData.HistInterpolation = oldEmitter.StripeData.HistoryInterpolation;

                            // 方向補間率.
                            userData.DirInterpolation = oldEmitter.StripeData.DirInterpolation;

                            stripeData.UserPageData = userData;
                            newEmitter.StripeData = stripeData;
                        }
                        break;

                    // 連結式.
                    case DataModel.StripeType.ComplexStripe:
                        {
                            EffectMaker.DataModel.DataModels.StripeSuperData stripeData = new EffectMaker.DataModel.DataModels.StripeSuperData();
                            EffectMaker.DataModel.DataModels.StripeSuperUserData userData = new EffectMaker.DataModel.DataModels.StripeSuperUserData();

                            // 計算方式.
                            userData.StripeType = (int)oldEmitter.StripeData.Type;

                            // エミッタに完全追従.
                            userData.FollowEmitter = oldEmitter.StripeData.FollowEmitter;

                            // オプション.
                            userData.Option = oldEmitter.StripeData.HLType;

                            // テクスチャリング.
                            userData.TexCoordType = (int)oldEmitter.StripeData.TexCoordType;

                            // 分割数.
                            userData.NumDivisions = oldEmitter.StripeData.NumDivisions;

                            // 先端α.
                            userData.StartSliceAlpha = oldEmitter.StripeData.StartAlpha;

                            // 末端α.
                            userData.EndSliceAlpha = oldEmitter.StripeData.EndAlpha;

                            // 履歴補間パラメータ.
                            userData.HistInterpolation = oldEmitter.StripeData.HistoryInterpolation;

                            // 方向補間率.
                            userData.DirInterpolation = oldEmitter.StripeData.DirInterpolation;

                            // 接続タイプ
                            userData.ConnectionType = (int)oldEmitter.StripeData.ConnectType;

                            stripeData.UserPageData = userData;
                            newEmitter.StripeData = stripeData;
                        }
                        break;
                }
            }

            // パーティクルタイプ.
            switch (oldEmitter.ParticleType)
            {
                case NWCore.DataModel.BillboardType.Billboard:    //  普通のビルボード
                    {
                        newEmitter.EmitterParticleData.EmitterParticleShapeData.ParticleType = 0;
                    }
                    break;
                case NWCore.DataModel.BillboardType.PolygonXY:    //  XY平面のポリゴン
                    {
                        newEmitter.EmitterParticleData.EmitterParticleShapeData.ParticleType = 3;
                    }
                    break;
                case NWCore.DataModel.BillboardType.PolygonXZ:    //  XZ平面のポリゴン
                    {
                        newEmitter.EmitterParticleData.EmitterParticleShapeData.ParticleType = 4;
                    }
                    break;
                case NWCore.DataModel.BillboardType.VelLook:     //  速度方向を向くビルボード
                    {
                        newEmitter.EmitterParticleData.EmitterParticleShapeData.ParticleType = 5;
                    }
                    break;
                case NWCore.DataModel.BillboardType.VelLookPolygon:    //  速度方向を向くポリゴン
                    {
                        newEmitter.EmitterParticleData.EmitterParticleShapeData.ParticleType = 6;
                    }
                    break;
                case NWCore.DataModel.BillboardType.YAxisBillboard:    //  Y軸ビルボード
                    {
                        newEmitter.EmitterParticleData.EmitterParticleShapeData.ParticleType = 2;
                    }
                    break;
                case NWCore.DataModel.BillboardType.ComplexBillboard:    //  高機能ビルボード
                    {
                        newEmitter.EmitterParticleData.EmitterParticleShapeData.ParticleType = 1;
                    }
                    break;
            }

            // パーティクル中心.
            newEmitter.EmitterParticleData.EmitterParticleShapeData.Scale.X = oldEmitter.UI_rotBasis.X;
            newEmitter.EmitterParticleData.EmitterParticleShapeData.Scale.Y = oldEmitter.UI_rotBasis.Y;


            //----- オフセットサブグループ -----
            // 処理方法
            switch (oldEmitter.ParticleShapeOffsetType)
            {
                case DataModel.ParticleShapeOffsetTypes.CameraDir:
                    {
                        newEmitter.EmitterParticleData.EmitterParticleShapeData.CameraOffsetProcessType = 1;
                    }
                    break;

                case DataModel.ParticleShapeOffsetTypes.DepthDir:
                    {
                        newEmitter.EmitterParticleData.EmitterParticleShapeData.CameraOffsetProcessType = 0;
                    }
                    break;
            }

            // オフセット値.
            newEmitter.EmitterParticleData.EmitterParticleShapeData.CameraOffsetValue = oldEmitter.BillboardZOffset;

            //*******************************
            //  揺らぎ
            //*******************************
            if (oldEmitter.FluctuationFlag.Enable == true && oldEmitter.Type == DataModel.EmitterType.Complex)
            {
                //　アルファ適応
                newEmitter.EmitterParticleData.EmitterParticleFluctutaionData.ApplyAlpha = oldEmitter.FluctuationFlag.ApplyAlpha;

                // スケール適応
                newEmitter.EmitterParticleData.EmitterParticleFluctutaionData.ApplyScale = oldEmitter.FluctuationFlag.ApplyScale;

                // 波形タイプ
                newEmitter.EmitterParticleData.EmitterParticleFluctutaionData.WaveType = (int)oldEmitter.FluctuationFlag.WaveType;

                // スケールYを個別に設定
                newEmitter.EmitterParticleData.EmitterParticleFluctutaionData.ApplyScaleY = oldEmitter.FluctuationFlag.EnableScaleY;

                // 振幅
                var amplitude = new EffectMaker.Foundation.Primitives.Vector2f();
                amplitude.X = oldEmitter.FluctuationData.ScaleX;
                amplitude.Y = oldEmitter.FluctuationData.ScaleY;
                newEmitter.EmitterParticleData.EmitterParticleFluctutaionData.Amplitude = amplitude;

                // 周期
                var frequency = new EffectMaker.Foundation.Primitives.Vector2i();
                frequency.X = oldEmitter.FluctuationData.FrequencyX;
                frequency.Y = oldEmitter.FluctuationData.FrequencyY;
                newEmitter.EmitterParticleData.EmitterParticleFluctutaionData.Frequency = frequency;

                // 初期位相
                var initPhase = new EffectMaker.Foundation.Primitives.Vector2i();
                initPhase.X = oldEmitter.FluctuationData.InitPhaseX;
                initPhase.Y = oldEmitter.FluctuationData.InitPhaseY;
                newEmitter.EmitterParticleData.EmitterParticleFluctutaionData.InitPhase = initPhase;

                // 位相ランダム
                var phaseRandom = new EffectMaker.Foundation.Primitives.Vector2i();
                phaseRandom.X = (int)oldEmitter.FluctuationData.PhaseRangeX;
                phaseRandom.Y = (int)oldEmitter.FluctuationData.PhaseRangeY;
                newEmitter.EmitterParticleData.EmitterParticleFluctutaionData.PhaseRandom = phaseRandom;
            }
            //*******************************
            //  レンダーステート.
            //*******************************
            // ブレンドタイプ.
            switch (oldEmitter.BlendType)
            {
                case DataModel.BlendType.Normal:
                    {
                        newEmitter.EmitterBasicSettingData.EmitterBasicRenderData.BlendType = 0;
                    }
                    break;

                case DataModel.BlendType.Add:
                    {
                        newEmitter.EmitterBasicSettingData.EmitterBasicRenderData.BlendType = 1;
                    }
                    break;

                case DataModel.BlendType.Sub:
                    {
                        newEmitter.EmitterBasicSettingData.EmitterBasicRenderData.BlendType = 2;
                    }
                    break;

                case DataModel.BlendType.Multiply:
                    {
                        newEmitter.EmitterBasicSettingData.EmitterBasicRenderData.BlendType = 3;
                    }
                    break;

                case DataModel.BlendType.Screen:
                    {
                        newEmitter.EmitterBasicSettingData.EmitterBasicRenderData.BlendType = 4;
                    }
                    break;
            }

            // Zバッファ, αテスト.
            newEmitter.EmitterBasicSettingData.EmitterBasicRenderData.ZbufferAlphaTestType = (int)oldEmitter.ZBufATestType;

            // 表示面.
            newEmitter.EmitterBasicSettingData.EmitterBasicRenderData.DisplaySurface = (int)oldEmitter.DisplaySide;
        }

        #endregion

        #region コンバイナ

        /// <summary>
        /// 【エミッタ】コンバイナタブの変換処理です.
        /// </summary>
        /// <param name="oldEmitter">旧形式.</param>
        /// <param name="newEmitter">新形式</param>
        private void ConvertEmitterCombinerTab(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            //***********************************
            //  パーティクルシェーダ.
            //***********************************
            // シェーダタイプ.
            newEmitter.EmitterCombinerData.EmitterCombinerShaderData.ShaderType = (int)oldEmitter.ShaderType;

            //----- 屈折/歪みパーティクルサブグループ -----
            // 係数.
            newEmitter.EmitterCombinerData.EmitterCombinerShaderData.Coefficient.X = oldEmitter.ShaderParams01.X;
            newEmitter.EmitterCombinerData.EmitterCombinerShaderData.Coefficient.Y = oldEmitter.ShaderParams01.Y;

            // 係数アニメ.
            // XXX = oldEmitter.ShaderParamAnim;
            newEmitter.EmitterCombinerData.EmitterCombinerShaderData.CoefficientAnimationTable.Clear();

            // 時間の重なりや優先する値の順番を考えつつコンバート
            var key = new EffectMaker.Foundation.Primitives.ColorRgba(1.0f, 1.0f, 1.0f, 1.0f);
            if (oldEmitter.AnimShaderTime3 == 0)
            {
                // 0フレに3つ重なるパターン
                key.A = oldEmitter.AnimShaderParam2;
                newEmitter.EmitterCombinerData.EmitterCombinerShaderData.CoefficientAnimationTable.AddKeyFrame(0, key, false);
                newEmitter.EmitterCombinerData.EmitterCombinerShaderData.CoefficientAnimationTable.AddKeyFrame(1, key, false);
                newEmitter.EmitterCombinerData.EmitterCombinerShaderData.CoefficientAnimationTable.AddKeyFrame(2, key, false);
                key.A = oldEmitter.AnimShaderParam3;
                newEmitter.EmitterCombinerData.EmitterCombinerShaderData.CoefficientAnimationTable.AddKeyFrame(100, key, false);
            }
            else if (oldEmitter.AnimShaderTime2 == 0)
            {
                // 0フレに2つ重なるパターン
                key.A = oldEmitter.AnimShaderParam2;
                newEmitter.EmitterCombinerData.EmitterCombinerShaderData.CoefficientAnimationTable.AddKeyFrame(0, key, false);
                newEmitter.EmitterCombinerData.EmitterCombinerShaderData.CoefficientAnimationTable.AddKeyFrame(1, key, false);
                if (oldEmitter.AnimShaderTime3 == 1)
                {
                    // 2キーが1フレで隣接していたら2フレにずらす
                    newEmitter.EmitterCombinerData.EmitterCombinerShaderData.CoefficientAnimationTable.AddKeyFrame(2, key, false);
                    key.A = oldEmitter.AnimShaderParam3;
                    newEmitter.EmitterCombinerData.EmitterCombinerShaderData.CoefficientAnimationTable.AddKeyFrame(100, key, false);
                }
                else if (oldEmitter.AnimShaderTime3 == 100)
                {
                    // 2キーが100フレで重なってたら99フレにずらす
                    newEmitter.EmitterCombinerData.EmitterCombinerShaderData.CoefficientAnimationTable.AddKeyFrame(99, key, false);
                    newEmitter.EmitterCombinerData.EmitterCombinerShaderData.CoefficientAnimationTable.AddKeyFrame(100, key, false);
                }
                else
                {
                    // 重なりがなければそのまま追加
                    newEmitter.EmitterCombinerData.EmitterCombinerShaderData.CoefficientAnimationTable.AddKeyFrame(oldEmitter.AnimShaderTime3, key, false);
                    key.A = oldEmitter.AnimShaderParam3;
                    newEmitter.EmitterCombinerData.EmitterCombinerShaderData.CoefficientAnimationTable.AddKeyFrame(100, key, false);
                }
            }
            else
            {
                // 0フレには重ならないパターン
                key.A = oldEmitter.AnimShaderParamInit;
                newEmitter.EmitterCombinerData.EmitterCombinerShaderData.CoefficientAnimationTable.AddKeyFrame(0, key, false);

                key.A = oldEmitter.AnimShaderParam2;
                if (oldEmitter.AnimShaderTime2 == 100)
                {
                    // 100フレに3つ重なるパターン
                    newEmitter.EmitterCombinerData.EmitterCombinerShaderData.CoefficientAnimationTable.AddKeyFrame(98, key, false);
                    newEmitter.EmitterCombinerData.EmitterCombinerShaderData.CoefficientAnimationTable.AddKeyFrame(99, key, false);
                    newEmitter.EmitterCombinerData.EmitterCombinerShaderData.CoefficientAnimationTable.AddKeyFrame(100, key, false);
                }
                else if (oldEmitter.AnimShaderTime3 == 100)
                {
                    // 100フレに2つ重なるパターン
                    if (oldEmitter.AnimShaderTime2 == 99)
                    {
                        // 2キー目が99フレだったら98フレにずらす
                        newEmitter.EmitterCombinerData.EmitterCombinerShaderData.CoefficientAnimationTable.AddKeyFrame(98, key, false);
                    }
                    else
                    {
                        // それ以外ならそのまま追加
                        newEmitter.EmitterCombinerData.EmitterCombinerShaderData.CoefficientAnimationTable.AddKeyFrame(oldEmitter.AnimShaderTime2, key, false);
                    }

                    newEmitter.EmitterCombinerData.EmitterCombinerShaderData.CoefficientAnimationTable.AddKeyFrame(99, key, false);
                    newEmitter.EmitterCombinerData.EmitterCombinerShaderData.CoefficientAnimationTable.AddKeyFrame(100, key, false);
                }
                else
                {
                    // 両端には重ならないパターン
                    if (oldEmitter.AnimShaderTime2 != oldEmitter.AnimShaderTime3)
                    {
                        // 時間が異なるならそのまま追加
                        newEmitter.EmitterCombinerData.EmitterCombinerShaderData.CoefficientAnimationTable.AddKeyFrame(oldEmitter.AnimShaderTime2, key, false);
                        newEmitter.EmitterCombinerData.EmitterCombinerShaderData.CoefficientAnimationTable.AddKeyFrame(oldEmitter.AnimShaderTime3, key, false);
                    }
                    else
                    {
                        // 99フレで重なっていたら前ずらし、それ以外なら後ずらし
                        if (oldEmitter.AnimShaderTime2 == 99)
                        {
                            newEmitter.EmitterCombinerData.EmitterCombinerShaderData.CoefficientAnimationTable.AddKeyFrame(98, key, false);
                            newEmitter.EmitterCombinerData.EmitterCombinerShaderData.CoefficientAnimationTable.AddKeyFrame(99, key, false);
                        }
                        else
                        {
                            newEmitter.EmitterCombinerData.EmitterCombinerShaderData.CoefficientAnimationTable.AddKeyFrame(oldEmitter.AnimShaderTime2, key, false);
                            newEmitter.EmitterCombinerData.EmitterCombinerShaderData.CoefficientAnimationTable.AddKeyFrame(oldEmitter.AnimShaderTime2 + 1, key, false);
                        }
                    }

                    key.A = oldEmitter.AnimShaderParam3;
                    newEmitter.EmitterCombinerData.EmitterCombinerShaderData.CoefficientAnimationTable.AddKeyFrame(100, key, false);
                }
            }

            // アルファ値を判定.
            newEmitter.EmitterCombinerData.EmitterCombinerShaderData.ParticleApplyAlphaValue = oldEmitter.UI_ApplyAlphaToRefraction;


            //**********************************
            // アルファ処理.
            //**********************************
            //----- ソフトパーティクルサブグループ. -----
            // 処理方法.
            switch (oldEmitter.SoftParticleProcessingSettings)
            {
                case NWCore.DataModel.SoftParticleProcessingSettings.Disable:
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.SoftParticleProcessType = 0;
                    break;
                case NWCore.DataModel.SoftParticleProcessingSettings.Vertex:
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.SoftParticleProcessType = 1;
                    break;
                case NWCore.DataModel.SoftParticleProcessingSettings.Fragment:
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.SoftParticleProcessType = 2;
                    break;
                case NWCore.DataModel.SoftParticleProcessingSettings.LensFlare:
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.SoftParticleProcessType = 3;
                    break;
            }
         //   newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.SoftParticleProcessType = (int)oldEmitter.SoftParticleProcessingSettings;

            // フェード開始α値オフセット
            newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.AlphaFadeStartAlphaOffset = oldEmitter.SoftParticleAlphaOffset;

            // サンプリング半径.
            newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.AlphaSamplingRadius = oldEmitter.SoftParticleSampleParam;

            // フェード開始距離.
            newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.FeedStartDistance = oldEmitter.SoftParticleFadeDistance;

            // ボリューム値.
            newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.VolumeValue = oldEmitter.SoftParticleVolumeParam;

            //----- 近距離アルファサブグループ -----
            // 有効.
            newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.EnableNearAlpha = oldEmitter.NearAlphaEnabled;

            // アルファが0になる距離.
            newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.NearAlphaZeroDistance = oldEmitter.NearAlphaMin;

            // アルファが1になる距離.
            newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.NearAlphaOneDistance = oldEmitter.NearAlphaMax;

            //----- 遠距離アルファサブグループ -----
            // 有効.
            newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.EnableFarAlpha = oldEmitter.FarAlphaEnabled;

            // アルファが0になる距離.
            newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.FarAlphaZeroDistance = oldEmitter.FarAlphaMin;

            // アルファが1になる距離.
            newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.FarAlphaOneDistance = oldEmitter.FarAlphaMax;

            //----- フレネルアルファグループ -----
            // 処理方法.
            switch (oldEmitter.FresnelAlphaType)
            {
                case DataModel.FresnelAlphaTypes.Disable:
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.FresnelAlphaProcessType = 0;
                    break;
                case DataModel.FresnelAlphaTypes.Vertex:
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.FresnelAlphaProcessType = 1;
                    break;
                case DataModel.FresnelAlphaTypes.Fragment:
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.FresnelAlphaProcessType = 2;
                    break;
            }
   //         newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.FresnelAlphaProcessType = (int)oldEmitter.FresnelAlphaType;

            // アルファが最小になる角度.
            newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.MinimumAlphaAngle = oldEmitter.FresnelAlphaMin;

            // アルファが最大になる角度.
            newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.MaximumAlphaAngle = oldEmitter.FresnelAlphaMax;

            //----- デカール -----
            // 有効.
            newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.EnableDecal = oldEmitter.DecalEnabled;

            // アルファが1になる範囲.
            newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.DecalAlphaOneDistance = oldEmitter.DecalAlphaMin;


            //*******************************
            //  カラーコンバイナ.
            //*******************************
            // テクスチャ0 - RGB/定数1.0
            newEmitter.EmitterCombinerData.EmitterCombinerEquationData.ColorTexture0Channel = (int)oldEmitter.UI_combinerTextureColorUsage;

            // テクスチャ1 - 上段との合成.
            newEmitter.EmitterCombinerData.EmitterCombinerEquationData.ColorTexture1BlendType = (int)oldEmitter.UI_combinerSubTextureColorBlend;

            // テクスチャ1 - RGB/定数1.0
            newEmitter.EmitterCombinerData.EmitterCombinerEquationData.ColorTexture1Channel = (int)oldEmitter.UI_combinerSubTextureColorUsage;

            // テクスチャ2 - 上段との合成.
            newEmitter.EmitterCombinerData.EmitterCombinerEquationData.ColorTexture2BlendType = (int)oldEmitter.UI_combinerTexture2ColorBlend;

            // テクスチャ2 - RGB/定数1.0
            newEmitter.EmitterCombinerData.EmitterCombinerEquationData.ColorTexture2Channel = (int)oldEmitter.UI_combinerTexture2ColorUsage;

            // カラー計算式.
            newEmitter.EmitterCombinerData.EmitterCombinerEquationData.ColorTextureExpression = (int)oldEmitter.UI_colorType;

            // プリミティブ頂点 - 上段との合成
            newEmitter.EmitterCombinerData.EmitterCombinerEquationData.ColorPrimitiveExpression = (int)oldEmitter.UI_combinerPrimitiveColorBlend;

            // プリミティブ頂点 - RGB/定数1.0
            newEmitter.EmitterCombinerData.EmitterCombinerEquationData.ColorPrimitiveChannel = (int)oldEmitter.UI_combinerPrimitiveColorUsage;


            //********************************
            //  アルファコンバイナ.
            //********************************
            // テクスチャ0 - A/R/定数1.0
            newEmitter.EmitterCombinerData.EmitterCombinerEquationData.AlphaTexture0Channel = (int)oldEmitter.UI_combinerTextureAlphaUsage;

            // テクスチャ1 - 上段との合成.
            newEmitter.EmitterCombinerData.EmitterCombinerEquationData.AlphaTexture1BlendType = (int)oldEmitter.UI_combinerSubTextureAlphaBlend;

            // テクスチャ1 - A/R/定数1.0
            newEmitter.EmitterCombinerData.EmitterCombinerEquationData.AlphaTexture1Channel = (int)oldEmitter.UI_combinerSubTextureAlphaUsage;

            // テクスチャ2 - 上段との合成.
            newEmitter.EmitterCombinerData.EmitterCombinerEquationData.AlphaTexture2BlendType = (int)oldEmitter.UI_combinerTexture2AlphaBlend;

            // テクスチャ2 - A/R/定数1.0
            newEmitter.EmitterCombinerData.EmitterCombinerEquationData.AlphaTexture2Channel = (int)oldEmitter.UI_combinerTexture2AlphaUsage;

            // プリミティブ頂点 - 上段との合成
            newEmitter.EmitterCombinerData.EmitterCombinerEquationData.AlphaPrimitiveExpression = (int)oldEmitter.UI_combinerPrimitiveAlphaBlend;

            // プリミティブ頂点 - A/定数1.0
            newEmitter.EmitterCombinerData.EmitterCombinerEquationData.AlphaPrimitiveChannel = (int)oldEmitter.UI_combinerPrimitiveAlphaUsage;

            // アルファ計算式.
            switch (oldEmitter.UI_alphaDstType)
            {
                case NWCore.DataModel.AlphaDst.ModulateTex:
                    newEmitter.EmitterCombinerData.EmitterCombinerEquationData.AlphaExpression = 0;
                    break;
                case NWCore.DataModel.AlphaDst.ModulateTex0AndTex1:
                    newEmitter.EmitterCombinerData.EmitterCombinerEquationData.AlphaExpression = 1;
                    break;
                case NWCore.DataModel.AlphaDst.SubtractFromTex:
                    newEmitter.EmitterCombinerData.EmitterCombinerEquationData.AlphaExpression = 2;
                    break;
                case NWCore.DataModel.AlphaDst.SubtractFromTex0AlphaModulateTex1Alpha:
                    newEmitter.EmitterCombinerData.EmitterCombinerEquationData.AlphaExpression = 3;
                    break;
            }
        }

        #endregion

        #region テクスチャ0

        /// <summary>
        /// 【エミッタ】テクスチャ0タブの変換処理です.
        /// </summary>
        /// <param name="oldEmitter">旧型式</param>
        /// <param name="newEmitter">新形式</param>
        private void ConvertEmitterTexture0Tab(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            //******************************
            //  ファイル.
            //******************************
            // ファイルパス.
            newEmitter.Textures.Texture0.EmitterTextureFileData.FilePath = GetFileName(oldEmitter.TexPatData0.UI_texPatFileName);

            // パターン横分割.
            newEmitter.Textures.Texture0.EmitterTextureFileData.PatternDivU = oldEmitter.TexPatData0.UI_texPatDivX;

            // パターン縦分割.
            newEmitter.Textures.Texture0.EmitterTextureFileData.PatternDivV = oldEmitter.TexPatData0.UI_texPatDivY;

            // ラップ(U)
            newEmitter.Textures.Texture0.EmitterTextureFileData.WrapTypeU = (int)oldEmitter.TexPatData0.UI_texWrapU;

            // ラップ(V)
            newEmitter.Textures.Texture0.EmitterTextureFileData.WrapTypeV = (int)oldEmitter.TexPatData0.UI_texWrapV;

            // U座標ランダム反転.
            newEmitter.Textures.Texture0.EmitterTextureFileData.RandomInvU = oldEmitter.TexPatData0.UI_texIsURandomInv;

            // V座標ランダム反転.
            newEmitter.Textures.Texture0.EmitterTextureFileData.RandomInvV = oldEmitter.TexPatData0.UI_texIsVRandomInv;

            // スフィアマップ
            newEmitter.Textures.Texture0.EmitterTextureFileData.SphericalMap = oldEmitter.TexPatData0.UI_isSphereMap;
            // 繰り返し数.
            newEmitter.Textures.Texture0.EmitterTextureFileData.RepeatCount = (int)oldEmitter.TexPatData0.UI_textureAddressing;

            // フィルタモード.
            newEmitter.Textures.Texture0.EmitterTextureFileData.FilterMode = (int)oldEmitter.TexPatData0.UI_texFilterMode;

            // 有効なミップレベル.
            newEmitter.Textures.Texture0.EmitterTextureFileData.AvailableMipMapLevel = oldEmitter.TexPatData0.UI_availableMipLevel;

            // ミップマップLODバイアス.
            newEmitter.Textures.Texture0.EmitterTextureFileData.MipMapBias = oldEmitter.TexPatData0.UI_mipMapBias;

            //*******************************
            //  テクスチャ座標アニメーション
            //*******************************

            // テクスチャ座標アニメを固定(texture共通設定)
            newEmitter.Textures.CommonSettings.FixedUvAnim = oldEmitter.UseFixedTexUVRandomAnimation;

            // UVスクロールアニメ.
            newEmitter.Textures.Texture0.EmitterTextureUvAnimationData.EnableUvScrollAnimation = oldEmitter.TexScrollAnimEnabled;

            // UV回転アニメ.
            newEmitter.Textures.Texture0.EmitterTextureUvAnimationData.EnableUvRotAnimation = oldEmitter.TexRotateAnimEnabled;

            // UVスケールアニメ.
            newEmitter.Textures.Texture0.EmitterTextureUvAnimationData.EnableUvScaleAnimation = oldEmitter.TexScaleAnimEnabled;

            //----- UVスクロールアニメ -----
            // 加算値.
            newEmitter.Textures.Texture0.EmitterTextureUvAnimationData.UvScrollAddValue.X = oldEmitter.UVScroll.X;
            newEmitter.Textures.Texture0.EmitterTextureUvAnimationData.UvScrollAddValue.Y = oldEmitter.UVScroll.Y;

            // 初期値.
            newEmitter.Textures.Texture0.EmitterTextureUvAnimationData.UvScrollInitialValue.X = oldEmitter.UVScrollInit.X;
            newEmitter.Textures.Texture0.EmitterTextureUvAnimationData.UvScrollInitialValue.Y = oldEmitter.UVScrollInit.Y;

            // 初期値ランダム
            newEmitter.Textures.Texture0.EmitterTextureUvAnimationData.UvScrollInitialRandomValue.X = oldEmitter.UVScrollInitRand.X;
            newEmitter.Textures.Texture0.EmitterTextureUvAnimationData.UvScrollInitialRandomValue.Y = oldEmitter.UVScrollInitRand.Y;

            //----- UV回転 -----
            // 加算値.
            newEmitter.Textures.Texture0.EmitterTextureUvAnimationData.UvRotationAddValue = oldEmitter.UVRot;

            // 初期値.
            newEmitter.Textures.Texture0.EmitterTextureUvAnimationData.UvRotationInitialValue = oldEmitter.UVRotInit;

            // 初期値ランダム.
            newEmitter.Textures.Texture0.EmitterTextureUvAnimationData.UvRotationInitialRandomValue = oldEmitter.UVRotInitRand;

            //----- UVスケール ------
            // 加算値.
            newEmitter.Textures.Texture0.EmitterTextureUvAnimationData.UvScaleAddValue.X = oldEmitter.UVScale.X;
            newEmitter.Textures.Texture0.EmitterTextureUvAnimationData.UvScaleAddValue.Y = oldEmitter.UVScale.Y;

            // 初期値.
            newEmitter.Textures.Texture0.EmitterTextureUvAnimationData.UvScaleInitialValue.X = oldEmitter.UVScaleInit.X;
            newEmitter.Textures.Texture0.EmitterTextureUvAnimationData.UvScaleInitialValue.Y = oldEmitter.UVScaleInit.Y;

            // 初期値ランダム.
            newEmitter.Textures.Texture0.EmitterTextureUvAnimationData.UvScaleInitialRandomValue.X = oldEmitter.UVScaleInitRand.X;
            newEmitter.Textures.Texture0.EmitterTextureUvAnimationData.UvScaleInitialRandomValue.Y = oldEmitter.UVScaleInitRand.Y;


            //********************************
            //  テクスチャパターンアニメーション.
            //********************************

            //----- パターンテーブル -----
            switch (oldEmitter.TexPatData0.UI_texPatAnimMode)
            {
                // ライフフィット.
                case 1:
                    {
                        newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.PatternAnimType = 1;

                        newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.PatternTableCount = (int)oldEmitter.TexPatData0.UI_numUseTexPatTbl;
                        newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.RandomPatternTableCount = 0;
                    }
                    break;

                // クランプ
                case 4:
                    {
                        newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.PatternAnimType = 2;

                        newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.PatternTableCount = (int)oldEmitter.TexPatData0.UI_numUseTexPatTbl;
                        newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.RandomPatternTableCount = 0;
                    }
                    break;

                // ランダム
                case 3:
                    {
                        newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.PatternAnimType = 4;

                        newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.PatternTableCount = 0;
                        newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.RandomPatternTableCount = (int)oldEmitter.TexPatData0.UI_texPatNumPattern;
                    }
                    break;

                // ループ
                case 2:
                    {
                        newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.PatternAnimType = 3;

                        newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.PatternTableCount = (int)oldEmitter.TexPatData0.UI_numUseTexPatTbl;
                        newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.RandomPatternTableCount = 0;
                    }
                    break;

                default:
                    {
                        newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.PatternAnimType = 0;
                        newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.PatternTableCount = 0;
                        newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.RandomPatternTableCount = 0;
                    }
                    break;
            }

            // テーブル簡易入力.
            var temp = new EffectMaker.Foundation.Collections.Generic.ArrayCollection<int>(oldEmitter.TexPatData0.UI_texPatTbl.Length);
            for (int i = 0; i < temp.Count; ++i)
            {
                temp[i] = oldEmitter.TexPatData0.UI_texPatTbl[i];
            }
            newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.PatternTable = temp;

            // 1コマの継続フレーム数.
            newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.DurationCountPerFrame = (int)oldEmitter.TexPatData0.UI_texPatFreqFrame;

            // 開始テーブルランダム.
            newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.EnableSimpleTableRandom = oldEmitter.TexPatData0.UI_texPatIsRandom;
        }

        #endregion

        #region テクスチャ1

        /// <summary>
        /// 【エミッタ】テクスチャ1タブの変換処理です.
        /// </summary>
        /// <param name="oldEmitter">旧型式.</param>
        /// <param name="newEmitter">新形式.</param>
        private void ConvertEmitterTexture1Tab(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            //******************************
            //  ファイル.
            //******************************
            // ファイルパス.
            newEmitter.Textures.Texture1.EmitterTextureFileData.FilePath = GetFileName(oldEmitter.TexPatData1.UI_texPatFileName);

            // パターン横分割.
            newEmitter.Textures.Texture1.EmitterTextureFileData.PatternDivU = oldEmitter.TexPatData1.UI_texPatDivX;

            // パターン縦分割.
            newEmitter.Textures.Texture1.EmitterTextureFileData.PatternDivV = oldEmitter.TexPatData1.UI_texPatDivY;

            // ラップ(U)
            newEmitter.Textures.Texture1.EmitterTextureFileData.WrapTypeU = (int)oldEmitter.TexPatData1.UI_texWrapU;

            // ラップ(V)
            newEmitter.Textures.Texture1.EmitterTextureFileData.WrapTypeV = (int)oldEmitter.TexPatData1.UI_texWrapV;

            // U座標ランダム反転.
            newEmitter.Textures.Texture1.EmitterTextureFileData.RandomInvU = oldEmitter.TexPatData1.UI_texIsURandomInv;

            // V座標ランダム反転.
            newEmitter.Textures.Texture1.EmitterTextureFileData.RandomInvV = oldEmitter.TexPatData1.UI_texIsVRandomInv;

            // 繰り返し数.
            newEmitter.Textures.Texture1.EmitterTextureFileData.RepeatCount = (int)oldEmitter.TexPatData1.UI_textureAddressing;

            // スフィアマップ
            newEmitter.Textures.Texture1.EmitterTextureFileData.SphericalMap = oldEmitter.TexPatData1.UI_isSphereMap;

            // フィルタモード.
            newEmitter.Textures.Texture1.EmitterTextureFileData.FilterMode = (int)oldEmitter.TexPatData1.UI_texFilterMode;

            // 有効なミップレベル.
            newEmitter.Textures.Texture1.EmitterTextureFileData.AvailableMipMapLevel = oldEmitter.TexPatData1.UI_availableMipLevel;

            // ミップマップLODバイアス.
            newEmitter.Textures.Texture1.EmitterTextureFileData.MipMapBias = oldEmitter.TexPatData1.UI_mipMapBias;


            //*******************************
            //  テクスチャ座標アニメーション
            //*******************************
            // UVスクロールアニメ.
            newEmitter.Textures.Texture1.EmitterTextureUvAnimationData.EnableUvScrollAnimation = oldEmitter.SubTexScrollAnimEnabled;

            // UV回転アニメ.
            newEmitter.Textures.Texture1.EmitterTextureUvAnimationData.EnableUvRotAnimation = oldEmitter.SubTexRotateAnimEnabled;

            // UVスケールアニメ.
            newEmitter.Textures.Texture1.EmitterTextureUvAnimationData.EnableUvScaleAnimation = oldEmitter.SubTexScaleAnimEnabled;

            //----- UVスクロールアニメ -----
            // 加算値.
            newEmitter.Textures.Texture1.EmitterTextureUvAnimationData.UvScrollAddValue.X = oldEmitter.SubTexUVScroll.X;
            newEmitter.Textures.Texture1.EmitterTextureUvAnimationData.UvScrollAddValue.Y = oldEmitter.SubTexUVScroll.Y;

            // 初期値.
            newEmitter.Textures.Texture1.EmitterTextureUvAnimationData.UvScrollInitialValue.X = oldEmitter.SubTexUVScrollInit.X;
            newEmitter.Textures.Texture1.EmitterTextureUvAnimationData.UvScrollInitialValue.Y = oldEmitter.SubTexUVScrollInit.Y;

            // 初期値ランダム
            newEmitter.Textures.Texture1.EmitterTextureUvAnimationData.UvScrollInitialRandomValue.X = oldEmitter.SubTexUVScrollInitRand.X;
            newEmitter.Textures.Texture1.EmitterTextureUvAnimationData.UvScrollInitialRandomValue.Y = oldEmitter.SubTexUVScrollInitRand.Y;

            //----- UV回転 -----
            // 加算値.
            newEmitter.Textures.Texture1.EmitterTextureUvAnimationData.UvRotationAddValue = oldEmitter.SubTexUVRot;

            // 初期値.
            newEmitter.Textures.Texture1.EmitterTextureUvAnimationData.UvRotationInitialValue = oldEmitter.SubTexUVRotInit;

            // 初期値ランダム.
            newEmitter.Textures.Texture1.EmitterTextureUvAnimationData.UvRotationInitialRandomValue = oldEmitter.SubTexUVRotInitRand;

            //----- UVスケール ------
            // 加算値.
            newEmitter.Textures.Texture1.EmitterTextureUvAnimationData.UvScaleAddValue.X = oldEmitter.SubTexUVScale.X;
            newEmitter.Textures.Texture1.EmitterTextureUvAnimationData.UvScaleAddValue.Y = oldEmitter.SubTexUVScale.Y;

            // 初期値.
            newEmitter.Textures.Texture1.EmitterTextureUvAnimationData.UvScaleInitialValue.X = oldEmitter.SubTexUVScaleInit.X;
            newEmitter.Textures.Texture1.EmitterTextureUvAnimationData.UvScaleInitialValue.Y = oldEmitter.SubTexUVScaleInit.Y;

            // 初期値ランダム.
            newEmitter.Textures.Texture1.EmitterTextureUvAnimationData.UvScaleInitialRandomValue.X = oldEmitter.SubTexUVScaleInitRand.X;
            newEmitter.Textures.Texture1.EmitterTextureUvAnimationData.UvScaleInitialRandomValue.Y = oldEmitter.SubTexUVScaleInitRand.Y;


            //********************************
            //  テクスチャパターンアニメーション.
            //********************************

            //----- パターンテーブル -----
            switch (oldEmitter.TexPatData1.UI_texPatAnimMode)
            {
                // ライフフィット.
                case 1:
                    {
                        newEmitter.Textures.Texture1.EmitterTexturePatternAnimationData.PatternAnimType = 1;

                        newEmitter.Textures.Texture1.EmitterTexturePatternAnimationData.PatternTableCount = (int)oldEmitter.TexPatData1.UI_numUseTexPatTbl;
                        newEmitter.Textures.Texture1.EmitterTexturePatternAnimationData.RandomPatternTableCount = 0;
                    }
                    break;

                // クランプ
                case 4:
                    {
                        newEmitter.Textures.Texture1.EmitterTexturePatternAnimationData.PatternAnimType = 2;

                        newEmitter.Textures.Texture1.EmitterTexturePatternAnimationData.PatternTableCount = (int)oldEmitter.TexPatData1.UI_numUseTexPatTbl;
                        newEmitter.Textures.Texture1.EmitterTexturePatternAnimationData.RandomPatternTableCount = 0;
                    }
                    break;

                // ランダム
                case 3:
                    {
                        newEmitter.Textures.Texture1.EmitterTexturePatternAnimationData.PatternAnimType = 4;

                        newEmitter.Textures.Texture1.EmitterTexturePatternAnimationData.PatternTableCount = 0;
                        newEmitter.Textures.Texture1.EmitterTexturePatternAnimationData.RandomPatternTableCount = (int)oldEmitter.TexPatData1.UI_texPatNumPattern;
                    }
                    break;

                // ループ
                case 2:
                    {
                        newEmitter.Textures.Texture1.EmitterTexturePatternAnimationData.PatternAnimType = 3;

                        newEmitter.Textures.Texture1.EmitterTexturePatternAnimationData.PatternTableCount = (int)oldEmitter.TexPatData1.UI_numUseTexPatTbl;
                        newEmitter.Textures.Texture1.EmitterTexturePatternAnimationData.RandomPatternTableCount = 0;
                    }
                    break;

                default:
                    {
                        newEmitter.Textures.Texture1.EmitterTexturePatternAnimationData.PatternAnimType = 0;
                        newEmitter.Textures.Texture1.EmitterTexturePatternAnimationData.PatternTableCount = 0;
                        newEmitter.Textures.Texture1.EmitterTexturePatternAnimationData.RandomPatternTableCount = 0;
                    }
                    break;
            }

            // テーブル簡易入力.
            var temp = new EffectMaker.Foundation.Collections.Generic.ArrayCollection<int>(oldEmitter.TexPatData1.UI_texPatTbl.Length);
            for (int i = 0; i < temp.Count; ++i)
            {
                temp[i] = oldEmitter.TexPatData1.UI_texPatTbl[i];
            }
            newEmitter.Textures.Texture1.EmitterTexturePatternAnimationData.PatternTable = temp;

            // 1コマの継続フレーム数.
            newEmitter.Textures.Texture1.EmitterTexturePatternAnimationData.DurationCountPerFrame = (int)oldEmitter.TexPatData1.UI_texPatFreqFrame;

            // 開始テーブルランダム.
            newEmitter.Textures.Texture1.EmitterTexturePatternAnimationData.EnableSimpleTableRandom = oldEmitter.TexPatData1.UI_texPatIsRandom;
        }

        #endregion

        #region テクスチャ2

        /// <summary>
        /// 【エミッタ】テクスチャタブ2の変換処理です.
        /// </summary>
        /// <param name="oldEmitter">旧型式.</param>
        /// <param name="newEmitter">新形式.</param>
        private void ConvertEmitterTexture2Tab(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            //******************************
            //  ファイル.
            //******************************
            // ファイルパス.
            newEmitter.Textures.Texture2.EmitterTextureFileData.FilePath = GetFileName(oldEmitter.TexPatData2.UI_texPatFileName);

            // パターン横分割.
            newEmitter.Textures.Texture2.EmitterTextureFileData.PatternDivU = oldEmitter.TexPatData2.UI_texPatDivX;

            // パターン縦分割.
            newEmitter.Textures.Texture2.EmitterTextureFileData.PatternDivV = oldEmitter.TexPatData2.UI_texPatDivY;

            // ラップ(U)
            newEmitter.Textures.Texture2.EmitterTextureFileData.WrapTypeU = (int)oldEmitter.TexPatData2.UI_texWrapU;

            // ラップ(V)
            newEmitter.Textures.Texture2.EmitterTextureFileData.WrapTypeV = (int)oldEmitter.TexPatData2.UI_texWrapV;

            // U座標ランダム反転.
            newEmitter.Textures.Texture2.EmitterTextureFileData.RandomInvU = oldEmitter.TexPatData2.UI_texIsURandomInv;

            // V座標ランダム反転.
            newEmitter.Textures.Texture2.EmitterTextureFileData.RandomInvV = oldEmitter.TexPatData2.UI_texIsVRandomInv;

            // 繰り返し数.
            newEmitter.Textures.Texture2.EmitterTextureFileData.RepeatCount = (int)oldEmitter.TexPatData2.UI_textureAddressing;

            // スフィアマップ
            newEmitter.Textures.Texture2.EmitterTextureFileData.SphericalMap = oldEmitter.TexPatData2.UI_isSphereMap;

            // フィルタモード.
            newEmitter.Textures.Texture2.EmitterTextureFileData.FilterMode = (int)oldEmitter.TexPatData2.UI_texFilterMode;

            // 有効なミップレベル.
            newEmitter.Textures.Texture2.EmitterTextureFileData.AvailableMipMapLevel = oldEmitter.TexPatData2.UI_availableMipLevel;

            // ミップマップLODバイアス.
            newEmitter.Textures.Texture2.EmitterTextureFileData.MipMapBias = oldEmitter.TexPatData2.UI_mipMapBias;


            //*******************************
            //  テクスチャ座標アニメーション
            //*******************************
            // UVスクロールアニメ.
            newEmitter.Textures.Texture2.EmitterTextureUvAnimationData.EnableUvScrollAnimation = oldEmitter.Tex2ScrollAnimEnabled;

            // UV回転アニメ.
            newEmitter.Textures.Texture2.EmitterTextureUvAnimationData.EnableUvRotAnimation = oldEmitter.Tex2RotateAnimEnabled;

            // UVスケールアニメ.
            newEmitter.Textures.Texture2.EmitterTextureUvAnimationData.EnableUvScaleAnimation = oldEmitter.Tex2ScaleAnimEnabled;

            //----- UVスクロールアニメ -----
            // 加算値.
            newEmitter.Textures.Texture2.EmitterTextureUvAnimationData.UvScrollAddValue.X = oldEmitter.Tex2UVScroll.X;
            newEmitter.Textures.Texture2.EmitterTextureUvAnimationData.UvScrollAddValue.Y = oldEmitter.Tex2UVScroll.Y;

            // 初期値.
            newEmitter.Textures.Texture2.EmitterTextureUvAnimationData.UvScrollInitialValue.X = oldEmitter.Tex2UVScrollInit.X;
            newEmitter.Textures.Texture2.EmitterTextureUvAnimationData.UvScrollInitialValue.Y = oldEmitter.Tex2UVScrollInit.Y;

            // 初期値ランダム
            newEmitter.Textures.Texture2.EmitterTextureUvAnimationData.UvScrollInitialRandomValue.X = oldEmitter.Tex2UVScrollInitRand.X;
            newEmitter.Textures.Texture2.EmitterTextureUvAnimationData.UvScrollInitialRandomValue.Y = oldEmitter.Tex2UVScrollInitRand.Y;

            //----- UV回転 -----
            // 加算値.
            newEmitter.Textures.Texture2.EmitterTextureUvAnimationData.UvRotationAddValue = oldEmitter.Tex2UVRot;

            // 初期値.
            newEmitter.Textures.Texture2.EmitterTextureUvAnimationData.UvRotationInitialValue = oldEmitter.Tex2UVRotInit;

            // 初期値ランダム.
            newEmitter.Textures.Texture2.EmitterTextureUvAnimationData.UvRotationInitialRandomValue = oldEmitter.Tex2UVRotInitRand;

            //----- UVスケール ------
            // 加算値.
            newEmitter.Textures.Texture2.EmitterTextureUvAnimationData.UvScaleAddValue.X = oldEmitter.Tex2UVScale.X;
            newEmitter.Textures.Texture2.EmitterTextureUvAnimationData.UvScaleAddValue.Y = oldEmitter.Tex2UVScale.Y;

            // 初期値.
            newEmitter.Textures.Texture2.EmitterTextureUvAnimationData.UvScaleInitialValue.X = oldEmitter.Tex2UVScaleInit.X;
            newEmitter.Textures.Texture2.EmitterTextureUvAnimationData.UvScaleInitialValue.Y = oldEmitter.Tex2UVScaleInit.Y;

            // 初期値ランダム.
            newEmitter.Textures.Texture2.EmitterTextureUvAnimationData.UvScaleInitialRandomValue.X = oldEmitter.Tex2UVScaleInitRand.X;
            newEmitter.Textures.Texture2.EmitterTextureUvAnimationData.UvScaleInitialRandomValue.Y = oldEmitter.Tex2UVScaleInitRand.Y;


            //********************************
            //  テクスチャパターンアニメーション.
            //********************************

            //----- パターンテーブル -----
            switch (oldEmitter.TexPatData2.UI_texPatAnimMode)
            {
                // ライフフィット.
                case 1:
                    {
                        newEmitter.Textures.Texture2.EmitterTexturePatternAnimationData.PatternAnimType = 1;

                        newEmitter.Textures.Texture2.EmitterTexturePatternAnimationData.PatternTableCount = (int)oldEmitter.TexPatData2.UI_numUseTexPatTbl;
                        newEmitter.Textures.Texture2.EmitterTexturePatternAnimationData.RandomPatternTableCount = 0;
                    }
                    break;

                // クランプ
                case 4:
                    {
                        newEmitter.Textures.Texture2.EmitterTexturePatternAnimationData.PatternAnimType = 2;

                        newEmitter.Textures.Texture2.EmitterTexturePatternAnimationData.PatternTableCount = (int)oldEmitter.TexPatData2.UI_numUseTexPatTbl;
                        newEmitter.Textures.Texture2.EmitterTexturePatternAnimationData.RandomPatternTableCount = 0;
                    }
                    break;

                // ランダム
                case 3:
                    {
                        newEmitter.Textures.Texture2.EmitterTexturePatternAnimationData.PatternAnimType = 4;

                        newEmitter.Textures.Texture2.EmitterTexturePatternAnimationData.PatternTableCount = 0;
                        newEmitter.Textures.Texture2.EmitterTexturePatternAnimationData.RandomPatternTableCount = (int)oldEmitter.TexPatData2.UI_texPatNumPattern;
                    }
                    break;

                // ループ
                case 2:
                    {
                        newEmitter.Textures.Texture2.EmitterTexturePatternAnimationData.PatternAnimType = 3;

                        newEmitter.Textures.Texture2.EmitterTexturePatternAnimationData.PatternTableCount = (int)oldEmitter.TexPatData2.UI_numUseTexPatTbl;
                        newEmitter.Textures.Texture2.EmitterTexturePatternAnimationData.RandomPatternTableCount = 0;
                    }
                    break;

                default:
                    {
                        newEmitter.Textures.Texture2.EmitterTexturePatternAnimationData.PatternAnimType = 0;
                        newEmitter.Textures.Texture2.EmitterTexturePatternAnimationData.PatternTableCount = 0;
                        newEmitter.Textures.Texture2.EmitterTexturePatternAnimationData.RandomPatternTableCount = 0;
                    }
                    break;
            }

            // テーブル簡易入力.
            var temp = new EffectMaker.Foundation.Collections.Generic.ArrayCollection<int>(oldEmitter.TexPatData2.UI_texPatTbl.Length);
            for (int i = 0; i < temp.Count; ++i)
            {
                temp[i] = oldEmitter.TexPatData2.UI_texPatTbl[i];
            }
            newEmitter.Textures.Texture2.EmitterTexturePatternAnimationData.PatternTable = temp;

            // 1コマの継続フレーム数.
            newEmitter.Textures.Texture2.EmitterTexturePatternAnimationData.DurationCountPerFrame = (int)oldEmitter.TexPatData2.UI_texPatFreqFrame;

            // 開始テーブルランダム.
            newEmitter.Textures.Texture2.EmitterTexturePatternAnimationData.EnableSimpleTableRandom = oldEmitter.TexPatData2.UI_texPatIsRandom;
        }

        #endregion

        #region カラー

        /// <summary>
        /// 【エミッタ】カラータブの変換処理です.
        /// </summary>
        /// <param name="oldEmitter">旧型式</param>
        /// <param name="newEmitter">新形式.</param>
        private void ConvertEmitterColorTab(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            //*******************************
            //  カラー計算.
            //*******************************
            // カラースケール.
            newEmitter.EmitterColorData.ParticleColor.EmitterColorCalculationData.ColorScale = oldEmitter.UI_colorScale;

            // 固定フラグをFE2のデータモデルに繋ぐ
            newEmitter.EmitterColorData.ParticleColor.EmitterColorCalculationData.ColorAlphaFixFlag = oldEmitter.UI_colorAlphaFixFlag;

            //*******************************
            //  カラー・アルファ設定.
            //*******************************
            // カラー0挙動タイプ.

            switch (oldEmitter.AnimEditData.Color0.ColorType)
            {
                // 定数.
                case DataModel.Constances.ColorSettingType.Constant:
                    {
                        newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Color0BehaviorType = 0;
                    }
                    break;

                // ３キーアニメ.
                case DataModel.Constances.ColorSettingType.Animation:
                    {
                        newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Color0BehaviorType = 2;
                    }
                    break;

                // 8キーアニメ.
                case DataModel.Constances.ColorSettingType.AnimationTable:
                    {
                        newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Color0BehaviorType = 2;
                    }
                    break;

                // ランダム.
                case DataModel.Constances.ColorSettingType.Random:
                    {
                        newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Color0BehaviorType = 3;
                    }
                    break;
            }

            // カラー1挙動タイプ.
            switch (oldEmitter.AnimEditData.Color1.ColorType)
            {
                // 定数.
                case DataModel.Constances.ColorSettingType.Constant:
                    {
                        newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Color1BehaviorType = 0;
                    }
                    break;

                // ３キーアニメ.
                case DataModel.Constances.ColorSettingType.Animation:
                    {
                        newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Color1BehaviorType = 2;
                    }
                    break;

                // 8キーアニメ.
                case DataModel.Constances.ColorSettingType.AnimationTable:
                    {
                        newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Color1BehaviorType = 2;
                    }
                    break;

                // ランダム.
                case DataModel.Constances.ColorSettingType.Random:
                    {
                        newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Color1BehaviorType = 3;
                    }
                    break;
            }

            // アルファ0挙動タイプ.
            switch (oldEmitter.AnimEditData.Alpha0.AnimType)
            {
                case DataModel.ParticleAnimationTypes.NoAnim:
                    {
                        newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Alpha0BehaviorType = 0;
                    }
                    break;

                case DataModel.ParticleAnimationTypes.Anim3Key:
                    {
                        newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Alpha0BehaviorType = 0;
                    }
                    break;

                case DataModel.ParticleAnimationTypes.AnimTable:
                    {
                        newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Alpha0BehaviorType = 2;
                    }
                    break;

                case DataModel.ParticleAnimationTypes.InOutAnim:
                    {
                        newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Alpha0BehaviorType = 2;
                    }
                    break;
            }
             // アルファ1挙動タイプ.
            switch (oldEmitter.AnimEditData.Alpha1.AnimType)
            {
                case DataModel.ParticleAnimationTypes.NoAnim:
                    {
                        newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Alpha1BehaviorType = 0;
                    }
                    break;

                case DataModel.ParticleAnimationTypes.Anim3Key:
                    {
                        newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Alpha1BehaviorType = 0;
                    }
                    break;

                case DataModel.ParticleAnimationTypes.AnimTable:
                    {
                        newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Alpha1BehaviorType = 2;
                    }
                    break;

                case DataModel.ParticleAnimationTypes.InOutAnim:
                    {
                        newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Alpha1BehaviorType = 2;
                    }
                    break;
            }

            // カラー0設定.
            newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Color0Value.R = oldEmitter.AnimEditData.Color0.Color1.R;
            newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Color0Value.G = oldEmitter.AnimEditData.Color0.Color1.G;
            newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Color0Value.B = oldEmitter.AnimEditData.Color0.Color1.B;
            newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Color0Value.A = oldEmitter.AnimEditData.Color0.Color1.A;

            newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.ColorAnimation0.Clear();

            // 3キーアニメ
            if (oldEmitter.AnimEditData.Color0.ColorType == DataModel.Constances.ColorSettingType.Animation)
            {
                // ループの有効・無効.
                newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.ColorAnimationLoopData0.EnableLoop = oldEmitter.AnimEditData.Color0.NumRepeat != 1;

                // ループ数
                newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.ColorAnimationLoopData0.LoopFrame = oldEmitter.AnimEditData.UI_life/oldEmitter.AnimEditData.Color0.NumRepeat;

                // 開始位置ランダム.
                newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.ColorAnimationLoopData0.EnableStartRandom = oldEmitter.AnimEditData.Color0.RandInitColor;

                if (oldEmitter.AnimEditData.Color0.Time1 != oldEmitter.AnimEditData.Color0.Time2)
                {
                    float keyR = oldEmitter.AnimEditData.Color0.Color1.R;
                    float keyG = oldEmitter.AnimEditData.Color0.Color1.G;
                    float keyB = oldEmitter.AnimEditData.Color0.Color1.B;
                    int   keyFrame = oldEmitter.AnimEditData.Color0.Time1;
                    EffectMaker.Foundation.Primitives.ColorRgba keyValue = new EffectMaker.Foundation.Primitives.ColorRgba(keyR, keyG, keyB, 1.0f);
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.ColorAnimation0.AddKeyFrame(keyFrame, keyValue, false);
                }
                {
                    float keyR = oldEmitter.AnimEditData.Color0.Color2.R;
                    float keyG = oldEmitter.AnimEditData.Color0.Color2.G;
                    float keyB = oldEmitter.AnimEditData.Color0.Color2.B;
                    int   keyFrame = oldEmitter.AnimEditData.Color0.Time2;
                    EffectMaker.Foundation.Primitives.ColorRgba keyValue = new EffectMaker.Foundation.Primitives.ColorRgba(keyR, keyG, keyB, 1.0f);
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.ColorAnimation0.AddKeyFrame(keyFrame, keyValue, false);
                }
                if (oldEmitter.AnimEditData.Color0.Time2 != oldEmitter.AnimEditData.Color0.Time3)
                {
                    float keyR = oldEmitter.AnimEditData.Color0.Color3.R;
                    float keyG = oldEmitter.AnimEditData.Color0.Color3.G;
                    float keyB = oldEmitter.AnimEditData.Color0.Color3.B;
                    int   keyFrame = oldEmitter.AnimEditData.Color0.Time3;
                    EffectMaker.Foundation.Primitives.ColorRgba keyValue = new EffectMaker.Foundation.Primitives.ColorRgba(keyR, keyG, keyB, 1.0f);
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.ColorAnimation0.AddKeyFrame(keyFrame, keyValue, false);
                }

            }
            // 8キーアニメ
            else if (oldEmitter.AnimEditData.Color0.ColorType == DataModel.Constances.ColorSettingType.AnimationTable)
            {
                NWCore.DataModel.AnimTableData anim_8_key_color0_r = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.anim_8key_color0_r);
                NWCore.DataModel.AnimTableData anim_8_key_color0_g = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.anim_8key_color0_g);
                NWCore.DataModel.AnimTableData anim_8_key_color0_b = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.anim_8key_color0_b);

                if ((anim_8_key_color0_r != null)
                 && (anim_8_key_color0_g != null)
                 && (anim_8_key_color0_b != null))
                {
                    int keyFrame = anim_8_key_color0_r.KeyFrames.Count;

                    // ループの有効・無効.
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.ColorAnimationLoopData0.EnableLoop = anim_8_key_color0_r.IsLoop;

                    // ループ数
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.ColorAnimationLoopData0.LoopFrame = anim_8_key_color0_r.NumLoopFrames;

                    // 開始位置ランダム.
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.ColorAnimationLoopData0.EnableStartRandom = anim_8_key_color0_r.IsRandomStartFrameEnabled;

                    for (int i = 0; i < keyFrame; ++i)
                    {
                        NWCore.DataModel.FloatAnimTableKeyFrameData keyR = anim_8_key_color0_r.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;
                        NWCore.DataModel.FloatAnimTableKeyFrameData keyG = anim_8_key_color0_g.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;
                        NWCore.DataModel.FloatAnimTableKeyFrameData keyB = anim_8_key_color0_b.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;
                        EffectMaker.Foundation.Primitives.ColorRgba keyValue = new EffectMaker.Foundation.Primitives.ColorRgba(keyR.Value, keyG.Value, keyB.Value, 1.0f);
                        newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.ColorAnimation0.AddKeyFrame( keyR.KeyFrame, keyValue, false);
                    }
                }
            }

            if (oldEmitter.AnimEditData.Color0.ColorType == DataModel.Constances.ColorSettingType.Random)
            {
                int randomNum = oldEmitter.AnimEditData.Color0.RandomColors.Length;
                for (int i = 0; i < randomNum; i++)
                {
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.RandomColor0[i].Enabled = true;

                    var keyColor = new NintendoWare.ToolDevelopmentKit.RgbaColor(1.0f, 1.0f, 1.0f, 1.0f);
                    keyColor.R = oldEmitter.AnimEditData.Color0.RandomColors[i].R;
                    keyColor.G = oldEmitter.AnimEditData.Color0.RandomColors[i].G;
                    keyColor.B = oldEmitter.AnimEditData.Color0.RandomColors[i].B;
                    keyColor.A = oldEmitter.AnimEditData.Color0.RandomColors[i].A;

                    EffectMaker.Foundation.Primitives.ColorRgba keyValue = new EffectMaker.Foundation.Primitives.ColorRgba(keyColor.R, keyColor.G, keyColor.B, keyColor.A);
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.RandomColor0[i].Color = keyValue;
                }
            }

            // カラー1設定.
            newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Color1Value.R = oldEmitter.AnimEditData.Color1.Color1.R;
            newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Color1Value.G = oldEmitter.AnimEditData.Color1.Color1.G;
            newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Color1Value.B = oldEmitter.AnimEditData.Color1.Color1.B;
            newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Color1Value.A = oldEmitter.AnimEditData.Color1.Color1.A;

            newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.ColorAnimation1.Clear();
            // 3キーアニメ
            if (oldEmitter.AnimEditData.Color1.ColorType == DataModel.Constances.ColorSettingType.Animation)
            {
                // ループの有効・無効.
                newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.ColorAnimationLoopData1.EnableLoop = oldEmitter.AnimEditData.Color1.NumRepeat != 1;

                // ループ数
                newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.ColorAnimationLoopData1.LoopFrame = oldEmitter.AnimEditData.UI_life / oldEmitter.AnimEditData.Color1.NumRepeat;

                // 開始位置ランダム.
                newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.ColorAnimationLoopData1.EnableStartRandom = oldEmitter.AnimEditData.Color1.RandInitColor;

                if (oldEmitter.AnimEditData.Color1.Time1 != oldEmitter.AnimEditData.Color1.Time2)
                {
                    float keyR = oldEmitter.AnimEditData.Color1.Color1.R;
                    float keyG = oldEmitter.AnimEditData.Color1.Color1.G;
                    float keyB = oldEmitter.AnimEditData.Color1.Color1.B;
                    int keyFrame = oldEmitter.AnimEditData.Color1.Time1;
                    EffectMaker.Foundation.Primitives.ColorRgba keyValue = new EffectMaker.Foundation.Primitives.ColorRgba(keyR, keyG, keyB, 1.0f);
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.ColorAnimation1.AddKeyFrame(keyFrame, keyValue, false);
                }
                {
                    float keyR = oldEmitter.AnimEditData.Color1.Color2.R;
                    float keyG = oldEmitter.AnimEditData.Color1.Color2.G;
                    float keyB = oldEmitter.AnimEditData.Color1.Color2.B;
                    int keyFrame = oldEmitter.AnimEditData.Color1.Time2;
                    EffectMaker.Foundation.Primitives.ColorRgba keyValue = new EffectMaker.Foundation.Primitives.ColorRgba(keyR, keyG, keyB, 1.0f);
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.ColorAnimation1.AddKeyFrame(keyFrame, keyValue, false);
                }
                if (oldEmitter.AnimEditData.Color1.Time2 != oldEmitter.AnimEditData.Color1.Time3)
                {
                    float keyR = oldEmitter.AnimEditData.Color1.Color3.R;
                    float keyG = oldEmitter.AnimEditData.Color1.Color3.G;
                    float keyB = oldEmitter.AnimEditData.Color1.Color3.B;
                    int keyFrame = oldEmitter.AnimEditData.Color1.Time3;
                    EffectMaker.Foundation.Primitives.ColorRgba keyValue = new EffectMaker.Foundation.Primitives.ColorRgba(keyR, keyG, keyB, 1.0f);
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.ColorAnimation1.AddKeyFrame(keyFrame, keyValue, false);
                }

            }
            // 8キーアニメ
            else if (oldEmitter.AnimEditData.Color1.ColorType == DataModel.Constances.ColorSettingType.AnimationTable)
            {
                NWCore.DataModel.AnimTableData anim_8_key_color1_r = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.anim_8key_color1_r);
                NWCore.DataModel.AnimTableData anim_8_key_color1_g = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.anim_8key_color1_g);
                NWCore.DataModel.AnimTableData anim_8_key_color1_b = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.anim_8key_color1_b);

                if ((anim_8_key_color1_r != null)
                 && (anim_8_key_color1_g != null)
                 && (anim_8_key_color1_b != null))
                {
                    // ループの有効・無効.
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.ColorAnimationLoopData1.EnableLoop = anim_8_key_color1_r.IsLoop;

                    // ループ数
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.ColorAnimationLoopData1.LoopFrame = anim_8_key_color1_r.NumLoopFrames;

                    // 開始位置ランダム.
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.ColorAnimationLoopData1.EnableStartRandom = anim_8_key_color1_r.IsRandomStartFrameEnabled;

                    int keyFrame = anim_8_key_color1_r.KeyFrames.Count;

                    for (int i = 0; i < keyFrame; ++i)
                    {
                        NWCore.DataModel.FloatAnimTableKeyFrameData keyR = anim_8_key_color1_r.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;
                        NWCore.DataModel.FloatAnimTableKeyFrameData keyG = anim_8_key_color1_g.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;
                        NWCore.DataModel.FloatAnimTableKeyFrameData keyB = anim_8_key_color1_b.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;
                        EffectMaker.Foundation.Primitives.ColorRgba keyValue = new EffectMaker.Foundation.Primitives.ColorRgba(keyR.Value, keyG.Value, keyB.Value, 1.0f);
                        newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.ColorAnimation1.AddKeyFrame(keyR.KeyFrame, keyValue, false);
                    }
                }
            }

            if (oldEmitter.AnimEditData.Color1.ColorType == DataModel.Constances.ColorSettingType.Random)
            {
                int randomNum = oldEmitter.AnimEditData.Color1.RandomColors.Length;
                for (int i = 0; i < randomNum; i++)
                {
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.RandomColor1[i].Enabled = true;

                    var keyColor = new NintendoWare.ToolDevelopmentKit.RgbaColor(1.0f, 1.0f, 1.0f, 1.0f);
                    keyColor.R = oldEmitter.AnimEditData.Color1.RandomColors[i].R;
                    keyColor.G = oldEmitter.AnimEditData.Color1.RandomColors[i].G;
                    keyColor.B = oldEmitter.AnimEditData.Color1.RandomColors[i].B;
                    keyColor.A = oldEmitter.AnimEditData.Color1.RandomColors[i].A;

                    EffectMaker.Foundation.Primitives.ColorRgba keyValue = new EffectMaker.Foundation.Primitives.ColorRgba(keyColor.R, keyColor.G, keyColor.B, keyColor.A);
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.RandomColor1[i].Color = keyValue;
                }
            }

            // アルファ0設定.
            newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Alpha0Value = oldEmitter.AnimEditData.Alpha0.Alpha2;

            newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.AlphaAnimation0.Clear();

            // インアウトアニメ
            if (oldEmitter.AnimEditData.Alpha0.AnimType == DataModel.ParticleAnimationTypes.InOutAnim)
            {
                if (newEmitter.EmitterParticleData.EmitterParticleLifeData.EnableInfinityLife)
                {
                    // インアウトで無限寿命の場合は強制固定へ。
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Alpha0BehaviorType = 0;
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Alpha0Value = oldEmitter.AnimEditData.Alpha0.Alpha2;
                }
                else
                {
                    if (oldEmitter.AnimEditData.Alpha0.Time1 == 0 && oldEmitter.AnimEditData.Alpha0.Time2 == 100)
                    {
                        newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Alpha0BehaviorType = 0;
                    }
                    else
                    {
                        int keyFrame = 0;
                        float alphaValue = 0;
                        EffectMaker.Foundation.Primitives.ColorRgba keyValue = null;

                        // key1
                        if (oldEmitter.AnimEditData.Alpha0.Time1 != 0)
                        {
                            alphaValue = oldEmitter.AnimEditData.Alpha0.Alpha1;
                            keyValue = new EffectMaker.Foundation.Primitives.ColorRgba(1.0f, 1.0f, 1.0f, alphaValue);
                            newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.AlphaAnimation0.AddKeyFrame(keyFrame, keyValue, false);
                        }
                        //key2
                        {
                            keyFrame = oldEmitter.AnimEditData.Alpha0.Time1;
                            alphaValue = oldEmitter.AnimEditData.Alpha0.Alpha2;
                            keyValue = new EffectMaker.Foundation.Primitives.ColorRgba(1.0f, 1.0f, 1.0f, alphaValue);
                            newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.AlphaAnimation0.AddKeyFrame(keyFrame, keyValue, false);
                        }

                        // key3
                        if (oldEmitter.AnimEditData.Alpha0.Time2 != oldEmitter.AnimEditData.Alpha0.Time1)
                        {
                            keyFrame = oldEmitter.AnimEditData.Alpha0.Time2;
                            alphaValue = oldEmitter.AnimEditData.Alpha0.Alpha2;
                            keyValue = new EffectMaker.Foundation.Primitives.ColorRgba(1.0f, 1.0f, 1.0f, alphaValue);
                            newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.AlphaAnimation0.AddKeyFrame(keyFrame, keyValue, false);
                        }

                        // key4
                        if (oldEmitter.AnimEditData.Alpha0.Time2 != 100)
                        {
                            keyFrame = 100;
                            alphaValue = oldEmitter.AnimEditData.Alpha0.Alpha3;
                            keyValue = new EffectMaker.Foundation.Primitives.ColorRgba(1.0f, 1.0f, 1.0f, alphaValue);
                            newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.AlphaAnimation0.AddKeyFrame(keyFrame, keyValue, false);
                        }
                    }
                }
            }
            // 8キーアニメ
            else if (oldEmitter.AnimEditData.Alpha0.AnimType == DataModel.ParticleAnimationTypes.AnimTable)
            {
                NWCore.DataModel.AnimTableData anim_8_key_alpha0 = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.anim_8key_alpha0);

                if (anim_8_key_alpha0 != null)
                {
                    // ループの有効・無効.
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.AlphaAnimationLoopData0.EnableLoop = anim_8_key_alpha0.IsLoop;

                    // ループ数.
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.AlphaAnimationLoopData0.LoopFrame = anim_8_key_alpha0.NumLoopFrames;

                    // 開始位置ランダム.
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.AlphaAnimationLoopData0.EnableStartRandom = anim_8_key_alpha0.IsRandomStartFrameEnabled;

                    int keyFrame = anim_8_key_alpha0.KeyFrames.Count;

                    for (int i = 0; i < keyFrame; ++i)
                    {
                        NWCore.DataModel.FloatAnimTableKeyFrameData keyA = anim_8_key_alpha0.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;
                        EffectMaker.Foundation.Primitives.ColorRgba keyValue = new EffectMaker.Foundation.Primitives.ColorRgba(1.0f, 1.0f, 1.0f, keyA.Value);

                        newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.AlphaAnimation0.AddKeyFrame( keyA.KeyFrame, keyValue, false);
                    }
                }
            }

            // アルファ1設定.
            newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Alpha1Value = oldEmitter.AnimEditData.Alpha1.Alpha2;

            newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.AlphaAnimation1.Clear();

            // インアウトアニメ
            if (oldEmitter.AnimEditData.Alpha1.AnimType == DataModel.ParticleAnimationTypes.InOutAnim)
            {
                if (newEmitter.EmitterParticleData.EmitterParticleLifeData.EnableInfinityLife)
                {
                    // インアウトで無限寿命の場合は強制固定へ。
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Alpha1BehaviorType = 0;
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Alpha1Value = oldEmitter.AnimEditData.Alpha1.Alpha2;
                }
                else
                {
                    if (oldEmitter.AnimEditData.Alpha1.Time1 == 0 && oldEmitter.AnimEditData.Alpha1.Time2 == 100)
                    {
                        newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Alpha1BehaviorType = 0;
                    }
                    else
                    {
                        int keyFrame = 0;
                        float alphaValue = 0;
                        EffectMaker.Foundation.Primitives.ColorRgba keyValue = null;

                        // key1
                        if (oldEmitter.AnimEditData.Alpha1.Time1 != 0)
                        {
                            keyFrame = 0;
                            alphaValue = oldEmitter.AnimEditData.Alpha1.Alpha1;
                            keyValue = new EffectMaker.Foundation.Primitives.ColorRgba(1.0f, 1.0f, 1.0f, alphaValue);
                            newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.AlphaAnimation1.AddKeyFrame(keyFrame, keyValue, false);
                        }

                        //key2
                        {
                            keyFrame = oldEmitter.AnimEditData.Alpha1.Time1;
                            alphaValue = oldEmitter.AnimEditData.Alpha1.Alpha2;
                            keyValue = new EffectMaker.Foundation.Primitives.ColorRgba(1.0f, 1.0f, 1.0f, alphaValue);
                            newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.AlphaAnimation1.AddKeyFrame(keyFrame, keyValue, false);
                        }

                        // key3
                        if (oldEmitter.AnimEditData.Alpha1.Time2 != oldEmitter.AnimEditData.Alpha1.Time1)
                        {
                            keyFrame = oldEmitter.AnimEditData.Alpha1.Time2;
                            alphaValue = oldEmitter.AnimEditData.Alpha1.Alpha2;
                            keyValue = new EffectMaker.Foundation.Primitives.ColorRgba(1.0f, 1.0f, 1.0f, alphaValue);
                            newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.AlphaAnimation1.AddKeyFrame(keyFrame, keyValue, false);
                        }

                        // key4
                        if (oldEmitter.AnimEditData.Alpha1.Time2 != 100)
                        {
                            keyFrame = 100;
                            alphaValue = oldEmitter.AnimEditData.Alpha1.Alpha3;
                            keyValue = new EffectMaker.Foundation.Primitives.ColorRgba(1.0f, 1.0f, 1.0f, alphaValue);
                            newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.AlphaAnimation1.AddKeyFrame(keyFrame, keyValue, false);
                        }
                    }
                }
            }
            // 8キーアニメ
            else if (oldEmitter.AnimEditData.Alpha1.AnimType == DataModel.ParticleAnimationTypes.AnimTable)
            {
                NWCore.DataModel.AnimTableData anim_8_key_alpha1 = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.anim_8key_alpha1);

                if (anim_8_key_alpha1 != null)
                {
                    // ループの有効・無効.
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.AlphaAnimationLoopData1.EnableLoop = anim_8_key_alpha1.IsLoop;

                    // ループ数.
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.AlphaAnimationLoopData1.LoopFrame = anim_8_key_alpha1.NumLoopFrames;

                    // 開始位置ランダム.
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.AlphaAnimationLoopData1.EnableStartRandom = anim_8_key_alpha1.IsRandomStartFrameEnabled;

                    int keyFrame = anim_8_key_alpha1.KeyFrames.Count;

                    for (int i = 0; i < keyFrame; ++i)
                    {
                        NWCore.DataModel.FloatAnimTableKeyFrameData keyA = anim_8_key_alpha1.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;
                        EffectMaker.Foundation.Primitives.ColorRgba keyValue = new EffectMaker.Foundation.Primitives.ColorRgba(1.0f, 1.0f, 1.0f, keyA.Value);

                        newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.AlphaAnimation1.AddKeyFrame(keyA.KeyFrame, keyValue, false);
                    }
                }
            }
        }

        #endregion

        #region スケール

        /// <summary>
        /// 【エミッタ】スケールタブの変換処理です.
        /// </summary>
        /// <param name="oldEmitter">旧型式.</param>
        /// <param name="newEmitter">新形式.</param>
        private void ConvertEmitterScaleTab(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            //**********************************
            //  基本設定.
            //**********************************
            // 基準値.
            newEmitter.EmitterScaleData.EmitterScaleBasicData.ScaleBasisValue.X = oldEmitter.AnimEditData.UI_scale2.X;
            newEmitter.EmitterScaleData.EmitterScaleBasicData.ScaleBasisValue.Y = oldEmitter.AnimEditData.UI_scale2.Y;

            {
                NWCore.DataModel.AnimTableData ptcl_scale_x = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.ptcl_scale_x);
                NWCore.DataModel.AnimTableData ptcl_scale_y = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.ptcl_scale_y);

                // 基準値 - エミッタ時間アニメ.
                newEmitter.EmitterScaleData.EmitterScaleBasicData.BasicScaleAnimation.EnableAnimation = ptcl_scale_x.IsEnabled;

                newEmitter.EmitterScaleData.EmitterScaleBasicData.BasicScaleAnimation.LoopMode = (ptcl_scale_x.IsLoop) ? 1 : 0;

                switch (ptcl_scale_x.InterpolationType)
                {
                    case DataModel.AnimInterpolationTypes.Linear:
                        {
                            newEmitter.EmitterScaleData.EmitterScaleBasicData.BasicScaleAnimation.InterpolationMode = 0;
                        }
                        break;

                    case DataModel.AnimInterpolationTypes.Smooth:
                        {
                            newEmitter.EmitterScaleData.EmitterScaleBasicData.BasicScaleAnimation.InterpolationMode = 1;
                        }
                        break;
                }

                // 基準値 - アニメーションテーブル.
                newEmitter.EmitterScaleData.EmitterScaleBasicData.BasicScaleAnimation.AnimationTable.Clear();

                // 寿命 - アニメーションテーブル.
                for (int i = 0; i < ptcl_scale_x.KeyFrames.Count; ++i)
                {
                    NWCore.DataModel.FloatAnimTableKeyFrameData keyX = ptcl_scale_x.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;
                    NWCore.DataModel.FloatAnimTableKeyFrameData keyY = ptcl_scale_y.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;

                    newEmitter.EmitterScaleData.EmitterScaleBasicData.BasicScaleAnimation.AnimationTable.AddKeyFrame(
                        keyX.KeyFrame,
                        keyX.Value,
                        keyY.Value,
                        0.0f,
                        0.0f,
                        false);
                }

            }


            // ランダム量(%)
            newEmitter.EmitterScaleData.EmitterScaleBasicData.ScaleBasisRandom.X = oldEmitter.UI_scaleRandVec.X;
            newEmitter.EmitterScaleData.EmitterScaleBasicData.ScaleBasisRandom.Y = oldEmitter.UI_scaleRandVec.Y;


            //**********************************
            //  速度アニメ.
            //**********************************
            // 速度をYスケールに加算.
            newEmitter.EmitterScaleData.EmitterScaleAnimationData.ScaleVelAddY = oldEmitter.AnimEditData.UI_scaleVelAddY;

            // 速度に有効値が設定されていたらフラグをON
            if (Math.Abs(oldEmitter.AnimEditData.UI_scaleVelAddY) > 0.0001f)
            {
                newEmitter.EmitterScaleData.EmitterScaleAnimationData.EnableScaleVelAddY = true;
            }

            /* IN/OUT形式のアニメーションはRemakeで廃止 */

            //**********************************
            //  スケールアニメ.
            //**********************************
            // アニメタイプ.
            // InOutアニメエディタ
            if (oldEmitter.AnimEditData.UI_scaleAnimType == DataModel.ParticleAnimationTypes.InOutAnim)
            {
                newEmitter.EmitterScaleData.EmitterScaleAnimationCurveData.AnimationTable.Clear();

                // key1 初期スケール(%)
                if (oldEmitter.AnimEditData.UI_scaleTime1 != 0)
                {
                    newEmitter.EmitterScaleData.EmitterScaleAnimationCurveData.AnimationTable.AddKeyFrame(
                        0,
                        oldEmitter.AnimEditData.UI_scale1.X / 100.0f,
                        oldEmitter.AnimEditData.UI_scale1.Y / 100.0f,
                        0.0f,
                        0.0f,
                        false);
                }
                // key2 中間スケール(%)
                {
                    newEmitter.EmitterScaleData.EmitterScaleAnimationCurveData.AnimationTable.AddKeyFrame(
                        oldEmitter.AnimEditData.UI_scaleTime1,
                        oldEmitter.AnimEditData.UI_scale2Relative.X / 100.0f,
                        oldEmitter.AnimEditData.UI_scale2Relative.Y / 100.0f,
                        0.0f,
                        0.0f,
                        false);
                }
                // key3 中間スケール(%)
                {
                    newEmitter.EmitterScaleData.EmitterScaleAnimationCurveData.AnimationTable.AddKeyFrame(
                        oldEmitter.AnimEditData.UI_scaleTime2,
                        oldEmitter.AnimEditData.UI_scale2Relative.X / 100.0f,
                        oldEmitter.AnimEditData.UI_scale2Relative.Y / 100.0f,
                        0.0f,
                        0.0f,
                        false);
                }

                // key4 終了スケール(%)
                if (oldEmitter.AnimEditData.UI_scaleTime2 != 100)
                {
                    newEmitter.EmitterScaleData.EmitterScaleAnimationCurveData.AnimationTable.AddKeyFrame(
                        100,
                        oldEmitter.AnimEditData.UI_scale3.X / 100.0f,
                        oldEmitter.AnimEditData.UI_scale3.Y / 100.0f,
                        0.0f,
                        0.0f,
                        false);
                }
            }

            // 8キーアニメエディタ.
            if (oldEmitter.AnimEditData.UI_scaleAnimType == DataModel.ParticleAnimationTypes.AnimTable)
            {
                NWCore.DataModel.AnimTableData anim_8key_scale_x = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.anim_8key_scale_x);
                NWCore.DataModel.AnimTableData anim_8key_scale_y = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.anim_8key_scale_y);

                newEmitter.EmitterScaleData.EmitterScaleAnimationCurveData.Randomize = (anim_8key_scale_x.IsRandomStartFrameEnabled ) ? 1 : 0;

                newEmitter.EmitterScaleData.EmitterScaleAnimationCurveData.LoopMode = (anim_8key_scale_x.IsLoop) ? 1 : 0;

                newEmitter.EmitterScaleData.EmitterScaleAnimationCurveData.LoopEnd = (anim_8key_scale_x.NumLoopFrames);

                newEmitter.EmitterScaleData.EmitterScaleAnimationCurveData.AnimationTable.Clear();

                for (int i = 0; i < anim_8key_scale_x.KeyFrames.Count; ++i)
                {
                    NWCore.DataModel.FloatAnimTableKeyFrameData keyX = anim_8key_scale_x.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;
                    NWCore.DataModel.FloatAnimTableKeyFrameData keyY = anim_8key_scale_y.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;

                    newEmitter.EmitterScaleData.EmitterScaleAnimationCurveData.AnimationTable.AddKeyFrame(
                        keyX.KeyFrame,
                        keyX.Value,
                        keyY.Value,
                        0.0f,
                        0.0f,
                        false );
                }
            }

            // 2キーで定数値の場合、キー数を1にする。
            if (newEmitter.EmitterScaleData.EmitterScaleAnimationCurveData.AnimationTable.Count == 2)
            {
                var keyA = newEmitter.EmitterScaleData.EmitterScaleAnimationCurveData.AnimationTable[0];
                var keyB = newEmitter.EmitterScaleData.EmitterScaleAnimationCurveData.AnimationTable[1];
                if (keyA.Value.Equals(keyB.Value))
                {
                    newEmitter.EmitterScaleData.EmitterScaleAnimationCurveData.AnimationTable.RemoveKeyFrameAt(1);
                }
            }

        }

        #endregion

        #region 回転

        /// <summary>
        /// 【エミッタ】回転タブの変換処理です.
        /// </summary>
        /// <param name="oldEmitter">旧型式.</param>
        /// <param name="newEmitter">新形式.</param>
        private void ConvertEmitterRotationTab(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            //*********************************
            //  基本設定.
            //*********************************
            // 回転タイプ.
            switch (oldEmitter.UI_rotType)
            {
                case NWCore.DataModel.PtclRotType.RotateYZX:
                    newEmitter.EmitterRotationData.EmitterRotationBasicData.RotationType = 4;
                    break;
                case NWCore.DataModel.PtclRotType.RotateXYZ:
                    newEmitter.EmitterRotationData.EmitterRotationBasicData.RotationType = 5;
                    break;
                case NWCore.DataModel.PtclRotType.RotateZXY:
                    newEmitter.EmitterRotationData.EmitterRotationBasicData.RotationType = 6;
                    break;
            }

            //*********************************
            //  回転軸
            //*********************************
            if ( oldEmitter.UI_rotType == NWCore.DataModel.PtclRotType.RotateX )
            {
                newEmitter.EmitterRotationData.EmitterRotationBasicData.EnableRotationAxisX = true;
            }

            if (oldEmitter.UI_rotType == NWCore.DataModel.PtclRotType.RotateY)
            {
                newEmitter.EmitterRotationData.EmitterRotationBasicData.EnableRotationAxisY = true;
            }

            if (oldEmitter.UI_rotType == NWCore.DataModel.PtclRotType.RotateZ)
            {
                newEmitter.EmitterRotationData.EmitterRotationBasicData.EnableRotationAxisZ = true;
            }

            if ((oldEmitter.UI_rotType == NWCore.DataModel.PtclRotType.RotateYZX) ||
                (oldEmitter.UI_rotType == NWCore.DataModel.PtclRotType.RotateXYZ) ||
                (oldEmitter.UI_rotType == NWCore.DataModel.PtclRotType.RotateZXY) )
            {
                newEmitter.EmitterRotationData.EmitterRotationBasicData.EnableRotationAxisX = true;
                newEmitter.EmitterRotationData.EmitterRotationBasicData.EnableRotationAxisY = true;
                newEmitter.EmitterRotationData.EmitterRotationBasicData.EnableRotationAxisZ = true;
            }

            //*********************************
            //  回転方向ランダム.
            //*********************************
            // X
            newEmitter.EmitterRotationData.EmitterRotationXData.EnableRotationRandom = (oldEmitter.UI_rotDirRand.X > 0) ? true : false;

            // Y
            newEmitter.EmitterRotationData.EmitterRotationYData.EnableRotationRandom = (oldEmitter.UI_rotDirRand.Y > 0) ? true : false;

            // Z
            newEmitter.EmitterRotationData.EmitterRotationZData.EnableRotationRandom = (oldEmitter.UI_rotDirRand.Z > 0) ? true : false;

            //*********************************
            //  初期回転.
            //*********************************
            // 基準 - X
            newEmitter.EmitterRotationData.EmitterRotationXData.InitialRotation = IndexToDegree(oldEmitter.UI_initRot.X, true);

            // 基準 - Y
            newEmitter.EmitterRotationData.EmitterRotationYData.InitialRotation = IndexToDegree(oldEmitter.UI_initRot.Y, true);

            // 基準 - Z
            newEmitter.EmitterRotationData.EmitterRotationZData.InitialRotation = IndexToDegree(oldEmitter.UI_initRot.Z, true);

            // ランダム - X
            newEmitter.EmitterRotationData.EmitterRotationXData.InitialRotationRandom = IndexToDegree(oldEmitter.UI_initRotRand.X, false);

            // ランダム - Y
            newEmitter.EmitterRotationData.EmitterRotationYData.InitialRotationRandom = IndexToDegree(oldEmitter.UI_initRotRand.Y, false);

            // ランダム - Z
            newEmitter.EmitterRotationData.EmitterRotationZData.InitialRotationRandom = IndexToDegree(oldEmitter.UI_initRotRand.Z, false);


            //*********************************
            //  回転速度.
            //*********************************
            // 減衰率.
            newEmitter.EmitterRotationData.EmitterRotationBasicData.RotationAttenuation = oldEmitter.UI_rotResist;

            // 基準 - X
            newEmitter.EmitterRotationData.EmitterRotationXData.RotationSpeed = IndexToDegree(oldEmitter.UI_rotVel.X, true);

            // 基準 - Y
            newEmitter.EmitterRotationData.EmitterRotationYData.RotationSpeed = IndexToDegree(oldEmitter.UI_rotVel.Y, true);

            // 基準 - Z
            newEmitter.EmitterRotationData.EmitterRotationZData.RotationSpeed = IndexToDegree(oldEmitter.UI_rotVel.Z, true);

            // ランダム - X
            newEmitter.EmitterRotationData.EmitterRotationXData.RotationSpeedRandom = IndexToDegree(oldEmitter.UI_rotVelRand.X, false);

            // ランダム - Y
            newEmitter.EmitterRotationData.EmitterRotationYData.RotationSpeedRandom = IndexToDegree(oldEmitter.UI_rotVelRand.Y, false);

            // ランダム - Z
            newEmitter.EmitterRotationData.EmitterRotationZData.RotationSpeedRandom = IndexToDegree(oldEmitter.UI_rotVelRand.Z, false);
        }

        #endregion

        #region 終了処理

        /// <summary>
        /// 【エミッタ】終了処理タブの変換処理です.
        /// </summary>
        /// <param name="oldEmitter">旧型式.</param>
        /// <param name="newEmitter">新形式.</param>
        private void ConvertEmitterEndProcessTab(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            //*********************************
            // fadeが呼ばれた後の処理.
            //*********************************
            // フェードタイプ.
            switch (oldEmitter.UI_fadeOutType)
            {
                case DataModel.FadeOutType.AlphaFade:
                    {
                        newEmitter.EmitterBasicSettingData.EmitterBasicBasicData.EnableAlphaFade = true;
                        newEmitter.EmitterBasicSettingData.EmitterBasicBasicData.EnableEndProcess = false;
                    }
                    break;

                case DataModel.FadeOutType.StopEmit:
                    {
                        newEmitter.EmitterBasicSettingData.EmitterBasicBasicData.EnableAlphaFade = false;
                        newEmitter.EmitterBasicSettingData.EmitterBasicBasicData.EnableEndProcess = true;
                    }
                    break;

                case DataModel.FadeOutType.StopEmitAlphaFade:
                    {
                        newEmitter.EmitterBasicSettingData.EmitterBasicBasicData.EnableAlphaFade = true;
                        newEmitter.EmitterBasicSettingData.EmitterBasicBasicData.EnableEndProcess = true;
                    }
                    break;
            }

            // αフェード時間.
            newEmitter.EmitterBasicSettingData.EmitterBasicBasicData.AlphaFadeTime = oldEmitter.UI_fadeAlphaFrameInFade;
        }

        #endregion

        #region カスタムアクション

        /// <summary>
        /// 【エミッタ】カスタムアクションタブの変換処理です.
        /// </summary>
        /// <param name="oldEmitter">旧型式.</param>
        /// <param name="newEmitter">新形式.</param>
        private void ConvertEmitterCustomActionTab(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            bool usingDefaultSetting = false;
            if (oldEmitter.UserCallbackID == -1)
            {
                var oldGroup = oldEmitter.UserDataGroups[0];
                usingDefaultSetting = oldGroup.ParamsF.Any(f => Math.Abs(f) > 0.0001f);
                usingDefaultSetting |= oldGroup.ParamsI.Any(i => i != 0);
                usingDefaultSetting |= oldGroup.BitFlags.Any(b => b);
                if (!usingDefaultSetting)
                {
                    return;
                }
            }

            var newCustomActionData = new CustomActionData();
            newEmitter.CustomActionData = newCustomActionData;
            newCustomActionData.Setting0 = null;
            newCustomActionData.Setting1 = null;
            newCustomActionData.Setting2 = null;
            newCustomActionData.Setting3 = null;
            newCustomActionData.Setting4 = null;
            newCustomActionData.Setting5 = null;
            newCustomActionData.Setting6 = null;
            newCustomActionData.Setting7 = null;
            newCustomActionData.DefaultSetting = null;
            newCustomActionData.SelectedSettingIndex = oldEmitter.UserCallbackID;

            Action<DataModel.EmitterUserData, CustomActionConvertingData> convert = (oldGroup, convertingData) =>
            {
                //----- カスタムシェーダフラグ ----- (チェックボックス)
                uint flags = 0;
                for (uint j = 0; j < oldGroup.BitFlags.Length; ++j)
                {
                    if (oldGroup.BitFlags[j])
                    {
                        flags += (uint)Math.Pow(2, j);
                    }
                }

                convertingData.Flags = flags;

                //----- 数値(int) -----
                for (uint j = 0; j < oldGroup.ParamsI.Length; ++j)
                {
                    convertingData.IntParams.Add((int)oldGroup.ParamsI[j]);
                }

                //----- 数値(float) -----
                for (uint j = 0; j < oldGroup.ParamsF.Length; ++j)
                {
                    convertingData.FloatParams.Add(oldGroup.ParamsF[j]);
                }
            };

            if (usingDefaultSetting)
            {
                var convertingData = new CustomActionConvertingData
                {
                    Name = "CustomActionDefaultSettingData",
                    Visible = true,
                    FlagsVisible = true,
                    FlagsVisibleList = null,
                    IntParamsVisible = true,
                    IntParamsVisibleList = null,
                    FloatParamsVisible = true,
                    FloatParamsVisibleList = null,
                };

                convert(oldEmitter.UserDataGroups[0], convertingData);

                newCustomActionData.DefaultSetting = convertingData;
            }

            for (uint i = 1; i < oldEmitter.UserDataGroups.Length; ++i)
            {
                var uddDef = this.customActionTypeList[(int)i];
                if (uddDef == null || string.IsNullOrEmpty(uddDef.Name))
                {
                    continue;
                }

                var oldGroup = oldEmitter.UserDataGroups[i];
                var convertingData = new CustomActionConvertingData
                {
                    Name = uddDef.Name,
                    Visible = uddDef.Visible,
                    FlagsVisible = uddDef.FlagsVisible,
                    FlagsVisibleList = uddDef.FlagsVisibleList,
                    IntParamsVisible = uddDef.IntParamsVisible,
                    IntParamsVisibleList = uddDef.IntParamsVisibleList,
                    FloatParamsVisible = uddDef.FloatParamsVisible,
                    FloatParamsVisibleList = uddDef.FloatParamsVisibleList,
                };

                convert(oldGroup, convertingData);

                switch (i)
                {
                    case 1:
                        newCustomActionData.Setting0 = convertingData;
                        break;
                    case 2:
                        newCustomActionData.Setting1 = convertingData;
                        break;
                    case 3:
                        newCustomActionData.Setting2 = convertingData;
                        break;
                    case 4:
                        newCustomActionData.Setting3 = convertingData;
                        break;
                    case 5:
                        newCustomActionData.Setting4 = convertingData;
                        break;
                    case 6:
                        newCustomActionData.Setting5 = convertingData;
                        break;
                    case 7:
                        newCustomActionData.Setting6 = convertingData;
                        break;
                    case 8:
                        newCustomActionData.Setting7 = convertingData;
                        break;
                }
            }
        }

        #endregion

        #region カスタムシェーダ

        /// <summary>
        /// 【エミッタ】カスタムシェーダタブの変換処理です.
        /// </summary>
        /// <param name="oldEmitter">旧型式.</param>
        /// <param name="newEmitter">新形式.</param>
        private void ConvertEmitterCustomShaderTab(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            var newCustomShaderSetting = newEmitter.EmitterCustomShaderData;
            newCustomShaderSetting.Settings = new ArrayCollection<EmitterCustomShaderSettingData>(8);
            newCustomShaderSetting.SelectedSettingIndex = (int)oldEmitter.ShaderSetting - 1;

            for (uint i = 0; i < oldEmitter.UserShaderParamGroups.Length; ++i)
            {
                var usdDef = this.customShaderTypeList[(int)i];
                if (string.IsNullOrEmpty(usdDef.Name))
                {
                    if ((int)oldEmitter.ShaderSetting - 1 >= i)
                    {
                        --newCustomShaderSetting.SelectedSettingIndex;
                    }

                    continue;
                }

                var oldGroup = oldEmitter.UserShaderParamGroups[i];

                var convertingData = new EmitterCustomShaderConvertingData
                {
                    Name = usdDef.Name,
                    Visible = usdDef.Visible,
                    SwitchVisible = usdDef.SwitchVisible,
                    SwitchesVisibleList = usdDef.SwitchesVisibleList,
                    ParamVisible = usdDef.ParamVisible,
                    ParamsVisibleList = usdDef.ParamsVisibleList
                };

                //----- カスタムシェーダフラグ ----- (チェックボックス)
                uint flags = 0;
                for (uint j = 0; j < oldGroup.BitFlags.Length; ++j)
                {
                    if (oldGroup.BitFlags[j])
                    {
                        flags += (uint)Math.Pow(2, j);
                    }
                }

                convertingData.Flags = flags;

                //----- カスタムシェーダフラグ ----- (ラジオボタン)
                for (uint j = 0; j < oldGroup.Bytes.Length; ++j)
                {
                    convertingData.Switches.Add(oldGroup.Bytes[j] >> (8 * (int)j));
                }

                //----- カスタムシェーダパラメータ -----
                for (uint j = 0; j < oldGroup.Params.Length; ++j)
                {
                    convertingData.Params.Add(oldGroup.Params[j]);
                }

                newCustomShaderSetting.Settings[(int)i] = convertingData;
            }
        }

        #endregion

        #endregion

        #region Child

        /// <summary>
        /// チャイルドデータを新形式に変換します.
        /// </summary>
        /// <param name="oldEmitter">旧型式.</param>
        /// <param name="newEmitter">新型式.</param>
        private void ConvertChild(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            // チャイルド名.
            newEmitter.Name = string.Copy(oldEmitter.Name) + "_Child";

            // チャイルド基本設定.
            this.ConvertChildBasicSettingsTab(oldEmitter, newEmitter);

            // 放出.
            this.ConvertChildEmissionTab(oldEmitter, newEmitter);

            // パーティクル.
            this.ConvertChildParticleTab(oldEmitter, newEmitter);

            // コンバイナ.
            this.ConvertChildCombinerTab(oldEmitter, newEmitter);

            // テクスチャ.
            this.ConvertChildTextureTab(oldEmitter, newEmitter);

            // カラー.
            this.ConvertChildColorTab(oldEmitter, newEmitter);

            // スケール.
            this.ConvertChildScaleTab(oldEmitter, newEmitter);

            // 回転.
            this.ConvertChildRotationTab(oldEmitter, newEmitter);

            // カスタムシェーダ.
            this.ConvertChildCustomShaderTab(oldEmitter, newEmitter);
        }

        #region チャイルド基本設定

        /// <summary>
        /// 【チャイルド】チャイルド基本設定タブの変換処理です.
        /// </summary>
        /// <param name="oldEmitter">旧型式.</param>
        /// <param name="newEmitter">新形式.</param>
        private void ConvertChildBasicSettingsTab(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            //*********************************
            //  基本.
            //*********************************
            // チャイルド有効.
            // oldEmitter.ChildFlag.Enable;

            //*********************************
            // 親との関連.
            //*********************************
            // 親とチャイルドの描画順
            newEmitter.EmitterBasicSettingData.EmitterBasicInheritanceData.EnablePreDraw = oldEmitter.ChildFlag.PreChildDraw;

            // チャイルド追従タイプ.
            newEmitter.EmitterBasicSettingData.EmitterBasicBasicData.ParticleFollowType = 1;

            // 親のフィールドを適用.
            if (oldEmitter.ChildFlag.ParentField)
            {
                this.ConvertField(oldEmitter, newEmitter);
            }
        }

        #endregion

        #region 放出

        /// <summary>
        /// 【チャイルド】放出タブの変換処理です.
        /// </summary>
        /// <param name="oldEmitter">旧型式.</param>
        /// <param name="newEmitter">新形式.</param>
        private void ConvertChildEmissionTab(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            //********************************
            //  放出タイミング.
            //********************************
            // 放出量.
            newEmitter.EmitterEmissionData.EmitterEmissionTimingData.EmitTimeEmissionRate = oldEmitter.ChildData.EmitRate;

            // 放出タイミング(%).
            newEmitter.EmitterEmissionData.EmitterEmissionTimingData.EmitTimingRate = (uint)oldEmitter.ChildData.EmitTiming;

            // 放出間隔.
            newEmitter.EmitterEmissionData.EmitterEmissionTimingData.EmitTimeDistUnit = oldEmitter.ChildData.EmitStep;


            //********************************
            //  放出時の位置と初速.
            //********************************
            // ランダム位置.
            newEmitter.EmitterEmitterData.EmitterEmitterShapeData.RandomPosition = oldEmitter.ChildData.InitPosRand;

            // 親の速度を継承.
            newEmitter.EmitterBasicSettingData.EmitterBasicInheritanceData.EnableVelocity = oldEmitter.ChildFlag.VelInherit;

            // 速度継承率.
            newEmitter.EmitterBasicSettingData.EmitterBasicInheritanceData.VelocityInheritanceRate = oldEmitter.ChildData.VelInheritRate;

            // 全方向初速.
            newEmitter.EmitterEmissionData.EmitterEmissionVelocityData.OmnidirectionalVelocity = oldEmitter.ChildData.FigureVelocity;

            // 拡散初速.
            newEmitter.EmitterEmissionData.EmitterEmissionVelocityData.DiffusionVelocity.X = oldEmitter.ChildData.RandVel.X;
            newEmitter.EmitterEmissionData.EmitterEmissionVelocityData.DiffusionVelocity.Y = oldEmitter.ChildData.RandVel.Y;
            newEmitter.EmitterEmissionData.EmitterEmissionVelocityData.DiffusionVelocity.Z = oldEmitter.ChildData.RandVel.Z;

            // 空気抵抗.
            newEmitter.EmitterEmissionData.EmitterEmissionGravityData.AirRegist = oldEmitter.ChildData.AirRegist;


            //********************************
            //  重力.
            //********************************
            // 座標系
            newEmitter.EmitterEmissionData.EmitterEmissionGravityData.EnableWorldCoordinate = oldEmitter.ChildFlag.WorldField;

            // 大きさ
            newEmitter.EmitterEmissionData.EmitterEmissionGravityData.GravityVolume = oldEmitter.ChildAssistData.UI_childGravity;

            {
                NWCore.DataModel.AnimTableData gravity = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.gravity_child);

                // 大きさ - エミッタ時間アニメ
                newEmitter.EmitterEmissionData.EmitterEmissionGravityData.GravityAnimation.EnableAnimation = gravity.IsEnabled;

                newEmitter.EmitterEmissionData.EmitterEmissionGravityData.GravityAnimation.LoopMode = (gravity.IsLoop) ? 1 : 0;

                switch (gravity.InterpolationType)
                {
                    // リニア.
                    case DataModel.AnimInterpolationTypes.Linear:
                        {
                            newEmitter.EmitterEmissionData.EmitterEmissionGravityData.GravityAnimation.InterpolationMode = 0;
                        }
                        break;

                    // スムース.
                    case DataModel.AnimInterpolationTypes.Smooth:
                        {
                            newEmitter.EmitterEmissionData.EmitterEmissionGravityData.GravityAnimation.InterpolationMode = 1;
                        }
                        break;
                }

                // テーブルを一端クリア.
                newEmitter.EmitterEmissionData.EmitterEmissionGravityData.GravityAnimation.AnimationTable.Clear();

                // 放出レート - アニメーションテーブル.
                for (int i = 0; i < gravity.KeyFrames.Count; ++i)
                {
                    NWCore.DataModel.FloatAnimTableKeyFrameData key = gravity.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;

                    newEmitter.EmitterEmissionData.EmitterEmissionGravityData.GravityAnimation.AnimationTable.AddKeyFrame(
                        key.KeyFrame,
                        key.Value,
                        0.0f,
                        0.0f,
                        0.0f,
                        false);
                }
            }

            // 向き.
            newEmitter.EmitterEmissionData.EmitterEmissionGravityData.GravityDir.X = oldEmitter.ChildAssistData.UI_childGravityDir.X;
            newEmitter.EmitterEmissionData.EmitterEmissionGravityData.GravityDir.Y = oldEmitter.ChildAssistData.UI_childGravityDir.Y;
            newEmitter.EmitterEmissionData.EmitterEmissionGravityData.GravityDir.Z = oldEmitter.ChildAssistData.UI_childGravityDir.Z;
        }

        #endregion

        #region パーティクル

        /// <summary>
        /// 【チャイルド】パーティクルタブの変換処理です.
        /// </summary>
        /// <param name="oldEmitter">旧型式.</param>
        /// <param name="newEmitter">新形式.</param>
        private void ConvertChildParticleTab(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            //********************************
            //  寿命設定.
            //********************************
            // 寿命.
            newEmitter.EmitterParticleData.EmitterParticleLifeData.Life = oldEmitter.ChildAssistData.UI_childLife;

            //********************************
            //  運動量ランダム.
            //********************************
            // ランダム値.
            newEmitter.EmitterEmissionData.EmitterEmissionGravityData.MomentumRandom =
                MathUtility.Clamp(oldEmitter.ChildData.DynamicsRandom, 0.0f, 1.0f);

            //********************************
            //  パーティクル形状.
            //********************************
            // 形状.
            newEmitter.EmitterParticleData.EmitterParticleShapeData.ShapeType = (int)oldEmitter.ChildData.MeshType;

            // トリミング.
            newEmitter.EmitterParticleData.EmitterParticleShapeData.EnableTrimming = oldEmitter.ChildData.TextureTrimming;

            // ファイルパス.
            newEmitter.EmitterParticleData.EmitterParticleShapeData.PrimitiveFilePath = GetFileName(oldEmitter.ChildData.BillboardPrimitiveFileSource);

            // プリミティブスケールZ
            newEmitter.EmitterParticleData.EmitterParticleShapeData.PrimitiveScaleZType = (int)oldEmitter.ChildData.PrimitiveScaleType;

            // パーティクルタイプ.
            newEmitter.EmitterParticleData.EmitterParticleShapeData.ParticleType = (int)oldEmitter.ChildData.BillboardType;
            switch (oldEmitter.ChildData.BillboardType)
            {
                case NWCore.DataModel.BillboardType.Billboard:           //  普通のビルボード
                    {
                        newEmitter.EmitterParticleData.EmitterParticleShapeData.ParticleType = 0;
                    }
                    break;
                case NWCore.DataModel.BillboardType.PolygonXY:          //  XY平面のポリゴン
                    {
                        newEmitter.EmitterParticleData.EmitterParticleShapeData.ParticleType = 3;
                    }
                    break;
                case NWCore.DataModel.BillboardType.PolygonXZ:          //  XZ平面のポリゴン
                    {
                        newEmitter.EmitterParticleData.EmitterParticleShapeData.ParticleType = 4;
                    }
                    break;
                case NWCore.DataModel.BillboardType.VelLook:           //  速度方向を向くビルボード
                    {
                        newEmitter.EmitterParticleData.EmitterParticleShapeData.ParticleType = 5;
                    }
                    break;
                case NWCore.DataModel.BillboardType.VelLookPolygon:    //  速度方向を向くポリゴン
                    {
                        newEmitter.EmitterParticleData.EmitterParticleShapeData.ParticleType = 6;
                    }
                    break;
                case NWCore.DataModel.BillboardType.YAxisBillboard:    //  Y軸ビルボード
                    {
                        newEmitter.EmitterParticleData.EmitterParticleShapeData.ParticleType = 2;
                    }
                    break;
                case NWCore.DataModel.BillboardType.ComplexBillboard:   //  高機能ビルボード
                    {
                        newEmitter.EmitterParticleData.EmitterParticleShapeData.ParticleType = 1;
                    }
                    break;
            }

            // パーティクル中心.
            newEmitter.EmitterParticleData.EmitterParticleShapeData.Scale.X = oldEmitter.ChildAssistData.UI_childRotBasis.X;
            newEmitter.EmitterParticleData.EmitterParticleShapeData.Scale.Y = oldEmitter.ChildAssistData.UI_childRotBasis.Y;


            //********************************
            //  レンダーステート.
            //********************************
            // ブレンドタイプ.
            switch (oldEmitter.ChildAssistData.UI_childBlendType)
            {
                case DataModel.BlendType.Normal:
                    {
                        newEmitter.EmitterBasicSettingData.EmitterBasicRenderData.BlendType = 0;
                    }
                    break;

                case DataModel.BlendType.Add:
                    {
                        newEmitter.EmitterBasicSettingData.EmitterBasicRenderData.BlendType = 1;
                    }
                    break;

                case DataModel.BlendType.Sub:
                    {
                        newEmitter.EmitterBasicSettingData.EmitterBasicRenderData.BlendType = 2;
                    }
                    break;

                case DataModel.BlendType.Multiply:
                    {
                        newEmitter.EmitterBasicSettingData.EmitterBasicRenderData.BlendType = 3;
                    }
                    break;

                case DataModel.BlendType.Screen:
                    {
                        newEmitter.EmitterBasicSettingData.EmitterBasicRenderData.BlendType = 4;
                    }
                    break;
            }
        //    newEmitter.EmitterBasicSettingData.EmitterBasicRenderData.BlendType = (int)oldEmitter.ChildAssistData.UI_childBlendType;

            // Zバッファー，αテスト.
            newEmitter.EmitterBasicSettingData.EmitterBasicRenderData.ZbufferAlphaTestType = (int)oldEmitter.ChildData.ZBufATestType;

            // 表示面.
            newEmitter.EmitterBasicSettingData.EmitterBasicRenderData.DisplaySurface = (int)oldEmitter.ChildData.DisplaySide;
        }

        #endregion

        #region コンバイナ

        /// <summary>
        /// 【チャイルド】コンバイナタブの変換処理です.
        /// </summary>
        /// <param name="oldEmitter"></param>
        /// <param name="newEmitter"></param>
        private void ConvertChildCombinerTab(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            //*******************************
            //  パーティクルシェーダ.
            //*******************************
            // シェーダタイプ.
            newEmitter.EmitterCombinerData.EmitterCombinerShaderData.ShaderType = (int)oldEmitter.ChildData.ShaderType;


            //*******************************
            //  アルファ処理.
            //*******************************
            //----- ソフトパーティクル -----
            // 処理方法
            switch (oldEmitter.ChildData.SoftParticleProcessingSettings)
            {
                case NWCore.DataModel.SoftParticleProcessingSettings.Disable:
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.SoftParticleProcessType = 0;
                    break;
                case NWCore.DataModel.SoftParticleProcessingSettings.Vertex:
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.SoftParticleProcessType = 1;
                    break;
                case NWCore.DataModel.SoftParticleProcessingSettings.Fragment:
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.SoftParticleProcessType = 2;
                    break;
                case NWCore.DataModel.SoftParticleProcessingSettings.LensFlare:
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.SoftParticleProcessType = 3;
                    break;
            }

    //        newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.SoftParticleProcessType = (int)oldEmitter.ChildData.SoftParticleProcessingSettings;

            // フェード開始α値オフセット
            newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.AlphaFadeStartAlphaOffset = oldEmitter.ChildData.SoftParticleAlphaOffset;

            // サンプリング半径.
            newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.AlphaSamplingRadius = oldEmitter.ChildData.SoftParticleSampleParam;

            // フェード開始距離.
            newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.FeedStartDistance = oldEmitter.ChildData.SoftParticleFadeDistance;

            // ボリューム値.
            newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.VolumeValue = oldEmitter.ChildData.SoftParticleVolumeParam;

            //----- 近距離アルファ -----
            // 有効.
            newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.EnableNearAlpha = oldEmitter.ChildData.NearAlphaEnabled;

            // アルファが0になる距離.
            newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.NearAlphaZeroDistance = oldEmitter.ChildData.NearAlphaMin;

            // アルファが1になる距離.
            newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.NearAlphaOneDistance = oldEmitter.ChildData.NearAlphaMax;

            //----- フレネルアルファ -----
            // 処理方法.
            switch (oldEmitter.ChildData.FresnelAlphaType)
            {
                case DataModel.FresnelAlphaTypes.Disable:
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.FresnelAlphaProcessType = 0;
                    break;
                case DataModel.FresnelAlphaTypes.Vertex:
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.FresnelAlphaProcessType = 1;
                    break;
                case DataModel.FresnelAlphaTypes.Fragment:
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.FresnelAlphaProcessType = 2;
                    break;
            }

    //        newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.FresnelAlphaProcessType = (int)oldEmitter.ChildData.FresnelAlphaType;

            // アルファが最小になる角度.
            newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.MinimumAlphaAngle = oldEmitter.FresnelAlphaMin;

            // アルファが最大になる角度.
            newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.MaximumAlphaAngle = oldEmitter.FresnelAlphaMax;

            //----- デカール -----
            // 有効.
            newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.EnableDecal = oldEmitter.ChildData.DecalEnabled;

            // アルファが1になる範囲.
            newEmitter.EmitterColorData.ParticleColor.EmitterColorAlphaProcessData.DecalAlphaOneDistance = oldEmitter.ChildData.DecalAlphaMin;


            //******************************
            //  カラーコンバイナ.
            //******************************
            // [テクスチャ0] - RGB/定数1.0
            newEmitter.EmitterCombinerData.EmitterCombinerEquationData.ColorTexture0Channel = (int)oldEmitter.ChildData.UI_combinerTextureColorUsage;

            // [カラー計算式]
            newEmitter.EmitterCombinerData.EmitterCombinerEquationData.ColorTextureExpression = (int)oldEmitter.ChildAssistData.UI_childColorType;

            // [プリミティブ頂点] - 上段との合成.
            newEmitter.EmitterCombinerData.EmitterCombinerEquationData.ColorPrimitiveExpression = (int)oldEmitter.ChildData.UI_combinerPrimitiveColorBlend;

            // [プリミティブ頂点] - RGB/定数1.0
            newEmitter.EmitterCombinerData.EmitterCombinerEquationData.ColorPrimitiveChannel = (int)oldEmitter.ChildData.UI_combinerPrimitiveColorUsage;


            //******************************
            // アルファコンバイナ.
            //******************************
            // [テクスチャ0】- A/R/定数1.0
            newEmitter.EmitterCombinerData.EmitterCombinerEquationData.AlphaTexture0Channel = (int)oldEmitter.ChildData.UI_combinerTextureAlphaUsage;

            // [プリミティブ頂点] - 上段との合成.
            newEmitter.EmitterCombinerData.EmitterCombinerEquationData.AlphaPrimitiveExpression = (int)oldEmitter.ChildData.UI_combinerPrimitiveAlphaBlend;

            // [プリミティブ頂点] - A/定数1.0
            // XXX = oldEmitter.ChildData.UI_combinerPrimitiveAlphaUsage;

            // [アルファ計算式]
            switch (oldEmitter.ChildAssistData.UI_childAlphaDstType)
            {
                case NWCore.DataModel.AlphaDst.ModulateTex:
                    newEmitter.EmitterCombinerData.EmitterCombinerEquationData.AlphaExpression = 0;
                    break;
                case NWCore.DataModel.AlphaDst.ModulateTex0AndTex1:
                    newEmitter.EmitterCombinerData.EmitterCombinerEquationData.AlphaExpression = 1;
                    break;
                case NWCore.DataModel.AlphaDst.SubtractFromTex:
                    newEmitter.EmitterCombinerData.EmitterCombinerEquationData.AlphaExpression = 2;
                    break;
                case NWCore.DataModel.AlphaDst.SubtractFromTex0AlphaModulateTex1Alpha:
                    newEmitter.EmitterCombinerData.EmitterCombinerEquationData.AlphaExpression = 3;
                    break;
            }
        }

        #endregion

        #region テクスチャ

        /// <summary>
        /// 【チャイルド】テクスチャタブの変換処理を行います.
        /// </summary>
        /// <param name="oldEmitter">旧型式.</param>
        /// <param name="newEmitter">新形式.</param>
        private void ConvertChildTextureTab(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            //*****************************
            //  ファイル
            //*****************************
            // ファイルパス.
            newEmitter.Textures.Texture0.EmitterTextureFileData.FilePath = GetFileName(oldEmitter.ChildTexPatData.UI_texPatFileName);

            // パターン横分割.
            newEmitter.Textures.Texture0.EmitterTextureFileData.PatternDivU = oldEmitter.ChildTexPatData.UI_texPatDivX;

            // パターン縦分割.
            newEmitter.Textures.Texture0.EmitterTextureFileData.PatternDivV = oldEmitter.ChildTexPatData.UI_texPatDivY;

            // ラップ(U)
            newEmitter.Textures.Texture0.EmitterTextureFileData.WrapTypeU = (int)oldEmitter.ChildTexPatData.UI_texWrapU;

            // ラップ(V)
            newEmitter.Textures.Texture0.EmitterTextureFileData.WrapTypeV = (int)oldEmitter.ChildTexPatData.UI_texWrapV;

            // U座標ランダム反転.
            newEmitter.Textures.Texture0.EmitterTextureFileData.RandomInvU = oldEmitter.ChildTexPatData.UI_texIsURandomInv;

            // V座標ランダム反転
            newEmitter.Textures.Texture0.EmitterTextureFileData.RandomInvV = oldEmitter.ChildTexPatData.UI_texIsVRandomInv;

            // スフィアマップ
            newEmitter.Textures.Texture0.EmitterTextureFileData.SphericalMap = oldEmitter.ChildTexPatData.UI_isSphereMap;

            // 繰り返し数.
            newEmitter.Textures.Texture0.EmitterTextureFileData.RepeatCount = (int)oldEmitter.ChildTexPatData.UI_textureAddressing;

            // フィルタモード.
            newEmitter.Textures.Texture0.EmitterTextureFileData.FilterMode = (int)oldEmitter.ChildTexPatData.UI_texFilterMode;

            // 有効なミップレベル.
            newEmitter.Textures.Texture0.EmitterTextureFileData.AvailableMipMapLevel = oldEmitter.ChildTexPatData.UI_availableMipLevel;

            // ミップマップLODバイアス.
            newEmitter.Textures.Texture0.EmitterTextureFileData.MipMapBias = oldEmitter.ChildTexPatData.UI_mipMapBias;


            //****************************
            //  テクスチャパターンアニメーション.
            //****************************

            //----- パターンテーブル -----
            switch (oldEmitter.ChildTexPatData.UI_texPatAnimMode)
            {
                // ライフフィット.
                case 1:
                    {
                        newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.PatternAnimType = 1;

                        newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.PatternTableCount = (int)oldEmitter.ChildTexPatData.UI_numUseTexPatTbl;
                        newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.RandomPatternTableCount = 0;
                    }
                    break;

                // クランプ
                case 4:
                    {
                        newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.PatternAnimType = 2;

                        newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.PatternTableCount = (int)oldEmitter.ChildTexPatData.UI_numUseTexPatTbl;
                        newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.RandomPatternTableCount = 0;
                    }
                    break;

                // ランダム
                case 3:
                    {
                        newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.PatternAnimType = 4;

                        newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.PatternTableCount = 0;
                        newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.RandomPatternTableCount = (int)oldEmitter.ChildTexPatData.UI_texPatNumPattern;
                    }
                    break;

                // ループ
                case 2:
                    {
                        newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.PatternAnimType = 3;

                        newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.PatternTableCount = (int)oldEmitter.ChildTexPatData.UI_numUseTexPatTbl;
                        newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.RandomPatternTableCount = 0;
                    }
                    break;

                default:
                    {
                        newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.PatternAnimType = 0;
                        newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.PatternTableCount = 0;
                        newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.RandomPatternTableCount = 0;
                    }
                    break;
            }

            // テーブル簡易入力.
            var temp = new EffectMaker.Foundation.Collections.Generic.ArrayCollection<int>(oldEmitter.ChildTexPatData.UI_texPatTbl.Length);
            for (int i = 0; i < temp.Count; ++i)
            {
                temp[i] = oldEmitter.ChildTexPatData.UI_texPatTbl[i];
            }
            newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.PatternTable = temp;

            // 1コマの継続フレーム数.
            newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.DurationCountPerFrame = (int)oldEmitter.ChildTexPatData.UI_texPatFreqFrame;

            // 開始テーブルランダム.
            newEmitter.Textures.Texture0.EmitterTexturePatternAnimationData.EnableSimpleTableRandom = oldEmitter.ChildTexPatData.UI_texPatIsRandom;
        }

        #endregion

        #region カラー

        /// <summary>
        /// 【チャイルド】カラータブの変換処理です.
        /// </summary>
        /// <param name="oldEmitter">旧型式.</param>
        /// <param name="newEmitter">新形式.</param>
        private void ConvertChildColorTab(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            //*********************************
            //  カラー計算.
            //*********************************
            // カラー継承.
            newEmitter.EmitterBasicSettingData.EmitterBasicInheritanceData.EnableColorScale = oldEmitter.ChildAssistData.UI_isChildColorScaleInherited;

            // カラースケール.
            newEmitter.EmitterColorData.ParticleColor.EmitterColorCalculationData.ColorScale = oldEmitter.ChildAssistData.UI_childColorScale;

            // 固定フラグをFE2のデータモデルに繋ぐ
            newEmitter.EmitterColorData.ParticleColor.EmitterColorCalculationData.ColorAlphaFixFlag = oldEmitter.ChildAssistData.UI_childColorAlphaFixFlag;

            //*********************************
            //  カラー・アルファ継承.
            //*********************************
            // 親のカラー0を継承.
            newEmitter.EmitterBasicSettingData.EmitterBasicInheritanceData.EnableColor0 = oldEmitter.ChildFlag.Color0Inherit;

            // 親のカラー1を継承.
            newEmitter.EmitterBasicSettingData.EmitterBasicInheritanceData.EnableColor1 = oldEmitter.ChildFlag.Color1Inherit;

            // 親のアルファ0を継承.
            newEmitter.EmitterBasicSettingData.EmitterBasicInheritanceData.EnableAlpha0 = oldEmitter.ChildFlag.Alpha0Inherit;

            // 親のアルファ1を継承.
            newEmitter.EmitterBasicSettingData.EmitterBasicInheritanceData.EnableAlpha1 = oldEmitter.ChildFlag.Alpha1Inherit;


            //*********************************
            //  カラー・アルファ設定.
            //*********************************

            newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Color0BehaviorType = 0;
            newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Color1BehaviorType = 0;


            // アルファ0挙動タイプ.
            if (oldEmitter.ChildAssistData.Alpha0.AnimType == DataModel.ParticleAnimationTypes.InOutAnim)
                newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Alpha0BehaviorType = 2;
            else
                newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Alpha0BehaviorType = 0;

            //oldEmitter.ChildAssistData.Alpha0.AnimType;

            // アルファ1挙動タイプ.
            if (oldEmitter.ChildAssistData.Alpha1.AnimType == DataModel.ParticleAnimationTypes.InOutAnim)
                newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Alpha1BehaviorType = 2;
            else
                newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Alpha1BehaviorType = 0;
            //oldEmitter.ChildAssistData.Alpha1.AnimType;

            // カラー0設定.
            if (oldEmitter.ChildFlag.Color0Inherit)
            {
                newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Color0Value.R = 1.0f;
                newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Color0Value.G = 1.0f;
                newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Color0Value.B = 1.0f;
            }
            else
            {
                newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Color0Value.R = oldEmitter.ChildAssistData.UI_childConstColor0.R;
                newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Color0Value.G = oldEmitter.ChildAssistData.UI_childConstColor0.G;
                newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Color0Value.B = oldEmitter.ChildAssistData.UI_childConstColor0.B;
            }

            // カラー1設定.
            if (oldEmitter.ChildFlag.Color1Inherit)
            {
                newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Color1Value.R = 1.0f;
                newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Color1Value.G = 1.0f;
                newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Color1Value.B = 1.0f;
            }
            else
            {
                newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Color1Value.R = oldEmitter.ChildAssistData.UI_childConstColor1.R;
                newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Color1Value.G = oldEmitter.ChildAssistData.UI_childConstColor1.G;
                newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Color1Value.B = oldEmitter.ChildAssistData.UI_childConstColor1.B;
            }

            // アルファ0設定.
            if (oldEmitter.ChildFlag.Alpha0Inherit)
                newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Alpha0Value = 1.0f;
            else
                newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Alpha0Value = oldEmitter.ChildAssistData.Alpha0.Alpha2;


            newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.AlphaAnimation0.Clear();

            // インアウトアニメ
            if (oldEmitter.ChildAssistData.Alpha0.AnimType == DataModel.ParticleAnimationTypes.InOutAnim)
            {
                int keyFrame = 0;
                float alphaValue = 0;
                EffectMaker.Foundation.Primitives.ColorRgba keyValue = null;

                // key1
                if (oldEmitter.ChildAssistData.Alpha0.Time1 != 0)
                {
                    alphaValue = oldEmitter.ChildAssistData.Alpha0.Alpha1;
                    keyValue = new EffectMaker.Foundation.Primitives.ColorRgba(1.0f, 1.0f, 1.0f, alphaValue);
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.AlphaAnimation0.AddKeyFrame(keyFrame, keyValue, false);
                }

                // key2
                {
                    keyFrame = oldEmitter.ChildAssistData.Alpha0.Time1;
                    alphaValue = oldEmitter.ChildAssistData.Alpha0.Alpha2;
                    keyValue = new EffectMaker.Foundation.Primitives.ColorRgba(1.0f, 1.0f, 1.0f, alphaValue);
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.AlphaAnimation0.AddKeyFrame(keyFrame, keyValue, false);
                }

                // key3
                if (oldEmitter.ChildAssistData.Alpha0.Time2 != oldEmitter.ChildAssistData.Alpha0.Time1)
                {
                    keyFrame = oldEmitter.ChildAssistData.Alpha0.Time2;
                    alphaValue = oldEmitter.ChildAssistData.Alpha0.Alpha2;
                    keyValue = new EffectMaker.Foundation.Primitives.ColorRgba(1.0f, 1.0f, 1.0f, alphaValue);
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.AlphaAnimation0.AddKeyFrame(keyFrame, keyValue, false);
                }

                // key4
                if (oldEmitter.ChildAssistData.Alpha0.Time2 != 100)
                {
                    keyFrame = 100;
                    alphaValue = oldEmitter.ChildAssistData.Alpha0.Alpha3;
                    keyValue = new EffectMaker.Foundation.Primitives.ColorRgba(1.0f, 1.0f, 1.0f, alphaValue);
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.AlphaAnimation0.AddKeyFrame(keyFrame, keyValue, false);
                }
            }

            // アルファ1設定.
            if (oldEmitter.ChildFlag.Alpha1Inherit)
                newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Alpha1Value = 1.0f;
            else
                newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.Alpha1Value = oldEmitter.ChildAssistData.Alpha1.Alpha2;


            // インアウトアニメ
            if (oldEmitter.ChildAssistData.Alpha1.AnimType == DataModel.ParticleAnimationTypes.InOutAnim)
            {
                int keyFrame = 0;
                float alphaValue = 0;
                EffectMaker.Foundation.Primitives.ColorRgba keyValue = null;

                // key1
                if (oldEmitter.ChildAssistData.Alpha1.Time1 != 0)
                {
                    alphaValue = oldEmitter.ChildAssistData.Alpha1.Alpha1;
                    keyValue = new EffectMaker.Foundation.Primitives.ColorRgba(1.0f, 1.0f, 1.0f, alphaValue);
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.AlphaAnimation1.AddKeyFrame(keyFrame, keyValue, false);
                }

                // key2
                {
                    keyFrame = oldEmitter.ChildAssistData.Alpha1.Time1;
                    alphaValue = oldEmitter.ChildAssistData.Alpha1.Alpha2;
                    keyValue = new EffectMaker.Foundation.Primitives.ColorRgba(1.0f, 1.0f, 1.0f, alphaValue);
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.AlphaAnimation1.AddKeyFrame(keyFrame, keyValue, false);
                }

                // key3
                if (oldEmitter.ChildAssistData.Alpha1.Time2 != oldEmitter.ChildAssistData.Alpha1.Time1)
                {
                    keyFrame = oldEmitter.ChildAssistData.Alpha1.Time2;
                    alphaValue = oldEmitter.ChildAssistData.Alpha1.Alpha2;
                    keyValue = new EffectMaker.Foundation.Primitives.ColorRgba(1.0f, 1.0f, 1.0f, alphaValue);
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.AlphaAnimation1.AddKeyFrame(keyFrame, keyValue, false);
                }

                // key4
                if (oldEmitter.ChildAssistData.Alpha1.Time2 != 100)
                {
                    keyFrame = 100;
                    alphaValue = oldEmitter.ChildAssistData.Alpha1.Alpha3;
                    keyValue = new EffectMaker.Foundation.Primitives.ColorRgba(1.0f, 1.0f, 1.0f, alphaValue);
                    newEmitter.EmitterColorData.ParticleColor.EmitterColorSettingsData.AlphaAnimation1.AddKeyFrame(keyFrame, keyValue, false);
                }
            }
        }

        #endregion

        #region スケール

        /// <summary>
        /// 【チャイルド】スケールタブの変換処理です.
        /// </summary>
        /// <param name="oldEmitter">旧型式.</param>
        /// <param name="newEmitter">新形式.</param>
        private void ConvertChildScaleTab(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            //*******************************
            //  初期値.
            //*******************************
            // 親のスケールを継承.
            newEmitter.EmitterBasicSettingData.EmitterBasicInheritanceData.EnableScale = oldEmitter.ChildFlag.ScaleInherit;

            // スケール継承率.
            newEmitter.EmitterBasicSettingData.EmitterBasicInheritanceData.ScaleInheritanceRate = oldEmitter.ChildData.ScaleInheritRate;

            // 固有スケール.
            newEmitter.EmitterScaleData.EmitterScaleBasicData.ScaleBasisValue.X = oldEmitter.ChildData.Scale.X;
            newEmitter.EmitterScaleData.EmitterScaleBasicData.ScaleBasisValue.Y = oldEmitter.ChildData.Scale.Y;

            // ランダム量(%).
            newEmitter.EmitterScaleData.EmitterScaleBasicData.ScaleBasisRandom.X = (float)Math.Round(oldEmitter.ChildData.RandomScale * 100.0f, MidpointRounding.AwayFromZero);
            newEmitter.EmitterScaleData.EmitterScaleBasicData.ScaleBasisRandom.Y = oldEmitter.ChildData.RandomScale * 100.0f;


            //*******************************
            //  スケールアニメ.
            //*******************************
            // key1 初期スケール(%)
            if (oldEmitter.ChildAssistData.UI_childScaleSec != 0)
            {
                newEmitter.EmitterScaleData.EmitterScaleAnimationCurveData.AnimationTable.AddKeyFrame(
                    0,
                    100.0f / 100.0f,
                    100.0f / 100.0f,
                    0.0f,
                    0.0f,
                    false);
            }

            // key2 中間スケール(%)
            {
                newEmitter.EmitterScaleData.EmitterScaleAnimationCurveData.AnimationTable.AddKeyFrame(
                    oldEmitter.ChildAssistData.UI_childScaleSec,    // アウト開始タイミング(%)
                    100.0f / 100.0f,
                    100.0f / 100.0f,
                    0.0f,
                    0.0f,
                    false);
            }

            // key3 ターゲットスケール(%)
            if (oldEmitter.ChildAssistData.UI_childScaleSec != 100)
            {
                newEmitter.EmitterScaleData.EmitterScaleAnimationCurveData.AnimationTable.AddKeyFrame(
                    100,
                    oldEmitter.ChildAssistData.UI_childScaleTarget.X / 100.0f,
                    oldEmitter.ChildAssistData.UI_childScaleTarget.Y / 100.0f,
                    0.0f,
                    0.0f,
                    false);
            }
        }

        #endregion

        #region 回転

        /// <summary>
        /// 【チャイルド】回転タブの変換処理です.
        /// </summary>
        /// <param name="oldEmitter">旧型式.</param>
        /// <param name="newEmitter">新形式.</param>
        private void ConvertChildRotationTab(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            //*******************************
            //  基本設定.
            //*******************************
            // 回転タイプ.
            newEmitter.EmitterRotationData.EmitterRotationBasicData.RotationType = (int)oldEmitter.ChildAssistData.UI_childRotType;
            switch (oldEmitter.ChildAssistData.UI_childRotType)
            {
                case NWCore.DataModel.PtclRotType.RotateYZX:
                    newEmitter.EmitterRotationData.EmitterRotationBasicData.RotationType = 4;
                    break;
                case NWCore.DataModel.PtclRotType.RotateXYZ:
                    newEmitter.EmitterRotationData.EmitterRotationBasicData.RotationType = 5;
                    break;
                case NWCore.DataModel.PtclRotType.RotateZXY:
                    newEmitter.EmitterRotationData.EmitterRotationBasicData.RotationType = 6;
                    break;
            }

            //*********************************
            //  回転軸
            //*********************************
            if (oldEmitter.ChildAssistData.UI_childRotType == NWCore.DataModel.PtclRotType.RotateX)
            {
                newEmitter.EmitterRotationData.EmitterRotationBasicData.EnableRotationAxisX = true;
            }

            if (oldEmitter.ChildAssistData.UI_childRotType == NWCore.DataModel.PtclRotType.RotateY)
            {
                newEmitter.EmitterRotationData.EmitterRotationBasicData.EnableRotationAxisY = true;
            }

            if (oldEmitter.ChildAssistData.UI_childRotType == NWCore.DataModel.PtclRotType.RotateZ)
            {
                newEmitter.EmitterRotationData.EmitterRotationBasicData.EnableRotationAxisZ = true;
            }

            if ((oldEmitter.ChildAssistData.UI_childRotType == NWCore.DataModel.PtclRotType.RotateYZX) ||
                (oldEmitter.ChildAssistData.UI_childRotType == NWCore.DataModel.PtclRotType.RotateXYZ) ||
                (oldEmitter.ChildAssistData.UI_childRotType == NWCore.DataModel.PtclRotType.RotateZXY))
            {
                newEmitter.EmitterRotationData.EmitterRotationBasicData.EnableRotationAxisX = true;
                newEmitter.EmitterRotationData.EmitterRotationBasicData.EnableRotationAxisY = true;
                newEmitter.EmitterRotationData.EmitterRotationBasicData.EnableRotationAxisZ = true;
            }

            //*******************************
            //  回転方向ランダム.
            //*******************************
            // X
            newEmitter.EmitterRotationData.EmitterRotationXData.EnableRotationRandom = (oldEmitter.ChildAssistData.UI_childRotDirRand.X > 0) ? true : false;

            // Y
            newEmitter.EmitterRotationData.EmitterRotationYData.EnableRotationRandom = (oldEmitter.ChildAssistData.UI_childRotDirRand.Y > 0) ? true : false;

            // Z
            newEmitter.EmitterRotationData.EmitterRotationZData.EnableRotationRandom = (oldEmitter.ChildAssistData.UI_childRotDirRand.Z > 0) ? true : false;


            //*******************************
            //  初期回転.
            //*******************************
            // 親の回転を継承.
            newEmitter.EmitterBasicSettingData.EmitterBasicInheritanceData.EnableRotation = oldEmitter.ChildFlag.RotateInherit;

            // 基準 - X
            newEmitter.EmitterRotationData.EmitterRotationXData.InitialRotation = IndexToDegree(oldEmitter.ChildAssistData.UI_childInitRot.X, true);

            // 基準 - Y
            newEmitter.EmitterRotationData.EmitterRotationYData.InitialRotation = IndexToDegree(oldEmitter.ChildAssistData.UI_childInitRot.Y, true);

            // 基準 - Z
            newEmitter.EmitterRotationData.EmitterRotationZData.InitialRotation = IndexToDegree(oldEmitter.ChildAssistData.UI_childInitRot.Z, true);

            // ランダム - X
            newEmitter.EmitterRotationData.EmitterRotationXData.InitialRotationRandom = IndexToDegree(oldEmitter.ChildAssistData.UI_childInitRotRand.X, false);

            // ランダム - Y
            newEmitter.EmitterRotationData.EmitterRotationYData.InitialRotationRandom = IndexToDegree(oldEmitter.ChildAssistData.UI_childInitRotRand.Y, false);

            // ランダム - Z
            newEmitter.EmitterRotationData.EmitterRotationZData.InitialRotationRandom = IndexToDegree(oldEmitter.ChildAssistData.UI_childInitRotRand.Z, false);


            //******************************
            //  回転速度.
            //******************************
            // 減衰率.
            newEmitter.EmitterRotationData.EmitterRotationBasicData.RotationAttenuation = oldEmitter.ChildAssistData.UI_childRotResist;

            // 基準 - X
            newEmitter.EmitterRotationData.EmitterRotationXData.RotationSpeed = IndexToDegree(oldEmitter.ChildAssistData.UI_childRotVel.X, true);

            // 基準 - Y
            newEmitter.EmitterRotationData.EmitterRotationYData.RotationSpeed = IndexToDegree(oldEmitter.ChildAssistData.UI_childRotVel.Y, true);

            // 基準 - Z
            newEmitter.EmitterRotationData.EmitterRotationZData.RotationSpeed = IndexToDegree(oldEmitter.ChildAssistData.UI_childRotVel.Z, true);

            // ランダム - X
            newEmitter.EmitterRotationData.EmitterRotationXData.RotationSpeedRandom = IndexToDegree(oldEmitter.ChildAssistData.UI_childRotVelRand.X, false);

            // ランダム - Y
            newEmitter.EmitterRotationData.EmitterRotationYData.RotationSpeedRandom = IndexToDegree(oldEmitter.ChildAssistData.UI_childRotVelRand.Y, false);

            // ランダム - Z
            newEmitter.EmitterRotationData.EmitterRotationZData.RotationSpeedRandom = IndexToDegree(oldEmitter.ChildAssistData.UI_childRotVelRand.Z, false);
        }

        #endregion

        #region カスタムシェーダ

        /// <summary>
        /// 【チャイルド】カスタムシェーダタブの変換処理です.
        /// </summary>
        /// <param name="oldEmitter">旧型式.</param>
        /// <param name="newEmitter">新形式.</param>
        private void ConvertChildCustomShaderTab(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            var newCustomShaderSetting = newEmitter.EmitterCustomShaderData;
            newCustomShaderSetting.Settings = new ArrayCollection<EmitterCustomShaderSettingData>(8);
            newCustomShaderSetting.SelectedSettingIndex = (int)oldEmitter.ChildData.ShaderSetting - 1;

            for (uint i = 0; i < oldEmitter.ChildData.UserShaderParamGroups.Length; ++i)
            {
                var usdDef = this.customShaderTypeList[(int)i];
                if (string.IsNullOrEmpty(usdDef.Name))
                {
                    if ((int)oldEmitter.ChildData.ShaderSetting - 1 >= i)
                    {
                        --newCustomShaderSetting.SelectedSettingIndex;
                    }

                    continue;
                }

                var oldGroup = oldEmitter.ChildData.UserShaderParamGroups[i];

                var convertingData = new EmitterCustomShaderConvertingData
                {
                    Name = usdDef.Name,
                    Visible = usdDef.Visible,
                    SwitchVisible = usdDef.SwitchVisible,
                    SwitchesVisibleList = usdDef.SwitchesVisibleList,
                    ParamVisible = usdDef.ParamVisible,
                    ParamsVisibleList = usdDef.ParamsVisibleList
                };

                //----- カスタムシェーダフラグ ----- (チェックボックス)
                uint flags = 0;
                for (uint j = 0; j < oldGroup.BitFlags.Length; ++j)
                {
                    if (oldGroup.BitFlags[j])
                    {
                        flags += (uint)Math.Pow(2, j);
                    }
                }

                convertingData.Flags = flags;

                //----- カスタムシェーダフラグ ----- (ラジオボタン)
                for (uint j = 0; j < oldGroup.Bytes.Length; ++j)
                {
                    convertingData.Switches.Add(oldGroup.Bytes[j] >> (8 * (int)j));
                }

                //----- カスタムシェーダパラメータ -----
                for (uint j = 0; j < oldGroup.Params.Length; ++j)
                {
                    convertingData.Params.Add(oldGroup.Params[j]);
                }

                newCustomShaderSetting.Settings[(int)i] = convertingData;
            }
        }

        #endregion

        #endregion

        #region Field

        /// <summary>
        /// フィールドデータを新形式に変換します.
        /// </summary>
        /// <param name="oldEmitter">旧型式.</param>
        /// <param name="newField">新型式.</param>
        private void ConvertField(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            if (oldEmitter.Type == DataModel.EmitterType.Complex)
            {
                // ランダム.
                this.ConvertFieldRandomTab(oldEmitter, newEmitter);

                // 磁力.
                this.ConvertFieldMagnetTab(oldEmitter, newEmitter);

                // スピン.
                this.ConvertFiledSpinTab(oldEmitter, newEmitter);

                // 収束.
                this.ConvertFieldConvergeTab(oldEmitter, newEmitter);

                // 位置に加算.
                this.ConvertFieldAddLocationTab(oldEmitter, newEmitter);

                // シンプルコリジョン.
                this.ConvertFieldCollisionTab(oldEmitter, newEmitter);

                // カールノイズ.
                this.ConvertFieldCurlNoiseTab(oldEmitter, newEmitter);
            }
        }

        #region ランダム

        /// <summary>
        /// 【フィールド】ランダムタブの変換処理です.
        /// </summary>
        /// <param name="oldEmitter">旧型式.</param>
        /// <param name="newField">新形式.</param>
        private void ConvertFieldRandomTab(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            if (oldEmitter.FieldFlag.Random)
            {
                EffectMaker.DataModel.DataModels.RandomFe1Data randomData = new EffectMaker.DataModel.DataModels.RandomFe1Data();

                //******************************
                //  基本設定.
                //******************************
                // 有効.
                /* 対応項目なし = oldEmitter.FieldFlag.Random; */

                //******************************
                //  フィールドパラメータ.
                //******************************
                // ランダム適用間隔(フレーム).
                randomData.RandomFe1ParamData.BlankFrames = oldEmitter.FieldRandomData.Blank;

                // ランダム幅.
                randomData.RandomFe1ParamData.RandomRange.X = oldEmitter.FieldRandomData.VelocityAddition.X;
                randomData.RandomFe1ParamData.RandomRange.Y = oldEmitter.FieldRandomData.VelocityAddition.Y;
                randomData.RandomFe1ParamData.RandomRange.Z = oldEmitter.FieldRandomData.VelocityAddition.Z;

                {
                    NWCore.DataModel.AnimTableData field_random_x = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.field_random_x);
                    NWCore.DataModel.AnimTableData field_random_y = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.field_random_y);
                    NWCore.DataModel.AnimTableData field_random_z = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.field_random_z);

                    // ランダム幅 - パーティクル時間アニメ.
                    randomData.RandomFe1ParamData.EnableAnimation = (field_random_x.IsEnabled) ? 1 : 0;

                    randomData.RandomFe1ParamData.LoopMode = (field_random_x.IsLoop) ? 1 : 0;

                    randomData.RandomFe1ParamData.LoopEnd = (field_random_x.NumLoopFrames);

                    randomData.RandomFe1ParamData.Randomize = (field_random_x.IsRandomStartFrameEnabled) ? 1 : 0;

                    randomData.RandomFe1ParamData.AnimationTable.Clear();

                    // ランダム幅 - カーブエディタ.
                    for (int i = 0; i < field_random_x.KeyFrames.Count; ++i)
                    {
                        NWCore.DataModel.FloatAnimTableKeyFrameData keyX = field_random_x.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;
                        NWCore.DataModel.FloatAnimTableKeyFrameData keyY = field_random_y.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;
                        NWCore.DataModel.FloatAnimTableKeyFrameData keyZ = field_random_z.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;

                        randomData.RandomFe1ParamData.AnimationTable.AddKeyFrame(
                            keyX.KeyFrame,
                            keyX.Value,
                            keyY.Value,
                            keyZ.Value,
                            0.0f,
                            false);
                    }
                }

                // フィールドリストに追加.
                newEmitter.FieldList.Add(randomData);
            }
        }

        #endregion

        #region 磁力

        /// <summary>
        /// 【フィールド】磁力タブの変換処理です.
        /// </summary>
        /// <param name="oldEmitter">旧型式.</param>
        /// <param name="newField">新形式.</param>
        private void ConvertFieldMagnetTab(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            if (oldEmitter.FieldFlag.Magnet)
            {
                EffectMaker.DataModel.DataModels.MagnetData magnetData = new EffectMaker.DataModel.DataModels.MagnetData();

                //******************************
                //  基本設定.
                //******************************
                // 有効.
                /* 対応項目なし = oldEmitter.FieldFlag.Magnet; */

                // デバッグ表示.
                /* 対応項目なし = oldEmitter.UI_hostioEmitterFlg.DispFieldMagnet; */


                //******************************
                //  フィールドパラメータ.
                //******************************
                // 磁力.
                magnetData.MagnetParamData.MagnetPower = oldEmitter.FieldMagnetData.FieldMagnetPower;

                {
                    NWCore.DataModel.AnimTableData field_magnet = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.field_magnet);

                    // ランダム幅 - パーティクル時間アニメ.
                    magnetData.MagnetParamData.EnableAnimation = (field_magnet.IsEnabled) ? 1 : 0;

                    magnetData.MagnetParamData.LoopMode = (field_magnet.IsLoop) ? 1 : 0;

                    magnetData.MagnetParamData.LoopEnd = (field_magnet.NumLoopFrames);

                    magnetData.MagnetParamData.Randomize = (field_magnet.IsRandomStartFrameEnabled) ? 1 : 0;

                    magnetData.MagnetParamData.AnimationTable.Clear();

                    // ランダム幅 - カーブエディタ.
                    for (int i = 0; i < field_magnet.KeyFrames.Count; ++i)
                    {
                        NWCore.DataModel.FloatAnimTableKeyFrameData key = field_magnet.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;

                        magnetData.MagnetParamData.AnimationTable.AddKeyFrame(
                            key.KeyFrame,
                            key.Value,
                            0.0f,
                            0.0f,
                            0.0f,
                            false);
                    }
                }


                // 位置.
                magnetData.MagnetParamData.MagnetPos.X = oldEmitter.FieldMagnetData.FieldMagnetPos.X;
                magnetData.MagnetParamData.MagnetPos.Y = oldEmitter.FieldMagnetData.FieldMagnetPos.Y;
                magnetData.MagnetParamData.MagnetPos.Z = oldEmitter.FieldMagnetData.FieldMagnetPos.Z;

                // 位置の追従タイプ.
                magnetData.MagnetParamData.FollowEmitter = oldEmitter.FieldMagnetData.Flag.FollowEmitter;

                // 対象座標軸X軸.
                magnetData.MagnetParamData.EnableAxisX = oldEmitter.FieldMagnetData.Flag.X;

                // 対象座標軸Y軸.
                magnetData.MagnetParamData.EnableAxisY = oldEmitter.FieldMagnetData.Flag.Y;

                // 対象座標軸Z軸.
                magnetData.MagnetParamData.EnableAxisZ = oldEmitter.FieldMagnetData.Flag.Z;


                // フィールドリストに追加.
                newEmitter.FieldList.Add(magnetData);
            }
        }

        #endregion

        #region スピン

        /// <summary>
        /// 【フィールド】スピンタブの変換処理です.
        /// </summary>
        /// <param name="oldEmitter">旧型式.</param>
        /// <param name="newField">新形式.</param>
        private void ConvertFiledSpinTab(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            if (oldEmitter.FieldFlag.Spin)
            {
                EffectMaker.DataModel.DataModels.SpinData spinData = new EffectMaker.DataModel.DataModels.SpinData();

                //******************************
                //  基本設定.
                //******************************
                // 有効.
                /* 対応項目なし = oldEmitter.FieldFlag.Spin; */

                // デバッグ表示.
                /* 対応項目なし = oldEmitter.UI_hostioEmitterFlg.DispFieldSpin; */


                //******************************
                //  フィールドパラメータ.
                //******************************
                // 回転速度.
                spinData.SpinParamData.SpinVelocity = IndexToDegree(oldEmitter.FieldSpinData.FieldSpinRotation, true);

                {
                    NWCore.DataModel.AnimTableData field_spin_rot = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.field_span_rot);

                    // 回転速度 - パーティクル時間アニメ.
                    spinData.SpinParamData.EnableVelocityAnimation = (field_spin_rot.IsEnabled) ? 1 : 0;

                    spinData.SpinParamData.VelocityLoopMode = (field_spin_rot.IsLoop) ? 1 : 0;

                    spinData.SpinParamData.VelocityLoopEnd = (field_spin_rot.NumLoopFrames);

                    spinData.SpinParamData.VelocityRandomize = (field_spin_rot.IsRandomStartFrameEnabled) ? 1 : 0;

                    spinData.SpinParamData.VelocityAnimationTable.Clear();

                    // 回転速度 - カーブエディタ.
                    for (int i = 0; i < field_spin_rot.KeyFrames.Count; ++i)
                    {
                        NWCore.DataModel.FloatAnimTableKeyFrameData key = field_spin_rot.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;

                        spinData.SpinParamData.VelocityAnimationTable.AddKeyFrame(
                            key.KeyFrame,
                            key.Value,
                            0.0f,
                            0.0f,
                            0.0f,
                            false);
                    }
                }


                // 軸.
                spinData.SpinParamData.SpinAxis = oldEmitter.FieldSpinData.FieldSpinAxis;

                // 拡散速度.
                spinData.SpinParamData.SpinOuter = oldEmitter.FieldSpinData.FieldSpinOuter;

                {
                    NWCore.DataModel.AnimTableData field_spin_outer = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.field_span_outer);

                    // 拡散速度 - パーティクル時間アニメ.
                    spinData.SpinParamData.EnableOuterAnimation = (field_spin_outer.IsEnabled) ? 1 : 0;

                    spinData.SpinParamData.OuterLoopMode = (field_spin_outer.IsLoop) ? 1 : 0;

                    spinData.SpinParamData.OuterLoopEnd = (field_spin_outer.NumLoopFrames);

                    spinData.SpinParamData.OuterRandomize = (field_spin_outer.IsRandomStartFrameEnabled) ? 1 : 0;

                    spinData.SpinParamData.OuterAnimationTable.Clear();

                    // 拡散速度 - カーブエディタ.
                    for (int i = 0; i < field_spin_outer.KeyFrames.Count; ++i)
                    {
                        NWCore.DataModel.FloatAnimTableKeyFrameData key = field_spin_outer.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;

                        spinData.SpinParamData.OuterAnimationTable.AddKeyFrame(
                            key.KeyFrame,
                            key.Value,
                            0.0f,
                            0.0f,
                            0.0f,
                            false);
                    }
                }


                // フィールドリストに追加.
                newEmitter.FieldList.Add(spinData);
            }
        }

        #endregion

        #region 収束

        /// <summary>
        /// 【フィールド】収束タブの変換処理です.
        /// </summary>
        /// <param name="oldEmitter">旧型式.</param>
        /// <param name="newField">新形式.</param>
        private void ConvertFieldConvergeTab(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            if (oldEmitter.FieldFlag.Convergence)
            {
                EffectMaker.DataModel.DataModels.ConvergeData convergeData = new EffectMaker.DataModel.DataModels.ConvergeData();

                //*******************************
                //  基本設定.
                //*******************************
                // 有効.
                /* 対応項目なし = oldEmitter.FieldFlag.Convergence; */

                // デバッグ表示.
                /* 対応項目なし = oldEmitter.UI_hostioEmitterFlg.DispFieldConvergence; */


                //*******************************
                //  フィールドパラメータ.
                //*******************************
                // 収束力.
                convergeData.ConvergeParamData.Ratio = oldEmitter.FieldConvergenceData.Ratio;

                {
                    NWCore.DataModel.AnimTableData field_convergence = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.field_convergence);

                    // 収束力 - パーティクル時間アニメ.
                    convergeData.ConvergeParamData.EnableAnimation = (field_convergence.IsEnabled) ? 1 : 0;

                    convergeData.ConvergeParamData.LoopMode = (field_convergence.IsLoop) ? 1 : 0;

                    convergeData.ConvergeParamData.LoopEnd = (field_convergence.NumLoopFrames);

                    convergeData.ConvergeParamData.Randomize = (field_convergence.IsRandomStartFrameEnabled) ? 1 : 0;

                    convergeData.ConvergeParamData.AnimationTable.Clear();

                    // 収束力 - カーブエディタ.
                    for (int i = 0; i < field_convergence.KeyFrames.Count; ++i)
                    {
                        NWCore.DataModel.FloatAnimTableKeyFrameData key = field_convergence.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;

                        convergeData.ConvergeParamData.AnimationTable.AddKeyFrame(
                            key.KeyFrame,
                            key.Value,
                            0.0f,
                            0.0f,
                            0.0f,
                            false);
                    }
                }

                // 位置.
                convergeData.ConvergeParamData.Position.X = oldEmitter.FieldConvergenceData.Position.X;
                convergeData.ConvergeParamData.Position.Y = oldEmitter.FieldConvergenceData.Position.Y;
                convergeData.ConvergeParamData.Position.Z = oldEmitter.FieldConvergenceData.Position.Z;

                // 位置の追従タイプ.
                convergeData.ConvergeParamData.FollowEmitter = oldEmitter.FieldConvergenceData.FollowEmitter;


                // フィールドリストに追加.
                newEmitter.FieldList.Add(convergeData);
            }
        }

        #endregion

        #region 位置に加算

        /// <summary>
        /// 【フィールド】位置に加算タブの変換処理です.
        /// </summary>
        /// <param name="oldEmitter">旧型式.</param>
        /// <param name="newField">新形式.</param>
        private void ConvertFieldAddLocationTab(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            if (oldEmitter.FieldFlag.PosAdd)
            {
                EffectMaker.DataModel.DataModels.AddLocationData addLocationData = new EffectMaker.DataModel.DataModels.AddLocationData();

                //*******************************
                //  基本設定.
                //*******************************
                // 基本設定.
                /* 対応項目なし = oldEmitter.FieldFlag.PosAdd; */

                //*******************************
                //  フィールドパラメータ.
                //*******************************
                // 座標系.
                addLocationData.FieldAddLocationParam.AddGlobal = oldEmitter.FieldPosAddData.AddGlobal;

                // 加算値.
                addLocationData.FieldAddLocationParam.Addition.X = oldEmitter.FieldPosAddData.Addition.X;
                addLocationData.FieldAddLocationParam.Addition.Y = oldEmitter.FieldPosAddData.Addition.Y;
                addLocationData.FieldAddLocationParam.Addition.Z = oldEmitter.FieldPosAddData.Addition.Z;

                {
                    NWCore.DataModel.AnimTableData field_pos_add_x = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.field_pos_add_x);
                    NWCore.DataModel.AnimTableData field_pos_add_y = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.field_pos_add_y);
                    NWCore.DataModel.AnimTableData field_pos_add_z = oldEmitter.AnimTableList.FindAnimationTableData(DataModel.AnimTableTargetTypes.field_pos_add_z);

                    // 加算値 - パーティクル時間アニメ.
                    addLocationData.FieldAddLocationParam.EnableAnimation = (field_pos_add_x.IsEnabled) ? 1 : 0;

                    addLocationData.FieldAddLocationParam.LoopMode = (field_pos_add_x.IsLoop) ? 1 : 0;

                    addLocationData.FieldAddLocationParam.LoopEnd = (field_pos_add_x.NumLoopFrames);

                    addLocationData.FieldAddLocationParam.Randomize = (field_pos_add_x.IsRandomStartFrameEnabled) ? 1 : 0;

                    addLocationData.FieldAddLocationParam.AnimationTable.Clear();

                    // 加算値 - カーブエディタ.
                    for (int i = 0; i < field_pos_add_x.KeyFrames.Count; ++i)
                    {
                        NWCore.DataModel.FloatAnimTableKeyFrameData keyX = field_pos_add_x.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;
                        NWCore.DataModel.FloatAnimTableKeyFrameData keyY = field_pos_add_y.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;
                        NWCore.DataModel.FloatAnimTableKeyFrameData keyZ = field_pos_add_z.KeyFrames[i] as NWCore.DataModel.FloatAnimTableKeyFrameData;

                        addLocationData.FieldAddLocationParam.AnimationTable.AddKeyFrame(
                            keyX.KeyFrame,
                            keyX.Value,
                            keyY.Value,
                            keyZ.Value,
                            0.0f,
                            false);
                    }
                }

                // フィールドリストに追加.
                newEmitter.FieldList.Add(addLocationData);
            }
        }

        #endregion

        #region シンプルコリジョン

        /// <summary>
        /// 【フィールド】シンプルコリジョンタブの変換処理です.
        /// </summary>
        /// <param name="oldEmitter">旧型式.</param>
        /// <param name="newField">新形式.</param>
        private void ConvertFieldCollisionTab(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            if (oldEmitter.FieldFlag.Collision)
            {
                EffectMaker.DataModel.DataModels.CollisionData collisionData = new EffectMaker.DataModel.DataModels.CollisionData();

                //********************************
                //  基本設定.
                //********************************
                // 有効.
                /* 対応項目なし = oldEmitter.FieldFlag.Collision; */

                // デバッグ表示.
                /* 対応項目なし oldEmitter.UI_hostioEmitterFlg.DispFieldCollision; */


                //********************************
                //  フィールドパラメータ.
                //********************************
                // ヒット時の挙動.
                collisionData.CollisionParamData.ActionType = oldEmitter.FieldCollisionData.Type;

                // 座標系.
                collisionData.CollisionParamData.IsWorld = oldEmitter.FieldCollisionData.IsWorld;

                // 共有平面.
                collisionData.CollisionParamData.UseSharedPlane = oldEmitter.FieldCollisionData.UseSharedPlane;

                // コリジョン面Y座標.
                collisionData.CollisionParamData.Coordinate = oldEmitter.FieldCollisionData.Coordinate;

                // 反射率.
                collisionData.CollisionParamData.BounceRate = oldEmitter.FieldCollisionData.Coefficient;

                // 反射回数.
                collisionData.CollisionParamData.BounceCount = oldEmitter.FieldCollisionData.BounceCount;

                // 摩擦係数.
                collisionData.CollisionParamData.FrictionCoefficient = oldEmitter.FieldCollisionData.FrictionCoefficient;


                // フィールドリストに追加.
                newEmitter.FieldList.Add(collisionData);
            }
        }

        #endregion

        #region カールノイズ

        /// <summary>
        /// 【フィールド】カールノイズタブの変換処理です.
        /// </summary>
        /// <param name="oldEmitter">旧型式.</param>
        /// <param name="newField">新形式.</param>
        private void ConvertFieldCurlNoiseTab(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterData newEmitter)
        {
            if (oldEmitter.FieldFlag.CurlNoise)
            {
                EffectMaker.DataModel.DataModels.CurlNoiseData curlNoiseData = new EffectMaker.DataModel.DataModels.CurlNoiseData();

                //********************************
                //  基本設定.
                //********************************
                // 有効.
                /* 対応項目なし = oldEmitter.FieldFlag.CurlNoise; */


                //********************************
                //  フィールドパラメータ.
                //********************************
                // 影響の強さ.
                curlNoiseData.CurlNoiseParamData.CurlNoiseInfluence.X = oldEmitter.FieldCurlNoiseData.Influence.X;
                curlNoiseData.CurlNoiseParamData.CurlNoiseInfluence.Y = oldEmitter.FieldCurlNoiseData.Influence.Y;
                curlNoiseData.CurlNoiseParamData.CurlNoiseInfluence.Z = oldEmitter.FieldCurlNoiseData.Influence.Z;

                // ノイズテーブルを補間.
                curlNoiseData.CurlNoiseParamData.Interpolation = oldEmitter.FieldCurlNoiseData.Interpolation;

                // ノイズのスケール.
                curlNoiseData.CurlNoiseParamData.CurlNoiseScale = oldEmitter.FieldCurlNoiseData.BorderSize;

                // ノイズの移動速度.
                curlNoiseData.CurlNoiseParamData.Speed.X = oldEmitter.FieldCurlNoiseData.Speed.X;
                curlNoiseData.CurlNoiseParamData.Speed.Y = oldEmitter.FieldCurlNoiseData.Speed.Y;
                curlNoiseData.CurlNoiseParamData.Speed.Z = oldEmitter.FieldCurlNoiseData.Speed.Z;

                // ノイズのオフセット.
                curlNoiseData.CurlNoiseParamData.InitialValue = oldEmitter.FieldCurlNoiseData.InitialValue;

                // ノイズのオフセットランダム.
                curlNoiseData.CurlNoiseParamData.RandomOffset = oldEmitter.FieldCurlNoiseData.RandomOffset;


                // フィールドリストに追加.
                newEmitter.FieldList.Add(curlNoiseData);
            }
        }

        #endregion

        #endregion

        #region Fluctuation

        /// <summary>
        /// 揺らぎデータを新形式に変換します.
        /// </summary>
        /// <param name="oldEmitter">旧型式.</param>
        /// <param name="newFluctuation">新型式.</param>
        private void ConvertFluctuation(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterParticleFluctuationData newFluctuation)
        {
            // パラメータ.
            this.ConvertFluctuationParamterTab(oldEmitter, newFluctuation);
        }

        #region パラメータ

        /// <summary>
        /// 【揺らぎ】パラメータタブの変換処理です.
        /// </summary>
        /// <param name="oldEmitter">旧型式.</param>
        /// <param name="newFluctuation">新形式.</param>
        private void ConvertFluctuationParamterTab(NWCore.DataModel.EmitterData oldEmitter, EffectMaker.DataModel.DataModels.EmitterParticleFluctuationData newFluctuation)
        {
            //*********************************
            //  基本設定.
            //*********************************
            // 有効.
            if (oldEmitter.FluctuationFlag.Enable && oldEmitter.Type == DataModel.EmitterType.Complex)
            {
                //*********************************
                //  揺らぎパラメータ.
                //*********************************
                // 波形タイプ.
                switch (oldEmitter.FluctuationFlag.WaveType)
                {
                    case DataModel.FluctuationWaveTypes.SinWave:
                        {
                            newFluctuation.WaveType = 0;
                        }
                        break;

                    case DataModel.FluctuationWaveTypes.SawToothWave:
                        {
                            newFluctuation.WaveType = 1;
                        }
                        break;

                    case DataModel.FluctuationWaveTypes.RectangularWave:
                        {
                            newFluctuation.WaveType = 2;
                        }
                        break;
                }

                // アルファに適用.
                newFluctuation.ApplyAlpha = oldEmitter.FluctuationFlag.ApplyAlpha;

                // スケールに適用.
                newFluctuation.ApplyScale = oldEmitter.FluctuationFlag.ApplyScale;

                // 振幅.
                newFluctuation.Amplitude.X = oldEmitter.FluctuationData.ScaleX;

                // 周期.
                newFluctuation.Frequency.X = oldEmitter.FluctuationData.FrequencyX;

                // 位相.
                newFluctuation.PhaseRandom.X = (int)oldEmitter.FluctuationData.PhaseRangeX;
            }
        }

        #endregion

        #endregion
    }
}
