﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Drawing;

namespace EffectMaker.Foundation.Utility
{
    /// <summary>
    /// Math utility class.
    /// </summary>
    public class MathUtility
    {
        /// <summary>
        /// MinPowerOfTenF.
        /// </summary>
        private const int MinPowerOfTenF = -7;

        /// <summary>
        /// MaxPowerOfTenF.
        /// </summary>
        private const int MaxPowerOfTenF = 7;

        /// <summary>
        /// powerOfTenF.
        /// </summary>
        private static float[] powerOfTenF = new float[]
        {
            0.0000001f,
            0.000001f,
            0.00001f,
            0.0001f,
            0.001f,
            0.01f,
            0.1f,
            1.0f,
            10.0f,
            100.0f,
            1000.0f,
            10000.0f,
            100000.0f,
            1000000.0f,
            10000000.0f
        };

        /// <summary>
        /// Static method for clamping the value to the given range.
        /// </summary>
        /// <param name="value">The value to clamp.</param>
        /// <param name="min">The minimum value.</param>
        /// <param name="max">The maximum value.</param>
        /// <returns>The clamped value.</returns>
        public static float Clamp(float value, float min, float max)
        {
            return Math.Min(max, Math.Max(min, value));
        }

        /// <summary>
        /// Static method for clamping the value to the given range.
        /// </summary>
        /// <param name="value">The value to clamp.</param>
        /// <param name="min">The minimum value.</param>
        /// <param name="max">The maximum value.</param>
        /// <returns>The clamped value.</returns>
        public static int Clamp(int value, int min, int max)
        {
            return Math.Min(max, Math.Max(min, value));
        }

        /// <summary>
        /// Checks whether a value is zero or very close to it.
        /// </summary>
        /// <param name="value">The value to check upon.</param>
        /// <param name="epsilon">The epsilon allowed for the zero range.</param>
        /// <returns>Returns true if value is zero or close to it, or false otherwise.</returns>
        public static bool IsZero(float value, float epsilon)
        {
            return -epsilon < value && value < epsilon;
        }

        /// <summary>
        /// Checks whether a value is zero or very close to it.
        /// </summary>
        /// <param name="value">The value to check upon.</param>
        /// <param name="epsilon">The epsilon allowed for the zero range.</param>
        /// <returns>Returns true if value is zero or close to it, or false otherwise.</returns>
        public static bool IsZero(double value, double epsilon)
        {
            return -epsilon < value && value < epsilon;
        }

        /// <summary>
        /// Compute the distance from a point to a line that passes the given vertices.
        /// </summary>
        /// <param name="point">The point.</param>
        /// <param name="lineVertex1">A vertex of the line.</param>
        /// <param name="lineVertex2">Another vertex of the line.</param>
        /// <returns>The distance from the point to the line.</returns>
        public static float ComputePointLineDistance(
            PointF point,
            PointF lineVertex1,
            PointF lineVertex2)
        {
            if (lineVertex1.X == lineVertex2.X)
            {
                return Math.Abs(point.X - lineVertex1.X);
            }

            // Slope m = (y1 - y2) / (x1 - x2)
            float m = (lineVertex1.Y - lineVertex2.Y) / (lineVertex1.X - lineVertex2.X);
            if (m == 0.0f)
            {
                return Math.Abs(point.Y - lineVertex1.Y);
            }

            // Line equation : y = mx + b => b = y - mx
            float b = lineVertex1.Y - (m * lineVertex1.X);

            return Math.Abs((m * point.X) - point.Y + b) / (float)Math.Sqrt((m * m) + 1.0f);
        }

        /// <summary>
        /// Utility method for finding the most significant digit of the given value.
        /// </summary>
        /// <param name="fvalue">The value.</param>
        /// <returns>Number of digits after decimal point.</returns>
        public static int FindMostSignificantDigit(float fvalue)
        {
            int numDecimalDigits = 0;
            double origValue = (double)Math.Abs(fvalue);
            double currValue = origValue;

            if (currValue < 1.0 &&
                 currValue > 0.0)
            {
                while (Math.Round(currValue, 6) < 1.0)
                {
                    currValue = origValue * (double)PowerOfTenF(numDecimalDigits + 1);
                    ++numDecimalDigits;
                }
            }

            return numDecimalDigits;
        }

        /// <summary>
        /// Computer the given power of 10 as floating point number.
        /// The power can be ranged from -7 to 7.
        /// </summary>
        /// <param name="power">The power.</param>
        /// <returns>The result.</returns>
        public static float PowerOfTenF(int power)
        {
            // System.Diagnostics.Debug.Assert(power >= MinPowerOfTenF && power <= MaxPowerOfTenF);
            return powerOfTenF[power - MinPowerOfTenF];
        }
    }
}
