﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;

namespace EffectMaker.Foundation.Collections
{
    /// <summary>
    /// A simple class that separates tokens in a path and stores them in a list structure.
    /// </summary>
    public class PathNodeList : IEnumerable<string>
    {
        /// <summary>The internal list that stores the path nodes.</summary>
        private List<string> pathNodes = new List<string>();

        /// <summary>The path string.</summary>
        private string originalPath = string.Empty;

        /// <summary>
        /// Constructor.
        /// </summary>
        public PathNodeList()
        {
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="path">The period-separated path.</param>
        public PathNodeList(string path)
        {
            this.UpdatePath(path);
        }

        /// <summary>
        /// Get the count of the path nodes.
        /// </summary>
        public int Count
        {
            get { return this.pathNodes.Count; }
        }

        /// <summary>
        /// Get or set the path string
        /// </summary>
        public string OriginalPath
        {
            get { return this.originalPath; }
            set { this.UpdatePath(value); }
        }

        /// <summary>
        /// Get or set a path node at the specified index.
        /// </summary>
        /// <param name="index">The index to the path node.</param>
        /// <returns>The path node at the index.</returns>
        public string this[int index]
        {
            get { return this.pathNodes[index]; }
            set { this.pathNodes[index] = value; }
        }

        /// <summary>
        /// Get an enumerator for the path nodes.
        /// </summary>
        /// <returns>The enumerator.</returns>
        public IEnumerator<string> GetEnumerator()
        {
            return this.pathNodes.GetEnumerator();
        }

        /// <summary>
        /// Get an enumerator for the path nodes.
        /// </summary>
        /// <returns>The enumerator.</returns>
        IEnumerator IEnumerable.GetEnumerator()
        {
            return this.GetEnumerator();
        }

        /// <summary>
        /// Update the path nodes with the given path string.
        /// </summary>
        /// <param name="path">The path string.</param>
        private void UpdatePath(string path)
        {
            this.pathNodes.Clear();
            this.pathNodes.AddRange(
                path.Split(new char[] { '.' }, StringSplitOptions.RemoveEmptyEntries));
        }
    }
}
