﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.CodeDom.Compiler;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Xml.Serialization;
using EffectMaker.DataModel.DataModels;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Serialization;
using EffectMaker.Foundation.Utility;
using Microsoft.CSharp;

namespace EffectMaker.DataModel.Manager
{
    /// <summary>
    /// ユーザデータページマネージャ
    /// </summary>
    public class UserDataModelManager
    {
        /// <summary>
        /// シングルトンインスタンス
        /// </summary>
        public static readonly UserDataModelManager Instance = new UserDataModelManager();

        /// <summary>
        /// データ定義ファイル内の文字列とクラスタイプの対応マップ.
        /// </summary>
        private static readonly Dictionary<string, Type> TargetDataModelMap =
                            new Dictionary<string, Type>()
        {
            { "Emitter", typeof(EmitterData) },
            { "Preview", typeof(PreviewData) },
////            { "CustomAction", typeof(CustomActionData) },
            { "StripeComplex", typeof(StripeSuperData) },
            { "StripeHistory", typeof(StripeHistoryData) },
        };

        /// <summary>
        /// データモデルコンテナ.
        /// </summary>
        private static Dictionary<Type, List<UserDataBase>> dataModelContainer =
                   new Dictionary<Type, List<UserDataBase>>();

        /// <summary>
        /// ユーザーデータ情報コンテナ.
        /// </summary>
        private static Dictionary<Type, UserDataModelInfo> userDataInfoContainer =
                   new Dictionary<Type, UserDataModelInfo>();

        /// <summary>
        /// ユーザーデータ情報.
        /// </summary>
        private UserDataInfo userDataInfo = null;

        /// <summary>
        /// ユーザーデータのアセンブリです.
        /// </summary>
        private AssemblyInfo userAssemblyInfo = null;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        private UserDataModelManager()
        {
        }

        /// <summary>
        /// エミッターユーザーデータ更新時イベント
        /// </summary>
        public event EventHandler UserDataUpdated;

        /// <summary>
        /// 実行ファイルディレクトリパス
        /// </summary>
        public static string ExeDirPath
        {
            get
            {
                return Path.GetDirectoryName(Assembly.GetEntryAssembly().Location);
            }
        }

        /// <summary>
        /// ユーザーページ情報コンテナ.
        /// </summary>
        public List<UserDataModelInfo> UserPageInfoContainer
        {
            get
            {
                return this.userDataInfo.UserDataModelList;
            }
        }

        /// <summary>
        /// ユーザーデータのアセンブリです.
        /// </summary>
        public Assembly UserAssembly
        {
            get
            {
                if (this.userAssemblyInfo != null)
                {
                    return this.userAssemblyInfo.Assembly;
                }

                return null;
            }
        }

        /// <summary>
        /// インスタンスの作成.
        /// </summary>
        public static void Initialize()
        {
            Instance.InitializeInternal();
        }

        /// <summary>
        /// データモデル定義ソースコードを読み込む
        /// </summary>
        /// <param name="filepath">ソースコードファイルパス</param>
        /// <returns>アセンブリ情報</returns>
        public static AssemblyInfo LoadDataModelSourceCode(string filepath)
        {
            if (File.Exists(filepath) == false)
            {
                return null;
            }

            string scriptText;
            {
                using (var sr = new StreamReader(filepath))
                {
                    scriptText = sr.ReadToEnd();
                }
            }

            if (string.IsNullOrEmpty(scriptText))
            {
                return null;
            }

            using (
                var codeProvider = new CSharpCodeProvider(new Dictionary<string, string> { { "CompilerVersion", "v4.0" } }))
            {
                var param = new CompilerParameters
                {
                    GenerateInMemory = true,
                    IncludeDebugInformation = true
                };

                // 参照アセンブリを追加
                {
                    param.ReferencedAssemblies.Add(Path.Combine(ExeDirPath, "EffectMaker.DataModel.dll"));
                    param.ReferencedAssemblies.Add(Path.Combine(ExeDirPath, "EffectMaker.Foundation.dll"));

                    // ユーザーが使いたそうなもの
                    param.ReferencedAssemblies.Add("System.dll");
                    param.ReferencedAssemblies.Add("System.Xml.dll");
                    param.ReferencedAssemblies.Add("System.Windows.Forms.dll");
                }

                var result = codeProvider.CompileAssemblyFromSource(param, scriptText);

                if (result.Errors.Count > 0)
                {
                    Logger.Log(LogLevels.Error, "Compile Error : " + filepath);

                    // エラーがあった場合、エラーの情報を表示する.
                    foreach (var e in result.Errors)
                    {
                        Logger.Log(LogLevels.Error, e.ToString());
                    }

                    return null;
                }

                if (result.CompiledAssembly == null)
                {
                    return null;
                }

                // 保存する
                return
                    new AssemblyInfo
                    {
                        Assembly = result.CompiledAssembly,
                        LastWriteTime = File.GetLastWriteTime(filepath)
                    };
            }
        }

        /// <summary>
        /// データモデル定義ソースコードを読み込み，コンパイルします.
        /// </summary>
        /// <param name="filePaths">読み込むソースコード各ファイルパス</param>
        /// <returns>アセンブリ情報</returns>
        public static AssemblyInfo LoadDataModelFromFiles(string[] filePaths)
        {
            // nullチェック.
            if (filePaths == null)
            {
                return null;
            }

            // ファイル名が格納されていることをチェック.
            if (filePaths.Length <= 0)
            {
                return null;
            }

            // 現在時刻を取得.
            System.DateTime dt = System.DateTime.Now;

            // 出力アセンブリ名を取得.
            string outputAssembly = "EffectMaker_UserData_" + dt.ToString("yyyy-MM-dd-HHmmss");

            using (
                var codeProvider = new CSharpCodeProvider(
                    new Dictionary<string, string> { { "CompilerVersion", "v4.0" } }))
            {
                // コンパイルパラメータを設定.
                var param = new CompilerParameters
                {
                    GenerateInMemory = true,
                    IncludeDebugInformation = true,
                };

                // 参照アセンブリを追加
                {
                    param.ReferencedAssemblies.Add(Path.Combine(ExeDirPath, "EffectMaker.DataModel.dll"));
                    param.ReferencedAssemblies.Add(Path.Combine(ExeDirPath, "EffectMaker.Foundation.dll"));
                    param.ReferencedAssemblies.Add(Path.Combine(ExeDirPath, "EffectMaker.DataModelLogic.dll"));

                    // ユーザーが使いたそうなもの
                    param.ReferencedAssemblies.Add("System.dll");
                    param.ReferencedAssemblies.Add("System.Xml.dll");
                    param.ReferencedAssemblies.Add("System.Windows.Forms.dll");
                }

                // ファイルからコンパイル.
                var result = codeProvider.CompileAssemblyFromFile(param, filePaths);

                // エラーチェック.
                if (result.Errors.Count > 0)
                {
                    // エラーログ出力.
                    Logger.Log(LogLevels.Error, "Compile Error : ");

                    // エラーがあった場合、エラーの情報を表示する.
                    foreach (var e in result.Errors)
                    {
                        Logger.Log(LogLevels.Error, e.ToString());
                    }

                    return null;
                }

                // アセンブリがあることをチェック.
                if (result.CompiledAssembly == null)
                {
                    return null;
                }

                // コンパイル結果を返却.
                return
                    new AssemblyInfo
                    {
                        Assembly = result.CompiledAssembly,
                        LastWriteTime = dt
                    };
            }
        }

        /// <summary>
        /// ユーザーデータモデルをロードする.
        /// </summary>
        public void LoadUserDataModels()
        {
            // クリアする.
            DataModelSerializerExtraDataManager.ClearUserDataType();
            dataModelContainer.Clear();
            userDataInfoContainer.Clear();
            DataModelManager.ClearUserDataModel();

            // ファイル名リストを作成.
            List<string> fileList = new List<string>();
            foreach (var file in this.userDataInfo.SourceCodeFilePaths)
            {
                // "Exeのディレクトリ\\Addins\\ファイル名" で追加.
                fileList.Add(Path.Combine(ExeDirPath, "Addins", file));
            }

            // タイプ名とユーザー情報の連想マップを生成.
            Dictionary<string, UserDataModelInfo> dictionary =
                new Dictionary<string, UserDataModelInfo>();
            foreach (var info in this.UserPageInfoContainer)
            {
                // ファイルを開いてクラス名を取ってくる.
                string clazz = this.GetClassNameFromSourceCode(Path.Combine(ExeDirPath, "Addins", info.DataModelFileName));

                // クラス名が取れたのを確認して登録する.
                if (!string.IsNullOrEmpty(clazz))
                {
                    dictionary.Add(clazz, info);
                }
            }

            // ソースコードを読み込みコンパイルする.
            AssemblyInfo asmInfo = LoadDataModelFromFiles(fileList.ToArray());

            if (asmInfo != null)
            {
                // ユーザーアセンブリ情報を設定.
                this.userAssemblyInfo = asmInfo;

                // UserDataBaseが親クラス.
                Type baseClassType = typeof(UserDataBase);

                // 親クラスがUserDataBaseであるものを引っ張り出す.
                var types = from t in asmInfo.Assembly.GetExportedTypes()
                            where t.IsClass && t.IsSubclassOf(baseClassType)
                            select t;

                foreach (var type in types.ToList())
                {
                    if (type == null)
                    {
                        continue;
                    }

                    // ユーザーデータモデルを追加.
                    DataModelSerializerExtraDataManager.AddUserDataType(type);

                    // データモデルのインスタンスを生成.
                    UserDataBase dm = Activator.CreateInstance(type) as UserDataBase;

                    // タイプ名が登録されているはずなのでチェックする.
                    if (dictionary.ContainsKey(type.ToString()))
                    {
                        // ユーザーデータページ情報を引っ張り出す.
                        UserDataModelInfo info = dictionary[type.ToString()];

                        // タブ名が含まているかチェック.
                        if (TargetDataModelMap.ContainsKey(info.TargetTabName))
                        {
                            // ターゲットタイプを取得.
                            Type targetType = TargetDataModelMap[info.TargetTabName];

                            // 含まれていなかったら作成.
                            if (!dataModelContainer.ContainsKey(targetType))
                            {
                                dataModelContainer[targetType] = new List<UserDataBase>();
                            }

                            // データモデルのインスタンスを登録する.
                            dataModelContainer[targetType].Add(dm);
                        }

                        // ユーザーデータ情報を設定.
                        userDataInfoContainer[dm.GetType()] = info;
                    }

                    // データモデルマネージャにタイプを登録.
                    DataModelManager.AddUserDataModel(type);
                }
            }

            if (this.UserDataUpdated != null)
            {
                this.UserDataUpdated(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// エミッターユーザーデータモデルを作る.
        /// </summary>
        /// <param name="type">ターゲット名.</param>
        /// <returns>The created user data instances.</returns>
        public IEnumerable<UserDataBase> CreateUserDataModel(Type type)
        {
            if (type == null)
            {
                yield break;
            }

            List<UserDataBase> list;

            if (dataModelContainer.TryGetValue(type, out list))
            {
                foreach (UserDataBase data in list)
                {
                    yield return data.Clone() as UserDataBase;
                }
            }
        }

        /// <summary>
        /// クラスタイプからXAMLのパスを取得する.
        /// </summary>
        /// <param name="type">クラスタイプ.</param>
        /// <returns>成功した場合、有効なリソース情報を返す.</returns>
        public ResourceInfo GetXamlInfo(Type type)
        {
            UserDataModelInfo info;
            if (userDataInfoContainer.TryGetValue(type, out info))
            {
                return new ResourceInfo()
                {
                    Kind = ResourceInfo.ResourceKind.External,
                    ResourceName = Path.Combine(
                    ExeDirPath, "Addins", info.XamlFileName),
                };
            }

            return null;
        }

        /// <summary>
        /// インスタンスの作成.
        /// </summary>
        private void InitializeInternal()
        {
            string filePath = Path.Combine(ExeDirPath, "Addins\\Addins.xml");

            // ファイルからユーザーページ情報をロードする.
            this.userDataInfo = SerializationHelper.Load<UserDataInfo>(filePath);

            this.LoadUserDataModels();
        }

        /// <summary>
        /// ソースコードからクラス名を取得します.
        /// </summary>
        /// <param name="filePath">The file path of the source code.</param>
        /// <returns>The class name.</returns>
        private string GetClassNameFromSourceCode(string filePath)
        {
            // ファイルの存在チェック.
            if (File.Exists(filePath) == false)
            {
                return null;
            }

            // がばっと読み込む.
            string scriptText;
            {
                using (var sr = new StreamReader(filePath))
                {
                    scriptText = sr.ReadToEnd();
                }
            }

            // 空でないことを確認.
            if (string.IsNullOrEmpty(scriptText))
            {
                return null;
            }

            string className     = string.Empty;
            string namespaceName = string.Empty;
            {
                string classTag = "public class ";
                string namespaceTag = "namespace ";

                // 正規表現を使ってクラス名を見つける.
                System.Text.RegularExpressions.Regex regex = new System.Text.RegularExpressions.Regex(classTag + @"\w+");
                System.Text.RegularExpressions.Match match = regex.Match(scriptText);
                if (match.Success)
                {
                    // クラス名のみを取り出し.
                    className = match.Value.Remove(0, classTag.Length);
                }

                // 正規表現を使って名前空間を見つける.
                regex = new System.Text.RegularExpressions.Regex(namespaceTag + @"[A-Za-z0-9.]+");
                match = regex.Match(scriptText);
                if (match.Success)
                {
                    // 名前空間のみを取り出し.
                    namespaceName = match.Value.Remove(0, namespaceTag.Length);
                }
            }

            if ((string.IsNullOrEmpty(className) == false) &&
                (string.IsNullOrEmpty(namespaceName) == false))
            {
                // フルネームにして返す.
                return namespaceName + "." + className;
            }

            return null;
        }

        /// <summary>
        /// アセンブリ情報
        /// </summary>
        public class AssemblyInfo
        {
            /// <summary>
            /// アセンブリ
            /// </summary>
            public Assembly Assembly { get; set; }

            /// <summary>
            /// 最終書き込み時間
            /// </summary>
            public DateTime LastWriteTime { get; set; }
        }
    }
}
