﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Windows.Forms;

namespace NWCore.DataModel
{
    /// <summary>
    /// プロジェクト用ドキュメントです。
    /// </summary>
    public interface IProjectDocument : IDocument
    {
        #region Property

        /// <summary>
        /// TreeNode
        /// </summary>
        IList<TreeNode> Nodes { get; }

        /// <summary>
        /// 選択されている（アクティブな）ドキュメント
        /// </summary>
        IDocument ActiveDocument { get; set; }

        /// <summary>
        /// ツリーノードの更新を有効にします。（DEF:=true）
        /// </summary>
        bool EnableBuildTreeNodes { get; set; }

        /// <summary>
        /// 環境設定データ
        /// </summary>
        EnvConfigData EnvConfigData { get; }

        /// <summary>
        /// Emitter set count
        /// </summary>
        int NumEmitterSetDocuments { get; }

        /// <summary>
        /// エミッタセットを列挙
        /// </summary>
        IEnumerable<IEmitterSetDocument> EmitterSetDocuments { get; }

        /// <summary>
        /// 要求ドキュメントを列挙
        /// </summary>
        IEnumerable<IRequireDocument> RequireDocuments { get; }

        /// <summary>
        /// プレビュードキュメント
        /// </summary>
        IPreviewDocument PreviewDocument { get; }

        #endregion

        #region Event

        /// <summary>
        /// Event triggered when effect list of the project has changed.
        /// </summary>
        event EventHandler EmitterSetListChanged;

        /// <summary>
        /// Event triggered when effect name has been changed.
        /// </summary>
        event EventHandler EffectNameChanged;

        /// <summary>
        /// Event triggered when selected document has been changed.
        /// </summary>
        event DocumentChangedEventHandler ActiveDocumentChanged;

        #endregion

        #region Method

        /// <summary>
        /// UpdateTreeNode
        /// </summary>
        void UpdateTreeNode();

        /// <summary>
        /// Build tree nodes
        /// </summary>
        void BuildTreeNodes();

        /// <summary>
        /// Reset the project document to the default values.
        /// </summary>
        void Reset();

        /// <summary>
        /// EmitterSetDocumentを追加します
        /// </summary>
        /// <param name="index">挿入先のインデックス/ =-1..後尾に追加</param>
        /// <param name="document">追加するEmitterSetDocumentです</param>
        void AddEmitterSetDocument(int index, IEmitterSetDocument document);

        /// <summary>
        /// EmitterSetDocumentを削除します
        /// </summary>
        int RemoveEmitterSetDocument(IEmitterSetDocument document);

        /// <summary>
        /// EmitterSetDocumentを取得します。
        /// </summary>
        IEmitterSetDocument GetEmitterSetDocumentByIndex(int index);

        /// <summary>
        /// 既に登録されているかチェックします。
        /// </summary>
        /// <param name="findItem">検索するドキュメント</param>
        /// <returns></returns>
        bool Contains(IEmitterSetDocument findItem);

        /// <summary>
        /// 要求項目を追加します。
        /// </summary>
        void PushRequireItem(RequireDocData item);

        #endregion
    }

    #region イベント用

    /// <summary>
    /// ドキュメント切り替え用イベント引数
    /// </summary>
    public class DocumentChangedEventArgs : EventArgs
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public DocumentChangedEventArgs()
        {
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public DocumentChangedEventArgs(IDocument oldDoc, IDocument newDoc)
        {
            this.OldDocument = oldDoc;
            this.NewDocument = newDoc;
        }

        /// <summary>
        /// 以前のドキュメント
        /// </summary>
        public IDocument OldDocument { get; set; }

        /// <summary>
        /// 選択された新しいドキュメント
        /// </summary>
        public IDocument NewDocument { get; set; }
    }

    /// <summary>
    /// ドキュメント切り替え用のイベントハンドら
    /// </summary>
    public delegate void DocumentChangedEventHandler(
        object sender, DocumentChangedEventArgs e);

    #endregion
}

