﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using NintendoWare.ToolDevelopmentKit;

namespace NWCore.DataModel.Major_1.Minor_6.Build_0.Revision_0
{
    /// <summary>
    /// プロジェクトヘッダー
    /// </summary>
    public sealed class StripeData : BaseDataModel<StripeData>
    {
        #region Properties

        /// <summary>
        /// タイプ
        /// </summary>
        public StripeComputeType Type { get; set; }

        /// <summary>
        /// 高機能版タイプ
        /// </summary>
        public int HLType { get; set; }

        /// <summary>
        /// 高機能版 分割数
        /// </summary>
        public int SubDivs { get; set; }

        /// <summary>
        /// ヒストリを何個とるか？
        /// </summary>
        public int NumHistory { get; set; }

        /// <summary>
        /// 開始α
        /// </summary>
        public float StartAlpha { get; set; }

        /// <summary>
        /// 終了α
        /// </summary>
        public float EndAlpha { get; set; }

        /// <summary>
        /// ＵＶスクロール加算値
        /// </summary>
        public Vector2 UVScroll { get; set; }

        /// <summary>
        /// 履歴をたどる間隔
        /// </summary>
        public int HistoryStep { get; set; }

        /// <summary>
        /// 履歴補完パラメータ
        /// </summary>
        public float HistoryInterpolation { get; set; }

        /// <summary>
        /// 方向補間率
        /// </summary>
        public float DirInterpolation { get; set; }

        /// <summary>
        /// 分割数
        /// </summary>
        public int NumDivisions { get; set; }

        /// <summary>
        /// 履歴をエミッタに完全追従
        /// </summary>
        public bool FollowEmitter { get; set; }

        /// <summary>
        /// 接続タイプ
        /// </summary>
        public StripeConnectTypes ConnectType { get; set; }

        /// <summary>
        /// ストライプテクスチャ座標
        /// </summary>
        public StripeTexCoordTypes TexCoordType { get; set; }

        #endregion

        #region Construtors

        /// <summary>
        /// デフォルトコンストラクタ
        /// </summary>
        public StripeData()
        {
            this.UVScroll = new Vector2();    // ＵＶスクロール加算値
        }

        /// <summary>
        /// コピーコンストラクタ
        /// </summary>
        /// <param name="source">コピー元のベクトル</param>
        public StripeData(StripeData source)
            : this()
        {
            this.Set(source);
        }

        /// <summary>
        /// 初期化
        /// </summary>
        public override void Initialize()
        {
            this.Type = StripeComputeType.Billboard;
            this.HLType = 0;
            this.SubDivs = 1;
            this.NumHistory = 60;
            this.StartAlpha = 1.0f;
            this.EndAlpha = 1.0f;
            this.UVScroll.SetZero();
            this.HistoryStep = 0;
            this.HistoryInterpolation = 1.0f;
            this.DirInterpolation = 1.0f;
            this.FollowEmitter = false;
            this.NumDivisions = 0;
            this.ConnectType = StripeConnectTypes.Normal;
            this.TexCoordType = StripeTexCoordTypes.Full;
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// ベクトルを設定します。
        /// </summary>
        /// <param name="source">設定するベクトルです。</param>
        public override void Set(StripeData source)
        {
            Ensure.Argument.NotNull(source);

            this.UVScroll.Set(source.UVScroll);

            this.Type = source.Type;
            this.HLType = source.HLType;
            this.SubDivs = source.SubDivs;
            this.NumHistory = source.NumHistory;
            this.StartAlpha = source.StartAlpha;
            this.EndAlpha = source.EndAlpha;
            this.HistoryStep = source.HistoryStep;
            this.HistoryInterpolation = source.HistoryInterpolation;
            this.DirInterpolation = source.DirInterpolation;
            this.FollowEmitter = source.FollowEmitter;
            this.ConnectType = source.ConnectType;
            this.NumDivisions = source.NumDivisions;
            this.TexCoordType = source.TexCoordType;
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public override StripeData Clone()
        {
            return new StripeData(this);
        }

        #endregion
    }
}
