﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using NintendoWare.ToolDevelopmentKit;

namespace NWCore.DataModel.Major_0.Minor_9.Build_0.Revision_0
{
    /// <summary>
    /// プロジェクトヘッダー
    /// </summary>
    public sealed class StripeData : BaseDataModel<StripeData>
    {
        #region Properties

        /// <summary>
        /// タイプ
        /// </summary>
        public StripeType Type { get; set; }

        /// <summary>
        /// ヒストリを何個とるか？
        /// </summary>
        public int NumHistory { get; set; }

        /// <summary>
        /// 開始α
        /// </summary>
        public float StartAlpha { get; set; }

        /// <summary>
        /// 終了α
        /// </summary>
        public float EndAlpha { get; set; }

        /// <summary>
        /// ＵＶスクロール加算値
        /// </summary>
        public Vector2 UVScroll { get; set; }

        /// <summary>
        /// 履歴をたどる間隔
        /// </summary>
        public int HistoryStep { get; set; }

        /// <summary>
        /// 方向補間率
        /// </summary>
        public float DirInterpolation { get; set; }

        #endregion

        #region Construtors

        /// <summary>
        /// デフォルトコンストラクタ
        /// </summary>
        public StripeData()
        {
            this.UVScroll = new Vector2();    // ＵＶスクロール加算値
        }

        /// <summary>
        /// コピーコンストラクタ
        /// </summary>
        /// <param name="source">コピー元のベクトル</param>
        public StripeData(StripeData source)
            : this()
        {
            this.Set(source);
        }

        /// <summary>
        /// 初期化
        /// </summary>
        public override void Initialize()
        {
            this.Type = StripeType.Billboard;
            this.NumHistory = 60;
            this.StartAlpha = 1.0f;
            this.EndAlpha = 1.0f;
            this.UVScroll.SetZero();
            this.HistoryStep = 0;
            this.DirInterpolation = 1.0f;
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// ベクトルを設定します。
        /// </summary>
        /// <param name="source">設定するベクトルです。</param>
        public override void Set(StripeData source)
        {
            Ensure.Argument.NotNull(source);

            this.UVScroll.Set(source.UVScroll);

            this.Type = source.Type;
            this.NumHistory = source.NumHistory;
            this.StartAlpha = source.StartAlpha;
            this.EndAlpha = source.EndAlpha;
            this.HistoryStep = source.HistoryStep;
            this.DirInterpolation = source.DirInterpolation;
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public override StripeData Clone()
        {
            return new StripeData(this);
        }

        #endregion
    }
}
