﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.IO;
using System.Xml;
using System.Xml.Schema;
using System.Xml.Xsl;

namespace App.IO
{
    /// <summary>
    /// バージョンアップデータ。
    /// </summary>
    public sealed class XMLVersionUpdater
    {
        /// <summary>
        /// スキーマ解決
        /// </summary>
        public delegate Stream ResolveSchema(string version);

        /// <summary>
        /// スタイルシート解決
        /// </summary>
        public delegate Stream ResolveStylesheet(string version);

        private bool _updated;

        /// <summary>
        /// バージョンが上がったか
        /// </summary>
        public bool Updated { get { return _updated; } }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public XMLVersionUpdater(
            ResolveSchema schemaResolver, ResolveStylesheet stylesheetResolver)
        {
            Debug.Assert((schemaResolver != null) && (stylesheetResolver != null));
            _schemaResolver = schemaResolver;
            _stylesheetResolver = stylesheetResolver;
        }

        /// <summary>
        /// アップデート。
        /// </summary>
        public Stream Update(Stream source, string targetVersion)
        {
            Debug.Assert(targetVersion != null);
            Stream result = source;
            string version;
            _updated = false;

            while (true)
            {

                // バージョン取得
                version = GetVersion(result);
                if (version == null) { return null; }
                if (version == targetVersion) { break; }

                // バリデート
                Validate(result, version);

                // バージョンのインクリメント
                result = IncrementVersion(result, version);
                if (result == null) { return null; }
                _updated = true;
            }
            return result;
        }

        //---------------------------------------------------------------------
        // バージョン取得
        private string GetVersion(Stream source)
        {
            string result = null;

            using (XmlReader reader = XmlReader.Create(source))
            {
                while (reader.Read())
                {
                    if (reader.NodeType == XmlNodeType.Element)
                    {
                        // 最初に見つかったバージョン属性を返す
                        string version = reader.GetAttribute("version");
                        if (version != null)
                        {
                            result = version;
                            break;
                        }
                    }
                }
            }

            source.Seek(0, SeekOrigin.Begin);
            return result;
        }

        //---------------------------------------------------------------------
        // バリデート
        private void Validate(Stream source, string version)
        {
            XmlReaderSettings settings = new XmlReaderSettings();
            settings.ValidationType = ValidationType.Schema;
            XmlSchema schema = XmlSchema.Read(_schemaResolver(version), null);
            settings.Schemas.Add(schema);

            using (XmlReader reader = XmlReader.Create(source, settings))
            {
                while (reader.Read()) { }
            }

            source.Seek(0, SeekOrigin.Begin);
        }

        //---------------------------------------------------------------------
        // バージョンのインクリメント
        private Stream IncrementVersion(Stream source, string version)
        {
            XslCompiledTransform xslt = new XslCompiledTransform();
            try
            {
                xslt.Load(XmlReader.Create(_stylesheetResolver(version)));
            }
            catch (DirectoryNotFoundException exception)
            {
                throw new XmlException("Unknown version", exception);
            }
            catch (FileNotFoundException exception)
            {
                throw new XmlException("Unknown version", exception);
            }

            Stream result = new MemoryStream();
            xslt.Transform(XmlReader.Create(source), XmlWriter.Create(result));
            source.Close();

#if DEBUG
            result.Seek(0, SeekOrigin.Begin);
            TextReader reader = new StreamReader(result);
            while (true)
            {
                string line = reader.ReadLine();
                if (line == null) { break; }
                Debug.WriteLine(line);
            }
#endif

            result.Seek(0, SeekOrigin.Begin);
            return result;
        }

        private ResolveSchema _schemaResolver;
        private ResolveStylesheet _stylesheetResolver;
    }
}
