﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Windows.Forms;
using System.Threading;
using System.IO;
using App.PropertyEdit.BoneLinkageDataTable;
using App.Command;
using App.Controls;
using App.Data;
using NintendoWare.ToolDevelopmentKit;
using NintendoWare.ToolDevelopmentKit.Collections;
using NintendoWare.ToolDevelopmentKit.Xml;
using NW4F.DataModel;
using NW4F.Viewer;

namespace App.IO
{
    /// <summary>
    /// プロジェクトのオープナ
    /// </summary>
    public class ProjectDocumentOpener : DocumentOpener
    {
        private const int TIMEOUT_MS = 5000;

        #region Inner Class - DataLoader
        /// <summary>
        /// Loader class to be launched by the thread
        /// </summary>
        private class DataLoader
        {
            /// <summary>
            /// Constructor of DataLoader class
            /// </summary>
            /// <param name="progressDialog">GUI to display loading progress</param>
            /// <param name="rootDir">root directory of the file</param>
            /// <param name="project">project file</param>
            /// <param name="changed">change event argument</param>
            public DataLoader(ProgressDialog progressDialog,
                              string rootDir,
                              IProjectDocument project,
                              DocumentComponentChangedEventArgs changed)

            {
                m_progressDialog = progressDialog;
                m_rootDir = rootDir;
                m_project = project;
                m_argChanged = changed;
                CurrentState = LoadState.Loading;
            }

            #region Public Methods

            /// <summary>
            /// Load data function is supposed to be called by a thread
            /// </summary>
            public void LoadData()
            {
                if (m_project == null)
                    return;
#if false ////TODO:未対応

                // try to load all the effects in the project
                try
                {
                    while (!m_progressDialog.IsHandleCreated);

                    foreach (ProjectNodeParameterXml item in m_project.XmlDocument.SerializeNodes)
                    {
                        // pause and wait if the state is not loading
                        while(CurrentState != LoadState.Loading)
                        {
                            if (CurrentState == LoadState.Canceled)
                            {
                                OnLoadFinished();
                                return;
                            }

                            // don't waste process power
                            Thread.Sleep(10);
                        }

                        CptlDocumentOpener opener = new CptlDocumentOpener();
                        string path = Path.Combine(m_rootDir, item.RelativePath);
                        path = Path.GetFullPath(path);

                        m_progressDialog.BeginTask(Path.GetFileName(path));
                        opener.Open(path, m_argChanged);
                        m_progressDialog.EndTask();
                    }

                    foreach (IDocument document in m_argChanged.OpenedDocuments)
                    {
                        ProjectNodeParameterXml foundItem =
                            m_project.XmlDocument.SerializeNodes.Find
                            (
                                delegate(ProjectNodeParameterXml item)
                                {
                                    return document.Name == item.Name;
                                }
                            );
                        if (foundItem != null)
                        {
                            EffectDocument emitterDoc = document as EffectDocument;
                            if (emitterDoc != null)
                            {
                                emitterDoc.AfterDeserialize(foundItem);
                            }
                        }
                    }

                    Thread.Sleep(500);
                    OnLoadFinished();

                    // Uncomment this to restore delayed loading of effect document
#if false
                                                                                                                                                                                                    // save the children effects into dictionary for display
                    foreach (ProjectNodeParameterXml item in newProject.XmlDocument.SerializeNodes)
                    {
                        string path = Path.Combine(rootDir, item.RelativePath);
                        path = Path.GetFullPath(path);

                        IProjectDocument.EffectState effectState =
                            new IProjectDocument.EffectState(item.RelativePath, false);
                        // 読み込み前のダミーノードを追加します。
                        newProject.AddEffectState(path, effectState);
                    }
#endif
                }
                catch (Exception ex)
                {
                    UIMessageBox.Error(ex.Message);

                    OnLoadFinished();
                    return;
                }
#endif
            }

            #endregion

            #region Private Methods

            /// <summary>
            /// /// Handles cancel loading
            /// </summary>
            /// <param name="sender">send of the event</param>
            /// <param name="e">parameter of the event</param>
            private void HandleCancelLoading(object sender, EventArgs e)
            {
                CurrentState = LoadState.Pending;
                m_progressDialog.TimerStopThreadSafe();
            }

            /// <summary>
            /// /// Called when loading is finished
            /// </summary>
            private void OnLoadFinished()
            {
                if (!m_progressDialog.IsDisposed)
                    m_progressDialog.DisposeThreadSafe();
            }

            #endregion

            #region Properties

            /// <summary>
            /// Accessor for ProgessDialog
            /// </summary>
            public ProgressDialog ProgressDialog
            {
                get {return m_progressDialog;}
            }

            /// <summary>
            /// Accessor for CancelLoading
            /// </summary>
            public LoadState CurrentState
            {
                get;
                set;
            }

            public enum LoadState
            {
                Loading,
                Pending,
                Canceled
            }

            #endregion

            #region Memeber Variables
            /// <summary>
            /// private variables
            /// </summary>
            private ProgressDialog m_progressDialog;
            private string m_rootDir;
            private IProjectDocument m_project;
            private DocumentComponentChangedEventArgs m_argChanged;
            #endregion
        }
        #endregion

        #region public methods
        /// <summary>
        /// プロジェクトを開く。
        /// </summary>
        public IProjectDocument Open(string filePath)
        {
            DocumentComponentChangedEventArgs changed = new DocumentComponentChangedEventArgs();
            IProjectDocument result = Open(filePath, changed);
            Document.NotifyComponentChanged(this, changed);
            return result;
        }

        /// <summary>
        /// プロジェクトを開く。
        /// </summary>
        public IProjectDocument Open(string filePath, DocumentComponentChangedEventArgs changed)
        {
            if (!CheckFileExists(filePath))
            {
                return null;
            }
            return OpenDocument(filePath, changed) as IProjectDocument;
        }

        /// <summary>
        /// ドキュメントを開く（TemplateMethod）。
        /// </summary>
        /// <param name="filePath">file path of the project file</param>
        /// <param name="changed">document changed event</param>
        /// <returns>new project created</returns>
        protected override IDocument OpenDocumentImplement(string filePath,
                                                          DocumentComponentChangedEventArgs changed)
        {
            // 新しいプロジェクトを生成
            FileNewProjectArgs args = new FileNewProjectArgs(false);
            DocumentIO.Command_FileNewProject(new MenuCommandArgs(false, null, args));
            IProjectDocument newProject = ProjectManager.ActiveProject;

#if false ////TODO:未対応
            try
            {
                using (MCSDisableBlock disableBlock = new MCSDisableBlock())
                using (Document.PropertyChangedSuppressBlock block =
                    new Document.PropertyChangedSuppressBlock())
                // 時間表示
                using (Utility.ScopedStopwatch stopwatch =
                    new Utility.ScopedStopwatch(Path.GetFileName(filePath) + " -- <Project> --"))
                {
            #region Project Loading

                    // プロジェクトの読み込みです。
                    newProject.XmlDocument =
                        XmlUtility.DeserializeFromFile<IProjectDocumentXml>(filePath);

                    // デシリアライズ
                    newProject.AfterDeserialize(newProject.XmlDocument);

                    // もしも同じ名前のプロジェクトがすでに存在していたら名前を変更する
                    {
                        string name = GetProjectName(newProject, newProject.Name);
                        if (newProject.Name != name)
                        {
                            newProject.Name = name;
                        }
                    }

                    // プロジェクトファイルのフォルダをルートにして相対パスで読み込みます。
                    string rootDir = Path.GetDirectoryName(filePath);
                    rootDir = rootDir.Replace("\\", "/");

                    if (rootDir.EndsWith("/") == true)
                        rootDir = rootDir.Remove(rootDir.Length - 1, 1);

                    // プロジェクトの基準フォルダを設定
                    newProject.SetFileLocation(rootDir);
                    #endregion

            #region Effect Loading
                    int iNumEffect = newProject.XmlDocument.SerializeNodes.Count;

                    m_progressDialog =
                        new ProgressDialog(iNumEffect,
                                           res.Strings.PROJECT_FILE_LOADING_PROGRESS_TITLE);
                    m_progressDialog.CancelTask += new EventHandler(this.HandleCancelLoading);

                    m_dataLoader = new DataLoader(m_progressDialog,
                                                  rootDir,
                                                  newProject,
                                                  changed);

                    // create and start the thread to load effect documents
                    m_dataLoaderThread = new Thread(new ThreadStart(m_dataLoader.LoadData));
                    m_dataLoaderThread.Name = "LoadingThread";
                    m_dataLoaderThread.Start();
                    while (!m_dataLoaderThread.IsAlive);

                    m_progressDialog.ShowDialogThreadSafe(TheApp.MainFrame);

                    m_dataLoaderThread.Join(TIMEOUT_MS);
                    if (m_dataLoaderThread.IsAlive)
                    {
                        m_dataLoaderThread.Abort();
                    }

                    if (m_dataLoader.CurrentState == DataLoader.LoadState.Canceled)
                    {
                        ForceDocumentCloser docCloser = new ForceDocumentCloser();
                        docCloser.Close(newProject);
                    }

                    if (m_progressDialog != null)
                        m_progressDialog.Dispose();

                    #endregion

            #region Preview Loading
                    string previewDir = Path.GetDirectoryName(filePath) +
                                        "\\" +
                                        DocumentConstants.PreviewDirectory;

                    DirectoryInfo previewDirInfo = new DirectoryInfo(previewDir);

                    if(previewDirInfo.Exists)
                    {
                        FileInfo[] previewFiles = previewDirInfo.GetFiles();

                        foreach (FileInfo fileInfo in previewFiles)
                        {
                            if (DocumentConstants.IsPreviewPath(fileInfo.FullName))
                            {
                                IPreviewDocument previewDoc =
                                    new PreviewDocument(fileInfo.Name,
                                                        fileInfo.FullName,
                                                        new BoneLinkageDataSet(),
                                                        newProject);

                                previewDoc.LoadDocument(fileInfo.FullName);
                                newProject.AddPreviewDocument(previewDoc);
                            }
                        }
                    }
                    #endregion

            #region  Update All
                    // プロジェクト全体を更新します。
                    foreach (IProjectDocument proj in ProjectManager.Projects)
                    {
                        if (proj.WindowForm != null)
                        {
                            App.PropertyEdit.ProjectWindow form =
                                proj.WindowForm as App.PropertyEdit.ProjectWindow;
                            if (form != null)
                                form.UpdateProjectWindowForm(this, null);
                        }
                    }
                    #endregion
                }

            }
            catch (Exception ex)
            {
                UIMessageBox.Error(ex.Message);
                UIMessageBox.Error(String.Format(res.Strings.IO_LOADFILE_FAILED_MSG, filePath));
            }
#endif
            return newProject;
        }
        #endregion

        #region Private Methods
        /// <summary>
        /// プロジェクト名を取得します。（同じプロジェクト名があったら、名前を変更して取得します。）
        /// </summary>
        /// <param name="newProject">現在のプロジェクト</param>
        /// <param name="projectName">プロジェクト名</param>
        /// <returns>新しいプロジェクト名</returns>
        private string GetProjectName(IProjectDocument newProject, string projectName)
        {
            int count = 0;
            string name = projectName;
            ReadOnlyList<IProjectDocument> list = ProjectManager.Projects;
            while (IsExistSameProjectName(list, newProject, name) == true)
            {
                name = String.Format("{0}_{1}", projectName, count.ToString("D02"));
                count++;
            }
            return name;
        }

        /// <summary>
        /// 同じ名前のプロジェクト名が存在するかチェックします。
        /// </summary>
        /// <param name="list">プロジェクトリスト</param>
        /// <param name="newProject">現在のプロジェクト</param>
        /// <param name="projectName">チェックするプロジェクト名</param>
        /// <returns>=true..存在する /=false..存在しない</returns>
        private bool IsExistSameProjectName(
            ReadOnlyList<IProjectDocument> list,
            IProjectDocument newProject, string projectName)
        {
            foreach (IProjectDocument item in list)
            {
                if (item != newProject && item.Name.Equals(
                    projectName, StringComparison.CurrentCultureIgnoreCase) == true)
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// /// Handles cancel loading
        /// </summary>
        /// <param name="sender">send of the event</param>
        /// <param name="e">parameter of the event</param>
        private void HandleCancelLoading(object sender, EventArgs e)
        {
            if (m_dataLoaderThread == null)
                return;

            if (m_dataLoader == null)
                return;

            m_dataLoader.CurrentState = DataLoader.LoadState.Pending;
            m_progressDialog.TimerStopThreadSafe();

            if (m_dataLoaderThread.IsAlive)
            {
                if (m_progressDialog.ConfirmCancel())
                    m_dataLoader.CurrentState = DataLoader.LoadState.Canceled;
                else
                {
                    m_dataLoader.CurrentState = DataLoader.LoadState.Loading;
                    m_progressDialog.TimerStartThreadSafe();
                }
            }
        }
        #endregion

        #region Member Variables
        /// <summary>
        /// Thread to load project file
        /// </summary>
        private Thread m_dataLoaderThread = null;

        /// <summary>
        /// Data loader
        /// </summary>
        private DataLoader m_dataLoader = null;

        /// <summary>
        /// Progress bar dialog
        /// </summary>
        private ProgressDialog m_progressDialog = null;

        #endregion
    }
}
